package Bookings::Security::CVSS::v2;

use 5.006;
use strict;
use warnings;

use Math::Round qw(nearest_ceil);

use Carp qw( croak );

=head1 NAME

Bookings::Security::CVSS::v2 - Calculate CVSSv2 values (Common Vulnerability Scoring System)

=head1 DESCRIPTION

CVSSv2 allows you to calculate all three types of score described
under the CVSS system: Base, Temporal and Environmental.

You can modify any parameter via its setter and recalculate
at any time.

The temporal score depends on the base score, and the environmental
score depends on the temporal score. Therefore you must remember
to supply all necessary parameters.

SetVector allows you to parse a CVSSv2 vector as described at:
http://nvd.nist.gov/cvss.cfm?vectorinfo

Vector() will return the CVSS vector as a string.

=head1 POSSIBLE VALUES

For meaning of these values see the official CVSS FAQ
at https://www.first.org/cvss/faq/#c7

=head1 SEE ALSO

This module is based on the formulas supplied at:
http://www.first.org/cvss/

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';

=head1 SYNOPSIS

  use Bookings::Security::CVSS::v2;

  my $CVSS = new Bookings::Security::CVSS::v2;

  # Set metrics individually:
  $CVSS->AccessVector('Local');
  $CVSS->AccessComplexity('High');
  $CVSS->Authentication('None');
  $CVSS->ConfidentialityImpact('Complete');
  $CVSS->IntegrityImpact('Complete');
  $CVSS->AvailabilityImpact('Complete');

  my $BaseScore = $CVSS->BaseScore();

  $CVSS->Exploitability('Proof-Of-Concept');
  $CVSS->RemediationLevel('Official-Fix');
  $CVSS->ReportConfidence('Confirmed');

  my $TemporalScore = $CVSS->TemporalScore()

  $CVSS->CollateralDamagePotential('None');
  $CVSS->TargetDistribution('None');
  $CVSS->ConfidentialityRequirement('High');
  $CVSS->IntegrityRequirement('Medium');
  $CVSS->AvailabilityRequirement('Low');

  my $EnvironmentalScore = $CVSS->EnvironmentalScore();

  # Set lots of metrics simultaneously in the constructor:
  my $CVSS = new Bookings::Security::CVSS::v2({
        AccessVector => 'Local',
        AccessComplexity => 'High',
        Authentication => 'None',
        ConfidentialityImpact => 'Complete',
        IntegrityImpact => 'Complete',
        AvailabilityImpact => 'Complete',
  });

  my $BaseScore = $CVSS->BaseScore();

  # Update lots of metrics simultaneously with a hashref:
  $CVSS->UpdateFromHash({
        AccessVector => 'Network',
        AccessComplexity => 'Low'
  });


  my $NewBaseScore = $CVSS->BaseScore();

  # Set the entire vector at once:
  $CVSS->SetVector('(AV:L/AC:H/Au:S/C:N/I:P/A:C)');
  my $BaseScore = $CVSS->BaseScore();

  # Get metrics individually:
  $CVSS->AccessVector; # 'local'

  # Get the whole vector:
  my $Vector = $CVSS->Vector();

=cut

our %BASE_PARAMS = (
    AccessVector          => { Params => {'local' => 0.395, 'adjacent-network' => 0.646, 'network' => 1},
        P2V    => {'local' => 'L',   'adjacent-network' => 'A',   'network' => 'N'},
        Abbrev => 'AV',
    },

    AccessComplexity      => { Params => {'low' => 0.71, 'medium' => 0.61, 'high' => 0.35},
        P2V    => {'low' => 'L',  'medium' => 'M',  'high' => 'H'},
        Abbrev => 'AC',
    },

    Authentication        => { Params => {'multiple' => 0.45, 'single' => 0.56, 'none' => 0.704},
        P2V    => {'multiple' => 'M',  'single' => 'S',  'none' => 'N'},
        Abbrev => 'Au',
    },

    ConfidentialityImpact => { Params => {'none' => 0,   'partial' => 0.275, 'complete' => 0.660},
        P2V    => {'none' => 'N', 'partial' => 'P',   'complete' => 'C'},
        Abbrev => 'C',
    },

    IntegrityImpact       => { Params => {'none' => 0,   'partial' => 0.275, 'complete' => 0.660},
        P2V    => {'none' => 'N', 'partial' => 'P',   'complete' => 'C'},
        Abbrev => 'I',
    },

    AvailabilityImpact    => { Params => {'none' => 0,   'partial' => 0.275, 'complete' => 0.660},
        P2V    => {'none' => 'N', 'partial' => 'P',   'complete' => 'C'},
        Abbrev => 'A',
    },
);
_CreateV2P(\%BASE_PARAMS);

our %TEMPORAL_PARAMS = (
    Exploitability   => { Params => {'not-defined' => 1,    'unproven' => 0.85, 'proof-of-concept' => 0.9, 'functional' => 0.95, 'high' => 1},
        P2V    => {'not-defined' => 'ND', 'unproven' => 'U',  'proof-of-concept' => 'POC', 'functional' => 'F',  'high' => 'H'},
        Abbrev => 'E',
    },

    RemediationLevel => { Params => {'not-defined' => 1,    'official-fix' => 0.87, 'temporary-fix' => 0.9, 'workaround' => 0.95, 'unavailable' => 1},
        P2V    => {'not-defined' => 'ND', 'official-fix' => 'OF',  'temporary-fix' => 'TF', 'workaround' => 'W',  'unavailable' => 'U'},
        Abbrev => 'RL',
    },

    ReportConfidence => { Params => {'not-defined' => 1,    'unconfirmed' => 0.9,  'uncorroborated' => 0.95, 'confirmed' => 1},
        P2V    => {'not-defined' => 'ND', 'unconfirmed' => 'UC', 'uncorroborated' => 'UR', 'confirmed' => 'C'},
        Abbrev => 'RC',
    },
);
_CreateV2P(\%TEMPORAL_PARAMS);

our %ENVIRONMENTAL_PARAMS = (
    CollateralDamagePotential  => { Params => {'not-defined' => 0,    'none' => 0,   'low' => 0.1, 'low-medium' => 0.3,  'medium-high' => 0.4,  'high' => 0.5},
        P2V    => {'not-defined' => 'ND', 'none' => 'N', 'low' => 'L', 'low-medium' => 'LM', 'medium-high' => 'MH', 'high' => 'H'},
        Abbrev => 'CDP',
    },

    TargetDistribution         => { Params => {'not-defined' => 1,    'none' => 0,   'low' => 0.25, 'medium' => 0.75, 'high' => 1},
        P2V    => {'not-defined' => 'ND', 'none' => 'N', 'low' => 'L',  'medium' => 'M',  'high' => 'H'},
        Abbrev => 'TD',
    },

    ConfidentialityRequirement => { Params => {'not-defined' => 1,    'low' => 0.5, 'medium' => 1,   'high' => 1.51},
        P2V    => {'not-defined' => 'ND', 'low' => 'L', 'medium' => 'M', 'high' => 'H'},
        Abbrev => 'CR',
    },

    IntegrityRequirement       => { Params => {'not-defined' => 1,    'low' => 0.5, 'medium' => 1,   'high' => 1.51},
        P2V    => {'not-defined' => 'ND', 'low' => 'L', 'medium' => 'M', 'high' => 'H'},
        Abbrev => 'IR',
    },

    AvailabilityRequirement    => { Params => {'not-defined' => 1,    'low' => 0.5, 'medium' => 1,   'high' => 1.51},
        P2V    => {'not-defined' => 'ND', 'low' => 'L', 'medium' => 'M', 'high' => 'H'},
        Abbrev => 'AR',
    },
);
_CreateV2P(\%ENVIRONMENTAL_PARAMS);

our %ALL_PARAMS = (%BASE_PARAMS, %TEMPORAL_PARAMS, %ENVIRONMENTAL_PARAMS);

# Create getters and setters for all parameters
foreach my $key (keys %ALL_PARAMS) {
    no strict 'refs';

    # Long-name getter.  For example, $self->TargetDistribution might return 'low'
    *{"Bookings::Security::CVSS::v2::$key"} = sub {
        my $self = shift;
        return ($self->{$key} // 'not-defined');
    };

    # And create getters for the 'short' values used in the vector.
    # For example, $CVSS->TD might return 'L'.
    my $abbrev = $ALL_PARAMS{$key}->{'Abbrev'};
    *{"Bookings::Security::CVSS::v2::$abbrev"} = sub {
        my $self = shift;
        if ($self->{$key}) {
            return ($ALL_PARAMS{$key}->{P2V}->{ $self->{$key} });
        } else {
            return undef;
        }
    };

    # setter
    *{"Bookings::Security::CVSS::v2::Set$key"} = sub {
        my $self = shift;
        $self->_ValidateParam($key, @_);
    };
}


# Create the Vector-to-param hash from the P2V hash
sub _CreateV2P {
    my $params = shift;

    foreach my $param (keys %$params) {
        $params->{$param}->{V2P} = { map { $params->{$param}->{P2V}->{$_} => $_ } keys %{$params->{$param}->{P2V}} };
    }
}

sub _ValidateParam {
    my $self  = shift;
    my $param = shift;
    my $value = shift;

    # If vector value - convert to full value
    if (exists($ALL_PARAMS{$param}->{V2P}->{$value})) {
        $value = $ALL_PARAMS{$param}->{V2P}->{$value};
    } else {
        $value = lc($value);
    }

    if (!grep(/^$value$/i, keys %{$ALL_PARAMS{$param}->{Params}})) {
        croak("Invalid value '$value' for $param");
    }

    $self->{$param} = $value;
}

sub UpdateFromHash {
    my ($self, $params) = @_;

    if (ref($params) ne 'HASH') {
        croak 'Parameter must be a hash reference';
    }

    foreach my $param (keys %$params) {
        if (!exists($ALL_PARAMS{$param})) {
            croak "$param is not a valid parameter";
        }

        my $setter_name = "Set$param";
        $self->$setter_name($params->{$param});
    }
}

=head1 EXPORT

  new
  Vector
  SetVector
  OverallScore
  BaseScore
  TemporalScore
  EnvironmentalScore

=cut

=head1 SUBROUTINES/METHODS

=head2 new

=cut

sub new {
    my $class  = shift;
    my $params = shift;

    my $self   = bless({}, $class);

    if (defined($params)) {
        $self->UpdateFromHash($params);
    }
    return $self;
}

=head2 SetVector

=cut

sub SetVector {

    my ($self, $vector) = @_;

    if (defined($vector)) {

        if ($vector !~ m/
            # Base vector
                ^\(AV:([LAN])\/AC:([HML])\/Au:([MSN])\/C:([NPC])\/I:([NPC])\/A:([NPC])

                    # Temporal vector
                    (\/E:(ND|U|POC|F|H)\/RL:(ND|OF|TF|W|U)\/RC:(ND|UC|UR|C))?

                    # Environmental vector
                    (\/CDP:(ND|N|L|LM|MH|H)\/TD:(ND|N|L|M|H)\/CR:(ND|L|M|H)\/IR:(ND|L|M|H)\/AR:(ND|L|M|H))?\)

            /x) {
            croak('Invalid CVSS vector');
        }

        my %values = (
            AccessVector          => $1,
            AccessComplexity      => $2,
            Authentication        => $3,
            ConfidentialityImpact => $4,
            IntegrityImpact       => $5,
            AvailabilityImpact    => $6,
        );

        # optional temporal portion
        if (defined($7)) {
            %values = (
                %values,
                Exploitability   => $8,
                RemediationLevel => $9,
                ReportConfidence => $10
            );
        }

        # optional environmental portion.
        if (defined($11)) {
            %values = (
                %values,
                CollateralDamagePotential   => $12,
                TargetDistribution          => $13,
                ConfidentialityRequirement  => $14,
                IntegrityRequirement        => $15,
                AvailabilityRequirement     => $16,
            );
        }

        $self->UpdateFromHash(\%values);

    }
    else {
        croak('Must call SetVector() with a $vector!');
    }
}

=head2 Vector

=cut

sub Vector {
    my ($self) = @_;

    # Check all parameters exist
    foreach my $param (keys %BASE_PARAMS) {
        if (!defined($self->{$param})) {
            croak("You must set '$param' to output the CVSS vector");
        }
    }

    my $vectorValue = sub {
        return $ALL_PARAMS{$_[0]}->{P2V}->{$self->{$_[0]}};
    };

    my $vector = sprintf('AV:%s/AC:%s/Au:%s/C:%s/I:%s/A:%s',
        &$vectorValue('AccessVector'),
        &$vectorValue('AccessComplexity'),
        &$vectorValue('Authentication'),
        &$vectorValue('ConfidentialityImpact'),
        &$vectorValue('IntegrityImpact'),
        &$vectorValue('AvailabilityImpact'));

    my $temporal = 1;
    foreach my $param (keys %TEMPORAL_PARAMS) {
        if (!defined($self->{$param})) {
            $temporal = 0;
            last;
        }
    }

    if ($temporal) {
        $vector .= sprintf('/E:%s/RL:%s/RC:%s',
            &$vectorValue('Exploitability'),
            &$vectorValue('RemediationLevel'),
            &$vectorValue('ReportConfidence'));
    }

    my $environmental = 1;
    foreach my $param (keys %ENVIRONMENTAL_PARAMS) {
        if (!defined($self->{$param})) {
            $environmental = 0;
            last;
        }
    }

    if ($environmental) {
        $vector .= sprintf('/CDP:%s/TD:%s/CR:%s/IR:%s/AR:%s',
            &$vectorValue('CollateralDamagePotential'),
            &$vectorValue('TargetDistribution'),
            &$vectorValue('ConfidentialityRequirement'),
            &$vectorValue('IntegrityRequirement'),
            &$vectorValue('AvailabilityRequirement'));
    }

    return "($vector)";
}

=head2 BaseScore

=cut

sub BaseScore {
    my $self = shift;

    # Check all parameters exist
    foreach my $param (keys %BASE_PARAMS) {
        if (!defined($self->{$param})) {
            return "Not Defined";
        }
    }

    my $vectorValue = sub {
        return $ALL_PARAMS{$_[0]}->{Params}->{$self->{$_[0]}};
    };

    # Calculate the impact portion of the score taking into account the weighting bias
    my $impact = 10.41 * (1 - (1 - &$vectorValue('ConfidentialityImpact')) * (1 - &$vectorValue('IntegrityImpact')) * (1 - &$vectorValue('AvailabilityImpact')));

    my $exploitability = 20 * &$vectorValue('AccessComplexity') * &$vectorValue('Authentication') * &$vectorValue('AccessVector');

    my $f = 0;
    $f = 1.176 if ($impact != 0);

    my $score = (0.6 * $impact + 0.4 * $exploitability - 1.5) * ( $f );

    # Round to one sig fig
    return nearest_ceil(0.1, $score);
}

=head2 TemporalScore

=cut

sub TemporalScore {
    my $self = shift;

    # Check all parameters exist
    foreach my $param (keys %TEMPORAL_PARAMS) {
        if (!defined($self->{$param})) {
            return "Not Defined";
        }
    }

    my $vectorValue = sub {
        return $ALL_PARAMS{$_[0]}->{Params}->{$self->{$_[0]}};
    };

    my $score = $self->BaseScore() * &$vectorValue('Exploitability') * &$vectorValue('RemediationLevel') * &$vectorValue('ReportConfidence');

    # Round to one sig fig
    return nearest_ceil(0.1, $score);
}

=head2 EnvironmentalScore

=cut

sub EnvironmentalScore {
    my $self = shift;

    # Check all parameters exist
    foreach my $param (keys %ENVIRONMENTAL_PARAMS) {
        if (!defined($self->{$param})) {
            return "Not Defined";
        }
    }

    my $vectorValue = sub {
        return $ALL_PARAMS{$_[0]}->{Params}->{$self->{$_[0]}};
    };

    my $adjusted_impact = 10.41 * (1 - (1 - &$vectorValue('ConfidentialityImpact') * &$vectorValue('ConfidentialityRequirement')) *
        (1 - &$vectorValue('IntegrityImpact') * &$vectorValue('IntegrityRequirement')) *
        (1 - &$vectorValue('AvailabilityImpact') * &$vectorValue('AvailabilityRequirement')));
    $adjusted_impact = 10 if $adjusted_impact > 10;

    my $exploitability = 20 * &$vectorValue('AccessComplexity') * &$vectorValue('Authentication') * &$vectorValue('AccessVector');

    my $f = 0;
    $f = 1.176 if ($adjusted_impact != 0);

    my $adjusted_base = (0.6 * $adjusted_impact + 0.4 * $exploitability - 1.5) * ( $f );
    $adjusted_base = nearest_ceil(0.1, $adjusted_base);

    my $adjusted_temporal = $adjusted_base * &$vectorValue('Exploitability') * &$vectorValue('RemediationLevel') * &$vectorValue('ReportConfidence');
    $adjusted_temporal = nearest_ceil(0.1, $adjusted_temporal);

    my $score = ($adjusted_temporal + ((10 - $adjusted_temporal) * &$vectorValue('CollateralDamagePotential'))) *  &$vectorValue('TargetDistribution');

    # Round to one sig fig
    return nearest_ceil(0.1, $score);
}

=head2 OverallScore

=cut

sub OverallScore {
    my $self = shift;

    my $overall_score = "Not Defined";
    my $base_score = $self->BaseScore;

    if ($base_score eq "Not Defined") {
        return "Not Defined";
    }

    my $environmental_score = $self->EnvironmentalScore;
    if ($environmental_score ne "Not Defined") {
        return $environmental_score;
    }

    my $temporal_score = $self->TemporalScore;
    if ($temporal_score ne "Not Defined") {
        return $temporal_score;
    }
    return $base_score;
}

1;

=head1 AUTHORS

Thomas Briggs, C<< <thomas.briggs at booking.com> >>
Daniel Ostermeier, C<< <daniel.ostermeier at booking.com> >>
Farshid Zaker, C<< <farshid.zaker at booking.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-bookings-security-cvss at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Bookings-Security-CVSS>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Bookings::Security::CVSS::v2


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Bookings-Security-CVSS>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Bookings-Security-CVSS>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Bookings-Security-CVSS>

=item * Search CPAN

L<http://search.cpan.org/dist/Bookings-Security-CVSS/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2017 booking.com

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Bookings::Security::CVSS::v2
__END__