package Auth::Kokolores::Request;

use Moose;

# ABSTRACT: saslauthd protocol request object
our $VERSION = '1.00'; # VERSION


has 'username' => ( is => 'rw', isa => 'Str', required => 1 );
has 'password' => ( is => 'rw', isa => 'Str', required => 1 );
has 'service' => ( is => 'rw', isa => 'Str', required => 1 );
has 'realm' => ( is => 'rw', isa => 'Str', required => 1 );


has 'server' => (
  is => 'ro',
  isa => 'Net::Server',
  required => 1,
  handles => {
    log => 'log',
  },
);


has 'userinfo' => (
  is => 'ro', isa => 'HashRef', lazy => 1,
  default => sub { {} },
  traits => [ 'Hash' ],
  handles => {
    get_info => 'get',
    set_info => 'set',
  },
);


sub _read_sasl_string {
  my ( $conn ) = @_;
  my $buf;
  $conn->read($buf, 2);
  my $size = unpack('n', $buf);
  $conn->read($buf, $size);
  return unpack("A$size", $buf);
}


sub new_from_conn {
  my ( $class, $conn, $server ) = @_;
  my %opts = (
    server => $server,
  );
  foreach my $field ('username', 'password', 'service', 'realm') {
    $opts{$field} = _read_sasl_string( $conn );
  }
  return $class->new( %opts );
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Auth::Kokolores::Request - saslauthd protocol request object

=head1 VERSION

version 1.00

=head1 DESCRIPTION

This class holds all information associated with an authentication request.
It is passed to all authentication plugins.

=head1 ATTRIBUTES

=head2 username

The username passed within the SASL request.

=head2 password

The password passed within the SASL request.

=head2 service

The service name passed within the SASL request.

=head2 realm

The sasl auth realm passed within the SASL request.

=head2 server

A reference to the L<Auth::Kokolores> server object.

=head2 userinfo

A hashref holding additional information to be passed between plugins.

Use get_info and set_info methods to access fields.

=head1 METHODS

=head2 get_info( $key )

Retrieve field $key from userinfo.

=head2 set_info( $key, $value )

Set field $key to $value in userinfo.

=head2 Auth::Kokolores::Request->new_from_conn( $conn, $server )

Constructor for Auth::Kokolores::Request for creating
an object by reading the parameters from a handle
passed in $conn.

=head1 AUTHOR

Markus Benning <ich@markusbenning.de>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2016 by Markus Benning <ich@markusbenning.de>.

This is free software, licensed under:

  The GNU General Public License, Version 2, June 1991

=cut
