(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Modula-2+ version created by John Ellis in ancient times.   *)
(* Simplified and ported to Modula-3 by J.Stolfi on May 1990.  *)
(* Last modified on Wed Feb 12 12:37:59 PST 1992 by muller     *)
(*      modified on Tue Oct 15 23:10:37 PDT 1991 by stolfi     *)

INTERFACE SxSymbol;
(*
  Facilities for manipulating symbols

  This interface provides operations on symbols, named entities that can
  be read by Sx.Read and printed by Sx.Print.
  
  Index: symbolic expressions

  Symbols are nodes in a directed tree structure, called a /symbol
  tree/.  Every symbol /s/, except the root of the tree, has a unique
  /parent symbol/, /s.parent/; the root symbol has /parent=NIL/.
  
  Each edge of a symbol tree is labeled by some TEXT string.  The /full
  name/ of a symbol is the list (not the concatenation) of all edge
  labels along the path from the root to that symbol; each label is
  a /component/ of the full name.  (This interface does not say how
  symbol names should be printed, but typically the component strings
  are printed in left to right orser, separated by periods.  See for
  example the Sx interface.)
  
  The last component of the full name is the /short name/ of the symbol,
  or its /name relative to its parent/.  All children of the same parent
  have distinct short names (that is, all edges out of a given node
  have distinct lables).  It follows that no two symbols in the same
  symbol tree have the same full name.

  Each symbol /s/ also has a /serial number/ s.number, which is assigned
  when the symbol is created, and is distinct from the serial number
  of all other symbols in the same address space.  The serial number
  is useful as an index in hash tables, especially in systems with
  relocating garbage collectors.
*)

TYPE 
  T <: OBJECT 
      parent:  (*CONST*) T;        (* The symbol's parent, or NIL if root *)
      name:    (*CONST*) TEXT;     (* Symbol's name rel. to parent, if parent#NIL *)
      number:  (*CONST*) CARDINAL; (* Symbol's serial number *)
    END;

VAR (*CONST*) DefaultRoot: T;
  (*
    The root of a default symbol tree. *)

PROCEDURE NewRoot(): T;
  (* 
    Creates a new root symbol, distinct from all other symbols.  
    The /parent/ will be NIL, and the /name/ is irrelevant.  *)

PROCEDURE FromName(name: TEXT; parent: T := NIL): T;
  (*
    Returns the symbol with given short name and given parent, creating 
    one if necessary (in Lisper's jargon, "interns the symbol"). 
    If parent = NIL, DefaultRoot is used instead. *)

PROCEDURE FromNameChars(READONLY name: ARRAY OF CHAR; parent: T := NIL): T;
  (* 
    Same as FromName, taking CHARS instead of TEXT. *)

PROCEDURE FromNames(READONLY name: ARRAY OF TEXT; root: T := NIL): T;
  (*
    Returns the symbol reached from the given root through
    the given sequence of short names.  Equivalent to
    
|     IF root = NIL THEN s := DefaultRoot ELSE s := root END; 
|     FOR i := 0 TO LAST(name) DO s := FromName(name[i], s) END;
|     RETURN s

    *)

END SxSymbol.
