(* Copyright (C) 1990, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Thu Apr  9 09:50:22 PDT 1992 by kalsow     *)
(*      modified on Wed Mar 13 01:21:53 1991 by muller         *)

INTERFACE RTMisc;

(* This interface provides access to miscellaneous runtime routines. *)


(*-------------------------------- program startup/shutdown -----------------*)

TYPE Exitor <: REFANY;

PROCEDURE RegisterExitor (p: PROCEDURE (n: INTEGER) RAISES ANY): Exitor;
(* Registers the procedure p to be executed when Exit is called; it is passed
   the argument of Exit.
   The registered procedures are executed in the reverse order. *)

PROCEDURE UnregisterExitor (e: Exitor);
(* removes e's procedure from the registered set. *)

PROCEDURE Exit (n: INTEGER);
(* call the registered exitors and terminate the program with status 'n' *)


(*------------------------------- byte copying ------------------------------*)

PROCEDURE Copy (src, dest: ADDRESS;  len: INTEGER);
(* copy len bytes from src to dest *)

PROCEDURE Zero (dest: ADDRESS;  len: INTEGER);
 (* zero len bytes begining at dest *)

 

(*------------------------------- rounded arithmetic ------------------------*)

PROCEDURE Align (a: ADDRESS; y: INTEGER): ADDRESS;
(* return the smallest integer greater or equal to x that is a multiple of y *)

PROCEDURE Upper (x, y: INTEGER): INTEGER;
(* return the smallest integer greater or equal to x that is a multiple of y *)



(*------------------------------- runtime error reporting -------------------*)

PROCEDURE FatalError (file: TEXT;  line: INTEGER;  a, b, c: TEXT := NIL);
(* report a non-classified fault *)

PROCEDURE FatalErrorI (msg: TEXT := NIL;  i: INTEGER);
(* report a non-classified fault with an integer argument *)

PROCEDURE FatalErrorPC (pc: INTEGER;  a, b, c: TEXT := NIL);
(* report a non-classified fault at the given PC *)

PROCEDURE AssertFault (file: TEXT;  line: INTEGER);
(* report an "assertion failure" runtime error *)

PROCEDURE ReturnFault (file: TEXT;  line: INTEGER);
(* report a "missing RETURN in function" runtime error *)

PROCEDURE CaseFault (file: TEXT;  line: INTEGER);
(* report a "missing CASE arm" runtime error *)

PROCEDURE TypecaseFault (file: TEXT;  line: INTEGER);
(* report a "missing TYPECASE arm" runtime error *)

PROCEDURE RangeFault (file: TEXT;  line: INTEGER);
(* report a "value out of range" runtime error *)

PROCEDURE SubscriptFault (file: TEXT;  line: INTEGER);
(* report a "subscript out of range" runtime error *)

PROCEDURE NarrowFault (file: TEXT;  line: INTEGER);
(* report a "narrow failure" runtime error *)

PROCEDURE NilFault (file: TEXT;  line: INTEGER);
(* report a "dereference NIL" runtime error *)

PROCEDURE HandlerFault (ex_name: TEXT);
(* report an "unhandled exception" runtime error *)

PROCEDURE RaisesFault (ex_name: TEXT);
(* report a "exception not in RAISES clause" runtime error *)

PROCEDURE StackOverflow (file: TEXT;  line: INTEGER);
(* report a "stack overflow" runtime error *)

END RTMisc.

