(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by stolfi on Tue Apr 25 20:53:33 1989               *)
(* Last modified on Fri Mar 13 14:55:02 PST 1992 by muller     *)
(*      modified on Thu Oct 25 13:40:52 PDT 1990 by stolfi     *)

MODULE RGBSort EXPORTS RGBSort, RGBSortPrivate;

IMPORT RGB, RGBDist, RandomPerm;

PROCEDURE Sort (
    READONLY color: Table;              (* Colors to be sorted *)
    READONLY model: Table;              (* "Ideal" color table *)
    READONLY weight: ARRAY OF REAL;     (* Relative importance of model entries *)
    dist: DistFn := RGBDist.Perceptual; (* Distance function to use *)
    parm: REAL := 1.0;                  (* Controlling parameter for /dist/ *)
    credit: NAT := LAST (NAT);          (* Max number of calls to /dist/ *)
    tolerance: REAL := 0.005;           (* An insignificant /dist/ *)
    check: CheckProc := NIL;            (* Client's progress-monitoring proc *)
    checkData: REFANY := NIL;           (* Data for same. *)
  ): REF ARRAY OF CARDINAL    (* Where each /color/ should go. *) 
  RAISES {} =
  VAR
    n: CARDINAL;           (* Size of extended arrays *)
    pos: REF ARRAY OF CARDINAL; (* pos[p] = where color[p] should go *)
    ev: REF ARRAY OF REAL;      (* Match edge energy: ev^[p] = D(p, pos[p]) *)
    downhillCredit: CARDINAL;
  BEGIN
    n := MAX (NUMBER (color), NUMBER (model));
    pos :=  NEW(REF ARRAY OF CARDINAL, n);
    FOR p := 0 TO n - 1 DO pos[p] := p END;
    ev := NEW(REF ARRAY OF REAL, n);
    FOR i := 0 TO LAST(ev^) DO ev[i] := 0.0 END;
    downhillCredit := credit;
    DownhillSort (
      color, model, weight, 
      dist, parm, 
      downhillCredit, tolerance, 
      (*IO*) pos^, (*OUT*) ev^, 
      check, checkData
    );
    RETURN pos
  END Sort;

PROCEDURE Mismatch (
    READONLY color: Table; 
    p: CARDINAL; 
    READONLY model: Table;
    READONLY weight: ARRAY OF REAL; 
    q: CARDINAL; 
    dist: DistFn; 
    parm: REAL;
    tolerance: REAL
  ): REAL RAISES {} =
  (* 
    Distance squared between color[p] and model[q] *)
  BEGIN
    IF p > LAST (color) 
    OR q > LAST (model)
    OR color[p] = RGB.Undefined
    OR model[q] = RGB.Undefined 
    OR weight[q] = 0.0 THEN
      RETURN 0.0
    ELSE
      WITH d = dist (color[p], model[q], parm) - tolerance DO
        IF d <= 0.0 THEN RETURN 0.0 ELSE RETURN weight[q] * d * d END;
      END
    END
  END Mismatch;

PROCEDURE DownhillSort (
    READONLY color: Table; 
    READONLY model: Table; 
    READONLY weight: ARRAY OF REAL; 
    dist: DistFn; 
    parm: REAL; 
    VAR (*IO*) credit: CARDINAL; 
    tolerance: REAL; 
    VAR (*IO*) pos: ARRAY OF CARDINAL; (* pos[p] where color[p] should go *)
    VAR (*IO*) ev: ARRAY OF REAL;      (* Cost of match edges *)
    check: CheckProc := NIL;
    checkData: REFANY := NIL;
  ) RAISES {} =
  VAR
    nColors: CARDINAL; (* Number of colors to place *)
    nSlots: CARDINAL;  (* Number of colors slots available *)
    n: CARDINAL;       (* MAX(nColors, nSlots) *)
    nBad: CARDINAL;    (* Num of imperfect matcing pairs *)
    q: CARDINAL;       (* Indices into color[], pos[] *)
    pp: CARDINAL;      (* Indices into model[], weight[] *)
    matchErg: REAL;    (* "Energy" of match *)
    swapErg: REAL;     (* Energy of match with p,q swapped *)
    perm: REF ARRAY OF CARDINAL;
    step: INTEGER;
    dpq, dqp: REAL;
    maxProbes: CARDINAL;
    probes: CARDINAL;
    pass: CARDINAL;
  BEGIN
    nColors := NUMBER (color);
    nSlots := NUMBER (model);
    <* ASSERT nSlots = NUMBER (weight) *>
    n := MAX (nColors, nSlots);
    <* ASSERT n = NUMBER (pos) *>
    <* ASSERT n = NUMBER (ev) *>
    (* Assumes pos already contains the initial match: *)
    (* Compute cost of match (per edge and total) *)
    matchErg := 0.0;
    nBad := 0;
    FOR p := 0 TO n - 1 DO
      ev[p] := Mismatch (
        color, p, model, weight, 
        pos[p], 
        dist, parm, tolerance
      );
      matchErg := matchErg + ev[p];
      IF ev[p] > 0.0 THEN INC (nBad) END;
    END;
    IF (check # NIL) THEN
      IF check (color, pos, matchErg, checkData) OR (nBad = 0) THEN RETURN  END
    END;
    IF (nBad = 0) THEN RETURN  END;
    perm := RandomPerm.NewArr (NIL, n - 1);
    (* Optimize match by downhill walk: *)
    maxProbes := MIN ((16 + nBad) * (16 + n), credit);
    probes := 0;
    pass := 0;
    REPEAT 
      (* One more pass: *)
      
      (* Pick another step size in [1..n-1]: *)
      step := 1 + perm[pass MOD NUMBER (perm^)];
      
      (* Perform one more pass over the unmatched colors: *)
      (* (Perhaps should consider p's in order of decreasing ev[p]?) *)
      FOR p := 0 TO nColors - 1 DO
        IF (ev[p] > 0.0) THEN
          
          (* Pick another vertex q distinct from p: *)
          q := (p + step) MOD n;
          
          (* Compute "energy" of match if p,q swapped: *)
          dpq := Mismatch (
            color, p, model, weight, 
            pos[q], 
            dist, parm, tolerance
          );
          dqp := Mismatch (
            color, q, model, weight, 
            pos[p], 
            dist, parm, tolerance
          );
          INC (probes, 2);
          swapErg := matchErg - ev[p] - ev[q] + dqp + dpq;
          
          (* Swap if better: *)
          IF swapErg < matchErg THEN
            (* swap pos[p], pos[q] *)
            pp := pos[p];
            pos[p] := pos[q];
            pos[q] := pp;
            ev[p] := dpq;
            ev[q] := dqp;
            matchErg := swapErg;
          ELSE
            (* stay put *)
          END;
        END
      END;
      
      (* Recompute matchErg to clear out accumulated rounding error *)
      matchErg := 0.0;
      nBad := 0;
      FOR p := 0 TO n - 1 DO
        matchErg := matchErg + ev[p];
        IF ev[p] > 0.0 THEN INC (nBad) END;
      END;
      IF (check # NIL) THEN
        IF check (color, pos, matchErg, checkData) THEN RETURN  END;
      END;
      INC (pass);
    UNTIL (probes >= maxProbes) OR (nBad = 0);
    IF credit < LAST (CARDINAL) THEN credit := MAX (0, credit - probes) END;
  END DownhillSort;

BEGIN
END RGBSort.

