/* pagercico.c
   Program to communicate with IXO-compatible servers.

   Copyright (C) 1995 Gregory Nickonov

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   The authors of the program may be contacted at gn@jet.msk.su or
   avk@jet.msk.su.

   $Log: pagercico.c,v $
   */

#include	<uucp.h>

#if USE_RCS_ID
const char pager_rcsid[] = "$Id: pagercico.c,v 0.90 1995/04/17 12:13:10 gn Exp $";
#endif

#include	<uudefs.h>
#include	<uuconf.h>
#include	<conn.h>
#include	<prot.h>
#include	<system.h>
#include	<sysdep.h>
#include	<getopt.h>

#include	<stdio.h>
#include	<ctype.h>
#include	<errno.h>



/* Local variables */

/* Shit */

const char	*zCuvar_escape = "~";
const char	*zCuvar_eol = "\r\025\003\017\004\023\021\022";

/* Global uuconf pointer */

static pointer				pPguuconf;

/* Connection */

static struct sconnection	*qPgconn;

/* Whether to close the connection */

static boolean				fPgclose_conn;

/* System was locked? */

static boolean				fPgsyslock;

/* Dialer used to dial out */

static struct uuconf_dialer	*qPgdialer;

/* Currently processing system */

static struct uuconf_system	*qPgsystem;

/* A structure used to pass information to ipgport_lock */

struct sconninfo {
	boolean				fMatched;
	boolean				fLocked;

	struct sconnection	*qconn;
	const char			*zline;
	};
	
/* Function icexpect, stolen from chat.c, sorry Ian, but I need it :) */

int					icexpect P(( struct sconnection *qconn, int cstrings,
								 char **azstrings, size_t *aclens,
								 int ctimeout, boolean fstrip ));

/* Local functions */

static void			upgusage P((void));
static void			upghelp P((void));
static void			upgabort P((void));

static int			ipgport_lock P((struct uuconf_port *qport, pointer pinfo ));

static void			PagerProtocol P(());
static void			ProcessMessages P(());

/* Long getopt options */

static const struct option asPglongopts[] = {

	{ "system", required_argument, NULL, 'z' },
	{ "config", required_argument, NULL, 'I' },
	{ "debug", required_argument, NULL, 'x' },
	{ "version", no_argument, NULL, 'v' },
	{ "help", no_argument, NULL, 1 },
	{ NULL, 0, NULL, 0 }
	};



int
main( argc, argv )
	int		argc;
	char	**argv; {
	
	char		*zphone = NULL;			/* -c: phone number */
	char		*zline = NULL;			/* -l: line */
	const char	*zport = NULL;			/* -p: port name */
	long		ibaud = 0L;				/* -s: speed */
	const char	*zsystem = NULL;		/* -z: system */
	const char	*zconfig = NULL;		/* -I: configuration file name */

	pointer		puuconf;
	int			iuuconf;

	struct uuconf_system		ssys;
	const struct uuconf_system	*qsys = NULL;

	struct uuconf_port			sport;
	struct sconnection			sconn;
	struct sconninfo			sinfo;

	long						ihighbaud;

	struct uuconf_dialer		sdialer;
	struct uuconf_dialer		*qdialer;

	int							i, iOpt;

	boolean						flooped;
	boolean						fNextSystem;

	enum tparitysetting			tparity;
	enum tstripsetting			tstrip;

	

	zProgram	= argv[0];



	/*
	 *	Parsing arguments
	 */
	
	while( (iOpt = getopt_long( argc, argv, "dI:vx:",
								asPglongopts, (int *) NULL )) != EOF ) {

		switch( iOpt ) {

			case 'd':		/* Set debugging level to maximum */

#if DEBUG > 1
				iDebug	= DEBUG_MAX;
#endif
				break;

			case 'I':		/* Configuration file name */
				if( fsysdep_other_config( optarg ) )
					zconfig	= optarg;
				break;

			case 'x':		/* Set debug level */
#if DEBUG > 1
				iDebug	|= idebug_parse( optarg );
#endif
				break;

			case 'v':		/* Print version and exit */
				fprintf( stderr,
						 "%s: Pager's messages delivery system, copyright (c) 1995 Gregory Nickonov\n",
						 zProgram );
				exit( EXIT_SUCCESS );

			case 1:			/* --help */
				upghelp();
				exit( EXIT_SUCCESS );

			case 0:			/* Long option found and flag set */
				break;

			default:
				upgusage();
			}
		}

	/*
	 * The only permitted non-option parameters is the system's names.
	 */

	if( optind == argc ) {

		fprintf( stderr, "%s: system name expected\n", zProgram );
		upgusage();
		}
	


	/*
	 *	Initializing configuration system and system dependend modules
	 */
	
	iuuconf	= uuconf_init( &puuconf, "pager", zconfig );

	if( iuuconf != UUCONF_SUCCESS )
		ulog_uuconf( LOG_FATAL, puuconf, iuuconf );

	pPguuconf	= puuconf;

#if DEBUG > 1
	{
		const char		*zdebug;

		iuuconf	= uuconf_debuglevel( puuconf, &zdebug );

		if( iuuconf != UUCONF_SUCCESS )
			ulog_uuconf( LOG_FATAL, puuconf, iuuconf );

		if( zdebug != NULL )
			iDebug	|= idebug_parse( zdebug );
		}
#endif

	usysdep_initialize( puuconf, INIT_SUID );

	
	ulog_to_file( puuconf, TRUE );
	ulog_fatal_fn( upgabort );

#ifdef SIGINT
	usysdep_signal( SIGINT );
#endif
#ifdef SIGHUP
	usysdep_signal( SIGHUP );
#endif
#ifdef SIGQUIT
	usysdep_signal( SIGQUIT );
#endif
#ifdef SIGTERM
	usysdep_signal( SIGTERM );
#endif
#ifdef SIGPIPE
	usysdep_signal( SIGPIPE );
#endif

	fPgsyslock	= FALSE;

	for( ; optind < argc; ++optind ) {

		zsystem	= argv[optind];

		/*
		 *	zsystem MUST be defined!
		 */
	
		iuuconf	= uuconf_system_info( puuconf, zsystem, &ssys );

		if( iuuconf != UUCONF_SUCCESS ) {
			if( iuuconf != UUCONF_NOT_FOUND )
				ulog_uuconf( LOG_FATAL, puuconf, iuuconf );

			ulog( LOG_ERROR, "%s: System not found", zsystem );
			continue;	/* Process next system */
			}
	
		/*
		 *	Is there any work for this system?
		 */

		if( !fsysdep_has_work( &ssys ) ) {
			/* ulog( LOG_NORMAL, "No work for system %s", zsystem ); */
			continue;
			}

		if( fPgsyslock == TRUE ) {	/* Unlocking previously locked system */

			(void) fsysdep_unlock_system( qPgsystem );
			fPgsyslock	= FALSE;
			}

		if( !fsysdep_lock_system( &ssys ) ) {

			ulog( LOG_NORMAL, "System %s already locked", zsystem );
			continue;
			}

		fPgsyslock	= TRUE;
		qPgsystem	= &ssys;

		qsys	= &ssys;

		flooped		= FALSE;
		fNextSystem	= FALSE;

		while( TRUE ) {

			sinfo.fMatched		= FALSE;
			sinfo.fLocked		= FALSE;
			sinfo.qconn			= &sconn;
			sinfo.zline			= zline;


			if( zport != NULL || zline != NULL || ibaud != 0L ) {

				iuuconf	= uuconf_find_port( puuconf, zport, ibaud, 0L,
											ipgport_lock, (pointer) &sinfo,
											&sport );

				if( iuuconf != UUCONF_SUCCESS ) {

					if( iuuconf != UUCONF_NOT_FOUND ) {

						if( sinfo.fLocked ) {

							(void) fconn_unlock( &sconn );
							uconn_free( &sconn );
							}

						ulog_uuconf( LOG_FATAL, puuconf, iuuconf );
						}

					if( zline == NULL || zport != NULL ||
						zphone != NULL || qsys != NULL ) {

						if( sinfo.fMatched )
							ulog( LOG_ERROR, "All matching ports in use" );
						else
							ulog( LOG_ERROR, "No matching ports" );

						fNextSystem	= TRUE;

						break;
						}

					sport.uuconf_zname			= zline;
					sport.uuconf_ttype			= UUCONF_PORTTYPE_DIRECT;
					sport.uuconf_zprotocols		= NULL;
					sport.uuconf_qproto_params	= NULL;
					sport.uuconf_ireliable		= 0;
					sport.uuconf_zlockname		= NULL;
					sport.uuconf_palloc			= NULL;
					sport.uuconf_u.uuconf_sdirect.uuconf_zdevice	= NULL;
					sport.uuconf_u.uuconf_sdirect.uuconf_ibaud		= ibaud;

					if( !fconn_init( &sport, &sconn,
									 UUCONF_PORTTYPE_UNKNOWN ) ){

						fNextSystem	= TRUE;

						if( sinfo.fLocked ) {
							(void) fconn_unlock( &sconn );
							uconn_free( &sconn );
							}

						break;
						}

					if( !fconn_lock( &sconn, FALSE ) ) {
						ulog( LOG_ERROR, "%s: Line in use", zline );
						fNextSystem	= TRUE;
						break;
						}

					qPgconn	= &sconn;

					if( !fsysdep_port_access( &sport ) ) {
						ulog( LOG_ERROR, "%s: Permission denied", zline );

						(void) fconn_unlock( &sconn );
						uconn_free( &sconn );

						fNextSystem	= TRUE;
						break;
						}
					}
				ihighbaud	= 0L;
				}
			else {

		
				for( ; qsys != NULL; qsys = qsys->uuconf_qalternate ) {

					if( !qsys->uuconf_fcall )
						continue;

					if( qsys->uuconf_qport != NULL ) {

						if( fconn_init( qsys->uuconf_qport, &sconn,
										UUCONF_PORTTYPE_UNKNOWN ) ) {
					
							if( fconn_lock( &sconn, FALSE ) ) {

								qPgconn		= &sconn;
								break;
								}

							uconn_free( &sconn );
							}
						}
					else {

						sinfo.fMatched	= FALSE;
						sinfo.fLocked	= FALSE;
						sinfo.qconn		= &sconn;

						iuuconf	= uuconf_find_port( puuconf, qsys->uuconf_zport,
													qsys->uuconf_ibaud,
													qsys->uuconf_ihighbaud,
													ipgport_lock,
													(pointer) &sinfo,
													&sport );

						if( iuuconf == UUCONF_SUCCESS )
							break;

						if( iuuconf != UUCONF_NOT_FOUND ) {

							if( sinfo.fLocked ) {

								(void) fconn_unlock( &sconn );
								uconn_free( &sconn );
								}

							ulog_uuconf( LOG_FATAL, puuconf, iuuconf );
							}
						}
					}

				if( qsys == NULL ) {
					const char		*zrem;

					if( flooped )
						zrem	= "remaining ";
					else
						zrem	= "";

					if( sinfo.fMatched )
						ulog( LOG_ERROR, "%s: All %smatching ports in use",
							  zsystem, zrem );
					else
						ulog( LOG_ERROR, "%s: No %smatching ports", zsystem, zrem );

					fNextSystem	= TRUE;
					break;
					}

				ibaud		= qsys->uuconf_ibaud;
				ihighbaud	= qsys->uuconf_ihighbaud;
				}

			/*
			 *
			 *	Here we have a locked connection to use
			 *
			 */
	
			if( !fconn_open( &sconn, ibaud, ihighbaud, FALSE ) ) {
				(void) fconn_unlock( &sconn );
				uconn_free( &sconn );
		
				fNextSystem	= TRUE;
				break;
				}
				
			fPgclose_conn	= TRUE;

			if( FGOT_SIGNAL() )
				upgabort();


			/*	Setting up the connection */

			tparity		= PARITYSETTING_EVEN;
			tstrip		= STRIPSETTING_SEVENBITS;

			if( !fconn_set( &sconn, tparity, tstrip, XONXOFF_ON ) ) {

				(void) fconn_unlock( &sconn );
				uconn_free( &sconn );

				fNextSystem	= TRUE;
				break;
				}

			if( qsys != NULL )
				zphone	= qsys->uuconf_zphone;

			if( qsys != NULL || zphone != NULL ) {
				enum tdialerfound		tdialer;

				if( !fconn_dial( &sconn, puuconf, qsys, zphone, &sdialer,
								 &tdialer ) ) {

					if( zport != NULL || zline != NULL ||
						ibaud != 0L || qsys == NULL ) {

						(void) fconn_unlock( &sconn );
						uconn_free( &sconn );

						fNextSystem	= TRUE;
						break;
						}

					qsys	= qsys->uuconf_qalternate;

					if( qsys == NULL ) {
						ulog( LOG_ERROR, "%s: No remaining alternates", zsystem );
						(void) fconn_unlock( &sconn );
						uconn_free( &sconn );

						fNextSystem	= TRUE;
						break;
						}


					fPgclose_conn	= FALSE;

					(void) fconn_close( &sconn, pPguuconf, qPgdialer, FALSE );
					qPgconn	= NULL;

					(void) fconn_unlock( &sconn );
					uconn_free( &sconn );

					flooped	= TRUE;
					continue;
					}

				if( tdialer == DIALERFOUND_FALSE )
					qdialer	= NULL;
				else
					qdialer	= &sdialer;
				}
			else {
			
				if( !fsysdep_port_access( sconn.qport ) ) {
					ulog( LOG_ERROR, "Access to port denied" );

					(void) fconn_unlock( &sconn );
					uconn_free( &sconn );

					fNextSystem	= TRUE;
					break;
					}
			
				qdialer	= NULL;

				if( !fconn_carrier( &sconn, FALSE ) )
					ulog( LOG_FATAL, "Can't turn off carrier" );
				}

			break;
			}
	
		if( fNextSystem == TRUE )
			break;				/* Processing next system */

		qPgdialer	= qdialer;

		if( FGOT_SIGNAL() )
			upgabort();
	
		/*
		 *
		 *	Here we have connected and can start the main IXO protocol.
		 *
		 */

		PagerProtocol();

		(void) fsysdep_unlock_system( qPgsystem );
		fPgsyslock	= FALSE;
		qPgsystem	= NULL;

		(void) fconn_close( &sconn, puuconf, qdialer, TRUE );
		(void) fconn_unlock( &sconn );
		uconn_free( &sconn );
		}

	ulog_close();

	usysdep_exit( TRUE );
	
	return	0;
	}
	

static int
ipgport_lock( qport, pinfo )
	struct uuconf_port	*qport;
	pointer				pinfo; {

	struct sconninfo	*q = (struct sconninfo *) pinfo;

	if( q->zline != NULL && !fsysdep_port_is_line( qport, q->zline ) )
		return	UUCONF_NOT_FOUND;
	
	q->fMatched	= TRUE;

	if( !fconn_init( qport, q->qconn, UUCONF_PORTTYPE_UNKNOWN ) )
		return	UUCONF_NOT_FOUND;
	else if( !fconn_lock( q->qconn, FALSE ) ) {
		uconn_free( q->qconn );
		return	UUCONF_NOT_FOUND;
		}
	else {
		qPgconn		= q->qconn;
		q->fLocked	= TRUE;
		return	UUCONF_SUCCESS;
		}
	}
	
	
static void
upgusage() {

	fprintf( stderr, "Usage: %s [options] [system name]\n", zProgram );
	fprintf( stderr, "Use %s --help for help\n", zProgram );

	exit( EXIT_FAILURE );
	}

static void
upghelp() {

	fprintf( stderr,
			 "Pager messages delivery system, copyright (C) 1995 Gregory Nickonov\n" );

	fprintf( stderr, "Usage: %s [options] [system name]\n", zProgram );

	fprintf( stderr, "-d: Set maximum debugging level\n" );
	fprintf( stderr, "-x,--debug debug: Set debugging type\n" );

#if HAVE_TAYLOR_CONFIG
	fprintf( stderr, "-I,--config file: Set configuration file to use\n" );
#endif /* HAVE_TAYLOR_CONFIG */

	fprintf( stderr, "-v,--version: Print version and exit\n" );
	fprintf( stderr, "--help: Print help and exit\n" );
	}

static void
upgabort() {

	if( qPgconn != NULL ) {
		struct sconnection	*qconn;

		if( fPgclose_conn ) {

			fPgclose_conn	= FALSE;
			(void) fconn_close( qPgconn, pPguuconf, qPgdialer, FALSE );
			}

		qconn	= qPgconn;
		qPgconn	= NULL;

		(void) fconn_unlock( qconn );
		uconn_free( qconn );
		}
	
	if( fPgsyslock && qPgsystem ) 
		(void) fsysdep_unlock_system( qPgsystem );

	ulog_close();

	usysdep_exit( FALSE );
	}



#define	STX			2
#define	EOT			4
#define	ACK			6
#define	LF			10
#define	CR			13
#define	NAK			21
#define	ESC			27
#define	RS			30

#define	MAX_PACKET	10000

int
GetByte( byte )
	char		*byte; {
	size_t		length = 1;

	if( fconn_read( qPgconn, byte, &length, 1, 1, TRUE ) && length == 1 )
		return	1;
	
	if( FGOT_SIGNAL() )
		upgabort();

	return	0;
	}
	
int
GetPacket( buffer )
	char					*buffer; {

	int			max, len, err;
	char		c, *buf = buffer;


	*buffer		= '\0';
	err			= 50;

	while( err-- ) {

		max		= MAX_PACKET;

		while( GetByte( &c ) ) {

			if( c == LF )
				continue;

			if( c == CR )
				break;

			*buf++	= c;
			--max;

			if( max == 0 ) {
				ulog( LOG_ERROR, "ReadPacket: packet too long" );
				return	-1;
				}

			}

		*buf	= '\0';
		len		= buf - buffer;

		if( len )
			return	0;
		}
	
	ulog( LOG_ERROR, "No data on line" );

	return	-1;
	}
	
char	*HandshakeMessages[] = {
		"ID=",
		};

size_t	HandshakeLengths[] = { 3, };

#define	N_HANDSHAKE_MESSAGES	(sizeof(HandshakeMessages)/sizeof(char *))

char	*LoginMessages[] = {
		"\6",						/* ACK, login ok */
		"PagerMode On",				/* Vessolink/Russia ACK */
		"AutoMode On",				/* Another Vessolink/Russia ACK */
		"\21",						/* NAK, login failed */
		"\27\4",					/* ESC+EOT, forced disconnection */
		};

size_t	LoginLengths[] = { 1, 12, 11, 1, 2 };

#define	N_LOGIN_MESSAGES		(sizeof(LoginMessages)/sizeof(char*))

#define	LOGIN_OK		2
#define	LOGIN_FAILED	3
#define	LOGIN_FD		4

char	*GoAheadMessages[] = {
		"\033[p",
		};

size_t	GoAheadLengths[] = { 3 };

#define	N_GOAHEAD_MESSAGES		1

static void
PagerProtocol() {
	int			retries;
	char		packet[MAX_PACKET];
	int			length, retcode;

	ulog( LOG_NORMAL, "Initiating login..." );

	retries		= 10;

	fconn_write( qPgconn, "\r", 1 );

	while( retries-- ) {

		retcode = icexpect( qPgconn, N_HANDSHAKE_MESSAGES,
							HandshakeMessages, HandshakeLengths, 2, TRUE );

		if( retcode == 0 )
			break;

		fconn_write( qPgconn, "\r", 1 );
		}
	
	if( retries < 0 ) {
		ulog( LOG_ERROR, "No ID= message received" );
		return;
		}
	
	ulog( LOG_NORMAL, "Got the ID= message, logging in..." );

	/*
	 *	Login procedure
	 */
	
	retries		= 5;

	while( retries-- ) {

		fconn_write( qPgconn, "\033", 1 );
		fconn_write( qPgconn, "PG1000000\r", 10 );

#if 0
		if( GetPacket( packet ) < 0 )
			return;

		length	= strlen( packet );

		if( length == 1 && *packet == ACK ) 
			break;

		if( length == 1 && *packet == NAK ) {
			ulog( LOG_NORMAL, "Retrying to log in" );
			continue;
			}

		if( length == 2 && (packet[0] == ESC && packet[1] == EOT) ) {
			ulog( LOG_ERROR, "Forced disconnection" );
			return;
			}

		if( !strcmp( packet, "PagerMode On" ) ) {
			ulog( LOG_NORMAL, "'PagerMode On' message received, assuming ACK" );
			break;
			}
#else
	
		retcode = icexpect( qPgconn, N_LOGIN_MESSAGES,
							LoginMessages, LoginLengths, 2, TRUE );

		if( retcode < 0 )
			continue;

		if( retcode <= LOGIN_OK )
			break;

		if( retcode <= LOGIN_FAILED ) {
			ulog( LOG_ERROR, "Login failed" );
			return;
			}

		if( retcode <= LOGIN_FD ) {
			ulog( LOG_ERROR, "Forced disconnection" );
			return;
			}
#endif

		ulog( LOG_ERROR, "Undefined protocol" );
		return;
		}
	
	if( retries < 0 ) {
		ulog( LOG_ERROR, "Unable to log in" );
		return;
		}
	

	ulog( LOG_NORMAL, "Waiting for go-ahead message" );

	retries	= 5;

	while( retries-- ) {

		retcode = icexpect( qPgconn, N_GOAHEAD_MESSAGES,
							GoAheadMessages, GoAheadLengths, 5, TRUE );

		if( retcode >= 0 )
			break;
		}
	
	if( retries < 0 ) {
		ulog( LOG_ERROR, "No go-ahead message received" );
		return;
		}
	

	ulog( LOG_NORMAL, "Go-ahead message received, starting sender." );

	ProcessMessages();

	ulog( LOG_NORMAL, "Disconnecting..." );

	fconn_write( qPgconn, "\4\r", 2 );

	retries	= 5;

	while( retries-- ) {

		if( GetPacket( packet ) < 0 )
			return;

		length	= strlen( packet );

		if( length == 2 && (packet[0] == ESC && packet[1] == EOT) )
			break;

		if( length == 1 && *packet == RS ) {
			ulog( LOG_ERROR, "Disconnection error -- ignored" );
			break;
			}
		}
	}

	

char *
Checksum( packet ) 
	char	*packet; {
	static char	check[10];
	int			sum = 0;

	for( ; *packet; ++packet )
		sum	+= *packet;

	check[2]	= '0' + (sum & 15);		sum		= sum >> 4;
	check[1]	= '0' + (sum & 15);		sum		= sum >> 4;
	check[0]	= '0' + (sum & 15);
	check[3]	= '\0';

	return	check;
	}

static int
ProcessMessage( pagerId, text )
	char				*pagerId;
	char				*text; {
	
	char				packet[MAX_PACKET];
	int					retries, length;

	
	ulog( LOG_NORMAL, "Delivering message to pager # %s", pagerId );

	retries	= 100;

	while( retries-- ) {

		packet[0]	= STX;
		packet[1]	= '\0';

		strcat( packet, pagerId );
		strcat( packet, "\r" );

		strcat( packet, text );
		strcat( packet, "\r\3" );

		strcat( packet, Checksum( packet ) );
		strcat( packet, "\r" );

		fconn_write( qPgconn, packet, strlen( packet ) );

yetAnotherPacket:		/* This is ONLY for debugging purposes */

		if( GetPacket( packet ) < 0 )
			return	-2;

		length	= strlen( packet );

		if( length == 1 && *packet == ACK )
			break;

		if( length == 1 && *packet == NAK ) {

			ulog( LOG_NORMAL, "Retrying" );
			continue;
			}

		if( length == 1 && *packet == RS ) {

			ulog( LOG_ERROR, "Message rejected" );
			return	-1;
			}

		if( length == 2 && (packet[0] == ESC && packet[1] == EOT) ) {
			ulog( LOG_ERROR, "Forced disconnection" );
			return	-2;
			}

		if( length > 0 )
			goto	yetAnotherPacket;
		}
	
	if( retries < 0 ) {
		ulog( LOG_NORMAL, "Too many retries" );
		return	-3;
		}
	
	return	0;
	}

static void
ProcessMessages() {
	struct scmd		qcmd;
	char			*ztext, *zfile;
	openfile_t		dataFile;
	size_t			ctextLength;
	int				retcode;
	
	/* Initializing works */

	if( fsysdep_get_work_init( qPgsystem, 'z' ) == FALSE ) {
		ulog( LOG_ERROR, "Can't initialize work processing" );
		return;
		}
	
	/* Processing messages */

	while( TRUE ) {

		if( fsysdep_get_work( qPgsystem, 'z', &qcmd ) == FALSE ) {
			ulog( LOG_ERROR, "Can't get work to do" );
			return;
			}

		if( qcmd.bcmd == 'H' )
			break;

		if( qcmd.bcmd != 'E' ) {
			ulog( LOG_ERROR, "Wrong command (%c) -- ignored", qcmd.bcmd );

			/* Bounce message here? */
			
			continue;
			}

		/* Processing this command */


		if( strchr( qcmd.zoptions, 'C' ) == NULL ) {
			ulog( LOG_ERROR, "Message not in the spool directory" );
			continue;
			}

		zfile	= zsysdep_spool_file_name( qPgsystem, qcmd.ztemp, qcmd.pseq );
		if( zfile == NULL ) {
			ulog( LOG_ERROR, "Can't expand data file name" );
			continue;
			}

		if( (dataFile = esysdep_user_fopen( zfile, TRUE, FALSE )) == NULL ) {
			ulog( LOG_ERROR, "Can't open data file %s", zfile );
			continue;
			}

		ztext		= NULL;
		ctextLength	= 0;

		retcode	= getline( &ztext, &ctextLength, dataFile );

		ffileclose( dataFile );
		
		if( retcode < 0 ) {
			ulog( LOG_ERROR, "Can't read from data file %s", zfile );
			continue;
			}

		if( ctextLength >= 0 ) {

			if( ctextLength > 255 )
				ztext[255]	= '\0';		/* Truncating long messages */

			retcode	= ProcessMessage( qcmd.zcmd, ztext );

			xfree( ztext );
			
			if( retcode == -2 ) {		/* Forced disconnection */
				usysdep_get_work_free( qPgsystem );
				return;
				}
			else if( retcode == -3 ) {	
				/* Too many retries, leaving message in the spool */
				}
			else if( retcode == -1 ) {
				/* Message rejected, we must send notification to sender */
				retcode	= 0;
				}
			}
		else
			retcode	= 0;
		
		/* Removing command from the queue */

		if( !retcode ) {
			if( fsysdep_did_work( qcmd.pseq ) == FALSE ) 
				ulog( LOG_ERROR, "Can't remove job from the queue" );
			}
		}
	
	usysdep_get_work_free( qPgsystem );
	}

