;;; $Id: vcs-rcs.el,v 1.6 26-Oct-1992 20:22:56 EST don Exp $
;;; 
;;; RCS support for vcs.el
;;; 
;;; Copyright (C) Donald Beaudry <don@vicorp.com> 1992
;;;
;;; This file is not part of GNU Emacs, but is made available under
;;; the same conditions.
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 1, or
;;; (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; $Log: vcs-rcs.el,v $
;;; Revision 1.6  26-Oct-1992 20:22:56 EST  don
;;; moved some stuff inside of a let
;;;
;;; Revision 1.5  25-Oct-1992 21:37:17 EST  don
;;; changed for use with the new form stuff
;;;
;;; Revision 1.4  17-Sep-1992 22:00:03 EDT  don
;;; Added rcs-set-comment-leader function... not really a vcs function
;;; but where else should I put it
;;;
;;; Revision 1.3  12-Sep-1992 19:23:38 EDT  don
;;; merged finish put and put
;;;
;;; Revision 1.2  12-Sep-1992 18:33:42 EDT  don
;;; Separated from vcs.el
;;;
;;;
;;; 

(require 'form)

;;;
;;;
;;;
(defun vcs-rcs-get (history-file working-file lock)
  (if (vcs-rcs-has-history-p history-file)
      (progn
	(vcs-execute-command nil "co -f" (if lock "-l") history-file
			     working-file)
	t)))

(vcs-add-hook 'vcs-get-hooks 'vcs-rcs-get)


;;;
;;;
;;;
(defun vcs-rcs-unlock (history-file working-file)
  (if (vcs-rcs-has-history-p history-file)
      (progn
	  (vcs-execute-command nil "rcs" "-u" history-file)
	  t)))

(vcs-add-hook 'vcs-unlock-hooks 'vcs-rcs-unlock)


;;;
;;;
;;;
(defun vcs-rcs-insert-info (history-file working-file)
  (if (vcs-rcs-has-history-p history-file)
      (progn
	(call-process vcs-shell-path nil buf nil "-c"
		      (concat "rlog " working-file " " history-file))
	(goto-char (point-min))
	(delete-char 1)
	t)))


(vcs-add-hook 'vcs-insert-info-hooks 'vcs-rcs-insert-info)


;;;
;;;
;;;
(defform vcs-rcs-put
  "Major mode checking files into RCS"
  (when-finished vcs-rcs-finish-put)
  (buffer vcs-put-buffer)
  (mode-name "VCS-rcs-put")
  (text "                       ====== RCS Put ======\n\n\n")
  (field history-file
	 (prompt "History File: ")
	 (default history-file)
  	 (verifier (lambda (b e)
		     (if (< b e)
			 t
		       (error "A history file must be specified")))))
  (text "\n")
  (field working-file
	 (prompt "Working File: ")
	 (default working-file)
	 (verifier (lambda (b e)
		     (if (file-exists-p (buffer-substring b e))
			 t
		       (error "Working file does not exist")))))
  (text "\n\n")
  (field comment
	 (prompt (if (not (file-exists-p history-file))
		     "--Descriptive text--\n"
		   "--Log message--\n"))
	 (verifier (lambda (b e)
		     (if (< b e)
			 t
		       (error "A log message must be specified")))))
  (text "\n\n--\n"))


;;;
;;;
;;;
(defun vcs-rcs-setup-for-put (history-file working-file)
  (if (vcs-rcs-working-file history-file)
      (progn
	(vcs-rcs-put-mode nil vcs-use-other-window 'comment)
	(setq default-directory (file-name-directory working-file))
	(run-hooks 'vcs-put-mode-hooks)
	t)))

(vcs-add-hook 'vcs-put-hooks 'vcs-rcs-setup-for-put)


;;;
;;; 
;;;
(defun vcs-rcs-locker-name (history-file)
  "Returns a strings describing the people who have FILE locked under RCS.
Or \"\" if there are no lockers or nil if the file is not under RCS."
  (if (vcs-rcs-has-history-p history-file)
      (let ((lockers "")
	    (buf (get-buffer-create vcs-temp-buffer)))
	(save-excursion
	  (set-buffer buf)
	  (goto-char (point-max))
	  (vcs-execute-command nil "rlog -L -h" history-file)
	  (if (not (re-search-forward "^locks:" (point-max) t))
	      nil
	    (forward-line)
	    (save-excursion
	      (if (re-search-forward "^access list:" (point-max))
		  (setq end (match-beginning 0))
		(setq end (point-max)))) ;; this should not happen
	    (while (re-search-forward "^[ \t]+\\([^:;]*\\):" end t)
	      (setq lockers
		    (if (not (string= lockers ""))
			(concat lockers ", "
				(buffer-substring (match-beginning 1)
						  (match-end 1)))
		      (buffer-substring (match-beginning 1)
					(match-end 1)))))))
	lockers)
    nil))

(vcs-add-hook 'vcs-locker-name-hooks 'vcs-rcs-locker-name)
	  

;;;
;;; Given an rcs HISTORY-NAME, return the working file name.  If the
;;; HISTORY-NAME is not an rcs history file name return nil.
;;;
(defun vcs-rcs-working-file (history-name)
  "Return the working file name for a given rcs FILE-NAME."
  (if (not history-name)
      nil
    (let* ((name (file-name-nondirectory history-name))
	   (directory (file-name-directory history-name)))
      (if (not (string-match ",v$" name))
	  nil
	(setq name (substring name 0 (match-beginning 0)))
	(if (string-match "/RCS/$" directory)
	    (setq directory (substring directory 0 (1+ (match-beginning 0)))))
	(concat directory name)))))

(vcs-add-hook 'vcs-working-file-hooks 'vcs-rcs-working-file)

      
;;;
;;; Given a FILE-NAME convert it to an rcs history file name.  If the
;;; name is already a history file name, just return it.  The history
;;; name is built by appending ",v" to the file name and RCS to the
;;; directory name iff the directory contains an RCS directory.
;;;
(defun vcs-rcs-history-file (file-name)
  "Return the name of the rcs history file name for FILE-NAME"
  (let ((case-fold-search nil))
    (if file-name
	(let ((directory (file-name-directory (expand-file-name file-name)))
	      (name (file-name-nondirectory file-name)))
	  (concat (if (and (file-directory-p (concat directory "RCS"))
			   (not (string-match "/RCS/$" directory)))
		      (concat directory "RCS/")
		    directory)
		  (if (string-match ",v$" name)
		      name
		    (concat name ",v")))))))

(vcs-add-hook 'vcs-history-file-hooks 'vcs-rcs-history-file)


;;;
;;;
;;;
(defun vcs-rcs-scan-for-error nil
  "Scan the shell command output buffer for an rcs error message
and signal an error if one was found."
  (save-excursion
    (if (re-search-forward "error:" nil t)
	(let ((start-pos (1+ (point))))
	  (search-forward "\n" nil t)
	  (buffer-substring start-pos (1- (point)))))))
    
(vcs-add-hook 'vcs-scan-for-error-hooks 'vcs-rcs-scan-for-error)
	  

;;;
;;;
;;;
(defun vcs-rcs-has-history-p (file-name)
  "Returns a non NIL value if FILE-NAME is already under rcs control."
  (if (not file-name)
      nil
    (file-exists-p (vcs-rcs-history-file file-name))))

;;;
;;;
;;;
(defun vcs-rcs-finish-put (put-data)
  (if (equal (current-buffer) (get-buffer vcs-put-buffer))
      nil
    (error "Wrong buffer"))
  (let* ((history-file (form-field-string 'history-file put-data))
	 (working-file (form-field-string 'working-file put-data))
	 (comment-region (form-field-region 'comment put-data))
	 (comment-file (concat vcs-temp-dir "/" (make-temp-name "vcs")))
	 (buf (get-file-buffer working-file)))
    (if (and buf (buffer-modified-p buf))
	(save-excursion
	  (set-buffer buf)
	  (if (y-or-n-p (concat "Save " buffer-file-name "? "))
	      (save-buffer))))
    (message "Checking in %s..." history-file)
    (write-region (car comment-region) (cdr comment-region)
		  comment-file nil 'no-message)
    (unwind-protect
	(let (initial (not (file-exists-p history-file)))
	  (if initial
	      (vcs-execute-command nil "ci -q -f"
				   (concat "-t" comment-file)
				   working-file history-file)
	    (vcs-execute-command nil "ci -q -f"
				 (concat "\"-m`cat " comment-file "`\"")
				 working-file history-file)))
      (delete-file comment-file))
    (message "Checked in %s..." history-file)
    (vcs-cleanup-after-put history-file working-file)
    t))


;;;
;;; set the comment leader
;;;
;;; This function doesn't have much to do with vcs but its pretty 
;;; handy.
;;; 
;;;
(defun rcs-set-comment-leader (&optional leader)
  "Set the comment leader for the RCS file in the current buffer."
  (interactive)
  (let ((fn (buffer-file-name)))
    (if leader
	t
      (save-excursion
	(goto-char (point-min))
	(if (re-search-forward "$\Log\$\\|$\Log:.*\$" nil t)
	    (progn
	      (beginning-of-line)
	      (setq leader (buffer-substring (point) (match-beginning 0))))))
      (setq leader (read-string "RCS Comment leader: " leader)))
    (rcs-execute-command (file-name-directory fn) "rcs"
			 (concat "'-c" leader "' ")
			 (file-name-nondirectory fn))))

