;; ========================================================================
;; hyerB.el -- Hypertext-style buttons
;; Author          : Mike Williams <mike-w@cs.aukuni.ac.nz>
;; Created On      : Thu Mar 28 13:42:44 1991
;; Last Modified By: Mike Williams
;; Last Modified On: Mon Aug 26 11:23:12 1991
;; RCS Info        : $Revision: 1.23 $ $Locker:  $
;; ========================================================================
;; [[ CheckMeOut ]] [[ CheckMeIn ]]
;; 
;; NOTE: this file must be recompiled if changed.
;;
;; Copyright (C) Mike Williams <mike-w@cs.aukuni.ac.nz> 1991
;;
;; This file is not part of GNU Emacs, but is made available under the
;; same conditions.
;;
;; GNU Emacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
;; to anyone for the consequences of using it or for whether it serves
;; any particular purpose or works at all, unless he says so in writing.
;; Refer to the GNU Emacs General Public License for full details.
;;
;; Everyone is granted permission to copy, modify and redistribute GNU
;; Emacs, but only under the conditions described in the GNU Emacs
;; General Public License.  A copy of this license is supposed to have
;; been given to you along with GNU Emacs so you can know your rights and
;; responsibilities.  It should be in a file named COPYING.  Among other
;; things, the copyright notice and this notice must be preserved on all
;; copies.

;; This package provides support for hypertext-like buttons.  The user may
;; define `button handlers' to handle hyperbutton `events'.  Handlers will
;; usually return nil, indicating that the preconditions for their
;; selection have not been fulfilled -- however, when conditions are right,
;; they may return a lisp form to be evaluated.  A list of handlers is
;; maintained ... the 'determine-hyperB-form function will iterate
;; over these until one returns a form to be evaluated.  The functions
;; hyperB:{add,remove}-{global,local}-handler make installing and
;; uninstalling handlers easier, eg;
;;
;;   (hyperB:remove-global-handler 'hyperB:view-file)
;;
;; One standard hyperbutton handler, 'hyperB:lookup-button, provides
;; an easier and more efficient (although less general) interface for
;; defining buttons.  This function uses an association list of regular
;; expressions to lisp forms to search for a form to evaluate.  See the
;; documentation for variable 'hyperB:global-button-alist for details.
;; 
;; The functions {define,undefine}-{global,local}-hyperB make
;; manipulating (adding to and deleting from) the button-alists a bit
;; easier, eg.  
;;
;;   (define-local-hyperB 'word "CheckMeIn" '(rcs-ci-buffer))
;;   (undefine-local-hyperB 'word "CheckMeIn")

;; Many thanks to Rick Mugridge <rick@cs.aukuni.ac.nz> for the idea.

(require 'thing@pt)

(provide 'hyperB)

;; LCD Archive Entry:
;; hyperB|Mike Williams|mike-w@cs.aukuni.ac.nz
;; |Run lisp code when arbitrary text patterns are clicked on
;; |91-06-21|$Revision: 1.23 $|~/interfaces/hyperB.shar.Z

;;=== Usage ===============================================================
;;
;; Either (require 'hyperB), or use some (or all) of these autoloads:
;; 
;; (autoload 'call-hyperB "hyperB" nil t)
;; (autoload 'determine-hyperB-form "hyperB")
;; (autoload 'x-mouse-hyperB-down "hyperB")
;; (autoload 'x-mouse-hyperB-up "hyperB")
;; 
;; (autoload 'define-global-hyperB "hyperB" nil t)
;; (autoload 'define-local-hyperB "hyperB" nil t)
;; (autoload 'undefine-global-hyperB "hyperB")
;; (autoload 'undefine-local-hyperB "hyperB")
;;
;; And for the adventurous ...
;; (autoload 'hyperB:add-global-handler "hyperB")
;; (autoload 'hyperB:add-local-handler "hyperB")
;; (autoload 'hyperB:remove-global-handler "hyperB")
;; (autoload 'hyperB:remove-local-handler "hyperB")
;;
;; You'll probably want to load hyperB-x as well.  Check out hyperB-x.el
;; for details.

;;=== Version =============================================================

(defconst hyperB:version (substring "$Revision: 1.23 $" 11 -2)
  "The revision number of hyperB (as string).  The complete RCS id is:

  $Id: hyperB.el,v 1.23 1991/08/25 23:23:52 mike-w Exp $")

;;=== How it works ========================================================
;;
;; Determine-hyperB-form runs thru the list of handler functions,
;; calling each in turn until one returns non-nil.  Each handler returns a
;; form to be evaluated, or nil.
;;
;; Call-hyperB evaluates the result of a call to
;; determine-hyperB-form.  The two are separated, as it is reasonable
;; to use determine-hyperB-form elsewhere.  For instance, I use a
;; mouse handler which determines the hyperB form at the point the
;; mouse is clicked, but returns to the original window before evaluating
;; it.  Here's a pair of x-mouse functions which should work with the
;; standard x-mouse.el. 

(cond 
 ((eq window-system 'x)
  
  (defconst hyperB:previous-window nil 
    "Window you were in before mouse-down")
  (defconst hyperB:previous-position (make-marker) 
    "Position in target buffer before mouse-down")

  (defun x-mouse-hyperB-down (arg)
    (setq hyperB:previous-window (selected-window))
    (x-mouse-select arg)
    (set-marker hyperB:previous-position (point))
    (x-mouse-set-point arg))

  (defun x-mouse-hyperB-up (arg)
    (let (form)
      (unwind-protect
	  (setq form (determine-hyperB-form))
	(goto-char (marker-position hyperB:previous-position))
	(select-window hyperB:previous-window)
	(setq x-previous-window nil)
	(set-marker hyperB:previous-position nil))
      (eval form)))

  ))

;; Bind the first to a mouse-down event, and the second to the
;; corresponding mouse-up event.  eg.
;;
;;   (define-key mouse-map x-button-c-left     'x-mouse-hyperB-down) 
;;   (define-key mouse-map x-button-c-left-up  'x-mouse-hyperB-up)
;;   
;; Writing similar functions for use with emacstool (under suntools), or
;; for epoch, should be straightforward.  Alternatively, you could bind
;; call-hyperB to a key. eg.
;;
;;   (global-set-key "\M-+" 'call-hyperB)

;;=== Standard buttons and handlers =======================================
;;
;; * File browsing [hyperB:view-file]
;;   Click on filename in any buffer to view the corresponding file.
;;
;; * Evaluate arbitrary elisp form:
;;   [[ Eval: (message "Hi there") ]]	<-- click me
;;
;; * Find or view a file:
;;   [[ Find: ~/.emacs ]]		<-- click me
;;   [[ View: /etc/motd ]]		<-- click me
;;
;; * Send mail:
;;   [[ Mail: groucho, chico ]]		<-- click me

;;=== Extras ==============================================================
;;
;; Handlers for
;;
;;   - Following #include references in C code;
;;   
;;   - Following (require ...) references in Elisp code;
;;   
;;   - Easy browsing in info;
;;     
;;   - Visiting errors in *Compilation* buffer;
;; 
;;   - Finding tags in C and Elisp source files;
;;
;; should have been included with hyperB in the file hyperB-x.el.

;;=== Ideas ===============================================================
;;
;;  * Define hyperbuttons for commonly executed functions, eg. reading
;;    mail [[ ReadMail ]], compiling an Emacs-Lisp file [[ CompileMe ]],
;;    banging your head against a brick wall [[ BrickWall ]].
;;
;;   (define-global-hyperB 'hyperB "ReadMail" 
;;     '(if (fboundp 'vm) (vm) (rmail)))
;;   (define-global-hyperB 'hyperB "CompileMe" 
;;     '(byte-compile-file buffer-file-name))
;;   (define-global-hyperB 'hyperB "BrickWall" '(doctor))
;;
;;   OR
;;   
;;   (setq hyperB:global-button-alist
;;         (append
;;          hyperB:global-button-alist
;;          (list
;;           '(hyperB
;;             ("\\s +ReadMail\\s +" . (if (fboundp 'vm) (vm) (rmail)))
;;             ("\\s +CompileMe\\s +" . (byte-compile-file buffer-file-name))
;;             ("\\s +BrickWall\\s +" . (doctor))
;;             ))))
;;             
;;  * Create a handler that will popup help on certain keywords when they
;;    are clicked on.
;;  
;;  * Define handlers/buttons for GNUS/VM to allow easy selection of
;;    groups, articles and mail messages.

;;=== Main user functions =================================================

(defun call-hyperB (&optional ARG)
  "Determine and evaluate hyperB form at point.  With a prefix 
argument ARG, evaluate the form in the ARG'th other window (default 1).

see documentation for determine-hyperB-form."
  (interactive "P")
  (if ARG (other-window (if (numberp ARG) ARG 1)))
  (eval (determine-hyperB-form)))

(defun determine-hyperB-form ()
  "Determine hyperB form to be evaluated by calling each of 
hyperB:local-handlers and hyperB:global-handlers until one returns non-nil.
The return value is an elisp form suitable for evaluation using 'eval."
  (let ((button-list (append hyperB:local-handlers
			     hyperB:global-handlers
			     '(hyperB:undefined)))
	return-val)
    (while (and (not return-val) button-list)
      (setq return-val (call-interactively (car button-list)))
      (setq button-list (cdr button-list)))
    return-val))

;;=== Handler variables ===================================================

(defvar hyperB:global-handlers 
  '(hyperB:lookup-button hyperB:view-file)
  "Global list of functions to be called in turn by determine-hyperB-form, 
until one returns a form to be evaluated.
Note that the hyperB:local-handlers take precedence.") 

(defvar hyperB:local-handlers nil
  "Buffer-local list of functions to be called in turn by 
determine-hyperB-form, until one returns a form to be evaluated.
These take precedence over the hyperB:global-handlers.") 
(make-variable-buffer-local 'hyperB:local-handlers)
(put 'hyperB:local-handlers 'preserved t)

;;=== Useful handlers =====================================================

;;--- Default handler ---

(defun hyperB:undefined ()
  (interactive)
  '(message "No matching button"))

;;--- Find/View files ---

(defun hyperB:find-file (FILE &optional VIEW)
  "Find filename at point."
  (interactive (list (thing-at-point 'filename)))
  (let ((path (substitute-in-file-name FILE)))
    (if (file-exists-p path)
	(if VIEW 
	    (` (view-file (, path))) 
	  (` (find-file (, path)))))
    ))

(defun hyperB:view-file (FILE)
  "View filename at point."
  (interactive (list (thing-at-point 'filename)))
  (if hyperB:hypercard-view-mode 
      (hyperB:find-file FILE 'view)
    (hyperB:find-file FILE)))
  
;;=== General button handler ==============================================

;;--- Hyperbutton name extraction ---

(defvar hyperB:start-re (concat (regexp-quote "[[") "[ \t]*"))
(defvar hyperB:end-re   (concat "[ \t]*" (regexp-quote "]]")))

(defun beginning-of-hyperB () 
  "Search backward for hyperB:start-re, and position point at end."
  (re-search-backward hyperB:start-re)
  (re-search-forward hyperB:start-re))

(defun end-of-hyperB () 
  "Search forward for hyperB:end-re, and position point at beginning."
  (re-search-forward hyperB:end-re)
  (re-search-backward hyperB:end-re))

;;--- Button/function association ---

(defvar hyperB:global-button-alist

  '((hyperB
     ("\\`\\s *Eval:" . 
      (message "%s"
	       (eval (read-from-whole-string 
		      (substring hyperB-name (match-end 0))))))
     ("^\\s *Find:\\s *\\(\\S +\\)\\s *$" . 
      (let ((path (substring hyperB-name 
			     (match-beginning 1) (match-end 1))))
	(if (file-exists-p path) ()
	  (if (featurep 'fnexpand)
	      (setq path (read-file-name-internal path nil nil))))
	(find-file path)))
;;     (find-file (substring hyperB-name 
;;		   (match-beginning 1) (match-end 1)))
     ("^\\s *View:\\s *\\(\\S +\\)\\s *$" . 
      (view-file (substring hyperB-name 
			    (match-beginning 1) (match-end 1))))
     ("^\\s *Mail:\\s *\\(.+\\)$" . 
      (mail nil (substring hyperB-name 
			   (match-beginning 1) (match-end 1))))))
  
  "Alist used by hyperB:lookup-button to determine a form to evaluate.

  ((THING 
    (REGEXP . BODY)
    (REGEXP . BODY)
    ...)
   (THING 
    (REGEXP . BODY)
    (REGEXP . BODY)
    ...))

  When the THING at point (cf. thing-at-point) matches associated regular 
expression REGEXP, execute BODY with the symbol 'hyperB-name 
dynamically bound to the THING matched.") 

(defvar hyperB:local-button-alist nil
  "Local alist used by hyperB:lookup-button to determine a form to 
evaluate.  See documentation for hyperB:global-button-alist for 
details.  Note that local definitions take precedence over global ones.")
(make-variable-buffer-local 'hyperB:local-button-alist)
(put 'hyperB:local-button-alist 'preserved t)

(defun hyperB:lookup-button ()
  "Determine a form to be evaluated using hyperB:local-button-alist 
and hyperB:global-button-alist."
  (interactive)
  (catch 'hyperB-form
    (let ((clauses (append hyperB:local-button-alist 
			   hyperB:global-button-alist)))
      (while clauses 
	(let ((button-name (thing-at-point (car (car clauses))))
	      (alist (cdr (car clauses))))
	  (if (not button-name) nil
	    (while alist
	      (if (string-match (car (car alist)) button-name)
		  (throw 'hyperB-form 
			 (` (let ((hyperB-name (, button-name)))
			      (eval (, (cdr (car alist))))))))
	      (setq alist (cdr alist)))))
	(setq clauses (cdr clauses))))))

;;=========================================================================
;;=== Utilities ===========================================================

(defun hyperB:filter (LIST PRED)
  "Return list of elements in LIST for which PRED is true."
  (cond 
   ((not LIST) nil)
   ((funcall PRED (car LIST))
    (cons (car LIST) (hyperB:filter (cdr LIST) PRED)))
   (t (hyperB:filter (cdr LIST) PRED))))

;;=== Add/remove handlers =================================================

(defun hyperB:add-handler (HANDLER-LIST HANDLER)
  "Update HANDLER-LIST to include HANDLER."
  (if (catch 'member
	(mapcar (function (lambda (elt) (if (equal elt HANDLER) 
					    (throw 'member t))))
		(symbol-value HANDLER-LIST))
	nil) nil
    (set HANDLER-LIST (append (symbol-value HANDLER-LIST) (list HANDLER)))))

(defun hyperB:add-global-handler (HANDLER)
  "Update hyperB:global-handlers to include HANDLER."
  (hyperB:add-handler 'hyperB:global-handlers HANDLER))

(defun hyperB:add-local-handler (HANDLER)
  "Update hyperB:local-handlers to include HANDLER."
  (hyperB:add-handler 'hyperB:local-handlers HANDLER))
			   
(defun hyperB:remove-handler (HANDLER-LIST HANDLER)
  "Update HANDLER-LIST to exclude HANDLER."
  (set HANDLER-LIST
       (hyperB:filter 
	(symbol-value HANDLER-LIST)
	(function (lambda (handler) (not (equal handler HANDLER)))))))

(defun hyperB:remove-global-handler (HANDLER)
  "Update hyperB:global-handlers to exclude HANDLER."
  (hyperB:remove-handler 'hyperB:global-handlers HANDLER))

(defun hyperB:remove-local-handler (HANDLER)
  "Update hyperB:local-handlers to exclude HANDLER."
  (hyperB:remove-handler 'hyperB:local-handlers HANDLER))
			
;;=== Easy button definition ==============================================

(defun define-hyperB (ALIST THING REGEXP BODY)
  "Update ALIST so that when the THING at point (cf. thing-at-point) 
matches REGEXP, BODY will be executed with hyperB-name bound to the 
value of the THING matched."
  (let ((entry (list THING (cons REGEXP BODY))))
    (undefine-hyperB ALIST THING REGEXP)
    (set ALIST (append (symbol-value ALIST) (list entry)))))

(defun define-global-hyperB (THING REGEXP BODY)
  "Update hyperB:global-button-alist so that when the THING at point
\(cf. thing-at-point\) matches REGEXP, BODY will be executed with 
hyperB-name bound to the value of the THING matched."
  (interactive 
   (list (read-from-whole-string 
	  (completing-read "Thing to define: " obarray))
	 (read-string "Regular expression: ")
	 (read-from-minibuffer "Form to eval: " nil nil 'read)))
  (define-hyperB 
   'hyperB:global-button-alist THING REGEXP BODY))

(defun define-local-hyperB (THING REGEXP BODY)
  "Update hyperB:local-button-alist so that when the THING at point
\(cf. thing-at-point\) matches REGEXP, BODY will be executed with 
hyperB-name bound to the value of the THING matched."
  (interactive 
   (list (read-from-whole-string 
	  (completing-read "Thing to define: " obarray))
	 (read-string "Regular expression: ")
	 (read-from-minibuffer "Form to eval: " nil nil 'read)))
  (define-hyperB 
   'hyperB:local-button-alist THING REGEXP BODY))

(defun undefine-hyperB (ALIST THING REGEXP)
  "Remove any entry in ALIST for THING matching REGEXP."
  (set ALIST
       (hyperB:filter 
	(mapcar 
	 (function 
	  (lambda (thing-assoc) 
	    (cons (car thing-assoc)
		  (hyperB:filter 
		   (cdr thing-assoc)
		   (function (lambda (regexp-assoc)
			       (not (equal (car regexp-assoc) REGEXP))))))))
	 (symbol-value ALIST))
	(function (lambda (thing-assoc)
		    (not (null (cdr thing-assoc))))))))

(defun undefine-global-hyperB (THING REGEXP)
  "Remove global hyperB entry for THING matching REGEXP."
  (undefine-hyperB 
   'hyperB:global-button-alist THING REGEXP))

(defun undefine-local-hyperB (THING REGEXP)
  "Remove local hyperB entry for THING matching REGEXP."
  (undefine-hyperB 
   'hyperB:local-button-alist THING REGEXP))

;;=== That's all folks ====================================================

(run-hooks 'hyperB-load-hooks)

;;=== END of hyperB.el ====================================================
