/*  $Revision: 1.1.1.1 $
**
**  Parse input to add to news overview database.
*/
#include <stdio.h>
#include <sys/types.h>
#include "configdata.h"
#include "clibrary.h"
#include <errno.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <fcntl.h>
#include "libinn.h"
#include "macros.h"
#include "paths.h"
#include "logging.h"
#include "qio.h"
#include "tree.h"

STATIC BOOL	InSpoolDir;

typedef struct _OVERLINE
{   char *newsgroup;
    long artnumber;
    char *line;
    int  size;
} OVERLINE;

#define MAXOVERLINES 	20000
#define BUFFERSIZE	5000000
#define MAXFLUSHTIME	180
#define MAX_IOV_LEN	16

STATIC BOOL WasAlarm;
OVERLINE overlines[MAXOVERLINES];
STATIC char msgid_ident[1024];

tree *newsgroups;
int  overlinesCount;
char *overlinesBuffer;
char *bufferPos;
time_t lastFlush;


/*
**  Try to make one directory.  Return FALSE on error.
*/
STATIC BOOL
MakeDir(Name)
    char		*Name;
{
    struct stat		Sb;

    if (mkdir(Name, GROUPDIR_MODE) >= 0)
	return TRUE;

    /* See if it failed because it already exists. */
    return stat(Name, &Sb) >= 0 && S_ISDIR(Sb.st_mode);
}


/*
**  Make overview directory if not in spool directory.  Return 0 if ok,
**  else -1.
*/
STATIC BOOL
MakeOverDir(Name)
    register char	*Name;
{
    register char	*p;
    BOOL		made;

    if (InSpoolDir)
	return FALSE;

    /* Optimize common case -- parent almost always exists. */
    if (MakeDir(Name))
	return TRUE;

    /* Try to make each of comp and comp/foo in turn. */
    for (p = Name; *p; p++)
	if (*p == '/') {
	    *p = '\0';
	    made = MakeDir(Name);
	    *p = '/';
	    if (!made)
		return FALSE;
	}

    return MakeDir(Name);
}


int overlinesCompare(ovl1, ovl2)
    OVERLINE *ovl1;
    OVERLINE *ovl2;
{
    if( ovl1->newsgroup<ovl2->newsgroup )
        return -1;
    else if( ovl1->newsgroup>ovl2->newsgroup )
        return 1;
    else if( ovl1->artnumber<ovl2->artnumber )
        return -1;
    else if( ovl1->artnumber>ovl2->artnumber )
        return 1;

    return 0;
}
    

STATIC void
flushBuffer()
{
    static struct iovec	*iov=NULL;
    static int		iovmax;
    struct iovec	*iovp;
    int			iovc;    
    register int	fd;
    register int	i, j;
    char		file[SPOOLNAMEBUFF];
    char		dir[SPOOLNAMEBUFF];
    struct stat		Sb;
    int			savedSeeks=0;
    time_t		theTime;
    
    syslog(L_TRACE, "start flushing (buffer:%d lines:%d seconds:%d)", 
    			bufferPos-overlinesBuffer, overlinesCount,
			time(&theTime)-lastFlush);
    
    if( iov==NULL )
    {   iovmax=32;
    
    	if( (iov=malloc(iovmax*sizeof(struct iovec)))==NULL )        
	{   (void)fprintf(stderr, "overchan cant malloc iov %s\n", 
			    strerror(errno));
	    exit(1);
	}
    }
    
    qsort(&overlines, overlinesCount, sizeof(OVERLINE), overlinesCompare);
	
    iovp=iov;
    iovc=0;
        
    for( i=0; i<overlinesCount; i++ )
    {   iovp->iov_base=overlines[i].line;
	iovp->iov_len=overlines[i].size;

	iovp++;
	iovc++;

	if( i==overlinesCount-1 
    	||  overlines[i].newsgroup!=overlines[i+1].newsgroup )
        {   /* Name the data file. */
	    sprintf(file, "%s/%s", overlines[i].newsgroup, _PATH_OVERVIEW);
	
	    /* Open and lock the file. */
	    for ( ; ; )
	    {	if ((fd = open(file, O_WRONLY | O_CREAT 
				| O_APPEND, ARTFILE_MODE)) < 0) {
		    if (errno == ENOENT)
		    {	strcpy(dir, overlines[i].newsgroup);
		    	MakeOverDir(dir);
			if ((fd = open(file, O_WRONLY | O_CREAT 
				| O_APPEND, ARTFILE_MODE)) < 0)
			{ (void)fprintf(stderr, "overchan cant open %s, %s\n",
				file, strerror(errno));
		    
			    break;
			}
		    }
		}
		if (LockFile(fd, FALSE) < 0)
		    /* Wait for it. */
		    (void)LockFile(fd, TRUE);
		else
		{   /* Got the lock; make sure the file is still there. */
		    if (fstat(fd, &Sb) < 0) {
			(void)fprintf(stderr, "overchan cant fstat %s, %s\n",
				file, strerror(errno));
			(void)close(fd);
			break;;
		    }
		    if (Sb.st_nlink > 0)
			break;
		}
	/* Close file -- expireover might have removed it -- and try again. */
		(void)close(fd);
	    }

	    savedSeeks+=iovc-1;
#if 1
	    /*ok = TRUE;*/
	    iovp=iov;
	    while( iovc>0 )
	    {   j=(iovc<MAX_IOV_LEN ? iovc : MAX_IOV_LEN);

	    	if (xwritev(fd, iovp, j) < 0) 
	    	{
		    (void)syslog(L_ERROR, "cant write %s %s\n",
			    file, strerror(errno));
		    /*ok = FALSE;*/
		}
		
		iovp+=j;
		iovc-=j;
	    }
#else
	    if (xwritev(fd, iov, iovc) < 0) 
	    {   (void)fprintf(stderr, "overchan cant write %s %s\n",
			file, strerror(errno));
	    }
#endif
	    /* Close up and return. */
	    if (close(fd) < 0) {
		(void)fprintf(stderr, "overchan cant close %s %s\n",
			file, strerror(errno));
		/*ok = FALSE;*/
	    }
	    
	    iovp=iov;
	    iovc=0;
	}

	if( iovc>=iovmax )
	{   iovmax*=2;
		
	    if( (iov=realloc(iov, iovmax*sizeof(struct iovec)))==NULL )
	    {   (void)fprintf(stderr, "overchan cant malloc iov %s\n", 
				strerror(errno));
		exit(1);
	    }
	    iovp=iov;
	    iovp+=iovc;
	}	
    }
    
    syslog(L_TRACE, "saved %d of %d open()/close()", savedSeeks, overlinesCount);
    overlinesCount=0;
    bufferPos=overlinesBuffer;
    time(&lastFlush);
    
/*    return ok;*/
}

STATIC void
nop()
{
}

/*
**  Get the lock for the group, then open the data file and append the
**  new data.  Return FALSE on error.
*/
STATIC void
WriteData(Dir, Art, Rest)
    char		*Dir;
    char		*Art;
    char		*Rest;
{
    time_t	theTime;
    char	*newsgroup;
    
    if( (newsgroup=tree_srch(&newsgroups, strcmp, Dir))==NULL )
    {	newsgroup=COPY(Dir);
	tree_add(&newsgroups, strcmp, newsgroup, nop);
    }
    
    sprintf(bufferPos, "%s\t%s\n", Art, Rest);
    
    overlines[overlinesCount].newsgroup=newsgroup;
    overlines[overlinesCount].artnumber=atol(Art);
    overlines[overlinesCount].line=bufferPos;
    overlines[overlinesCount].size=strlen(bufferPos);

    bufferPos+=overlines[overlinesCount].size+1;
    overlinesCount++;

    if( bufferPos-overlinesBuffer>BUFFERSIZE-QIO_BUFFER 
    ||	overlinesCount>=MAXOVERLINES
    ||  (time(&theTime)-lastFlush>MAXFLUSHTIME && overlinesCount)
    ||  strstr(Rest, msgid_ident)!=0 )
        flushBuffer();
}

STATIC void
CATCHalarm(sig)
    int sig;
{
    time_t theTime;
    
    if( time(&theTime)-lastFlush>MAXFLUSHTIME && overlinesCount )
    	flushBuffer();

    signal(SIGALRM, CATCHalarm);
    alarm(10);
    WasAlarm = TRUE;
}


/*
**  Process the input.  Data can come from innd:
**	news/group/name/<number> [space news/group/<number>]... \t data
**  or from mkov:
**	news/group/name \t number \t data
*/
STATIC void
ProcessIncoming(INNinput, qp)
    BOOL		INNinput;
    QIOSTATE		*qp;
{
    register char	*Dir;
    register char	*Art;
    register char	*Rest;
    register char	*p;

    for ( ; ; ) {
	/* Read the first line of data. */
		
	alarm(10);
	if ((p = QIOread(qp)) == NULL) {
	    if( errno==EINTR )
	    {	if( WasAlarm == TRUE)
		{   WasAlarm = FALSE;
		    continue;
		}
	    }

	    alarm(0);
	    if (QIOtoolong(qp)) {
		(void)fprintf(stderr, "overchan line too long\n");
		continue;
	    }
	    break;
	}
	alarm(0);
	
	/* If doing mkov output, process that line and continue. */
	if (!INNinput) {
	    /* Split up the fields. */
	    Dir = p;
	    if ((p = strchr(p, '\t')) == NULL || p[1] == '\0') {
		(void)fprintf(stderr, "overchan missing field 1: %s\n", Dir);
		return;
	    }
	    *p++ = '\0';
	    Art = p;
	    if ((p = strchr(p, '\t')) == NULL || p[1] == '\0') {
		(void)fprintf(stderr, "overchan missing field 2: %s\n", Dir);
		return;
	    }
	    *p++ = '\0';

	    /* Write data. */
	    WriteData(Dir, Art, p);

	    continue;
	}

	/* Nip off the first part. */
	Dir = p;
	if ((Rest = strchr(p, '\t')) == NULL) {
	    (void)fprintf(stderr, "overchan bad input\n");
	    continue;
	}
	*Rest++ = '\0';

	/* Process all fields in the first part. */
	for ( ; *Dir; Dir = p) {

	    /* Split up this field, then split it up. */
	    for (p = Dir; *p; p++)
		if (ISWHITE(*p)) {
		    *p++ = '\0';
		    break;
		}

	    if ((Art = strrchr(Dir, '/')) == NULL || Art[1] == '\0') {
		(void)fprintf(stderr, "overchan bad entry %s\n", Dir);
		continue;
	    }
	    *Art++ = '\0';

	    /* Write data. */
	    WriteData(Dir, Art, Rest);
	}
    }

    flushBuffer();

    if (QIOerror(qp))
	(void)fprintf(stderr, "overchan cant read %s\n", strerror(errno));
    QIOclose(qp);
}

STATIC void
Setup()
{
    overlinesBuffer=calloc(BUFFERSIZE, sizeof(char));
    bufferPos=overlinesBuffer;
    
    if( !overlinesBuffer )
    {	(void)fprintf(stderr, "overchan cant malloc buffer %s\n", 
    			strerror(errno));
	exit(1);
    }

    signal(SIGALRM, CATCHalarm);
    tree_init(&newsgroups);
    time(&lastFlush);
    overlinesCount=0;
    WasAlarm = FALSE;
    sprintf(msgid_ident, "@%s>\t", GetFQDN());

    openlog("overchan", L_OPENLOG_FLAGS | LOG_PID, LOG_INN_PROG);
}

STATIC NORETURN
Usage()
{
    (void)fprintf(stderr, "usage:  overchan [-c] [-D dir] [files...]\n");
    exit(1);
}


int
main(ac, av)
    int			ac;
    char		*av[];
{
    register int	i;
    QIOSTATE		*qp;
    char		*Dir;
    BOOL		INNinput;

    /* Set defaults. */
    Dir = _PATH_OVERVIEWDIR;
    INNinput = TRUE;
    (void)umask(NEWSUMASK);

    Setup();

    /* Parse JCL. */
    while ((i = getopt(ac, av, "cD:")) != EOF)
	switch (i) {
	default:
	    Usage();
	    /* NOTREACHED */
	case 'c':
	    INNinput = FALSE;
	    break;
	case 'D':
	    Dir = optarg;
	    break;
	}
    ac -= optind;
    av += optind;
    InSpoolDir = EQ(Dir, _PATH_SPOOL);

    if (chdir(Dir) < 0) {
	(void)fprintf(stderr, "overchan cant chdir %s %s\n",
		Dir, strerror(errno));
	exit(1);
    }

    if (ac == 0)
	ProcessIncoming(INNinput, QIOfdopen(STDIN, QIO_BUFFER));
    else {
	for ( ; *av; av++)
	    if (EQ(*av, "-"))
		ProcessIncoming(INNinput, QIOfdopen(STDIN, QIO_BUFFER));
	    else if ((qp = QIOopen(*av, QIO_BUFFER)) == NULL)
		(void)fprintf(stderr, "overchan cant open %s %s\n",
			*av, strerror(errno));
	    else
		ProcessIncoming(INNinput, qp);
    }

    exit(0);
    /* NOTREACHED */
}
