/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

#include <stdio.h>
#include <sys/param.h>
#include "hostenv.h"
#include "scheduler.h"
#include "mail.h"
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <sys/file.h>
#include NDIR_H
#include "zmsignal.h"

#include "prototypes.h"

static int scheduler_nofiles = -1; /* Will be filled below */
static void dotransport __((char **, int, struct vertex *, char*, char*));
static void runcommand __((char **, int, char **, struct vertex *, struct web *, struct web *));
static void stashprocess __((int, int, int, struct web *, struct web *, struct vertex *));
static void reclaim __((int));
static void waitandclose __((int));
static void readfrom __((int));

#ifdef	USE_UNIONWAIT
#include <sys/wait.h>
#endif
#ifdef	USE_WAITPID
#include <sys/wait.h>
#endif

struct procinfo *cpids = NULL;

#define	MAXFILESPERTRANSPORT	1000

int	numkids = 0;

extern char *getzenv();
extern char *strchr();
extern int cistrcmp(), ranny();
extern int epipe();
extern int errno;
extern time_t time();


/*
 * Collect up all the marked control file names (unmarking them in the process)
 * and send them to the command given by the parameter. Said command must be
 * forked off, and the appropriate structures set up to deal with the stdout
 * of that command.
 */

static int t_n, t_nmax;
static char **t_files;

static int tscan(spl)
	struct spblk *spl;
{
	struct ctlfile *cfp = (struct ctlfile *)spl->data;

	/* assert cfp != NULL */

	/* Is this marked ? */
	if (cfp->mark == V_NONE)
	  return -1; /* .. No. */

	/* If the count of jobs exceeds our high guesses.. */
	if (t_n >= t_nmax) {
	  /* Report only once per scan.. */
	  if (t_nmax > 0)
	    printf("**AARGH**  MAXFILESPERTRANSPORT(%d) TURNED OUT TO BE TOO SMALL!\n", t_nmax);
	  t_nmax = 0;
	  /* Clear the markings.. */
	  cfp->mark    = V_NONE;
	  return -1;
	}

	t_files[t_n++] = cfp->mid;
	cfp->mark    = V_NONE;

	return 0;
}

void
transport(vhead, channel, host)
	struct vertex *vhead;
	char *channel, *host;
{
	int i;
	char *files[MAXFILESPERTRANSPORT];	/* XX */

	/* generate the file names we care about */
	t_n     = 0;
	t_nmax  = MAXFILESPERTRANSPORT;
	t_files = files;

	if (verbose)
	  printf("transport(vhead,chan=\"%s\",host=\"%s\")\n",channel,host);

	sp_scan(tscan, (struct spblk *)NULL, spt_mesh[L_CTLFILE]);
	dotransport(files, t_n, vhead, channel, host);
}

static void dotransport(files, n, vhead, channel, host)
	char *files[];
	int n;
	struct vertex *vhead;
	char *channel, *host;
{
	struct spblk *spl;
	struct web *chwp, *howp;
	char	*av[30], *s, *os, *cp, *ocp, buf[MAXPATHLEN];
	int	i;

	if (n == 0)
		return;
	chwp = NULL;
	if (channel != NULL) {
	  spl = sp_lookup(symbol((u_char *)channel), spt_mesh[L_CHANNEL]);
	  if (spl == NULL || (chwp = (struct web *)spl->data) == NULL)
	    return;		/* yes, this can happen */
	}
	howp = NULL;
	if (host != NULL) {
	  spl = sp_lookup(symbol((u_char *)host), spt_mesh[L_HOST]);
	  if (spl == NULL || (howp = (struct web *)spl->data) == NULL)
	    return;		/* yes, this can happen */
	}
	if (vhead->ce.argv == NULL) {
	  fprintf(stderr, "No command defined for %s/%s!\n", channel, host);
	  return;
	}
	/*
	 * Replace the $host and $channel strings in the command line.
	 */
	os = buf;
	for (i = 0; vhead->ce.argv[i] != NULL; ++i) {
	  if (cistrcmp(vhead->ce.argv[i], replhost) == 0) {
	    av[i] = host;
	  } else if (cistrcmp(vhead->ce.argv[i], replchannel) == 0) {
	    av[i] = channel;
	  } else if (strchr(vhead->ce.argv[i], '$') != NULL) {
	    s = os;
	    for (cp = vhead->ce.argv[i]; *cp != '\0'; ++cp) {
	      if (*cp == '$' && *(cp+1) == '{') {
		cp += 2;
		ocp = cp;
		while (*cp != '\0' && *cp != '}')
		  ++cp;
		if (*cp == '}') {
		  *cp = '\0';
		  strcpy(s, getzenv(ocp));
		  s += strlen(s);
		  *cp = '}';
		} else
		  --cp;
	      } else
		*s++ = *cp;
	    }
	    *s = '\0';
	    av[i] = os;
	    os = s + 1;
	  } else
	    av[i] = vhead->ce.argv[i];
	}
	av[i] = NULL;
	/*
	 * randomize the file names if appropriate, this is a simpleminded
	 * but probably effective way of avoiding a single message blocking
	 * some queue.  XX: This should really be expanded to invoke a general
	 * strategy routine/mechanism.
	 */
	if (n > 1) {
	  char *temp;
	  u_int ni;

	  for (i = 0; i < n; ++i) {
	    ni = ranny(n-1);
	    temp = files[i];
	    files[i] = files[ni];
	    files[ni] = temp;
	  }
	}
	/* fork off the appropriate command with the appropriate stdin */
	if (verbose) {
	  printf("$");
	  for (i = 0; av[i] != NULL; ++i)
	    printf(" %s", av[i]);
	  printf(" <<\\EOF\n");
	  for (i = 0; i < n; ++i)
	    printf("%s\n", files[i]);
	  printf("\\EOF\n");
	}
	runcommand(files, n, av, vhead, chwp, howp);
}

static void runcommand(filearr, nelem, argv, vhead, chwp, howp)
	char	*filearr[];
	int	nelem;
	char	*argv[];
	struct vertex *vhead;
	struct web *chwp, *howp;
{
	int	i, pid, position, to[2], from[2], uid, gid;
	char	*cmd;
	FILE	*fp;
	static int pipesize = 0;


	uid = vhead->ce.uid;
	gid = vhead->ce.gid;
	cmd = vhead->ce.command;

	if (nelem <= 0)
	  return;
	if (epipe(to) < 0)
	  return;
	if (epipe(from) < 0) {
	  close(to[0]);
	  close(to[1]);
	  return;
	}
	if (pipesize == 0)
	  pipesize = resources_query_pipesize(to[0]);

	if (verbose)
	  fprintf(stderr, "to %d/%d from %d/%d\n",
		  to[0],to[1],from[0],from[1]);
	if ((pid = fork()) == 0) {	/* child */
	  int topfileno;
	  if (to[0] != 0) {
	    dup2(to[0],0);	/* child stdin */
	    close(to[0]);
	  }
	  if (from[1] != 1) {
	    dup2(from[1],1);	/* child stdout */
	    close(from[1]);
	  }
	  /* dup2(1,2); */

	  /* keep current stderr for child stderr */
	  /* close all other open filedescriptors */

	  /* ... if detach() did its job, there shouldn't be any! */
	  /* ... no, the 'querysock' is there somewhere!   */
	  if (scheduler_nofiles < 1)
	    scheduler_nofiles = resources_query_nofiles();
	  for (i = 3; i < scheduler_nofiles; ++i)
	    close(i);

	  resources_limit_nofiles(transportmaxnofiles);
	  setgid(gid);	/* Do GID setup while still UID 0..   */
	  setuid(uid);	/* Now discard all excessive powers.. */
	  execv(cmd, argv);
	  fprintf(stderr, "Exec of %s failed!\n", cmd);
	  _exit(1);
	} else if (pid < 0) {	/* fork failed - yell and forget it */
	  close(to[0]); close(to[1]);
	  close(from[0]); close(from[1]);
	  fprintf(stderr, "Fork failed!\n");
	  return;
	} else {		/* parent */
	  close(to[0]);
	  close(from[1]);
	  ++numkids;
	  if (chwp != NULL) chwp->kids += 1;
	  if (howp != NULL) howp->kids += 1;
	  /* save from[0] away as a descriptor to watch */
	  stashprocess(pid, from[0], to[1], chwp, howp, vhead);
	  /* write the list of files into to[1], and close it */
	  if ((fp = fdopen(to[1], "w")) == NULL) {
	    if (verbose)
	      fprintf(stderr, "cannot open fd %d for w.\n",
		      to[1]);
	    close(to[1]);
	    close(from[0]);
	    return;
	  }
	  for (i = position = 0; i < nelem; ++i) {
	    position += strlen(filearr[i]) + 1;
	    if (position + 1 >= pipesize)
	      break;
	    fprintf(fp, "%s\n", filearr[i]);
	  }
	  if (verbose)
	    fprintf(stderr, "wrote %d bytes to fd %d\n",
		    position, fileno(fp));
	  fputc('\n', fp);
	  fclose(fp);	/* This will also close the to[1] fd */
	}
}

static void stashprocess(pid, fromfd, tofd, chwp, howp, vhead)
	int pid, fromfd, tofd;
	struct web *chwp, *howp;
	struct vertex *vhead;
{
	int i;
	struct procinfo *proc;

	if (verbose)
	  fprintf(stderr, "stashprocess(%d, %d, %d, %s, %s)\n",
		  pid, fromfd, tofd, chwp ? chwp->name : "nil",
		  howp ? howp->name : "nil");
	if (cpids == NULL) {
	  if (scheduler_nofiles < 1)
	    scheduler_nofiles = resources_query_nofiles();
	  i = scheduler_nofiles;
	  cpids = (struct procinfo *)
	    emalloc((unsigned)(i * sizeof (struct procinfo)));
	  while (--i >= 0)
	    cpids[i].pid = 0;
	}

	proc = &cpids[fromfd];

	proc->pid    = pid;
	proc->reaped = 0;
	proc->ch     = chwp;
	proc->ho     = howp;
	proc->vertices = vhead;
	proc->tofd   = tofd;
	vhead->proc  = proc;
	proc->carryover = NULL;
	proc->hungry = 1;
#ifdef O_NONBLOCK /* POSIXy thing */
	/* set up non-blocking I/O */
	i = fcntl(fromfd, F_GETFL, 0);
	i |= O_NONBLOCK;
	i = fcntl(fromfd, F_SETFL, i);
#else
#ifdef	USE_BSDNODELAY
#ifdef	FNONBLOCK
	i |= FNONBLOCK;
#else
	i |= FNDELAY;
#endif
	i = fcntl(fromfd, F_SETFL, i);
#endif	/* USE_BSDNODELAY */
#endif	/* O_NONBLOCK  */
}

static void reclaim(fromfd)
	int fromfd;
{
	struct vertex *vp, *nvp;

	struct procinfo *proc = &cpids[fromfd];

	proc->pid = 0;
	proc->reaped = 0;
	if (proc->carryover != NULL) {
	  fprintf(stderr, "%s: HELP! Lost %d bytes: '%s'\n",
		  progname, strlen(proc->carryover),
		  proc->carryover);
	  free(proc->carryover);
	  proc->carryover = NULL;
	}
	if (proc->ch != NULL) {
	  --proc->ch->kids;
	  if (proc->ch->fifohead)
	    fifoschedule(proc->ch);
	  else if (proc->ch->kids == 0 && proc->ch->link == NULL) {
	    unweb(L_CHANNEL, proc->ch);
	    proc->ch = NULL;
	  }
	}
	if (proc->ho != NULL) {
	  --proc->ho->kids;
	  if (proc->ho->fifohead)
	    fifoschedule(proc->ho);
	  else if (proc->ho->kids == 0 && proc->ho->link == NULL) {
	    unweb(L_HOST, proc->ho);
	    proc->ho = NULL;
	  }
	}
	close(fromfd);

	/* Reschedule the vertices that are left
	   (that were not reported on).		*/

	nvp = proc->vertices;
	proc->vertices = NULL;
	for (vp = nvp; vp != NULL ; vp = nvp) {
	  nvp = vp->nextitem;
	  if (vp->ngroup == 0)
	    abort();
	  reschedule(vp, 0, -1);
	}
}

static void waitandclose(fd)
	int	fd;
{
#ifndef USE_SIGREAPER
	int	n, pid;

#if 0
	/* This is a quick and dirty hack (for A/UX I think). BEWARE */
	if (cpids[fd].pid <= 0) {
	  fprintf(stderr,"fd %d: pid is %d ??\n",fd,cpids[fd].pid);
	  return;
	}
	n = waitpid(cpids[fd].pid, (int *)NULL, 0);
	if(n != cpids[fd].pid && (n != -1 || errno != 10)) {
	  fprintf(stderr,"fd %d, pid %d: waitpid says %d, errno %d\n",
		  fd,cpids[fd].pid, n, errno);
	  return;
	}
	--numkids;
	reclaim(fd);
#else
	if (cpids[fd].pid > 0) {
#ifdef	USE_UNIONWAIT
	  while ((pid = wait((union wait *)0)) > 0)
#else	/* !USE_UNIONWAIT */
	  while ((pid = wait((int *)0)) > 0)
#endif	/* USE_UNIONWAIT */
	    {
	      --numkids;
	      if (verbose)
		printf("reaped %d\n", pid);
	      if (pid == cpids[fd].pid) {
		reclaim(fd);
		break;
	      } else {
		for (n =; n < scheduler_nofiles; ++n) {
		  if (cpids[n].pid == pid) {
		    cpids[n].pid = -pid;
		    ++numkids;
		    break;
		  }
		}
	      }
	    }
	} else if (cpids[fd].pid < 0) {
	  --numkids;
	  reclaim(fd);
	}
#endif
#else	/* USE_SIGREAPER */
	/* This is called when
	   - fd return 0 (EOF)
	   - fd returns -1, and errno != {EAGAIN|EWOULDBLOCK|EINTR}
	 */
	--numkids;
	reclaim(fd);
#endif
}

#ifdef	USE_SELECT

#if	defined(BSD4_3) || defined(sun)
#include <sys/file.h>
#endif
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/time.h>

#ifndef	NFDBITS
/*
 * This stuff taken from the 4.3bsd /usr/include/sys/types.h, but on the
 * assumption we are dealing with pre-4.3bsd select().
 */

typedef long	fd_mask;

#ifndef	NBBY
#define	NBBY	8
#endif	/* NBBY */
#define	NFDBITS		((sizeof fd_mask) * NBBY)

/* SunOS 3.x and 4.x>2 BSD already defines this in /usr/include/sys/types.h */
#ifdef	notdef
typedef	struct fd_set { fd_mask	fds_bits[1]; } fd_set;
#endif	/* notdef */

#ifndef	FD_SET
#define	FD_SET(n, p)	((p)->fds_bits[0] |= (1 << (n)))
#define	FD_CLR(n, p)	((p)->fds_bits[0] &= ~(1 << (n)))
#define	FD_ISSET(n, p)	((p)->fds_bits[0] & (1 << (n)))
#define FD_ZERO(p)	bzero((char *)(p), sizeof(*(p)))
#endif	/* !FD_SET */
#endif	/* !NFDBITS */

#ifdef _AIX
#include <sys/types.h>
#include <sys/select.h>
	struct {
		fd_set  fdsmask;
		fd_set  msgsmask;
	} readfds, writefds;
#define _FD_SET(sock,var) FD_SET(sock,&var.fdsmask)
#define _FD_CLR(sock,var) FD_CLR(sock,&var.fdsmask)
#define _FD_ZERO(var) FD_ZERO(&var.fdsmask)
#define _FD_ISSET(i,var) FD_ISSET(i,&var.fdsmask)
#else
#ifdef FD_SET
#define _FD_SET(sock,var) FD_SET(sock,&var)
#define _FD_CLR(sock,var) FD_CLR(sock,&var)
#define _FD_ZERO(var) FD_ZERO(&var)
#define _FD_ISSET(i,var) FD_ISSET(i,&var)
#else
#define _FD_SET(sock,var) var |= (1 << sock)
#define _FD_CLR(sock,var) var &= ~(1 << sock)
#define _FD_ZERO(var) var = 0
#define _FD_ISSET(i,var) ((var & (1 << i)) != 0)
#endif
#endif

int
mux(timeout)
time_t timeout;
{
	int	i, n, rc, maxf;
#ifdef _AIX
	struct { fd_set fdsmask; fd_set msgsmask; } mask;
#else
	fd_set	mask;
#endif
	struct sockaddr_in raddr;
	int	raddrlen;
	struct timeval tv;

	time(&now);

	tv.tv_sec = timeout - now;
	if (timeout < now)
	  tv.tv_sec = 0;
	tv.tv_usec = 0;

	maxf = -1;
	_FD_ZERO(mask);
	if (cpids != NULL)
	  for (i = 0; i < scheduler_nofiles ; ++i)
	    if (cpids[i].pid != 0) {
	      _FD_SET(i, mask);
	      maxf = i;
	    }
	if (querysocket >= 0) {
	  _FD_SET(querysocket, mask);
	  if (maxf < querysocket)
	    maxf = querysocket;
	}
	if (maxf < 0)
	  return -1;
	++maxf;
	/*fprintf(stderr, "about to select on %x [%d]\n",
			  mask.fds_bits[0], maxf); */
	if ((n = select(maxf, &mask, NULL, NULL, &tv)) < 0) {
	  /* fprintf(stderr, "got an interrupt (%d)\n", errno); */
	  if (errno == EINTR)
	    return 0;
	} else if (n == 0) {
	  /* fprintf(stderr, "abnormal 0 return from select!\n"); */
	  /* -- just a timeout -- fast or long */
	  return 1;
	} else {
	  /*fprintf(stderr, "got %d ready (%x)\n", n, mask.fds_bits[0]);*/
	  if (querysocket >= 0 && _FD_ISSET(querysocket, mask)) {
	    --n;
	    _FD_CLR(querysocket, mask);
	    raddrlen = sizeof raddr;
	    i = accept(querysocket, (struct sockaddr *)&raddr, &raddrlen);
	    if (i < 0) {
	      perror("accept");
	    } else {
	      rc = fork();
	      if (rc == 0) { /* Child! */
		close(querysocket);
		qprint(i);
		execl("/bin/false",NULL);
		_exit(0);
	      }
	      if (rc > 0)
		++numkids;
	      close(i);
	    }
	  }
	  if (cpids != NULL)
	    for (i = 0; i < maxf; ++i) {
	      if (cpids[i].pid != 0 && _FD_ISSET(i, mask)) {
		--n;
		_FD_CLR(i, mask);
		/*fprintf(stderr,"that is fd %d\n",i);*/
		/* do non-blocking reads from this fd */
		readfrom(i);
	      }
	    }
	}
	/* fprintf(stderr, "return from mux\n"); */
	return 0;
}

void
queryipcinit()
{
#ifdef	AF_INET
	struct servent *serv;
	struct sockaddr_in sad;
	int on = 1;

	if (querysocket >= 0)
		return;
	time(&now);
	if ((serv = getservbyname("mailq", "tcp")) == NULL) {
	  fprintf(stderr, "No 'mailq' tcp service defined!\n");
	  /* try again in 5 minutes or so */
	  qipcretry = now + 300;
	  return;
	}
	qipcretry = now + 5;
	sad.sin_port        = serv->s_port;
	sad.sin_family      = AF_INET;
	sad.sin_addr.s_addr = htonl(INADDR_ANY);
	if ((querysocket = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	  perror("socket");
	  return;
	}
	setsockopt(querysocket, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));

	if (bind(querysocket, (struct sockaddr *)&sad, sizeof sad) < 0) {
	  perror("bind");
	  close(querysocket);
	  querysocket = -1;
	  return;
	}
	if (listen(querysocket, 5) < 0) {
	  perror("listen");
	  close(querysocket);
	  querysocket = -1;
	  return;
	}
	qipcretry = 0;
#endif	/* AF_INET */
}
#else	/* !HAVE_SELECT */
int
mux(timeout)
time_t timeout;
{
	int	fd;

	/*
	 * Nice'n easy and simpleminded: grab a random file descriptor,
	 * and sit and read off it until something happens.
	 * Some very complicated mux'ing schemes (with shared pipes'n stuff)
	 * are possible in the absence of async i/o like select() or the
	 * simulation USG supplies, but it ain't worth the hassle.
	 */
	if (cpids != NULL)
	  for (fd = 0; fd < scheduler_nofiles ; ++fd)
	    if (cpids[fd].pid != 0)
	      readfrom(fd);

	time(&now);
	if (timeout > now)
	  sleep(1);
	return 1;
}

void
queryipcinit()
{
}
#endif	/* HAVE_SELECT */

static void readfrom(fd)
	int fd;
{
	int	n, dontbreak, bufsize = 2048;
	char	*cp, *pcp, *eobuf, *buf;

	dontbreak = 0;
	cp = pcp = NULL;

	buf = (char *)emalloc(bufsize);

	if (cpids[fd].carryover != NULL) {
	  int carrylen = strlen(cpids[fd].carryover);
	  if (carrylen > bufsize) {
	    while (carrylen > bufsize)
	      bufsize += 1024;
	    buf = erealloc(buf,bufsize);
	  }
	  strcpy(buf, cpids[fd].carryover);
	  cp = buf+strlen(buf);
	  pcp = buf;
	  free(cpids[fd].carryover);
	  cpids[fd].carryover = NULL;
	  dontbreak = 1;
	}
	/* Note that if we get an alarm() call, the read will return -1, TG */
	errno = 0;
	while ((n = read(fd, dontbreak ? cp : buf,
			 bufsize - (dontbreak ? (cp - buf) : 0))) > 0) {
	  if (verbose)
	    fprintf(stderr, "read from %d returns %d\n", fd, n);
	  eobuf = (dontbreak ? cp : buf) + n;

	  for (cp = buf, pcp = buf; cp < eobuf; ++cp) {
	    if (*cp == '\n') {
	      *cp = '\0';
	      if (verbose)
		fprintf(stderr, "%d processed: %s\n",
			fd, pcp);
	      update(fd,pcp);
	      *cp = '_';
	      pcp = cp + 1;
	      dontbreak = 0;
	    } else
	      dontbreak = 1;
	  }
	  if (dontbreak && cp == buf + bufsize) {
	    if (pcp == buf) {
	      /* XX:
	       * can't happen, this would mean a status report line 
	       * that is rather long...
	       * (oh no! it did happen, it did, it did!...)
	       */
	      bufsize += 1024;
	      pcp = buf = erealloc(buf,bufsize);
	      cp = buf + (bufsize - 1024);
	      *cp = '\0';
	    } else {
	      memcpy(buf, pcp, cp-pcp);
	      cp = buf + (cp-pcp);
	      *cp = '\0';
	      pcp = buf;	/* may be used below */
	    }
	  }
#ifndef	USE_BSDNODELAY
	  if (!dontbreak)
	    break;
#endif				/* !USE_BSDNODELAY */
	}
	if (verbose) {
	  fprintf(stderr, "read from %d returns %d, errno=%d\n", fd, n, errno);
	}
	if (n == 0 || (n < 0 && errno != EWOULDBLOCK && errno != EAGAIN &&
		       errno != EINTR)) {
	  /*printf("about to call waitandclose(), n=%d, errno=%d\n",n,errno);*/
	  waitandclose(fd);
	}
	/* fprintf(stderr, "n = %d, errno = %d\n", n, errno); */
	/*
	 * if n < 0, then either we got an interrupt or the read would
	 * block (EINTR or EWOULDBLOCK). In both cases we basically just
	 * want to get back to whatever we were doing. We just need to
	 * make darned sure that a newline was the last character we saw,
	 * or else some report may get lost somewhere.
	 */
	if (dontbreak) {
	  if (cpids[fd].pid != 0) {
	    cpids[fd].carryover = emalloc(cp-pcp+1);
	    memcpy(cpids[fd].carryover, pcp, cp-pcp);
	    cpids[fd].carryover[cp-pcp] = '\0';
	  } else
	    fprintf(stderr,
		    "HELP! Lost %d bytes (n=%d/%d, off=%d): '%s'\n",
		    cp - pcp, n, errno, pcp-buf, pcp);
	}
	free(buf);
}

#if defined(USE_BINMKDIR) || defined(USE_BINRMDIR)

/*
 * From Ross Ridge's Xenix port:
 * - A nasty problem occurs with scheduler if rmdir (and mkdir also I think),
 *   is implented as system("/bin/rmdir ...").  When system() calls wait()
 *   it can reap the scheduler's children without it knowing.  I fixed this
 *   problem by writing a replacement system() function for scheduler.
 *
 */

int
system(name)
	char *name;
{
	char *sh;
	int st, r;
	int pid;
	int i;

	pid = fork();
	switch(pid) {
	case -1:
		return -1;
	case 0:
		sh = getenv("SHELL");
		if (sh == NULL) {
		  sh = "/bin/sh";
		}
		execl(sh, sh, "-c", name, NULL);
		_exit(1);
	default:
#ifndef USE_SIGREAPER
		while(1) {
		  r = wait(&st);
		  if (r == -1) {
		    if (errno != EINTR) {
		      return -1;
		      if (errno != EINTR) {
			return -1;
		      }
		    } else if (r == pid) {
		      break;
		    }
		    for(i = 0; i < scheduler_nofiles; i++) {
		      if (cpids[i].pid == r) {
			cpids[i].pid = -r;
			break;
		      }
		    }
		  }

		  if ((st & 0x00ff) == 0) {
		    return st >> 8;
		  }
		  return 1;
		}
#endif
		break;
	   }
}

#endif


#ifdef USE_SIGREAPER
/*
 *	Catch each child-process death, and reap them..
 */
SIGNAL_TYPE sig_chld(signum)
int signum;
{
	int pid;
	int ok = 0;
	int i;
#ifdef USE_UNIONWAIT
	union wait statloc;
#else
	int statloc;
#endif
#ifdef USE_UNIONWAIT
	while ((pid = wait3(&statloc,WNOHANG,NULL)) > 0) ;
#else
#ifdef	USE_WAITPID
	while ((pid = waitpid(-1,&statloc,WNOHANG)) > 0) ;
#else
	pid = wait(&statloc);
#endif
#endif


#if 0	/* DO NOT do anything excessive -- we don't care WHO died.. */

#ifdef	WIFEXITED
	if (WIFEXITED(statloc)) ok = 1;
#else
      ERROR:ERROR:ERROR
#endif
#ifdef WIFSIGNALED
	if (WIFSIGNALED(statloc)) ok = 1;
#else
      ERROR:ERROR:ERROR
#endif

	if (verbose)
	  fprintf(stderr,"sig_chld() pid=%d, ok=%d\n",pid,ok);

	if (ok && cpids != NULL) {
	  /* Only EXIT and SIGxx DEATHS accepted */

	  for (i = scheduler_nofiles-1; i >= 0; --i) {
	    if (cpids[i].pid == pid) {
	      cpids[i].pid = -pid;	/* Mark it as reaped.. */
	      ok = 0;
	      break;
	    }
	    if (cpids[i].pid == -pid) {
printf(" .. already reaped ??\n");
	      cpids[i].pid = -pid;	/* Mark it as reaped.. */
	      ok = 0;
	      break;
	    }
	  }
	}
#endif
	/* re-instantiate the signal handler.. */
#ifdef SIGCLD
	SIGNAL_HANDLE(SIGCLD,  sig_chld);
#else
	SIGNAL_HANDLE(SIGCHLD, sig_chld);
#endif
}
#endif /* USE_SIGREAPER */
