/*
 *	Copyright 1990 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

#include "mailer.h"
#include <ctype.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/times.h>
#include "interpret.h"
#include "io.h"		/* redefines stdio routines */
#include "shconfig.h"
#ifdef	USE_UNIONWAIT
#include <sys/wait.h>
#endif	/* USE_UNIONWAIT */

#ifndef	HZ
#define	HZ	60
#endif	/* HZ */

extern struct conscell *sh_car(), *sh_cdr(), *sh_list(), *sh_grind();
extern struct conscell *sh_elements(), *sh_setf(), *sh_get(), *sh_length();
extern struct conscell *sh_return(),*sh_returns(), *sh_last();

extern int sh_read(), sh_echo(), sh_read(), sh_test(), sh_include();
extern int sh_cd(), sh_colon(), sh_hash(), sh_bce(), sh_eval(), sh_export();
extern int sh_getopts(), sh_shift(), sh_umask(), sh_set(), sh_unset();
extern int sh_wait(), sh_times(), sh_trap(), sh_type(), sh_builtin();
extern int sh_sleep();


extern char *ifs;
extern char *optarg;
extern char *path_hash();
extern char *prepath();
extern char *strerror();
extern char *sys_siglist[];
extern char *strchr();
extern int errno;
extern int eval();
extern int loadeval();
extern int optind, opterr;
extern struct codedesc *interpret();
extern struct conscell *envarlist;
extern struct osCmd *globalcaller;
extern void assign();
extern void functype();
extern void ifs_flush();
extern void path_flush();
extern void v_export();
extern void v_set();
extern void v_touched();


struct shCmd builtins[] = {
{	"car",		0,		sh_car,		SH_ARGV		},
{	"first",	0,		sh_car,/*sic*/	SH_ARGV		},
{	"cdr",		0,		sh_cdr,		SH_ARGV		},
{	"rest",		0,		sh_cdr,/*sic*/	SH_ARGV		},
{	"last",		0,		sh_last,	SH_ARGV		},
{	"list",		0,		sh_list,	SH_ARGV		},
{	"grind",	0,		sh_grind,	SH_ARGV		},
{	"elements",	0,		sh_elements,	SH_ARGV		},
{	"setf",		0,		sh_setf,	SH_ARGV		},
{	"get",		0,		sh_get,		SH_ARGV		},
{	"length",	0,		sh_length,	SH_ARGV		},
{	"[",		sh_test,	0,		0,		},
{	"test",		sh_test,	0,		0,		},
{	"echo",		sh_echo,	0,		0,		},
{	"cd",		sh_cd,		0,		0,		},
{	"hash",		sh_hash,	0,		0,		},
{	"read",		sh_read,	0,		0,		},
{	":",		sh_colon,	0,		0,		},
{	".",		sh_include,	0,		0,		},
{	"break",	sh_bce,		0,		SH_INTERNAL,	},
{	"continue",	sh_bce,		0,		SH_INTERNAL,	},
{	"return",	0,		sh_return,	SH_ARGV,	},
{	"returns",	0,		sh_returns,	SH_ARGV,	},
{	"exit",		sh_bce,		0,		SH_INTERNAL,	},
{	"eval",		sh_eval,	0,		0,		},
{	"export",	sh_export,	0,		0,		},
{	"getopts",	sh_getopts,	0,		0,		},
{	"shift",	sh_shift,	0,		0,		},
{	"umask",	sh_umask,	0,		0,		},
{	"set",		sh_set,		0,		0,		},
{	"unset",	sh_unset,	0,		0,		},
{	"wait",		sh_wait,	0,		0,		},
{	"times",	sh_times,	0,		0,		},
{	"trap",		sh_trap,	0,		0,		},
{	"type",		sh_type,	0,		0,		},
{	"builtin",	sh_builtin,	0,		0,		},
{	"sleep",	sh_sleep,	0,		0,		},
{	0,		0,		0,		0,		},
};

struct conscell *
sh_car(avl, il)
	struct conscell *avl, *il;
{
	struct conscell *tmp;

	il = cdar(avl);
	if (il == NULL)
		return NULL;
	if (STRING(il))
		return conststring(il->string);
	/* setf preparation */
	if (car(il) == NULL) {
		if (il->prev) {
			il->prev = cdr(il->prev);
			il->pflags = 0;
		}
		return il;
	}
	car(il) = copycell(car(il));	/* don't modify malloc'ed memory! */
	cdar(il) = NULL;
	return car(il);
}

struct conscell *
sh_cdr(avl, il)
	struct conscell *avl, *il;
{
	il = cdar(avl);
	if (il == NULL || STRING(il) || car(il) == NULL)
		return NULL;
	/* setf preparation */
	if (cdar(il)) {
		il->prev = cdar(il)->prev;
	} else if (car(il)->prev) {
		if (car(il)->pflags)
			il->prev = cdr(car(il)->prev);
		else
			il->prev = car(car(il)->prev);
	}
	il->pflags = 3;
	car(il) = cdar(il);
	return il;
}

struct conscell *
sh_last(avl, il)
	struct conscell *avl, *il;
{
	il = cdar(avl);
	if (il == NULL || STRING(il) || car(il) == NULL)
		return NULL;
	/* setf preparation */
	while (cdar(il) != NULL)
		car(il) = cdar(il);
	if (car(il)->prev) {
		/* if (car(il)->pflags)*/
			il->prev = car(il)->prev;
		/*else
			il->prev = car(car(il)->prev);*/
	}
	il->pflags = 3;
	return il;
}

struct conscell *
sh_list(avl, il)
	struct conscell *avl, *il;
{
	car(avl) = cdar(avl);
	return avl;
}

struct conscell *
sh_grind(avl, il)
	struct conscell *avl, *il;
{
	return cdar(avl);
}

struct conscell *
sh_elements(avl, il)
	struct conscell *avl, *il;
{
	if (LIST(cdar(avl))) {
		for (il = cadar(avl); il != NULL; il = cdr(il))
			il->flags |= ELEMENT;
		return cadar(avl);
	}
	return cdar(avl);
}

struct conscell *
sh_setf(avl, il)
	struct conscell *avl, *il;
{
	struct conscell *place, *value, *tmp = NULL;
	int oval = stickymem;

	/*
	 * The `place' argument to setf should NOT be malloc'ed stuff,
	 * minor tweaks may be necessary in car/cdr/get (certainly the latter)
	 * to ensure this.
	 */
	if ((place = cdar(avl)) == NULL) {
		fprintf(stderr, "Usage: %s variable-reference [new-value]\n",
				car(avl)->string);
		return NULL;
	}
	value = cddar(avl);

	if (place->prev == NULL)
		return value;

	stickymem = MEM_MALLOC;

	value = s_copy_tree(value);
	if (place->pflags) {
		if (value != NULL && (place->pflags & 02)) {
			if (LIST(value)) {
				tmp = car(value);
				free((char *)value);
				value = tmp;
			}
			place->pflags &= ~02;
		}
		if (place->pflags & 04) {
			if ((tmp = cdr(place->prev))) {
				tmp = cdr(tmp);
				if (value != NULL) {
					cddr(place->prev) = NULL;
					s_free_tree(cdr(place->prev));
				}
			}
		} else {
			if (cdr(place->prev) != NULL)
				s_free_tree(cdr(place->prev));
			cdr(place->prev) = NULL;
		}
		if (value != NULL) {
			cdr(place->prev) = value;
			s_set_prev(place->prev, value);
			value->pflags = 1;
			if (place->pflags & 04) {
				value = s_last(value);
				if (cdr(value))
					s_free_tree(cdr(value));	/* XX */
				cdr(value) = tmp;
			}
		}
	} else {
		if (value != NULL && car(place->prev) != NULL) {
			if (cdar(place->prev))
				cdr(s_last(value)) = cdar(place->prev);
			cdar(place->prev) = NULL; /* for free() below */
		}
		s_free_tree(car(place->prev));
		car(place->prev) = value;
		s_set_prev(place->prev, value);
	}
	stickymem = oval;
#ifdef	MAILER
	v_touched();
#endif	/* MAILER */
	return cddar(avl);
}

struct conscell *
sh_get(avl, il)
	struct conscell *avl, *il;
{
	struct conscell *plist, *key, *d, *tmp;

	if ((plist = cdar(avl)) == NULL
	    || (key = cddar(avl)) == NULL
	    || !STRING(key)) {
		fprintf(stderr, "Usage: %s variable-name key\n",
				car(avl)->string);
		return NULL;
	}
	if (STRING(plist)) {
		if ((d = v_find(plist->string)) == NULL) {
			/* (setq plist '(key nil)) */
			d = conststring(key->string);
			cdr(d) = NIL;
			d = ncons(d);
			assign(plist, d, (struct osCmd *)NULL);
			/* setf preparation */
			cdar(d)->prev = cadr(v_find(plist->string));
			cdar(d)->pflags = 1;
			return cdar(d);
		}
		plist = cdr(d);
		if (!LIST(plist))
			return NULL;
		plist = car(plist);
	}
	/* now we have a property list in plist, scan it */
	/* printf("plist = "); s_grind(plist, stdout); putchar('\n'); */
	for (d = plist; d != NULL; d = cdr(d)) {
		/* if (STRING(d))
			printf("comparing '%s' and '%s'\n",
				d->string, key->string);	*/
		if (STRING(d) && strcmp((char *)d->string, (char *)key->string) == 0) {
			d = copycell(cdr(d));
			cdr(d) = NULL;
			d->pflags |= 04;
			return d;
		}
		d = cdr(d);
		plist = d;
		if (d == NULL) {
			plist = NULL;
			break;
		}
	}
	if (plist) {
		int oval = stickymem;

		/* plist is now s_last(cadr(v_find(key->string))) */
		cdr(plist) = copycell(key);
		d = NIL;
		cddr(plist) = d;

		stickymem = MEM_MALLOC;
		cdr(plist) = s_copy_tree(cdr(plist));
		s_set_prev(plist, cdr(plist));
		cdr(plist)->pflags = 1;
		stickymem = oval;

		/* setf preparation */
		d->prev = cdr(plist);
		d->pflags = 01 | 04;
	}

	return d;
}

struct conscell *
sh_length(avl, il)
	struct conscell *avl, *il;
{
	char buf[10];
	struct conscell *tmp;
	int len = 0;

	if ((il = cdar(avl)) && LIST(il)) {
		for (il = car(il); il != NULL; il = cdr(il))
			++len;
	}
	sprintf(buf, "%d", len);

	avl = newstring((u_char *)strsave(buf));
	return avl;
}

/* returns -- return ALWAYS a string [mea] */
struct conscell *
sh_returns(avl, il, statusp)
	struct conscell *avl, *il;
	int *statusp;
{
	int n;
	char *cp;

	if ((il = cdar(avl)) && LIST(il) && cdr(il) == NULL)
		return il;
	else if (il == NULL)
		return NULL;
	else if (STRING(il)) {
		for (cp = (char *)il->string; *cp != 0; ++cp) {
			if (!isascii(*cp) || !isdigit(*cp))
				break;
		}
		if (*cp)
			return il;
		if ((n = atoi((char *)il->string)) < 0) {
			fprintf(stderr, "%s: %s: %d\n", car(avl)->string,
					NEGATIVE_VALUE, n);
			n = 1;
		}
		*statusp = n;
		return NULL;
	}
	/* NOTREACHED */
	return NULL;
}

struct conscell *
sh_return(avl, il, statusp)
	struct conscell *avl, *il;
	int *statusp;
{
	int n;
	char *cp;

	if ((il = cdar(avl)) && LIST(il) && cdr(il) == NULL)
		return il;
	else if (il == NULL)
		return NULL;
	else if (STRING(il)) {
		for (cp = (char *)il->string; *cp != 0; ++cp) {
			if (!isascii(*cp) || !isdigit(*cp))
				break;
		}
		if (*cp)
			return il;
		if ((n = atoi((char *)il->string)) < 0) {
			fprintf(stderr, "%s: %s: %d\n", car(avl)->string,
					NEGATIVE_VALUE, n);
			n = 1;
		}
		*statusp = n;
		return NULL;
	}
	/* NOTREACHED */
	return NULL;
}

/*-*/

int
sh_colon(argc, argv)
	int argc;
	char *argv[];
{
	return 0;
}

int
sh_builtin(argc, argv)
	int argc;
	char *argv[];
{
	return 0;
}

int
sh_echo(argc, argv)
	int argc;
	char *argv[];
{
	int n;

	--argc, ++argv;
	if (argc > 0
	    && argv[0][0] == '-' && argv[0][1] == 'n' && argv[0][2] == '\0') {
		--argc, ++argv;
		n = 0;
	} else
		n = 1;
	while (--argc >= 0) {
		fputs(*argv++, stdout);
		if (argc > 0)
			putchar(' ');
	}
	if (n)
		putchar('\n');
	fflush(stdout);
	return 0;
}

int
sh_cd(argc, argv)
	int argc;
	char *argv[];
{
	char *dir, *cddir, *path;
	struct conscell *d;
	u_int pathlen;

	if (argc == 1) {
		/* cd $HOME */
		d = v_find(HOME);
		if (d == NULL || cdr(d) == NULL || LIST(cdr(d))) {
			fprintf(stderr, "%s: %s\n", argv[0], NO_HOME_DIRECTORY);
			return 1;
		}
		dir = (char *)cdr(d)->string;
	} else if (argc == 2) {
		/* cd argv[1] */
		dir = argv[1];
	} else {
		fprintf(stderr, USAGE_CD, argv[0]);
		return 1;
	}
	if (!(dir[0] == '/' ||
	      (dir[0] == '.' &&
	       (dir[1] == '\0' || dir[1] == '/' ||
		(dir[1] == '.' && (dir[2] == '\0' || dir[2] == '/'))))) &&
	    (d = v_find(CDPATH)) != NULL &&
	    cdr(d) != NULL && STRING(cdr(d))) {
		cddir = (char *)cdr(d)->string;
		pathlen = strlen(cddir)+strlen(dir)+1+1;
		path = tmalloc(pathlen);
		while (cddir != NULL) {
			cddir = prepath(cddir, dir, path, pathlen);
			if (chdir(path) == 0) {
				if (!(path[0] == '.' && path[1] == '/'))
					printf("%s\n", path);
				return 0;
			}
		}
	} else if (chdir(dir) == 0)
		return 0;

	fprintf(stderr, "%s: %s: %s\n", argv[0], dir, strerror(errno));
	return 1;
}

int
sh_hash(argc, argv)
	int argc;
	char *argv[];
{
	if (argc == 1) {
		printf(NO_HASHING_INFORMATION);
		return 0;
	} else if (strcmp(argv[1], "-r") == 0) {
		path_flush();
		--argc, ++argv;
	}
	--argc, ++argv;
	while (argc-- > 0) {
		/* printf("hash '%s'\n", *argv); */
		path_hash(*argv++);
	}
	return 0;
}

int
sh_read(argc, argv)
	int argc;
	char *argv[];
{
	static char *buf = NULL;
	static int bufsize;
	char *cp, *value = NULL, *bp;
	int flag, offset;

	if (argc == 1) {
		fprintf(stderr, USAGE_READ, argv[0]);
		return 1;
	}
	if (ifs == NULL)
		ifs_flush();
	--argc, ++argv;
	if (buf == NULL) {
		bufsize = BUFSIZ - 24;
		buf = emalloc(bufsize);
	}
	flag = 0;
	bp = NULL;
	buf[0] = '\0';
	while (argc > 0 && fgets(buf, bufsize, stdin) != NULL) {
		for (cp = buf; argc > 0 && *cp != '\0'; ) {
			while (*cp != '\0' && WHITESPACE((unsigned)*cp))
				++cp;
			if (*cp == '\0') {
				if (cp > buf && *(cp-1) != '\n' &&
				    (offset = cp - buf)		&&
				    (bufsize = 2*bufsize)	&&
				    (buf = erealloc(buf, bufsize)) &&
				    fgets(cp, bufsize/2, stdin) != NULL) {
					cp = buf + offset;
					continue;
				} else
					goto eoinput;
			}
			if (!flag) {
				bp = cp;
				value = cp;
			}
			if (argc == 1)
				flag = 1;
			while (*cp == '\0' || !WHITESPACE((unsigned)*cp)) {
				if (*cp == '\0') {
					if (cp > buf && *(cp-1) != '\n' &&
					    (offset = cp - buf)		&&
					    (bufsize = 2*bufsize)	&&
					    (buf = erealloc(buf, bufsize)) &&
					    fgets(cp, bufsize/2, stdin)!=NULL) {
						cp = buf + offset;
						continue;
					} else
						break;
				}
				if (*cp == '\\' && *(cp+1) != '\0') { /* bug */
					if (*++cp == '\n') {
						*cp++ = '\0'; /* defeat above */
						continue;
					}
				}
				*bp++ = *cp++;
			}
			if (argc > 1) {
				--argc;
				if (*cp == '\0') {
					*bp = '\0';	/* bp might == cp */
					v_set(*argv++, value);
					bp = value;
					break;
				} else {
					*bp = '\0';
					v_set(*argv++, value);
					++cp;
				}
			} else if (*cp++ != '\0')
				*bp++ = ' ';
		}
		if (bp > value) {
			if (*(bp-1) == ' ')
				*--bp = '\0';
			else
				*bp = '\0';
			v_set(*argv++, value), --argc;
		}
	}
	if (buf[0] == '\0')
		return 1;
eoinput:
	while (argc-- > 0)
		v_set(*argv++, "");
	return 0;
}

int
sh_include(argc, argv)
	int argc;
	char *argv[];
{
	int fd, status;
	struct conscell *d;
	struct stat stbuf;
	u_int pathlen;
	char *dir, *buf, *path = NULL;

	if (argc < 2 || argv[1][0] == '\0') {
		fprintf(stderr, USAGE_INCLUDE, argv[0]);
		return 1;
	}
	fd = -1;
	if (strchr(argv[1], '/') == NULL) {
		d = v_find(PATH);
		if (d != NULL && cdr(d) != NULL && STRING(cdr(d))) {
			dir = (char *)cdr(d)->string;
			pathlen = strlen(dir)+strlen(argv[1])+1+1;
			path = tmalloc(pathlen);
			while (dir != NULL) {
				dir = prepath(dir, argv[1], path, pathlen);
				if ((fd = open(path, O_RDONLY, 0)) >= 0)
					break;
			}
		}
	} else {
		path = argv[1];
		fd = open(path, O_RDONLY, 0);
	}
	if (fd < 0) {
		fprintf(stderr, "%s: %s\n", argv[1], NOT_FOUND);
		return 1;
	}
	if (fstat(fd, &stbuf) < 0) {
		perror("fstat");
		close(fd);
		return 1;
	}
	status = loadeval(fd, path, &stbuf);
	if (status >= 0) {
		/* loadeval closes fd */
		return status;
	}
	buf = tmalloc(stbuf.st_size+1);
	if ((status = read(fd, buf, stbuf.st_size)) != stbuf.st_size) {
		perror("read");
		close(fd);
		return 1;
	}
	close(fd);
	buf[stbuf.st_size] = '\0';
	return eval(buf, argv[1], path);
}

int
sh_bce(argc, argv)		/* handle break, continue, and exit */
	int argc;
	char *argv[];
{
	int n;

	if (argc > 2 || (argc == 2 && !isdigit(argv[1][0])))
		fprintf(stderr, USAGE_BCE, argv[0]);
	else if (argc == 1)
		/* 0 for "return" and "exit", 1 o/w */
		return !(argv[0][0] == 'r' || argv[0][0] == 'e');
	if ((n = atoi(argv[1])) < 0) {
		fprintf(stderr, "%s: %s: %d\n", argv[0], NEGATIVE_VALUE, n);
		n = 1;
	} else if (n == 0) {
		switch (argv[0][0]) {
		case 'b': case 'c':	/* 0 for break and continue means 1 */
			n = 1;
			break;
		}
	}
	return n;
}

int
sh_eval(argc, argv)
	int argc;
	char *argv[];
{
	int len, i;
	int status;
	u_char *table, *eotable;
	char *cp, *buf;

	for (len = 0, i = 1; i < argc; ++i) {
		len += strlen(argv[i]);
		++len;
	}
	if (len < argc)
		return 0;
	buf = tmalloc(len+1+1/* trailing \n*/);
	for (cp = buf, i = 1; i < argc; ++i) {
		strcpy(cp, argv[i]);
		cp += strlen(argv[i]);
		*cp++ = ' ';
	}
	*cp++ = '\n';
	*cp = '\0';
	commandline = s_pushstack(commandline, buf);
	table = SslWalker("eval string", stdout, &eotable);
	status = 7;
	if (table != NULL) {
		if (isset('O')) {
			table = optimize(0, table, &eotable);
			if (isset('V'))
				table = optimize(1, table, &eotable);
		}
		interpret(table, eotable, (u_char *)NULL, globalcaller,
			  &status, (struct codedesc *)NULL);
	}
	return status;
}

int
sh_export(argc, argv)
	int argc;
	char *argv[];
{
	struct conscell *elist, *glist, *scope;

	if (argc > 1) {
		/* export the listed variables */
		--argc, ++argv;
		while (argc > 0) {
			v_export(argv[0]);
			--argc, ++argv;
		}
		return 0;
	}
	/* print which variables have been exported */
	for (scope = car(envarlist); cddr(scope) != NULL; scope = cdr(scope))
		continue;
	for (elist = cadr(scope); elist != NULL; elist = cddr(elist)) {
		if (LIST(elist) || elist->string == NULL)
			continue;
		for (glist = car(scope); glist != NULL; glist = cddr(glist)) {
			if (STRING(glist) &&
			    strcmp((char *)glist->string,
				   (char *)elist->string) == 0)
				break;
		}
		if (glist == NULL)
			printf("%s %s\n", EXPORT, elist->string);
	}
	return 0;
}

int
sh_getopts(argc, argv)
	int argc;
	char *argv[];
{
	char buf[20], *optstring, **av, *name;
	int c;
	struct conscell *d;

	if (argc < 3) {
		fprintf(stderr, USAGE_GETOPTS, argv[0]);
		return 1;
	}
	opterr = 0;
	--argc, ++argv;
	optstring = argv[0];
	--argc, ++argv;
	name = argv[0];
	if (argc == 1) {
		for (d = cdar(globalcaller->argv); d != NULL ; d = cdr(d))
			++argc;
		argv = av = (char **)tmalloc((argc+1)*sizeof (char *));
		for (d = car(globalcaller->argv); d != NULL ; d = cdr(d))
			if (STRING(d))
				*av++ = (char *)d->string;
		*av = NULL;
	}
	c = getopt(argc, argv, optstring);	/* this MUST be our getopt() */
	sprintf(buf, "%d", optind);
	v_set(OPTIND, buf);
	if (c == EOF)
		return 1;
	if (optarg != NULL)	/* our getopt() makes this reliable */
		v_set(OPTARG, optarg);
	sprintf(buf, "%c", c);
	v_set(name, buf);
	return 0;
}

int
sh_shift(argc, argv)
	int argc;
	char *argv[];
{
	int n;
	struct conscell *d;

	if (argc > 2 || (argc == 2 && !isdigit(argv[1][0]))) {
		fprintf(stderr, USAGE_SHIFT, argv[0]);
		return 1;
	}
	if (argc == 2) {
		if ((n = atoi(argv[1])) < 1)
			return 0;
	} else
		n = 1;
	for (d = cdar(globalcaller->argv); d != NULL && n > 0; d = cdr(d))
		--n;
	cdar(globalcaller->argv) = d;		/* XX: possible malloc leak */
	return 0;
}

int
sh_umask(argc, argv)
	int argc;
	char *argv[];
{
	int mask;
	char *cp;

	if (argc > 2 || (argc == 2 && !isdigit(argv[1][0]))) {
		fprintf(stderr, USAGE_UMASK, argv[0]);
		return 1;
	}
	if (argc == 1) {
		mask = umask(077);
		printf("%#o\n", mask);
		umask(mask);
		return 0;
	}
	mask = 0;
	for (cp = argv[1]; *cp != '\0'; ++cp)
		mask = mask*8 + (*cp - '0');		/* XX: ebcdic?? */
	umask(mask);
	return 0;
}

int
sh_set(argc, argv)
	int argc;
	char *argv[];
{
	int n, i, oval;
	char **kk, *cp, *ep;
	struct conscell *d, *pd = NULL, *scope, *tmp;

	if (argc == 1) {
		/* print variable values */
		/* first count how many there might be */
		n = 0;
		for (scope = car(envarlist); scope != NULL; scope = cdr(scope))
			for (d = car(scope); d != NULL; d = cddr(d))
				++n;
		/* allocate enough space to hold cache ptrs */
#ifdef	USE_ALLOCA
		kk = (char **)alloca((n+1) * sizeof (char *));
#else
		kk = (char **)tmalloc((n+1) * sizeof (char *));
#endif
		kk[0] = NULL;
		/* iterate through all variables */
		for (scope = car(envarlist); scope != NULL; scope = cdr(scope))
			for (d = car(scope); d != NULL; d = cddr(d)) {
				if (!STRING(d) || cdr(d) == envarlist)
					continue;
				/* if we already saw it, skip this one O(N^2) */
				for (i = 0; kk[i] != NULL && i < n; ++i)
					if (strcmp(kk[i], (char *)d->string) == 0)
						break;
				if (kk[i] == NULL) {
					kk[i] = (char *)d->string;
					kk[i+1] = NULL;
					printf("%s=", (char*)d->string);
					s_grind(cdr(d), stdout);
					putchar('\n');
				}
			}
		/* free(kk); -- tmalloc() on temp memory makes this unnecessary */
		return 0;
	}
	--argc, ++argv;
	if (argv[0][0] == '-' || argv[0][0] == '+') {
		i = (argv[0][0] == '-');
		for (cp = ep = &argv[0][1]; *cp != '\0'; ++cp) {
			switch (*cp) {
			case 'a':/* automatically export changed variables */
			case 'e':/* exit on error exit status of any command */
			case 'f':/* disable filename generation (no globbing) */
			case 'h':/* hash program locations */
			case 'n':/* read commands but do not execute them */
			case 't':/* read and execute one command only */
			case 'u':/* unset variables are error on substitution */
			case 'v':/* print shell input lines as they are read */
			case 'x':/* print commands as they are executed */
			case 'L':/* Trace LEXER processing (sslWalker) */
			case 'C':
			case 'P':
			case 'S':
				setopt(*cp, i);
				break;
			case '-':	/* do nothing */
				break;
			case 'k':	/* we do not support the k option */
			default:	/* complain */
				*ep++ = *cp;
				break;
			}
		}
		if (ep > &argv[0][1]) {
			*ep = '\0';
			fprintf(stderr, "%s: %s: %s\n",
					argv[-1], BAD_OPTIONS, argv[0]);
		}
		--argc, ++argv;
	}
	if (argc == 0)
		return 0;
	/* the remaining arguments should replace $@ */
	oval = stickymem;
	stickymem = MEM_MALLOC;
	if (globalcaller != NULL && globalcaller->argv != NULL)
		pd = car(globalcaller->argv);
	/* XX: possible memory leak here! */
	while (argc-- > 0) {
		d = newstring((u_char *)strsave(*argv));
		if (pd != NULL)
			cdr(pd) = d;
		++argv;
		pd = d;
	}
	stickymem = oval;
	return 0;
}

STATIC char *sacred[] = { PATH, PS1, PS2, IFS, MAILCHECK, 0 };

int
sh_unset(argc, argv)
	int argc;
	char *argv[];
{
	struct conscell *d, *pd = NULL, *scope, *next;
	char **cpp, *av0;

	if (argc == 1) {
		fprintf(stderr, USAGE_UNSET, argv[0]);
		return 1;
	}
	av0 = argv[0];
	--argc, ++argv;
	while (argc-- > 0) {
		for (cpp = sacred; *cpp != NULL; ++cpp) {
			if (strcmp(*cpp, *argv) == 0) {
				fprintf(stderr, "%s: %s %s\n",
					av0, CANNOT_UNSET, *argv);
				break;
			}
		}
		if (*cpp != NULL) {	/* couldn't unset that one */
			++argv;
			continue;
		}
		for (scope = car(envarlist); scope != NULL; scope = cdr(scope))
			for (d = car(scope); d != NULL; pd = d, d = next) {
				next = cddr(d);
				if (LIST(d) || strcmp((char *)d->string, *argv) != 0)
					continue;
				/* now we've got it! unlink and free */
				if (d == car(scope))
					car(scope) = next;
				else
					cddr(pd) = next;
				cddr(d) = NULL;
				s_free_tree(d);
				/* no point doing anything else in this scope */
				break;
			}
		++argv;
	}
	return 0;
}

int
sh_wait(argc, argv)
	int argc;
	char *argv[];
{
	int retcode, pid;
#if	defined(WNOHANG) && !defined(WLOBYTE)
	union wait status;

	status.w_termsig = 0;
	status.w_retcode = 0;
#else	/* !WNOHANG */
	int status = 0;
#endif	/* WNOHANG */
	if (argc > 2 || (argc == 2 && !isdigit(argv[1][0]))) {
		fprintf(stderr, USAGE_WAIT, argv[0]);
		return 1;
	}
	while ((pid = wait(&status)) > 0) {
		if (argc == 2 && pid == atoi(argv[1]))
			break;
	}
	if (pid < 0)		/* no more children */
		return 0;
#if	defined(WNOHANG) && !defined(WLOBYTE)
	if (status.w_termsig) {
		fprintf(stderr, "%s", sys_siglist[status.w_termsig]);
		if (status.w_coredump)
			fprintf(stderr, CORE_DUMPED);
		fprintf(stderr, "\n");
		retcode = 0200 + status.w_termsig;
	} else
		retcode = status.w_retcode;
#else	/* !WNOHANG */
	if ((status&0177) > 0) {
		fprintf(stderr, "%s", sys_siglist[status&0177]);
		if (status&0200)
			fprintf(stderr, CORE_DUMPED);
		fprintf(stderr, "\n");
		retcode = 0200 + (status&0177);
	} else
		retcode = (status>>8)&0377;
#endif	/* WNOHANG */
	return retcode;
}

int
sh_times(argc, argv)
	int argc;
	char *argv[];
{
	struct tms foo;

	if (argc > 1) {
		fprintf(stderr, USAGE_TIMES, argv[0]);
		return 1;
	}
	if (times(&foo) < 0)
		return 1;
	printf("%dm%ds %dm%ds\n",
	       foo.tms_cutime / HZ / 60, (foo.tms_cutime / HZ) % 60,
	       foo.tms_cstime / HZ / 60, (foo.tms_cstime / HZ) % 60);
	return 0;
}

int
sh_type(argc, argv)
	int argc;
	char *argv[];
{
	char *dir, *path;
	struct shCmd *shcmdp;
	struct sslfuncdef *sfdp;
	u_int pathlen;

	for (--argc, ++argv; argc-- > 0 ; ++argv) {
		if (*argv == '\0') {
			printf(NULL_NAME);
			continue;
		}
		functype(*argv, &shcmdp, &sfdp);
		/* defined function ? */
		if (sfdp != NULL) {
			printf("%s %s\n", *argv, IS_A_SHELL_FUNCTION);
			continue;
		}
		/* builtin ? */
		if (shcmdp != NULL) {
			printf("%s %s\n", *argv, IS_A_SHELL_BUILTIN);
			continue;
		}
		/* unix command */
		if ((dir = path_hash(*argv)) != NULL) {
			pathlen = strlen(dir)+strlen(*argv)+1+1;
			path = tmalloc(pathlen);
			dir = prepath(dir, *argv, path, pathlen);
			printf("%s %s %s\n", *argv, IS, path);
			continue;
		}
		printf("%s %s\n", *argv, NOT_FOUND);
	}
	return 0;
}


int
sh_sleep(argc, argv)
	int argc;
	char *argv[];
{
	if (argc != 2 || atoi(argv[1]) <= 0) {
		fprintf(stderr, USAGE_SLEEP, argv[0]);
		return 1;
	}
	sleep(atoi(argv[1]));
	return 0;
}

