/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */
                                       

#include "../exim.h"
#include "pipe.h"



/* Options specific to the pipe transport. They must be in alphabetic
order (note that "_" comes before the lower case letters). Those starting
with "*" are not settable by the user but are used by the option-reading
software for alternative value types. */

optionlist pipe_transport_options[] = {
  { "*expand_group",     opt_stringptr,
      (void *)(offsetof(pipe_transport_options_block, expand_gid)) }, 
  { "*expand_user",      opt_stringptr,
      (void *)(offsetof(pipe_transport_options_block, expand_uid)) }, 
  { "bsmtp",             opt_local_smtp,
      (void *)(offsetof(pipe_transport_options_block, local_smtp)) }, 
  { "command",           opt_stringptr,
      (void *)(offsetof(pipe_transport_options_block, cmd)) }, 
  { "delivery_date_add", opt_bool,     
      (void *)(offsetof(pipe_transport_options_block, delivery_date_add)) },
  { "directory",         opt_stringptr,
      (void *)(offsetof(pipe_transport_options_block, directory)) },      
  { "from_hack",         opt_bool,     
      (void *)(offsetof(pipe_transport_options_block, from_hack)) },
  { "group",             opt_expand_gid,     
      (void *)(offsetof(pipe_transport_options_block, gid)) },
  { "ignore_status",     opt_bool,
      (void *)(offsetof(pipe_transport_options_block, ignore_status)) },      
  { "max_output",        opt_mkint,
      (void *)(offsetof(pipe_transport_options_block, max_output)) },      
  { "path",              opt_stringptr,     
      (void *)(offsetof(pipe_transport_options_block, path)) },
  { "pipe_as_creator",   opt_bool,     
      (void *)(offsetof(pipe_transport_options_block, pipe_as_creator)) }, 
  { "prefix",            opt_stringptr,     
      (void *)(offsetof(pipe_transport_options_block, prefix)) },
  { "restrict_to_path",  opt_bool,     
      (void *)(offsetof(pipe_transport_options_block, restrict_to_path)) },
  { "return_output",     opt_bool,     
      (void *)(offsetof(pipe_transport_options_block, return_output)) },
  { "return_path_add",   opt_bool,     
      (void *)(offsetof(pipe_transport_options_block, return_path_add)) },
  { "suffix",            opt_stringptr,     
      (void *)(offsetof(pipe_transport_options_block, suffix)) },
  { "umask",             opt_octint,
      (void *)(offsetof(pipe_transport_options_block, umask)) },      
  { "user",              opt_expand_uid,     
      (void *)(offsetof(pipe_transport_options_block, uid)) },
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int pipe_transport_options_count =
  sizeof(pipe_transport_options)/sizeof(optionlist);

/* Default private options block for the pipe transport. */

pipe_transport_options_block pipe_transport_option_defaults = {
  NULL,           /* cmd */ 
  "/usr/bin",     /* path */ 
  "From ${return_path} ${tod_bsdinbox}\n", /* prefix */
  "\n",           /* suffix */     
  NULL,           /* expand_uid */
  NULL,           /* expand_gid */  
  NULL,           /* directory */ 
  local_smtp_off, /* local_smtp */ 
  -1,             /* uid */ 
  -1,             /* gid */ 
  022,            /* umask */ 
  20480,          /* max_output */ 
  FALSE,          /* from_hack */ 
  FALSE,          /* ignore_status */ 
  FALSE,          /* pipe_as_creator */ 
  FALSE,          /* return_path_add */ 
  FALSE,          /* delivery_date_add */ 
  FALSE,          /* return_output */ 
  FALSE           /* restrict_to_path */ 
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up. */

void pipe_transport_init(transport_instance *tblock)
{
pipe_transport_options_block *ob = 
  (pipe_transport_options_block *)(tblock->options_block);
  
/* If pipe_as_creator is set, then uid/gid should not be set. */

if (ob->pipe_as_creator && (ob->uid >= 0 || ob->gid >= 0 || 
  ob->expand_uid != NULL || ob->expand_gid != NULL))
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
      "both pipe_as_creator and an explicit uid/gid are set for the %s "
        "transport", tblock->name);  
        
/* If a fixed uid field is set, then a gid field must also be set. */

if (ob->uid >= 0 && ob->gid < 0)
  log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
    "user set without group for the %s transport", tblock->name);  
  
/* Copy the uid, gid, home directory, local_smtp, pipe_as_creator, and
return_output into the generic slots. */

tblock->uid = ob->uid;
tblock->gid = ob->gid; 
tblock->local_smtp = ob->local_smtp;
tblock->home_dir = ob->directory;
tblock->expand_uid = ob->expand_uid;
tblock->expand_gid = ob->expand_gid;
tblock->deliver_as_creator = ob->pipe_as_creator;
tblock->return_output = ob->return_output;
}




/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface details. */

void pipe_transport_entry(
  transport_instance *tblock,      /* data for this instantiation */
  address_item *addr)              /* address we are working on */
{
int i, fd, pid, outpid, rc;
int pipefd[2];
int envcount = 0;
pipe_transport_options_block *ob = 
  (pipe_transport_options_block *)(tblock->options_block);
BOOL smtp_dots = FALSE;
BOOL return_path_add = ob->return_path_add;
BOOL delivery_date_add = ob->delivery_date_add;
BOOL expand_arguments;
char *argv[60];   
char *envp[20];
char *env_local_part;
char *s, *ss, *cmd;

DEBUG(2) debug_printf("%s transport entered\n", tblock->name);

/* Set up for the good case */

addr->transport_return = OK;
addr->errno = 0;

/* Pipes are not accepted as general addresses, but they can be generated from 
.forward files or alias files. In those cases, the command to be obeyed is
pointed to by addr->local_part; it starts with the pipe symbol. In other cases,
the command is supplied as one of the pipe transport's options. */

if (addr->local_part[0] == '|') 
  {
  cmd = addr->local_part + 1; 
  while (isspace(*cmd)) cmd++; 
  expand_arguments = addr->expand_pipe; 
  } 
else
  {  
  cmd = ob->cmd; 
  expand_arguments = TRUE;   
  }   

/* Split the command up into arguments terminated by white space. Lose
trailing space at the start and end. Quoted arguments can contain \\ and
\" escapes. Copy each argument into a new string. The total length can
be no longer than the length of the original. */

i = 0;
s = cmd;
ss = store_malloc((int)strlen(s) + 1);
while (isspace(*s)) s++;

while (*s != 0 && i < sizeof(argv)/sizeof(char *) - 2)
  {
  argv[i++] = ss;
  
  /* Handle quoted arguments */
   
  if (*s == '\"')
    {
    s++; 
    while (*s != 0)
      {
      if (*s == '\"') break;
      
      /* Handle escape sequences */
       
      else if (*s == '\\')
        {
        char ch = *(++s);
        if (ch == 0) break; 

        if (isdigit(ch) && ch != '8' && ch != '9')
          {
          ch -= '0';
          if (isdigit(s[1]) && s[1] != '8' && s[1] != '9')
            {
            ch = ch * 8 + *(++s) - '0';
            if (isdigit(s[1]) && s[1] != '8' && s[1] != '9')
              ch = ch * 8 + *(++s) - '0';
            }
          }
        else switch(ch)
          {
          case 'n':  ch = '\n'; break;
          case 'r':  ch = '\r'; break;
          case 't':  ch = '\t'; break;
          case 'x':
          ch = 0;
          if (isxdigit(s[1]))
            {
            ch = ch * 16 +
              strchr(hex_digits, tolower(*(++s))) - hex_digits;
            if (isxdigit(s[1])) ch = ch * 16 +
              strchr(hex_digits, tolower(*(++s))) - hex_digits;
            }
          break;
          }
        *ss++ = ch;
        s++; 
        }
        
      /* Not an escape sequence */
       
      else *ss++ = *s++;          
      }
      
    /* Advance past terminator */
     
    if (*s != 0) s++;   
    }  
 
  /* Argument not in quotes is terminated by EOL or white space */
   
  else while (*s != 0 && !isspace(*s)) *ss++ = *s++;
  
  /* Terminate the current argument, and skip trailing spaces. */
      
  *ss++ = 0;  
  while (isspace(*s)) s++;
  }
   
argv[i] = (char *)0;

/* If *s != 0 we have run out of argument slots. */

if (*s != 0)
  {
  addr->transport_return = FAIL;
  addr->message = string_sprintf("Too many arguments to command \"%s\" in "
    "%d transport", cmd, tblock->name);
  return;
  }       
  
/* Expand each individual argument if required. Expansion happens for pipes set 
up in filter files and with directly-supplied commands. It does not happen if 
the pipe comes from a traditional .forward file. A failing expansion is a big
disaster if the command came from the transport's configuration; if it came
from a user it is just a normal failure. */

DEBUG(7) 
  {
  debug_printf("pipe command:\n"); 
  for (i = 0; argv[i] != (char *)0; i++)
    debug_printf("  argv[%d] = %s\n", i, string_printing(argv[i], FALSE)); 
  } 
 
if (expand_arguments)
  {
  for (i = 0; argv[i] != (char *)0; i++)
    {
    argv[i] = expand_string(argv[i]);
    if (argv[i] == NULL)
      {
      addr->transport_return = (addr->local_part[0] == '|')? FAIL : PANIC;
      addr->message = string_sprintf("Expansion of \"%s\" "
        "from command \"%s\" in %s transport failed", 
        argv[i], cmd, tblock->name);
      return;
      }    
    }
  
  DEBUG(7) 
    {
    debug_printf("pipe command after expansion:\n"); 
    for (i = 0; argv[i] != (char *)0; i++)
      debug_printf("  argv[%d] = %s\n", i, string_printing(argv[i], FALSE)); 
    } 
  }   
 
    
/* If restrict_to_path is set, check that the name of the command does not
contain any slashes. */

if (ob->restrict_to_path) 
  {
  if (strchr(argv[0], '/') != NULL) 
    {
    addr->transport_return = FAIL;
    addr->message = string_sprintf("\"/\" found in \"%s\" (command for %s "
      "transport) - failed for security reasons", cmd, tblock->name);
    return;
    }
  } 
  
/* If the command is not an absolute path, search the PATH directories
for it. */

if (argv[0][0] != '/')
  {
  char *p;
  for (p = string_nextinlist(ob->path, ':'); p != NULL; 
       p = string_nextinlist(NULL, ':'))
    {
    struct stat statbuf;
    sprintf(big_buffer, "%s/%s", p, argv[0]);
    if (stat(big_buffer, &statbuf) == 0)
      {  
      argv[0] = string_copy(big_buffer);
      break; 
      }
    }
  if (p == NULL)
    {
    addr->transport_return = FAIL;
    addr->message = string_sprintf("\"%s\" command not found for %s transport",
      argv[0], tblock->name);
    return;     
    }  
  }


/* Set up the environment for the command. The LOCAL_PART variable is either
the current local address (from a smartuser director) or the parent of a pipe
generated by another director. */

env_local_part = 
  ((addr->local_part[0] == '/' || addr->local_part[0] == '|') &&
    addr->parent != NULL)?
      addr->parent->local_part : addr->local_part;

envp[envcount++] = string_sprintf("LOCAL_PART=%s", env_local_part);
envp[envcount++] = string_sprintf("LOGNAME=%s", env_local_part);
envp[envcount++] = string_sprintf("USER=%s", env_local_part);
envp[envcount++] = string_sprintf("HOME=%s", (deliver_home == NULL)?
  "" : deliver_home);
envp[envcount++] = string_sprintf("DOMAIN=%s", addr->domain);
envp[envcount++] = string_sprintf("MESSAGE_ID=%s", message_id_external);
envp[envcount++] = string_sprintf("PATH=%s", ob->path);
envp[envcount++] = string_sprintf("QUALIFY_DOMAIN=%s", qualify_domain_sender);
envp[envcount++] = string_sprintf("SENDER=%s", sender_address);
envp[envcount++] = "SHELL=/bin/sh";

if (addr->host_list != NULL)
  envp[envcount++] = string_sprintf("HOST=%s", addr->host_list->name);

envp[envcount] = NULL;


/* If the -N option is set, can't do any more. */

if (dont_deliver)
  {
  debug_printf("*** delivery by %s transport bypassed by -N option",
    tblock->name);
  addr->transport_return = OK;
  return;     
  } 


/* Handling the output from the pipe is tricky. If a file for catching this 
output is provided, we could just hand that fd over to the pipe, but this isn't 
very safe because the pipe might loop and carry on writing for ever (which is 
exactly what happened in early versions of Exim). Therefore we must hand over a 
pipe fd, read our end of the pipe and count the number of bytes that come 
through, chopping the sub-process if it exceeds some limit. 

However, this means we want to run a sub-process with both its input and output 
attached to pipes. We can't handle that easily from a single parent process 
using straightforward code such as the transport_write_message() function 
because the subprocess might not be reading its input because it is trying to 
write to a full output pipe. The complication of redesigning the world to 
handle this is too great - simpler just to run another process to do the 
reading of the output pipe. */


/* Make the pipe for handling the output - do this always, even if a 
return_file is not provided. */

if (pipe(pipefd) < 0)
  {
  addr->transport_return = DEFER;
  addr->message = string_sprintf(
    "Failed to create pipe for handling output in %s transport",
      tblock->name);
  return;        
  } 

/* As this is a local transport, we are already running with the required
uid/gid, so pass -1 to child_open to indicate no change. */

if ((pid = child_open(argv, envp, ob->umask, -1, -1, &fd, 
     pipefd[pipe_write])) < 0)
  {
  addr->transport_return = DEFER;
  addr->message = string_sprintf(
    "Failed to create child process for %s transport", tblock->name);
  return;
  }      
  
/* Close off the end of the output pipe we are not using. */

close(pipefd[pipe_write]);

/* Now fork a process to handle the output that comes down the pipe. */

if ((outpid = fork()) < 0)
  {
  addr->errno = errno;
  addr->transport_return = DEFER;
  addr->message = string_sprintf(
    "Failed to create process for handling output in %s transport",
      tblock->name);
  close(pipefd[pipe_read]);
  return;          
  }
  
/* This is the code for the output-handling subprocess. Read from the pipe
in chunks, and write to the return file if one is provided. Keep track of
the number of bytes handled. If the limit is exceeded, try to kill the 
subprocess, and in any case close the pipe and exit, which should cause the
subprocess to fail. */

if (outpid == 0)
  {
  int count = 0;
  close(fd); 
  set_process_info("reading output from %s", cmd);
  while ((rc = read(pipefd[pipe_read], big_buffer, BIG_BUFFER_SIZE)) > 0)
    {
    if (addr->return_file >= 0)
      write(addr->return_file, big_buffer, rc); 
    count += rc;
    if (count > ob->max_output)
      {
      char *message = "\n\n*** Too much output - remainder discarded ***\n"; 
      DEBUG(2) debug_printf("Too much output from pipe - killed\n");
      if (addr->return_file >= 0) 
        write(addr->return_file, message, (int)strlen(message));
      kill(pid, SIGKILL);
      break; 
      }   
    }
  close(pipefd[pipe_read]);   
  _exit(0);
  }
  
close(pipefd[pipe_read]);  /* Not used in this process */
 

/* Carrying on now with the main parent process. Attempt to write the message
to it down the pipe. It is a fallacy to think that you can detect write errors
when the sub-process fails to read the pipe. The parent process may complete
writing and close the pipe before the sub-process completes. Sleeping for a bit
here lets the sub-process get going, but it may still not complete. So we
ignore all writing errors. */

DEBUG(2) debug_printf("Writing message to pipe\n");

/* If the local_smtp option is not unset, we need to write SMTP prefix 
information. The various different values for batching are handled outside; if 
there is more than one address available here, all must be included. Force
SMTP dot-handling. */

if (ob->local_smtp != local_smtp_off)
  {
  address_item *a;
  char *sender = (addr->errors_address != NULL)? 
    addr->errors_address : sender_address;
  smtp_dots = TRUE;
  return_path_add = delivery_date_add = FALSE; 
   
  write(fd, "MAIL FROM: <", 12);
  write(fd, sender, (int)strlen(sender));
  write(fd, ">\n", 2); 
   
  for (a = addr; a != NULL; a = a->next)
    {
    write(fd, "RCPT TO: <", 10);
    if (a->local_part[0] == ',' || a->local_part[0] == ':')
      { 
      write(fd, "@", 1);
      write(fd, a->domain, (int)strlen(a->domain));
      write(fd, a->local_part, (int)strlen(a->local_part));  
      }
    else
      {
      write(fd, a->local_part, (int)strlen(a->local_part));
      write(fd, "@", 1);
      write(fd, a->domain, (int)strlen(a->domain));
      }        
    write(fd, ">\n", 2);
    }          
     
  write(fd, "DATA\n", 5);
  } 

/* Now any other configured prefix. */
 
if (ob->prefix != NULL)
  {
  char *prefix = expand_string(ob->prefix);
  if (prefix == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (prefix for %s "
      "transport) failed", ob->prefix, tblock->name);
    return;
    }    
  write(fd, prefix, (int)strlen(prefix));
  } 

(void) transport_write_message(fd, return_path_add, delivery_date_add,
  ob->from_hack, FALSE, smtp_dots, addr->errors_address, 0);   /* no CRLF */

/* Now any configured suffix */
  
if (ob->suffix != NULL)
  {
  char *suffix = expand_string(ob->suffix);
  if (suffix == NULL)
    {
    addr->transport_return = PANIC;
    addr->message = string_sprintf("Expansion of \"%s\" (suffix for %s "
      "transport) failed", ob->suffix, tblock->name);
    return;
    }    
  write(fd, suffix, (int)strlen(suffix));
  } 
 
/* If local_smtp, write the terminating dot. */

if (ob->local_smtp != local_smtp_off) write(fd, ".\n", 2);

/* OK, the writing is now all done. Close the pipe. */

(void) close(fd);

/* Wait for the child process to complete. If requested, fail when the
called command failed. We need an exit code to indicate deferral. A number
of systems contain the following line in sysexits.h:

  #define EX_TEMPFAIL 75 temp failure; user is invited to retry
  
Based on this, use exit code 75 to mean "defer". */
   
if ((rc = child_close(pid)) != 0 && !ob->ignore_status)
  {
  addr->transport_return = (rc == 75)? DEFER : FAIL;
  addr->message = 
    string_sprintf("Child process of %s transport returned %d for command %s",
      tblock->name, rc, cmd);    
  } 
  
/* The child_close() function just calls wait() until the required pid
is returned. Therefore it might already have waited for the output handling
process. In case it hasn't, do some more waiting here. All subprocesses
should be done before we pass this point. */

while (wait(&rc) >= 0);

DEBUG(2) debug_printf("%s transport yielded %d\n", tblock->name, 
  addr->transport_return);
}

/* End of transport/pipe.c */
