/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */



#include "../exim.h"
#include "lookuphost.h"



/* Options specific to the lookuphost router. */

optionlist lookuphost_router_options[] = {
  { "gethostbyname",   opt_bool,
      (void *)(offsetof(lookuphost_router_options_block, gethostbyname)) },
  { "mx_domains",         opt_stringptr,
      (void *)(offsetof(lookuphost_router_options_block, mx_domains)) },
  { "non_mx_domains",     opt_stringptr,
      (void *)(offsetof(lookuphost_router_options_block, non_mx_domains)) },
  { "qualify_single",  opt_bool,
      (void *)(offsetof(lookuphost_router_options_block, qualify_single)) },
  { "rewrite_headers", opt_bool,
      (void *)(offsetof(lookuphost_router_options_block, rewrite_headers)) }, 
  { "search_parents",  opt_bool,
      (void *)(offsetof(lookuphost_router_options_block, search_parents)) },
  { "self_mx",         opt_stringptr,
      (void *)(offsetof(lookuphost_router_options_block, self_mx)) },      
  { "widen_domains",   opt_stringptr,
      (void *)(offsetof(lookuphost_router_options_block, widen_domains)) } 
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int lookuphost_router_options_count =
  sizeof(lookuphost_router_options)/sizeof(optionlist);

/* Default private options block for the lookuphost router. */

lookuphost_router_options_block lookuphost_router_option_defaults = {
  FALSE,           /* gethostbyname */
  TRUE,            /* qualify_single */
  TRUE,            /* search_parents */
  TRUE,            /* rewrite_headers */ 
  FALSE,           /* self_mx_rewrite */ 
  self_mx_freeze,  /* self_mx_code */ 
  NULL,            /* self_mx */ 
  NULL,            /* widen_domains */ 
  NULL,            /* mx_domains */
  NULL,            /* non_mx_domains */  
  NULL,            /* re_mx_domains */
  NULL             /* re_non_mx_domains */   
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up. */

void lookuphost_router_init(router_instance *rblock)
{
lookuphost_router_options_block *ob =
  (lookuphost_router_options_block *)(rblock->options_block);

/* There must be a transport. */

if (rblock->transport == NULL)
  log_write(LOG_PANIC_DIE, "Exim configuration error for %s router:\n  "
    "a transport specification is required", rblock->name);
    
/* Decode any self_mx setting */

if (ob->self_mx != NULL)
  {
  if (strcmp(ob->self_mx, "freeze") == 0) ob->self_mx_code = self_mx_freeze;
  else if (strcmp(ob->self_mx, "defer") == 0) ob->self_mx_code = self_mx_defer;
  else if (strcmp(ob->self_mx, "send") == 0) ob->self_mx_code = self_mx_send;
  else if (strcmp(ob->self_mx, "fail_soft") == 0) ob->self_mx_code = self_mx_fail;
  else if (strcmp(ob->self_mx, "fail_hard") == 0) ob->self_mx_code = self_mx_forcefail;
  else if (strncmp(ob->self_mx, "local:", 6) == 0)
    {
    char *s = ob->self_mx + 6;
    while (isspace(*s)) s++;
    if (strncmp(s, "rewrite:", 8) == 0)
      {
      ob->self_mx_rewrite = TRUE;
      s += 8;
      while (isspace(*s)) s++;
      }     
    if (match_isinlist(s, local_domains, &re_local_domains))
      { 
      ob->self_mx_code = self_mx_local;
      ob->self_mx = s;
      }  
    else log_write(LOG_PANIC_DIE, "Exim configuration error for %s router:\n  "
      "%s is not a local domain", rblock->name, ob->self_mx); 
    }
  else log_write(LOG_PANIC_DIE, "Exim configuration error for %s router:\n  "
      "%s is not valid for the self_mx option", rblock->name, ob->self_mx);     
  } 
}



/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface details */ 

int lookuphost_router_entry(
  router_instance *rblock,        /* data for this instantiation */
  address_item *addr,             /* address we are working on */
  address_item **addr_local,      /* add it to this if successful & local */
  address_item **addr_remote)     /* add it to this if successful & remote */
{
host_item h;
int rc;
lookuphost_router_options_block *ob =
  (lookuphost_router_options_block *)(rblock->options_block);
char *widen = NULL; 
char *fully_qualified_name;

DEBUG(2) 
  debug_printf("%s router called for %s: %s lookup: route_domain = %s\n", 
  rblock->name, addr->orig, (ob->gethostbyname)? "gethostbyname" : "dns",
  addr->route_domain);


/* Set up an initial host item, and then call the appropriate function to fill
in its address and chain on additional host_items if necessary. */

h.next = NULL;
h.name = addr->route_domain;
h.address = NULL;
h.mx = -1;
h.status = hstatus_unknown;
h.why = hwhy_unknown;
h.last_try = 0;


/* Loop to cope with explicit widening of domains as configured. */

if (ob->widen_domains != NULL)
  widen = string_nextinlist(ob->widen_domains, ':'); 

for (;;)
  {
  if (ob->gethostbyname) 
    rc = host_find_byname(&h, &fully_qualified_name);      
    
  /* Unfortunately, we cannot set the mx_only option in advance, because the 
  DNS lookup may extend an unqualified name. Therefore, we must do the test
  subsequently. The MX-only test cannot be applied to source-routed addresses,
  which can be identified by a local-part starting with ',' or ':'. */
    
  else
    { 
    rc = host_find_bydns(&h, FALSE, FALSE, ob->qualify_single, 
      ob->search_parents, &fully_qualified_name);
      
    if (rc == HOST_FOUND && h.mx < 0 &&        /* Found A records only */
        addr->local_part[0] != ',' &&          /* Not source routed */
        addr->local_part[0] != ':' &&  
        (ob->non_mx_domains == NULL ||         /* Not in non_mx_domains list */
        !match_isinlist(fully_qualified_name, ob->non_mx_domains,
        &(ob->re_non_mx_domains))) &&
        (ob->mx_domains != NULL &&             /* In mx-domains list */
        match_isinlist(fully_qualified_name, ob->mx_domains,
        &(ob->re_mx_domains))))
      {
      DEBUG(2) debug_printf("%s router rejected %s: no MX record(s)\n",
        rblock->name, fully_qualified_name);
      return FAIL;     
      }   
    }   
  
  /* Deferral returns forthwith, and anything other than failure breaks the 
  loop. */
  
  if (rc == HOST_FIND_AGAIN) 
    {
    addr->message = "host lookup did not complete"; 
    return DEFER;
    }
      
  if (rc != HOST_FIND_FAILED) break;

  /* If there are any configured widening domains, widen the name we
  have got and try again. Otherwise, fail. */ 

  if (widen == NULL) return FAIL; 
  h.name = string_sprintf("%s.%s", addr->route_domain, widen);
  widen = string_nextinlist(NULL, ':'); 
  DEBUG(2) debug_printf("%s router widened %s to %s\n", rblock->name,
    addr->route_domain, h.name);  
  }
 


/* If the original domain name has been changed as a result of the host lookup,
change the name in the address structure and request header rewrites if so
configured. Then check to see if the fully qualified name is in fact one of the
local domain names. If so, return ISLOCAL so that the address can be passed
back to the directors, and force header rewriting. */

if (strcmp(addr->route_domain, fully_qualified_name) != 0)
  {
  addr->route_domain = fully_qualified_name;
  addr->rewrite_headers = ob->rewrite_headers;
  if (match_isinlist(fully_qualified_name, local_domains, &re_local_domains))
    {
    addr->rewrite_headers = TRUE;  
    return ISLOCAL; 
    } 
  } 


/* If the yield is HOST_FOUND_LOCAL, the remote domain name found MX records
with the lowest numbered one pointing to a host with an IP address that is set
on one of the interfaces of this machine. This can happen quite legitimately if
the original name was a shortened form of a local domain, but if so, the fully
qualified name will be a local domain and will have been detected above. If it
is not, there may be some kind of configuration error or lookuphost error. 

The action to be taken can be configured by the self_mx option as follows:

.  freeze: Log the incident, freeze, and return ERROR
.  defer:  Log the incident and return ERROR
.  send:   Carry on with the delivery regardless - this makes sense only
           if the SMTP listener on this machine is a differently configured
           MTA
.  <local-domain>: Change the domain to the given local domain and return
   ISLOCAL so it gets passed back to the directors. The domain was checked
   for being local at initialization. 
  
The default is freeze, since this state is normally a serious error. */

if (rc == HOST_FOUND_LOCAL)
  {
  switch (ob->self_mx_code)
    {
    case self_mx_freeze:   
    log_write(LOG_MAIN, "lowest MX record for %s points to local host",
      addr->route_domain); 
    addr->message = "lowest numbered MX record points to local host";   
    addr->special_action = SPECIAL_FREEZE; 
    return ERROR;
    
    case self_mx_defer: 
    addr->message = "lowest numbered MX record points to local host";   
    return DEFER;
    
    case self_mx_local:
    DEBUG(2) debug_printf("lowest MX record for %s points to local host: "
      "domain changed to %s\n", addr->route_domain, ob->self_mx);  
    addr->route_domain = string_copy(ob->self_mx);
    addr->rewrite_headers = ob->self_mx_rewrite; 
    return ISLOCAL; 

    case self_mx_send:
    DEBUG(2) debug_printf("lowest MX record for %s points to local host: "
      "configured to try delivery anyway\n", addr->route_domain);  
    break;         /* Carry on processing */  
    
    case self_mx_fail:
    DEBUG(2) debug_printf("lowest MX record for %s points to local host: "
      "%s router failed soft\n", addr->route_domain, rblock->name);  
    addr->message = "lowest numbered MX record points to local host";   
    return FAIL; 
    break;
    
    case self_mx_forcefail:
    DEBUG(2) debug_printf("lowest MX record for %s points to local host: "
      "%s router failed hard\n", addr->route_domain, rblock->name);  
    addr->message = "lowest numbered MX record points to local host";   
    return FORCEFAIL; 
    break;    
    }
  } 

/* Get store in which to preserve the original host item, chained on
to the address. */

addr->host_list = store_malloc(sizeof(host_item));
addr->host_list[0] = h;

/* Fill in the transport (known to exist), queue the address for local or
remote delivery, and yield success. */

addr->transport = rblock->transport;
if (addr->transport->info->local)
  {
  addr->next = *addr_local;
  *addr_local = addr;
  }
else
  {
  addr->next = *addr_remote;
  *addr_remote = addr;
  }

DEBUG(9) debug_printf("%s router succeeded\n", rblock->name);
return OK;
}

/* End of routers/lookuphost.c */
