/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions for parsing addresses */


#include "exim.h"


static char  fixed_phrase[256];
static char *last_comment_position;



/* In stand-alone mode, provide a replacement for deliver_make_addr()
and rewrite_address so as to avoid having to drag in too much redundant
apparatus. */

#ifdef STAND_ALONE

static address_item *deliver_make_addr(char *address)
{
address_item *addr = store_malloc(sizeof(address_item));
addr->next = NULL;
addr->parent = NULL;
addr->orig = address;
return addr;
}

static char *rewrite_address(char *recipient, BOOL dummy1, BOOL dummy2)
{
return recipient;
}

#endif




/*************************************************
*             Find the end of an address         *
*************************************************/

/* Scan over an address looking for a terminating comma
or end of the string. */

char *parse_find_address_end(char *s)
{
while (*s != 0 && *s != ',')
  {
  /* Skip quoted items that are not inside brackets. Note that
  quoted pairs are allowed inside quoted strings. */

  if (*s == '\"')
    {
    while (*(++s) != 0)
      {
      if (*s == '\\' && s[1] != 0) s++;
        else if (*s == '\"') { s++; break; }
      }
    }

  /* Skip comments, which may include nested brackets, but quotes
  are not recognized inside comments, though quoted pairs are. */

  else if (*s == '(')
    {
    int level = 1;
    while (*(++s) != 0)
      {
      if (*s == '\\' && s[1] != 0) s++;
        else if (*s == '(') level++;
          else if (*s == ')' && --level <= 0) { s++; break; }
      }
    }

  /* Non-special character; just advance. */

  else s++;
  }

return s;
}



/*************************************************
*            Find last @ in an address           *
*************************************************/

/* We have to be a bit more clever than just a plain search, in order to handle
addresses like "thing@thong"@a.b.c correctly. Since quotes may not legally
be part of a domain name, we can give up on hitting the first quote when
searching from the right. */

char *parse_find_at(char *s)
{
char *t = s + (int)strlen(s);
while (--t >= s)
  {
  if (*t == '@') return t;
  if (*t == '\"') return NULL;
  }
return NULL;
}




/***************************************************************************
* In all the functions below that read a particular object type from       *
* the input, return the new value of the pointer s (the first argument),   *
* and put the object into the store pointed to by t (the second argument), *
* adding a terminating zero. If no object is found, t will point to zero   *
* on return.                                                               *
***************************************************************************/


/*************************************************
*          Skip white space and comment          *
*************************************************/

/* Algorithm:
  (1) Skip spaces.
  (2) If char not '(', return.
  (3) Skip till matching ')', not counting any characters
      escaped with '\'.
  (4) Move past ')' and goto (1).

The start of the last potential comment position is remembered to
make it possible to ignore comments at the end of compound items. */

static char *skip_comment(char *s)
{
last_comment_position = s;
while (*s)
  {
  int c, level;
  while (isspace(*s)) s++;
  if (*s != '(') break;
  level = 1;
  while((c = *(++s)) != 0)
    {
    if (c == '(') level++;
    else if (c == ')') { if (--level <= 0) { s++; break; } }
    else if (c == '\\' && s[1] != 0) s++;
    }
  }
return s;
}



/*************************************************
*                 Read an atom                   *
*************************************************/

/* An atom is a sequence of any characters except specials, space,
and controls. The specials are ( ) < > @ , ; : \ " . [ and ]. */

static char *read_atom(char *s, char *t)
{
s = skip_comment(s);
while (!iscntrl(*s) && strchr(" ()<>@,;:\\\".[]", *s) == NULL)
  *t++ = *s++;
*t = 0;
return skip_comment(s);
}



/*************************************************
*                 Read a word                    *
*************************************************/

/* A <word> is either a quoted string, or an <atom>. */

static char *read_word (char *s, char *t)
{
s = skip_comment(s);
if (*s == '\"')
  {
  int c;
  *t++ = '\"';
  while ((c = *(++s)) > 0 && c != '\"')
    {
    *t++ = c;
    if (c == '\\' && s[1] != 0) *t++ = *(++s);
    }
  *t++ = '\"';
  *t = 0;
  return skip_comment(s + ((c == '\"')? 1:0));
  }
else return read_atom(s, t);
}




/*************************************************
*             Read a domain                      *
*************************************************/

/* A domain is a sequence of sub-domains, separated by dots.
A subdomain is an atom. (Need to consider domain literals later.)
Make sure the output is set to the null string if there is a syntax
error as well as if there is no domain at all. */

static char *read_domain(char *s, char *t)
{
char *tt = t;
for (;;)
  {
  s = skip_comment(s);
  s = read_atom(s, t);
  if (*t == 0) { *tt = 0; return s; }
  t += (int)strlen(t);
  if (*s != '.') break;
  *t++ = *s++;
  }
return s;
}



/*************************************************
*            Read a local-part                   *
*************************************************/

/* A local-part is a sequence of words, separated by periods.
Make sure the output is set to the null string if there is a syntax
error as well as if there is no domain at all. */

static char *read_local_part(char *s, char *t)
{
char *tt = t;
for (;;)
  {
  s = skip_comment(s);
  s = read_word(s, t);
  if (*t == 0) { *tt = 0; return s; }
  t += (int)strlen(t);
  if (*s != '.') break;
  *t++ = *s++;
  }
return s;
}


/*************************************************
*            Read route part of route-addr       *
*************************************************/

/* The pointer is at the initial "@" on entry. Return it following
the terminating colon. */

static char *read_route(char *s, char *t, char **errorptr)
{
*errorptr = NULL;

while (*s == '@')
  {
  *t++ = '@';
  s = read_domain(s+1, t);
  if (*t == 0)
    {
    *errorptr = "missing or malformed route domain";
    return s;
    }
  t += (int)strlen(t);
  if (*s != ',') break;
  *t++ = *s++;
  s = skip_comment(s);
  }

if (*s == ':') *t++ = *s++;
  else *errorptr = "colon expected after route";

*t = 0;
return skip_comment(s);
}



/*************************************************
*                Read addr-spec                  *
*************************************************/

/* Addr-spec is local-part@domain. We make the domain optional -
the expected terminator for the whole thing is passed to check this.
This function is called only when we know we have a route-addr. */

static char *read_addr_spec(char *s, char *t, int term, char **errorptr,
   char **domainptr)
{
*errorptr = NULL;
s = read_local_part(s, t);
if (*t == 0) *errorptr = "local part missing in route address"; else
  {
  t += (int)strlen(t);
  if (*s != term)
    {
    if (*s != '@') *errorptr = "@ missing in route address"; else
      {
      *t++ = *s++;
      *domainptr = t;
      s = read_domain(s, t);
      if (*t == 0)
        *errorptr = "domain missing or malformed in route address";
      }
    }
  }
return s;
}



/*************************************************
*           Extract route address                *
*************************************************/

/* This function extracts an address from a full RFC822 mailbox and returns it
in a piece of malloc'ed store. We take the easy way and get a piece of store
the same size as the input, and then copy into it whatever is necessary. If we
cannot find a valid address (syntax error), return NULL, and point the error
pointer to the reason. The arguments "start" and "end" are used to return the
offsets of the first and one past the last characters in the original address
of the address that has been extracted, to aid in re-writing. The argument
"domain" is set to point to the first character after "@" in the final part of
the returned address, or zero if there is no @. Note: if the address is of the
form @domain,...:route-addr "domain" refers to the final route-addr. */


#define FAILED(s) { *errorptr = s; goto PARSE_FAILED; }

char *parse_extract_address(char *mailbox, char **errorptr,
     int *start, int *end, int *domain, BOOL allow_null)
{
char *yield = store_malloc((int)strlen(mailbox) + 1);
char *startptr, *endptr;
char *s = mailbox;
char *t = yield;

*domain = 0;

/* At the start of the string we expect either an addr-spec or a phrase
preceding a <route-addr>. If we find an initial word followed by
a dot, strict interpretation of the RFC would cause it to be taken
as the start of an addr-spec. However, many mailers break the rules
and use addresses of the form "a.n.other <ano@somewhere>" and so we
allow this case. */

s = skip_comment(s);
startptr = s;                 /* In case addr-spec */
s = read_local_part(s, t);    /* Dot separated words */

/* If the terminator is neither < nor @ then the format of the address
must either be a bare local-part (we are now at the end), or a phrase
followed by a route-addr (more words must follow). */

if (*s != '@' && *s != '<')
  {
  if (*s == 0)
    {
    if (*t == 0) FAILED("empty address");
    endptr = last_comment_position;
    goto PARSE_SUCCEEDED;              /* Bare local part */
    }

  /* Expect phrase route-addr, but allow dots in the phrase; complete
  the loop only when '<' is encountered - end of string will produce
  a null local_part and therefore fail. We don't need to keep updating
  t, as the phrase isn't to be kept. */

  while (*s != '<')
    {
    s = read_local_part(s, t);
    if (*t == 0) FAILED("expected word or '<'");
    }
  /* Assert *s == '<' */
  }

/* At this point the next character is either '@' or '<'. If it is '@',
only a single local-part has previously been read. An angle bracket
signifies the start of a <route-addr>. Throw away anything we have saved so
far before processing it. Note that this is "if" rather than "else if"
because it's also used after reading a preceding phrase. */

if (*s == '<')
  {
  char *domainptr = yield;

  t = yield;
  startptr = s + 1;
  s = skip_comment(s+1);

  /* Read an optional series of routes, each of which is a domain. They
  are separated by commas and terminated by a colon. */

  if (*s == '@')
    {
    s = read_route(s, t, errorptr);
    if (*errorptr != NULL) goto PARSE_FAILED;
    t += (int)strlen(t);
    }

  /* Now an addr-spec, terminated by '>'. If there is no preceding route,
  we must allow an empty addr-spec if allow_null is TRUE, to permit the
  address "<>" in some circumstances. */

  if (allow_null && t == yield && *s == '>')
    {
    *t = 0;
    *errorptr = NULL;
    }
  else
    {
    s = read_addr_spec(s, t, '>', errorptr, &domainptr);
    *domain = domainptr - yield;
    }

  endptr = s;
  if (*errorptr != NULL) goto PARSE_FAILED;
  if (*s++ != '>') FAILED("> missing at end of route address");
  s = skip_comment(s);
  }

/* Hitting '@' after the first local-part means we have definitely got an
addr-spec, on a strict reading of the RFC, and the rest of the string
should be the domain. However, for flexibility we allow for a route-address
not enclosed in <> as well, which is indicated by an empty first local
part preceding '@'. */

else if (*t == 0)
  {
  char *domainptr = yield;
  s = read_route(s, t, errorptr);
  if (*errorptr != NULL) goto PARSE_FAILED;
  t += (int)strlen(t);
  s = read_addr_spec(s, t, 0, errorptr, &domainptr);
  if (*errorptr != NULL) goto PARSE_FAILED;
  *domain = domainptr - yield;
  endptr = last_comment_position;
  }

/* This is the strict case of local-part@domain. */

else
  {
  t += (int)strlen(t);
  *t++ = *s++;
  *domain = t - yield;
  s = read_domain(s, t);
  if (*t == 0) FAILED("domain missing or malformed after @");
  endptr = last_comment_position;
  }

/* Use goto to get here from the bare local part case. Arrive by falling
through for other cases. Endptr may have been moved over whitespace, so
move it back past white space if necessary. */

PARSE_SUCCEEDED:
  if (*s != 0) FAILED("extraneous characters at end of address");
  *start = startptr - mailbox;      /* Return offsets */
  while (isspace(endptr[-1])) endptr--;
  *end = endptr - mailbox;
  return yield;

/* Use goto (via the macro FAILED) to get to here from a variety of places. */

PARSE_FAILED:
  store_free(yield);
  return NULL;
}

#undef FAILED



/*************************************************
*            Fix up an RFC 822 "phrase"          *
*************************************************/

/* This function is called to repair any syntactic defects in the "phrase" part
of an RFC822 address. In particular, it is applied to the user's name as read
from the passwd file when accepting a local message.

If the string contains existing quoted strings or comments containing
freestanding quotes, then we just quote those bits that need quoting -
otherwise it would get awfully messy and probably not look good. If not, we
quote the whole thing if necessary. Thus

   John Q. Smith            =>  "John Q. Smith"
   John "Jack" Smith        =>  John "Jack" Smith
   John "Jack" Q. Smith     =>  John "Jack" "Q." Smith
   John (Jack) Q. Smith     =>  "John (Jack) Q. Smith"
   John ("Jack") Q. Smith   =>  John ("Jack") "Q." Smith
but
   John (\"Jack\") Q. Smith =>  "John (\"Jack\") Q. Smith"

Sheesh! This is tedious code. It is a great pity that the syntax of RFC822 is
the way it is...

Non-printing characters are turned into question marks, which do not need to be
quoted. The result is passed back in a static buffer, and must therefore be
copied - this makes sense, since it is usually going to be added to some other
string. */

char *parse_fix_phrase(char *s)
{
int ch;
BOOL quoted = FALSE;
char *t = fixed_phrase + 1;
char *yield = t;

while (isspace(*s)) s++;

while ((ch = (*s++) & 127) != 0)
  {
  if (!isprint(ch)) ch = '?';

  /* Copy over quoted strings, remembering we encountered one */

  if (ch == '\"')
    {
    *t++ = '\"';
    while ((ch = *s++ & 127) > 0 && ch != '\"')
      {
      *t++ = ch;
      if (ch == '\\' && *s != 0) *t++ = *s++ & 127;
      }
    *t++ = '\"';
    if (ch == 0) break;
    quoted = TRUE;
    }

  /* Copy over comments, noting if they contain freestanding quote
  characters */

  else if (ch == '(')
    {
    int level = 1;
    *t++ = '(';
    while((ch = *s++ & 127) != 0)
      {
      *t++ = ch;
      if (ch == '(') level++;
      else if (ch == ')') { if (--level <= 0) break; }
      else if (ch == '\\' && *s != 0) *t++ = *s++ & 127;
      else if (ch == '\"') quoted = TRUE;
      }
    if (ch == 0)
      {
      while (level--) *t++ = ')';
      break;
      }
    }

  /* Handle special characters that need to be quoted */

  else if (strchr(")<>@,;:\\.[]", ch) != NULL)
    {
    /* If hit previous quotes just make one quoted "word" */

    if (quoted)
      {
      char *tt = t++;
      while (*(--tt) != ' ' && *tt != '\"' && *tt != ')') tt[1] = *tt;
      tt[1] = '\"';
      *t++ = ch;
      while((ch = *s++ & 127) != 0)
        if (ch == ' ' || ch == '\"') break; else *t++ = ch;
      *t++ = '\"';
      }

    /* Else quote the whole string so far, and the rest up to
    any following quotes. */

    else
      {
      fixed_phrase[0] = '\"';
      yield--;
      *t++ = ch;

      while ((ch = *s++ & 127) != 0)
        {
        /* If hit subsequent quotes, insert our quote before any
        trailing spaces and go into quoted mode. */

        if (ch == '\"')
          {
          int count = 0;
          while (t[-1] == ' ') { t--; count++; }
          *t++ = '\"';
          while (count-- > 0) *t++ = ' ';
          quoted = TRUE;
          break;
          }

        /* If hit a subsequent comment, check it for unescaped quotes,
        and if so, end our quote before it. */

        else if (ch == '(')
          {
          char *ss = s;     /* Char after '(' */
          int level = 1;
          while((ch = *ss++ & 127) != 0)
            {
            if (ch == '(') level++;
            else if (ch == ')') { if (--level <= 0) break; }
            else if (ch == '\\' && ss[1] != 0) ss++;
            else if (ch == '\"') { quoted = TRUE; break; }
            }

          /* Comment contains unescaped quotes; end our quote before
          the start of the comment. */

          if (quoted)
            {
            int count = 0;
            while (t[-1] == ' ') { t--; count++; }
            *t++ = '\"';
            while (count-- > 0) *t++ = ' ';
            break;
            }

          /* Comment does not contain unescaped quotes; include it in
          our quote. */

          else
            {
            if (ch == 0) ss--;
            *t++ = '(';
            strncpy(t, s, ss-s);
            t += ss-s;
            s = ss;
            }
          }

        /* Not a comment or quote; include this character in our quotes. */

        else *t++ = ch;
        }
      }

    /* Add a final quote if we hit the end of the string, and move back the
    pointer to the char that terminated the quote. */

    if (ch == 0) *t++ = '\"';
    s--;
    }

  /* Non-special character; just copy it over */

  else *t++ = ch;
  }

*t = 0;
return yield;
}



/*************************************************
*          Extract addresses from a list         *
*************************************************/

/* This function is called by the forwardfile and aliasfile (and maybe other)
directors to scan a string containing a list of addresses separated by commas
(with optional white space) or by newlines, and to generate a chain of address
items from them.

The SunOS5 documentation for alias files is not very clear on the syntax; it
does not say that either a comma or a newline can be used for separation.
However, that is the way smail does it, so we follow suit.

If a # character is encountered in a white space position, then characters from
there to the next newline are skipped.

If an unqualified address begins with '\', just skip that character. This gives
compatibility with sendmail's use of \ to prevent looping. Exim has its own
loop prevention scheme which handles other cases too - see the code in
direct_address(). If a qualified address begins with '\' it is a syntax error.

An "address" can be a specification of a file or a pipe; the latter may often
need to be quoted because it may contain spaces, but we don't want to retain
the quotes. Quotes may appear in normal addresses too, and should be retained.
We can distinguish between these cases, because in addresses, quotes are used
only for parts of the address, not the whole thing. Therefore, we remove quotes
from items when they entirely enclose them, but not otherwise.

An "address" can also be of the form :include:pathname to include a list of
addresses contained in the specified file.

Any unqualified addresses are qualified with and rewritten if necessary, via
the rewrite_address() function.

Yield 0 if the string was successfully parsed; non-zero otherwise, with
a message pointed to by the error variable. A positive value is given if
the failure is simply a failure to open or subsequently stat an :include: file
- some callers treat this as a soft error - and a negative value is given
otherwise. 

If optional is TRUE, then failure to generate any addresses, without a syntax 
error, is not an error. 

If directory is not NULL, then require any :include: files to begin with the 
directory string AND be regular files. */

int parse_extract_addresses(char *s, address_item **anchor, char **error, 
  BOOL optional, char *directory)
{
int count = 0;

DEBUG(9) debug_printf("parse_extract_addresses: %s\n", s);

for (;;)
  {
  int len;
  char *ss;
  address_item *addr;
  BOOL inquote = FALSE; 

  for (;;)
    {
    while (isspace(*s) || *s == ',') s++;
    if (*s == '#') { while (*s != 0 && *s != '\n') s++; } else break;
    }

  if (*s == 0)
    {
    if (count > 0 || optional) return 0;
    *error = "no addresses generated";
    return -1;
    }

  /* Find the end of the next address. Quoted strings in addresses may contain
  escaped characters; I haven't found a proper specification of .forward or
  alias files that mentions the quoting properties, but it seems right to do
  the escaping thing in all cases. */

  for (ss = s; *ss != 0 && *ss != ',' && *ss != '\n' && *ss != '#'; ss++)
    {
    if (*ss == '\"')
      {
      while (*(++ss) != 0 && *ss != '\"')
        {
        if (*ss == '\\' && ss[1] != 0) ss++;
        }
      if (*ss == 0) break;
      }
    }

  /* We now have s->start and ss->end of the next address. Remove quotes
  if they completely enclose, remembering the address started with a quote
  for handling pipes and files. */

  len = ss - s;
  if (*s == '\"' && ss[-1] == '\"')
    {
    s++;
    len -= 2;
    inquote = TRUE; 
    }

  /* If there is a leading \ remove it if there is no @. We search
  for @ from the end, stopping if quotes are encountered. */

  if (*s == '\\')
    {
    char *t = ss;
    while (--t > s)
      {
      if (*t == '@') break;
      if (*t == '\"') { t = s; break; }
      }
    if (t <= s) { s++; len--; }
    }
    
  DEBUG(9) 
    {
    int save = s[len];
    s[len] = 0;  
    debug_printf("extracted: %s\n", s);
    s[len] = save; 
    } 

  /* If the address is of the form :include:pathname, read the file, and call
  this function recursively to extract the addresses from it. There are 
  possible security implications when called from forwardfile. If directory is 
  NULL, do no checks (we are running under a suitable uid). Otherwise, insist
  that the file name starts with the given directory and is a regular file. 
  If the directory is "*", no included files are allowed. */

  if (strncmp(s, ":include:", 9) == 0)
    {
    char *filebuf;
    char filename[256];
    struct stat statbuf;
    address_item *last;
    int extracted;
    FILE *f;

    strncpy(filename, s+9, len-9);
    filename[len-9] = 0;
    
    /* Insist on absolute path */
     
    if (filename[0]!= '/')
      {
      *error = string_sprintf("included file %s is not an absolute path",
        filename);
      return -1;
      }
   
    /* Check file name if required - this is requested only when not running
    under a suitable uid. */ 
    
    if (directory != NULL)
      {
      int len = (int)strlen(directory); 
      char *p = filename + len; 
       
      if (directory[0] == '*') 
        {
        *error = "included files not permitted"; 
        return -1;
        }  
      else if (strncmp(filename, directory, len) != 0 || *p != '/')
        { 
        *error = string_sprintf("included file %s is not in directory %s", 
          filename, directory);
        return -1;
        }
    
      /* It is necessary to check that every component inside the directory
      is NOT a symbolic link, in order to keep the file inside the directory.
      This is mighty tedious. It is also not totally foolproof in that it
      leaves the possibility of a race attack, but I don't know how to do
      any better. */  
      
      while (*p != 0)
        {
        int temp; 
        while (*(++p) != 0 && *p != '/');
        temp = *p;
        *p = 0;  
        if (lstat(filename, &statbuf) != 0)
          {
          *error = string_sprintf("failed to stat %s (component of included "
            "file)", filename); 
          *p = temp;
          return -1;
          }     
           
        *p = temp;

        if ((statbuf.st_mode & S_IFMT) == S_IFLNK)
          {
          *error = string_sprintf("included file %s in the %s directory "
            "involves a symbolic link", filename, directory);
          return -1;     
          }   
        }   
      }  

    /* Open and stat the file */
     
    if ((f = fopen(filename, "r")) == NULL)
      {
      *error = string_sprintf("failed to open included file %s: %s",
        filename, strerror(errno));
      return +1;
      }

    if (fstat(fileno(f), &statbuf) != 0)
      {
      *error = string_sprintf("failed to stat included file %s: %s",
        filename, strerror(errno));
      fclose(f);
      return +1;
      }
      
    /* If directory was checked, double check that we opened a regular file */
    
    if (directory != NULL && (statbuf.st_mode & S_IFMT) != S_IFREG)
      {
      *error = string_sprintf("included file %s is not a regular file in "
        "the %s directory", filename, directory);
      return -1;     
      }   
      
    /* Get a buffer and read the contents */

    filebuf = store_malloc(statbuf.st_size + 1);
    if (fread(filebuf, 1, statbuf.st_size, f) != statbuf.st_size)
      {
      *error = string_sprintf("error while reading included file %s: %s",
        filename, strerror(errno));
      fclose(f);
      return -1;
      }
    filebuf[statbuf.st_size] = 0;   
    fclose(f);

    addr = NULL;
    extracted = parse_extract_addresses(filebuf, &addr, error, optional,
      directory);
    store_free(filebuf);
    if (extracted != 0) return extracted;    /* error already set */

    last = addr;
    while (last->next != NULL) { count++; last = last->next; }
    last->next = *anchor;
    *anchor = addr;
    count++;
    }

  /* Else ensure address is syntactically correct and fully qualified if not a
  pipe or a file. For those we must handle quoting. It's not quite clear
  exactly what to do for partially quoted things, but the common case of
  having the whole thing in quotes is straightforward. If this was the case,
  inquote will have been set TRUE above. */

  else
    {
    int save = s[len];
    s[len] = 0;
       
    if (*s == '|' || *s == '/')
      {
      char *t = store_malloc((int)strlen(s) + 1);
      char *p = t;  
      char *q = s; 
      while (*q != 0)
        {
        if (inquote)
          {
          *p++ = (*q == '\\')? *(++q) : *q;
          q++; 
          }  
        else *p++ = *q++;
        }   
      *p = 0;   
      addr = deliver_make_addr(t);
      } 
    else
      {
      int start, end, domain;
      char *recipient;

      recipient = parse_extract_address(s, error, &start, &end, &domain, FALSE);

      if (recipient == NULL) 
        {
        *error = string_sprintf("%s in \"%s\"", *error, s); 
        s[len] = save;
        return -1;
        } 
         
      if (domain == 0) recipient = rewrite_address(recipient, TRUE, TRUE);
      addr = deliver_make_addr(recipient);
      }
       
    s[len] = save;

    /* Add to the output chain */

    addr->next = *anchor;
    *anchor = addr;
    count++;
    }

  /* Advance pointer for the next address */

  s = ss;
  }
}




/*************************************************
**************************************************
*             Stand-alone test program           *
**************************************************
*************************************************/

#ifdef STAND_ALONE
int main(void)
{
int start, end, domain;
char buffer[256];

printf("Testing parse_extract_address\n");

while (fgets(buffer, 256, stdin) != NULL)
  {
  char *out;
  char *errmess;
  buffer[(int)strlen(buffer) - 1] = 0;
  out = parse_extract_address(buffer, &errmess, &start, &end, &domain, FALSE);
  if (out == NULL) printf("*** bad address: %s\n", errmess); else
    {
    char extract[100];
    strncpy(extract, buffer+start, end-start);
    extract[end-start] = 0;
    printf("%s %d %d %d \"%s\"\n", out, start, end, domain, extract);
    store_free(out);
    }
  }

printf("Testing parse_fix_phrase\n");

while (fgets(buffer, 256, stdin) != NULL)
  {
  buffer[(int)strlen(buffer)-1] = 0;
  printf("%s\n", parse_fix_phrase(buffer));
  }

printf("Testing parse_extract_addresses\n");

while (fgets(buffer, 256, stdin) != NULL)
  {
  char *out;
  char *errmess;
  int extracted; 
  address_item *anchor = NULL;
  buffer[(int)strlen(buffer) - 1] = 0;
  if ((extracted = parse_extract_addresses(buffer, &anchor, &errmess)) == 0)
    {
    while (anchor != NULL)
      {
      address_item *addr = anchor;
      anchor = anchor->next;
      printf("%s\n", addr->orig);
      store_free(addr);
      }
    }
  else printf("Failed: %d %s\n", extracted, errmess);
  }

return 0;
}

#endif

/* End of parse.c */
