/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions for matching strings */


#include "exim.h"



/*************************************************
*           Generalized string match             *
*************************************************/

/* We are passed the subject and pattern as strings, and a pointer to a pointer
to a regular expression block. If the pattern is a regular expression and the
pointer points to NULL, set up a new re_block and compile the r.e. For
non regular expression, if the first character of the pattern is *, the match
is on the tail of the item. If the pattern starts with <searchtype>; then
do a file lookup, using the remainder as the file name. Keep a list of open
files for efficiency. */

BOOL match_string(char *s, char *pattern, re_block **chain_ad)
{
re_block *p;

/* No regexp pointer given, or pattern is not a regular expression. */

if (chain_ad == NULL || pattern[0] != '^')
  {
  int search_type;
  BOOL freefound; 
  char *error; 
  void *handle; 

  /* Tail match */

  if (pattern[0] == '*')
    {
    int patlen = (int)strlen(++pattern);
    int slen = (int)strlen(s);
    if (patlen > slen) return FALSE;
    return strncmp(s + slen - patlen, pattern, patlen) == 0;
    }

  /* Exact string match */

  if (strchr(pattern, ';') == NULL) return strcmp(s, pattern) == 0;

  /* Match by file lookup */

  if (strncmp(pattern, "lsearch;", 8) == 0)
    {
    search_type = stype_lsearch;
    pattern += 8;
    }
  else if (strncmp(pattern, "dbm;", 4) == 0)
    {
    search_type = stype_dbm;
    pattern += 4;
    }
  else if (strncmp(pattern, "nis;", 4) == 0)
    {
    if (have_nis)
      {  
      search_type = stype_nis;
      pattern += 4;
      }
    else log_write(LOG_PANIC_DIE, "search type \"nis\" unavailable for use "
      "in string match: %s", pattern); 
    }
  else if (strncmp(pattern, "nis0;", 5) == 0)
    {
    if (have_nis)
      {  
      search_type = stype_nis0;
      pattern += 5;
      }
    else log_write(LOG_PANIC_DIE, "search type \"nis0\" unavailable for use "
      "in string match: %s", pattern); 
    }
  else log_write(LOG_PANIC_DIE, "unknown search type in string match: %s",
    pattern);

  while (isspace(*pattern)) pattern++;
  handle = search_open(pattern, search_type, 0, NULL, NULL, &error, 
    &stringmatch_tree); 
  if (handle == NULL) log_write(LOG_PANIC_DIE, "%s", error);
  s = search_find(handle, pattern, s, search_type);
  if (s == NULL) return FALSE;
  store_free(s);
  return TRUE; 
  }

/* Regular expression match: compile if necessary */

p = *chain_ad;
if (p == NULL)
  {
  p = store_malloc(sizeof(re_block));
  *chain_ad = p;
  p->next = NULL;
  regexp_compiling = pattern;
  p->re = regcomp(pattern);
  regexp_compiling = NULL;
  }

/* Perform a regular expression match */

return regexec(p->re, s);
}



/*************************************************
*            Match in colon-separated list       *
*************************************************/

/* We are passed the string form, and optionally the address of the pointer to
a chain of compiled regular expressions. */

BOOL match_isinlist(char *s, char *list, re_block **chain_ad)
{
char *ss = string_nextinlist(list, ':');

while (ss != NULL)
  {
  if (match_string(s, ss, chain_ad)) return TRUE;
  if (ss[0] == '^' && *chain_ad != NULL) chain_ad = &((*chain_ad)->next);
  ss = string_nextinlist(NULL, ':');
  }

return FALSE;
}


/*************************************************
*           Do file existence tests              *
*************************************************/

/* This function is given a colon-separated list of files whose existence
is to be tested. The string is first expanded, and the resulting file names
must be absolute. However, a preceding "!" indicates non-existence is what
is wanted. */

BOOL match_exists(char *s)
{
char *ss, *file;
BOOL yield = TRUE;
struct stat statbuf;

if (s == NULL) return TRUE;
ss = expand_string(s);
if (ss == NULL)
  log_write(LOG_MAIN|LOG_PANIC_DIE, "expansion of %s failed: %s", s, 
  expand_string_message);

for (file = string_nextinlist(ss, ':'); file != NULL;
     file = string_nextinlist(NULL, ':'))
  {
  BOOL invert = FALSE;
  if (*file == '!')
    {
    invert = TRUE;
    file++;  
    }     
  if (*file != '/')
    log_write(LOG_MAIN|LOG_PANIC_DIE, "file name for existence test is not "
      "fully qualified: %s", file);
  if ((stat(file, &statbuf) < 0) != invert)
    {
    yield = FALSE;
    break; 
    }
  }       

store_free(ss);
return yield;
}

/* End of match.c */
