/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */


#include <stdio.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>

#include <sys/stat.h>

#include "macros.h"
#include "dbhdr.h"

#define type_retry 1
#define type_wait 2
#define type_reject 3


/* Utility program to dump the contents of an exim database file. The first
argument is the name of the spool directory. The second argument is the name of
the database file. The available names are:

  retry:      retry delivery information       
  wait-<t>:   message waiting information - <t> is transport name
  reject:     record of message rejections 
  
Output formatting is rudimentary. We want this utility to be as fast
as possible so as not to hold the lock any longer than necessary. Fancy
formatting can be done by some other program subsequently. Don't use
a pipe! */


/* Need spool_directory from the globals module, but no other globals. */

extern char *spool_directory;


static char time_buffer[sizeof("09-xxx-1999 hh:mm:ss  ")];

char *print_time(time_t t)
{
struct tm *tmstr = localtime(&t);
strftime(time_buffer, sizeof(time_buffer), "%d-%b-%Y %H:%M:%S", tmstr);
return time_buffer;
}


int main(int argc, char **argv)
{
struct stat statbuf;
int db_type = 0;
EXIM_DB *dbm;
char *key;
char buffer[256];

/* Check up on the number of arguments */

if (argc != 3)
  {
  printf("Usage: exim_dumpdb <spool-directory> <database-name>\n");
  printf("<database-name> = retry | wait-<transport> | reject\n");
  exit(1);
  }
  
if (strcmp(argv[2], "retry") == 0) db_type = type_retry;
  else if (strcmp(argv[2], "reject") == 0) db_type = type_reject;
  else if (strncmp(argv[2], "wait-", 5) == 0) db_type = type_wait;

if (db_type == 0)
  {
  printf("Usage: exim_dumpdb <spool-directory> <database-name>\n");
  printf("<database-name> = retry | wait-<transport> | reject\n");
  exit(1);
  } 

/* Set up the spool directory in the global place, open the database - this
gets a read lock as well. */

spool_directory = argv[1]; 
dbm = db_open(argv[2], O_RDONLY);

if (dbm == NULL)
  {
  printf("Failed to open database file %s (spool directory = %s)\n", 
    argv[2], spool_directory);
  exit(1);
  }   

/* Scan the file, formatting the information for each entry. Note
that data is returned in a malloc'ed block, in order that it be
correctly aligned. */

key = db_scan(dbm, TRUE);
while (key != NULL)
  {
  db_retry *retry; 
  db_wait *wait; 
  db_reject *reject; 
  int i; 
  char *t; 
  char name[MESSAGE_ID_LENGTH + 1]; 
  void *value = db_read(dbm, key);
   
  if (value == NULL) 
    fprintf(stderr, "entry %s not found - shurely shome mistake!\n", key);
  else    
    { 
    /* Note: don't use print_time more than once in one statement, since
    is uses a single buffer. */
       
    switch(db_type)
      {
      case type_retry:
      retry = (db_retry *)value;
      printf("  %s %d %d %s\n%s  ", key, retry->errno, retry->more_errno,
        retry->text, print_time(retry->first_failed));
      printf("%s  ", print_time(retry->last_try));  
      printf("%s %s\n", print_time(retry->next_try),
        (retry->expired)? "*" : ""); 
      break;
      
      case type_wait:
      wait = (db_wait *)value; 
      printf("%s ", key);
      t = wait->text; 
      name[MESSAGE_ID_LENGTH] = 0; 
      for (i = 1; i <= wait->count; i++)  
        {
        strncpy(name, t, MESSAGE_ID_LENGTH);
        printf("%s ", name); 
        t += MESSAGE_ID_LENGTH; 
        } 
      printf("\n");
      break;  
      
      case type_reject:
      reject = (db_reject *)value;
      printf("%s %s\n", print_time(reject->time_stamp), key);   
      }  
    free(value); 
    }
  key = db_scan(dbm, FALSE);
  }

db_close(dbm);
return 0;    
}

/* End of exim_dumpdb.c */
