/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */


/* The main function: entry point, initialization, and high-level control. */


#include "exim.h"


static char process_info[256];





/*************************************************
*                Regerror function               *
*************************************************/

/* This function is called whenever one of the regular expression functions
detects an error. Most regular expressions are part of the configuration, in
which case a compiling error is a disaster. However, from filter files there
can be user regular expressions. If regcomp_error_pointer is not NULL, it
points at a pointer to point the message at. Otherwise, panic. */

void regerror(char *s)
{
if (regcomp_error_pointer == NULL)
  log_write(LOG_PANIC_DIE, "regular expression error: %s%s%s", s,
    (regexp_compiling == NULL)? "" : " while compiling ",
    (regexp_compiling == NULL)? "" : regexp_compiling);
else *regcomp_error_pointer = string_copy(s); 
}



/*************************************************
*             Handler for SIGUSR1                *
*************************************************/

/* SIGUSR1 causes any exim process to write to the process log details of
what it is currently doing. It will only be used if the OS is capable of
setting up a handler that causes automatic restarting of any system call
that is in progress at the time. */

static void usr1_handler(int sig)
{
log_write(LOG_PROCESS, "%s", process_info);
log_close();
os_restarting_signal(SIGUSR1, usr1_handler);
}



/*************************************************
*            Set up processing details           *
*************************************************/

/* Save a text string for dumping when SIGUSR1 is received. */

void set_process_info(char *format, ...)
{
int len;
va_list ap;
sprintf(process_info, "%5d %s ", getpid(), version_string);
len = strlen(process_info);
va_start(ap, format);
vsprintf(process_info + len, format, ap);
DEBUG(2) debug_printf("set_process_info: %s\n", process_info);
va_end(ap);
}





/*************************************************
*          Entry point and high-level code       *
*************************************************/

/* Entry point for the Exim mailer. Analyse the arguments and arrange to take
the appropriate action. All the necessary functions are present in the one
binary. I originally thought one should split it up, but it turns out that so
much of the apparatus is needed in each chunk that one might as well just have
it all available all the time, which then makes the coding easier as well. */


int main(int argc, char **argv)
{
int  arg_accept_timeout = -1;
int  arg_error_handling = error_handling;
int  filter_fd;
int  group_count;
int  i;
int  msg_action;
int  msg_action_arg = -1;
int  namelen = (int)strlen(argv[0]);
int  recipients_arg = argc;
int  sender_address_domain = 0;
BOOL address_test_mode = FALSE;
BOOL arg_queue_only;
BOOL arg_queue_smtp;
BOOL extract_recipients = FALSE;
BOOL forced_delivery = FALSE;
BOOL deliver_give_up = FALSE;
BOOL list_queue = FALSE;
BOOL list_variables = FALSE;
BOOL more = TRUE;
BOOL one_msg_action = FALSE;
BOOL queue_only_set = FALSE;
BOOL queue_smtp_set = FALSE;
BOOL smtp_first = TRUE;
BOOL synchronous_delivery = FALSE;
BOOL verify_only = FALSE;
BOOL version_printed = FALSE;
char *start_queue_run_id = NULL;
char *stop_queue_run_id = NULL;
transport_instance *ti;
struct passwd *pw;
struct stat statbuf;
gid_t group_list[NGROUPS_MAX];

/* Indicate to various shared functions that this is exim, not one of
its utilities. This affects some error processing. */

really_exim = TRUE;

/* Ensure we have a buffer for constructing log entries. Use malloc directly,
because store_malloc writes a log entry on failure. */

log_buffer = (char *)malloc(LOG_BUFFER_SIZE);
if (log_buffer == NULL)
  {
  fprintf(stderr, "exim: failed to get store for log buffer\n");
  exit(EXIT_FAILURE);
  }

/* Set up the handler for the data request signal, and set the initial
descriptive text. */

os_restarting_signal(SIGUSR1, usr1_handler);
set_process_info("initializing");

/* SIGHUP is used to get the daemon to reconfigure. It gets set as appropriate
in the daemon code. For the rest of exim's uses, we ignore it. */

signal(SIGHUP, SIG_IGN);

/* We don't want to die on pipe errors as the code is written to handle
the write error instead. */

signal(SIGPIPE, SIG_IGN);

/* Save the arguments for use if we re-exec exim as a daemon after receiving
SIGHUP. */

sighup_argv = argv;

/* Set up the version number. Set up the leading 'E' for the external form of
message ids, set the pointer to the internal form, and initialize it to
indicate no message being processed. */

version_init();
message_id_external[0] = 'E';
message_id = message_id_external + 1;
message_id[0] = 0;

/* Set the umask to zero so that any files that Exim creates are created
with the modes that it specifies. */

umask(0);


/* Precompile the regular expression for matching the name of a spool file.
Keep this in step with the code that generates such names in the accept.c
module. We need to do this here, because the -M options check their arguments
for syntactic validity using mac_msgid, which uses this. */

regexp_spoolfile = regcomp(
  "^[a-zA-Z0-9][a-zA-Z0-9][a-zA-Z0-9][a-zA-Z0-9][a-zA-Z0-9][a-zA-Z0-9]"
  "\\-[a-zA-Z0-9][a-zA-Z0-9][a-zA-Z0-9][a-zA-Z0-9][a-zA-Z0-9][a-zA-Z0-9]"
  "\\-[0-9][0-9]$");

/* Ditto for matching an IP address. */

regexp_ip_address = regcomp(
  "[0-9][0-9]?[0-9]?\\.[0-9][0-9]?[0-9]?\\.[0-9][0-9]?[0-9]?\\."
  "[0-9][0-9]?[0-9]?$");
  
/* Ditto for matching a "From_" line in an incoming message, in the form
   From ph10 Fri Jan  5 12:35 GMT 1996 
which the "mail" commands send to the MTA (undocumented, of course). Because
of variations in time formats, just match up to the end of the minutes. That
should be sufficient. */    

regexp_From = regcomp(
  "^From +([^ ]+) +[a-zA-Z][a-zA-Z][a-zA-Z] +[a-zA-Z][a-zA-Z][a-zA-Z] +"
  "[0-9]?[0-9] +[0-9][0-9]:[0-9][0-9]");
  

/* If the program is called as "mailq" treat it as equivalent to "exim -bp"; 
this seems to be a generally accepted convention, since one finds symbolic 
links called "mailq" in standard OS configurations. */

if ((namelen == 5 && strcmp(argv[0], "mailq") == 0) ||
    (namelen  > 5 && strncmp(argv[0] + namelen - 6, "/mailq", 6) == 0))
      list_queue = TRUE; 
      
/* If the program is called as "rsmtp" treat it as equivalent to "exim -bS";
this is a smail convention. */

if ((namelen == 5 && strcmp(argv[0], "rsmtp") == 0) ||
    (namelen  > 5 && strncmp(argv[0] + namelen - 6, "/rsmtp", 6) == 0))
      smtp_input = smtp_batched_input = TRUE; 
 

/* Scan the program's arguments. Some can be dealt with right away; others are
simply recorded for checking and handling afterwards. */

for (i = 1; i < argc; i++)
  {
  /* -bd: Run in daemon mode, awaiting SMTP connections. */

  if (strcmp(argv[i], "-bd") == 0) daemon_listen = TRUE;
  
  /* -bf: Run in mail filter testing mode */
  
  else if (strcmp(argv[i], "-bf") == 0) 
    {
    if(++i < argc) filter_test = argv[i]; else
      {
      fprintf(stderr, "exim: file name expected after -bf\n");
      exit(EXIT_FAILURE);
      } 
    } 

  /* -bm: Deliver message - the default option. */

  else if (strcmp(argv[i], "-bm") == 0) continue;

  /* -bp: List the contents of the mail queue */

  else if (strcmp(argv[i], "-bp") == 0) list_queue = TRUE;

  /* -bP: List the configuration variables given as the address list. */

  else if (strcmp(argv[i], "-bP") == 0) list_variables = TRUE;

  /* -bS: Read SMTP commands on standard input, but produce no replies -
  all errors are reported by sending messages. */ 

  else if (strcmp(argv[i], "-bS") == 0) smtp_input = smtp_batched_input = TRUE;

  /* -bs: Read SMTP commands on standard input and produce SMTP replies
  on standard output. */

  else if (strcmp(argv[i], "-bs") == 0) smtp_input = TRUE;

  /* -bt: address testing mode */

  else if (strcmp(argv[i], "-bt") == 0) address_test_mode = TRUE;

  /* -bv: verify addresses */

  else if (strcmp(argv[i], "-bv") == 0) verify_only = TRUE;

  /* -bV: Print version string */

  else if (strcmp(argv[i], "-bV") == 0)
    {
    printf("Exim version %s #%s built %s\n", version_string,
      version_cnumber, version_date);
    printf("%s\n", version_copyright);
    version_printed = TRUE;
    }

  /* -C: change configuration file */

  else if (strcmp(argv[i], "-C") == 0)
    {
    config_changed = TRUE;
    config_filename = argv[++i];
    }

  /* -d: Set debug level (see also -v below) or set memory tracing, or
  set up stderr to a file. */

  else if (strncmp(argv[i], "-d", 2) == 0)
    {
    if (strcmp(argv[i], "-dm") == 0) debug_trace_memory = TRUE;
    else if (strcmp(argv[i], "-df") == 0)
      {
      if (stderr_filename != NULL) freopen(stderr_filename, "a", stderr); 
      }   
    else if (argv[i][2] == 0 && (i+1 >= argc || !isdigit(argv[i+1][0])))
      debug_level = 1; 
    else
      {
      int n = 0;
      char *s;
      if (argv[i][2] == 0) s = argv[++i]; else s = argv[i] + 2;   
      (void)sscanf(s, "%d%n", &debug_level, &n);
      if (n+2 < (int)strlen(argv[i]))
        {
        fprintf(stderr, "exim: unknown option %s: abandoned\n", argv[i]);
        exit(EXIT_FAILURE);
        }
      }
    if (debug_level > 0)
      {
      debug_file = stderr;
      debug_printf("Debug level set to %d\n", debug_level); 
      } 
    }
    
  /* -E: This is a local error message. This option is not intended for
  external use at all, but is not restricted to trusted callers because it
  does no harm (just suppresses certain error messages) and if exim is run
  not setuid root it won't always be trusted when it generates error messages
  using this option. */
  
  else if (strcmp(argv[i], "-E") == 0)
    {
    local_error_message = TRUE; 
    }    
    
  /* -ex: The vacation program calls sendmail with the undocumented "-eq" 
  option, so it looks as if historically the -oex options are also callable 
  without the leading -o. So we have to accept them (see the -oex options 
  below). */
    
  /* -F: Set sender's full name, used instead of the gecos entry from
  the password file. Since users can usually alter their gecos entries,
  there's no security involved in using this instead. The data can follow
  the -F or be in the next argument. */
  
  else if (strncmp(argv[i], "-F", 2) == 0)
    {
    char *name = argv[i] + 2;
    if (name[0] == 0)
      {
      if(++i < argc) name = argv[i]; else
        {
        fprintf(stderr, "exim: string expected after -F\n");
        exit(EXIT_FAILURE);
        } 
      }  
    name = parse_fix_phrase(name);
    user_name = store_malloc((int)strlen(name) + 1);
    strcpy(user_name, name);
    }

  /* -f: Set sender's address - only actually used if run by a trusted user,
  except that the null address can be set by any user. For an untrusted user
  it is used only as the envelope address for outgoing SMTP and to prevent
  mail being sent from a filter; the actual sender is still put in Sender: if 
  it doesn't match the From: header. The data can follow the -f or be in the 
  next argument. The -r switch is an obsolete form of -f but since there 
  appear to be programs out there that use anything that sendmail ever 
  supported, better accept it. */

  else if (strncmp(argv[i], "-f", 2) == 0 || strncmp(argv[i], "-r", 2) == 0)
    {
    int start, end;
    char *errmess;
    char *name = argv[i] + 2;
    if (name[0] == 0)
      {
      if (i+1 < argc) name = argv[++i]; else  
        {
        fprintf(stderr, "exim: string expected after -%c\n", argv[i][1]);
        exit(EXIT_FAILURE);
        } 
      }     
    sender_address =
      parse_extract_address(name, &errmess, &start, &end,
        &sender_address_domain, TRUE);
    if (sender_address == NULL)
      {
      fprintf(stderr, "exim: %s - bad address: %s\n", name, errmess);
      return EXIT_FAILURE;
      }
    }
    
  /* -h: Set the hop count for an incoming message. Exim does not currently
  support this; it always computes it by counting the Received: headers.
  To put it in will require a change to the spool header file format. */
  
  else if (strncmp(argv[i], "-h", 2) == 0)
    {
    if (argv[i][2] == 0) i++;
    }      

  /* -i: Set flag so dot doesn't end non-SMTP input (same as -oi, seems
  not to be documented for sendmail but mailx (at least) uses it) */

  else if (strcmp(argv[i], "-i") == 0) dot_ends = FALSE;

  /* -MC:  continue delivery of another message via an existing open
  file descriptor. This option is used for an internal call by the
  smtp transport when there is a pending message waiting to go to an
  address to which it has got a connection. The caller is required to
  be root. Four subsequent arguments are required: transport name,
  host name, sequence number, and message_id. Transports may decline
  to create new processes if the sequence number gets too big. The channel is
  stdin + stdout. This must be the last argument. There's a subsequent
  check that the real-uid is privileged. */

  else if (strcmp(argv[i], "-MC") == 0)
    {
    if (argc != i + 5)
      {
      fprintf(stderr, "exim: too many or too few arguments after -MC\n");
      return EXIT_FAILURE;
      }

    if (msg_action_arg >= 0)
      {
      fprintf(stderr, "exim: incompatible arguments\n");
      return EXIT_FAILURE;
      }

    continue_transport = argv[++i];
    continue_hostname = argv[++i];
    continue_sequence = atoi(argv[++i]);
    msg_action = MSG_DELIVER;
    msg_action_arg = ++i;
    forced_delivery = TRUE;

    if (!mac_ismsgid(argv[i]))
      {
      fprintf(stderr, "exim: malformed message id %s after -MC option\n",
        argv[i]);
      return EXIT_FAILURE;
      }
    }
    
  /* -M[x]: various operations on the following list of message ids:
     -M    deliver the messages, ignoring next retry times
     -Mc   deliver the messages, checking next retry times 
     -Mf   freeze the messages
     -Mg   give up on the messages 
     -Mt   thaw the messages
     -Mrm  remove the messages
  In these cases, this must be the last option. There are also the following
  options which are followed by a single message id, and which act on
  that message. Some of them use the "recipient" addresses as well.
     -Mar  add recipient(s)
     -Mmd  mark recipients(s) received  
     -Mes  edit sender 
     -Meb  edit body 
  */

  else if (strncmp(argv[i], "-M", 2) == 0)
    {
    int j;
    if (strcmp(argv[i], "-M") == 0) 
      {
      msg_action = MSG_DELIVER;
      forced_delivery = TRUE;
      }
    else if (strcmp(argv[i], "-Mar") == 0)  
      { 
      msg_action = MSG_ADD_RECIPIENT;
      one_msg_action = TRUE;
      }  
    else if (strcmp(argv[i], "-Mc") == 0)  msg_action = MSG_DELIVER;
    else if (strcmp(argv[i], "-Meb") == 0)  
      {
      msg_action = MSG_EDIT_BODY;
      one_msg_action = TRUE;
      }  
    else if (strcmp(argv[i], "-Mes") == 0)  
      {
      msg_action = MSG_EDIT_SENDER;
      one_msg_action = TRUE;
      }  
    else if (strcmp(argv[i], "-Mf") == 0)  msg_action = MSG_FREEZE;
    else if (strcmp(argv[i], "-Mg") == 0)  
      {
      msg_action = MSG_DELIVER;
      deliver_give_up = TRUE;
      }  
    else if (strcmp(argv[i], "-Mmd") == 0)  
      {
      msg_action = MSG_MARK_DELIVERED;
      one_msg_action = TRUE;
      }  
    else if (strcmp(argv[i], "-Mrm") == 0) msg_action = MSG_REMOVE;
    else if (strcmp(argv[i], "-Mt") == 0)  msg_action = MSG_THAW;
    else
      {
      fprintf(stderr, "exim: unknown option %s: abandoned\n", argv[i]);
      exit(EXIT_FAILURE);
      }
    msg_action_arg = i + 1;
    
    /* All the -M options require at least one message id. */
     
    if (msg_action_arg >= argc)
      {
      fprintf(stderr, "exim: no message ids given after %s option\n", argv[i]);
      return EXIT_FAILURE;
      }
    
    /* Some require only message ids to follow */
    
    if (!one_msg_action)
      {    
      for (j = msg_action_arg; j < argc; j++) if (!mac_ismsgid(argv[j]))
        {
        fprintf(stderr, "exim: malformed message id %s after %s option\n",
          argv[j], argv[i]);
        return EXIT_FAILURE;
        }
      break;   /* Remaining args are ids */   
      }
      
    /* Others require only one message id, possibly followed by addresses,
    which will be handled as normal arguments. */
    
    else 
      {
      if (!mac_ismsgid(argv[msg_action_arg]))
        {
        fprintf(stderr, "exim: malformed message id %s after %s option\n",
          argv[msg_action_arg], argv[i]);
        return EXIT_FAILURE;
        }
      i++;         
      } 
    }
    
  /* -N: don't do delivery - a debugging option that stops transports doing
  their thing. It implies debugging. */
  
  else if (strcmp(argv[i], "-N") == 0)
    {
    dont_deliver = TRUE;
    if (debug_level <= 0)
      {
      debug_level = 1;
      debug_file = stderr;
      }
    }         
    
  /* -odb: background delivery */

  else if (strcmp(argv[i], "-odb") == 0)
    {
    synchronous_delivery = FALSE;
    arg_queue_only = FALSE;
    queue_only_set = TRUE; 
    }

  /* -odf: foreground delivery (smail-compatible option) */
  /* -odi: interactive (synchronous) delivery (sendmail-compatible option) */

  else if (strcmp(argv[i], "-odf") == 0 || strcmp(argv[i], "-odi") == 0)
    {
    synchronous_delivery = TRUE;
    arg_queue_only = FALSE;
    queue_only_set = TRUE; 
    }

  /* -odq: queue only */

  else if (strcmp(argv[i], "-odq") == 0)
    {
    synchronous_delivery = FALSE;
    arg_queue_only = TRUE;
    queue_only_set = TRUE; 
    }

  /* -odqs: queue SMTP only - do local deliveries */

  else if (strcmp(argv[i], "-odqs") == 0)
    {
    arg_queue_smtp = TRUE;
    queue_smtp_set = TRUE; 
    }
    
  /* -oex: Accept sendmail error flags, but exim does not support
  all of them. Accept these flags without the "o" prefix, because the
  vacation program calls sendmail with -eq, though this is nowhere
  documented. */

  else if (strcmp(argv[i], "-oem") == 0 || strcmp(argv[i], "-em") == 0) 
    arg_error_handling = ERRORS_SENDER;
  else if (strcmp(argv[i], "-oep") == 0 || strcmp(argv[i], "-ep") == 0) 
    arg_error_handling = ERRORS_STDERR;
  else if (strcmp(argv[i], "-oeq") == 0 || strcmp(argv[i], "-eq") == 0) 
    arg_error_handling = ERRORS_STDERR;
  else if (strcmp(argv[i], "-oew") == 0 || strcmp(argv[i], "-ew") == 0) 
    arg_error_handling = ERRORS_SENDER;

  /* -oi: Set flag so dot doesn't end non-SMTP input (same as -i) */

  else if (strcmp(argv[i], "-oi") == 0) dot_ends = FALSE;

  /* -oMa: Set sender host address (root or exim only) */

  else if (strcmp(argv[i], "-oMa") == 0)
    {
    i++;
    sender_host_address = argv[i];
    }

  /* -oMr: Received protocol (root or exim only) */

  else if (strcmp(argv[i], "-oMr") == 0)
    {
    i++;
    received_protocol = argv[i];
    }

  /* -oMs: Set sender host name (root or exim only) */

  else if (strcmp(argv[i], "-oMs") == 0)
    {
    i++;
    sender_host_name = argv[i];
    }

  /* -oMt: Set sender ident (root or exim only) */

  else if (strcmp(argv[i], "-oMt") == 0)
    {
    i++;
    sender_ident = argv[i];
    }
    
  /* -om: Me-too flag for aliases. Exim always does this. Some programs
  seem to call this as -m (undocumented) */
  
  else if (strcmp(argv[i], "-om") == 0 || strcmp(argv[i], "-m") == 0) {}  
    
  /* -or <n>: set timeout for non-SMTP acceptance */
  
  else if (strncmp(argv[i], "-or", 3) == 0)
    {
    if (argv[i][3] == 0)
      {
      if (i+1 < argc)
        arg_accept_timeout = readconf_readtime(argv[++i], 0);  
      }  
    else arg_accept_timeout = readconf_readtime(argv[i]+3, 0);
    if (arg_accept_timeout < 0)
      {
      fprintf(stderr, "exim: bad time value %s: abandoned\n", argv[i]);
      exit(EXIT_FAILURE);
      }     
    }    

  /* -oX <n>: Set SMTP listening port to <n> */

  else if (strcmp(argv[i], "-oX") == 0) smtp_port = atoi(argv[++i]);

  /* -q[<n>]: Run the queue, optionally at regular intervals or starting from
  a given message id. Either one option or the other, not both. */

  else if (strncmp(argv[i], "-q", 2) == 0)
    {
    /* No interval specified */
    if (argv[i][2] == 0 && 
        (i + 1 >= argc || argv[i+1][0] == '-' || mac_ismsgid(argv[i+1])))
      {
      queue_interval = 0;
      if (i+1 < argc && mac_ismsgid(argv[i+1]))
        start_queue_run_id = argv[++i];
      if (i+1 < argc && mac_ismsgid(argv[i+1]))
        stop_queue_run_id = argv[++i];
      }
    /* Interval specified */
    else
      {
      if (argv[i][2] != 0) 
        queue_interval = readconf_readtime(argv[i]+2, 0);
      else   
        queue_interval = readconf_readtime(argv[++i], 0);
      if (queue_interval <= 0)
        {
        fprintf(stderr, "exim: bad time value %s: abandoned\n", argv[i]);
        exit(EXIT_FAILURE);
        }
      }
    }
    
  /* -R: Set string to match in addresses for forced queue run to
  pick out particular messages. */
  
  else if (strcmp(argv[i], "-R") == 0) 
    {
    if (i+1 < argc) 
      {
      deliver_selectstring = argv[++i]; 
      if (queue_interval < 0) queue_interval = 0; 
      } 
    else
      {
      fprintf(stderr, "exim: string expected after -R\n");
      exit(EXIT_FAILURE);
      }  
    } 
    
  /* -r: an obsolete synonym for -f (see above) */

  /* -t: Set flag to extract recipients from body of message. */

  else if (strcmp(argv[i], "-t") == 0) extract_recipients = TRUE;

  /* -v: verify things - this is the same as -d or -d1. */

  else if (strcmp(argv[i], "-v") == 0)
    {
    if (debug_level <= 0)
      {
      debug_level = 1;
      debug_file = stderr;
      }
    }

  /* Either an unknown option, or the start of a recipients' list */

  else
    {
    if (argv[i][0] == '-')
      {
      fprintf(stderr, "exim: unknown option %s: abandoned\n", argv[i]);
      exit(EXIT_FAILURE);
      }
    else
      {
      recipients_arg = i;
      break;
      }
    }
  }


/* Arguments have been processed. Check for incompatibilities. */

if ((
    (smtp_input || extract_recipients || recipients_arg < argc) &&
    (daemon_listen || queue_interval >= 0 || list_queue || 
      filter_test != NULL || (msg_action_arg > 0 && !one_msg_action))
    ) ||
    (
    msg_action_arg > 0 &&
    (daemon_listen || queue_interval >= 0 || list_variables || verify_only ||
     address_test_mode)
    ) ||
    (
    (daemon_listen || queue_interval >= 0) &&
    (sender_address != NULL || list_variables || verify_only ||
     address_test_mode)
    ) ||
    (
    list_variables &&
    (verify_only || address_test_mode || smtp_input || extract_recipients ||
      filter_test != NULL)
    ) ||
    (
    verify_only &&
    (address_test_mode || smtp_input || extract_recipients ||
      filter_test != NULL)
    ) ||
    (
    address_test_mode && (smtp_input || extract_recipients || 
      filter_test != NULL) 
    ) ||  
    (
    smtp_input && (sender_address != NULL || filter_test != NULL)
    ) ||
    (
    deliver_selectstring != NULL && queue_interval < 0 
    )  
   )
  {
  fprintf(stderr, "exim: incompatible arguments\n");
  exit(EXIT_FAILURE);
  }


/* Exim is normally entered as root (except when called from inetd under its
own uid in order to receive a message, with security level >= 2). The security
level controls how it uses set{u,g}id and/or sete{g,u}id to reduce its
privilege when not necessary. However, it always spins off sub-processes that
set their uid and gid as required for local delivery. We don't want to pass on
any extra groups that root may belong to, so just get rid of them all. If this
process isn't running as root, setgroups() has no effect. 

We need to obey setgroups() at this stage, before possibly giving up root 
privilege for a changed configuration file, but later on we might need to check
on the additional groups for the admin user privilege - can't do that till
after reading the config, which might specify the exim gid. Therefore, save the
group list here. */

group_count = getgroups(NGROUPS_MAX, group_list);
setgroups(0, NULL);


/* Get the real uid and gid. If the configuration file name has been altered by
an argument on the command line, or if this is a filter testing run, remove any
setuid privilege the program has, and run as the underlying user. Otherwise,
set the real ids to the effective values (should be root unless run from inetd,
which it can either be root or the exim uid, if one is configured). When
running with the security options set, root privilege will subsequently be
relinquished as soon as possible. */

real_uid = getuid();
real_gid = getgid();

if (config_changed || filter_test != NULL)
  {
  setgid(real_gid);    /* Sets real and effective */
  setuid(real_uid);
  DEBUG(9) debug_printf("Removing setuid privilege\n");
  }
else
  {
  setgid(getegid());
  setuid(geteuid());
  } 

/* If testing a filter, open the file now, before wasting time doing other
setups and reading the message. */

if (filter_test != NULL)  
  {
  filter_fd = open(filter_test, O_RDONLY);
  if (filter_fd < 0)
    {
    printf("exim: failed to open %s: %s\n", filter_test, strerror(errno));
    return FALSE;
    }
  }   

/* Read the main runtime configuration data; this gives up if there
is a failure. It leaves the configuration file open so that the subsequent
configuration data for delivery can be read if needed. */

readconf_main();

/* The path for Exim may be changed by the configuration file. It is also
used in the argument list for calling Exim to send an error message. */

mailer_argv[0] = exim_path;

/* If debugging is turned on, arrange to pass the setting when we re-exec
exim for error messages, etc. Also pass on -N if set (-d is always set if -N 
is.) The argument list has three 0 entries at the end, the first two of which
can be overwritten. */

if (debug_level > 0)
  {
  int i = -1;
  while (mailer_argv[++i] != (char *)0);
  mailer_argv[i] = string_sprintf("-d%d", debug_level);   
  if (dont_deliver) mailer_argv[++i] = "-N"; 
  } 
 

/* If an action on specific messages is requested, set admin_user true if the
real user is root or exim, or if the real group is exim, or if one of the
supplementary groups is exim. Note: this is not the same as "trusted user"
status. We don't fail immediately if not admin_user, since some actions
can be performed by others. */

if (msg_action_arg > 0)
  {
  if (real_uid == root_uid || real_uid == exim_uid || real_gid == exim_gid)
    admin_user = TRUE;
  else if (exim_gid >= 0)
    {
    while (--group_count >= 0)
      if (group_list[group_count] == exim_gid) { admin_user = TRUE; break; }  
    }    
  }   
  
  
/* Only an admin user may request that a message be returned to
its sender forthwith. */

if (deliver_give_up && !admin_user)
  {
  fprintf(stderr, "exim: permission denied\n");
  exit(EXIT_FAILURE);  
  }  


/* Set the working directory to be the top-level spool directory. We don't rely
on this in the code, which always uses fully qualified names, but it's useful
for core dumps etc. Don't complain if it fails - the spool directory might not
be generally accessible and calls with the -C option have lost privilege by 
now. */

if (chdir(spool_directory) != 0)
  {
  directory_make(spool_directory, "", SPOOL_DIRECTORY_MODE);
  (void) chdir(spool_directory);
  }


/* If the real user is not root or the exim uid, the argument for passing
in an open TCP/IP connection for another message is not permitted, nor is
running with the -N option for any delivery action, unless this call to exim is
one that supplied an input message. */

if (real_uid != root_uid && real_uid != exim_uid &&
     (continue_hostname != NULL || 
       (dont_deliver && 
         (queue_interval >= 0 || daemon_listen || msg_action_arg > 0)
       )
     ))
  {
  fprintf(stderr, "exim: Permission denied\n");
  return EXIT_FAILURE;
  }
    
/* See if the caller is root or exim or is in the list of trusted uids or gids. 
Trusted callers are permitted to specify sender addresses with -f on the 
command line. */

if (real_uid == root_uid || real_uid == exim_uid) trusted_caller = TRUE; else
  {
  if (trusted_users != NULL)
    {
    int *nn = trusted_users;
    while (*nn >= 0) if (*nn++ == real_uid) 
      { trusted_caller = TRUE; break; } 
    }  
  if (!trusted_caller && trusted_groups != NULL)
    {
    int *nn = trusted_groups;
    while (*nn >= 0) if (*nn++ == real_gid) 
      { trusted_caller = TRUE; break; } 
    }  
  }  

/* If the caller is not trusted, certain arguments are ignored. Note that
authority for performing certain actions on messages is tested in the
queue_action() function. */

if (!trusted_caller)
  {
  sender_host_name = sender_host_address = sender_ident = NULL;
  received_protocol = NULL; 
  }  
  
/* The queue_only configuration option can be overridden by -odx on the
command line, the queue_smtp configuration option can be overridden by -odbs,
and the accept_timeout option can be overridden by -or. */

if (queue_only_set) queue_only = arg_queue_only;
if (queue_smtp_set) queue_smtp = arg_queue_smtp;
if (arg_accept_timeout >= 0) accept_timeout = arg_accept_timeout;


/* Ensure there is a big buffer for temporary use in several places. */

big_buffer = store_malloc(BIG_BUFFER_SIZE);


/* If an exim uid is specified, handle setuid/seteuid setting according
to the security level. The macros mac_setegid and mac_seteuid are defined as 0
on systems which do not have seteuid and setegid, so that this code will
compile on all systems. However, the security level won't be set to values that
cause the use of the sete{u,g}id functions on systems that don't have them (see
readconf.c). */

if (exim_uid > 0)
  {
  /* Level 1: Use seteuid to reduce privilege at all times it is not needed, 
  but use setuid if we know that privilege is never going to be required again, 
  typically for enquiry type calls. However, privilege might be needed for
  the forwardfile director, so can't lose it for verification. */
  
  if (security_level == 1)
    {
    if (queue_interval < 0 &&                  /* not running the queue */
        !daemon_listen &&                      /* and not starting the daemon */
        (msg_action_arg < 0 || 
          msg_action != MSG_DELIVER) &&        /* and not delivering */
        !verify_only &&                        /* and not verifying */   
        !address_test_mode &&                  /* and not testing addresses */ 
        (list_variables ||                     /* and either not receiving */
          (recipients_arg >= argc && !extract_recipients && !smtp_input) ||
          queue_only))                         /* or just queueing */
      {
      setgid(exim_gid);
      setuid(exim_uid); 
      }

    /* Running the queue, starting daemon, delivering, or receiving with
    queue_only FALSE. */
        
    else
      {
      mac_setegid(exim_gid);
      mac_seteuid(exim_uid);
      }       
    }     

  /* Level 2: if neither starting the daemon nor delivering messages, always
  use setuid to remove all root privilege. Level 2 makes no use of the 
  sete{g,u}id functions; subsequent re-execs are used to regain root privilege 
  for delivering. 
  
  Level 3: as level 2, but in addition use sete{g,u}id at other times. */ 

  else
    { 
    if (queue_interval != 0 &&    /* no queue run, or periodic queue run */
        !daemon_listen &&         /* not starting listener */
        (msg_action_arg < 0 || msg_action != MSG_DELIVER))
      {
      setgid(exim_gid);
      setuid(exim_uid); 
      } 
   
    /* Level 3 */
     
    else if (security_level > 2)
      {
      mac_setegid(exim_gid);
      mac_seteuid(exim_uid);  
      }  
    }   
  } 


/* Handle a request to list the delivery queue */

if (list_queue)
  {
  set_process_info("listing the queue");
  queue_list();
  exit(EXIT_SUCCESS);
  }


/* Handle actions on specific messages, except for the force delivery action,
which is done below. Some actions take a whole list of message ids, which
are known to continue up to the end of the arguments. Others take a single
message id and then operate on the recipients list. */

if (msg_action_arg > 0 && msg_action != MSG_DELIVER)
  { 
  int yield = EXIT_SUCCESS;
  set_process_info("acting on specified messages");
 
  if (!one_msg_action)
    { 
    for (i = msg_action_arg; i < argc; i++)
      if (!queue_action(argv[i], msg_action, stdout, NULL, 0, 0)) 
        yield = EXIT_FAILURE;
    }
    
  else if (!queue_action(argv[msg_action_arg], msg_action, stdout, argv, 
    argc, recipients_arg)) yield = EXIT_FAILURE;
    
  exit(yield);
  } 


/* All the modes below here require the delivery configuration options
to be set up. Doing it here saves effort when several processes are
subsequently spun off. */

transport_init();
direct_init();
route_init();
readconf_retries();
readconf_rewrites();
fclose(config_file);



/* Handle a request to list one or more configuration variables */

if (list_variables)
  {
  set_process_info("listing variables");
  if (recipients_arg >= argc) readconf_print("all", NULL);
    else for (i = recipients_arg; i < argc; i++) 
      {
      if (i < argc - 1 && 
          (strcmp(argv[i], "director") == 0 ||
           strcmp(argv[i], "router") == 0 ||
           strcmp(argv[i], "transport") == 0))
        {
        readconf_print(argv[i+1], argv[i]);
        i++;
        }         
      else readconf_print(argv[i], NULL);
      } 
  exit(EXIT_SUCCESS);
  }


/* If, when debugging, the debug_transport configuration option is set, find
the debug transport. This will then be substituted for the real transport for
all messages. */

#ifdef TRANSPORT_DEBUG
if (debug_transport_file != NULL)
  {
  transport_info *tt; 
  for (tt = transports_available; tt->driver_name[0] != 0; tt++)
    {
    if (strcmp("debug", tt->driver_name) == 0)
      {
      debug_transport = tt;
      break;
      }
    }
  if (debug_transport == NULL)
    {
    fprintf(stderr, "exim: debug_transport specified, but "
      "debug transport not found\n");  
    exit(EXIT_FAILURE);
    } 
  } 
#endif


/* Ensure the address of the mailmaster is fully qualified. Don't use
rewrite_address_qualify, as that frees the input store, and errors_address
may not be in dynamic store. */

if (strchr(errors_address, '@') == NULL)
  errors_address = string_sprintf("%s@%s", errors_address, 
    qualify_domain_sender);

/* Search the configured transports for the assumed ones that will be used
for generated pipe and file addresses and auto replies. If they are not found,
do not complain now. A complaint will be generated later if an address actually
causes a pipe or file or auto-reply delivery to be generated. */

for (ti = transports; ti != NULL; ti = ti->next)
  {
  if (strcmp(ti->name, address_pipe_transport) == 0)
    transport_address_pipe = ti;
  else if (strcmp(ti->name, address_file_transport) == 0)
    transport_address_file = ti;
  else if (strcmp(ti->name, address_reply_transport) == 0)
    transport_address_reply = ti;
  }


/* Handle a request to deliver one or more messages that are already on the
queue. Values of msg_action other than MSG_DELIVER are dealt with above. This
is typically used for a small number when prodding by hand. Message delivery
happens in a separate process, so we fork a process for each one, and run them
sequentially so that debugging output doesn't get intertwined, and to avoid
spawning too many processes if a long list is given. */

if (msg_action_arg > 0)
  {
  set_process_info("delivering specified messages");
  queue_smtp = FALSE; 
  for (i = msg_action_arg; i < argc; i++)
    {
    int status;
    if (fork() == 0)
      {
      if (deliver_give_up) queue_action(argv[i], MSG_THAW, NULL, NULL, 0, 0);
      (void)deliver_message(argv[i], forced_delivery, deliver_give_up);
      _exit(EXIT_SUCCESS);
      }
    else wait(&status);
    }
  exit(EXIT_SUCCESS);
  }



/* Handle a request to verify a list of addresses. */

if (verify_only)
  {
  if (recipients_arg >= argc)
    {
    fprintf(stderr, "exim: no addresses given for verification\n");
    return EXIT_FAILURE;
    }
  for (i = recipients_arg; i < argc; i++)
    (void) verify_address(argv[i], TRUE, TRUE, stdout, NULL, NULL, 
    FALSE, FALSE);
  direct_tidyup();
  route_tidyup();
  search_tidyup(&expand_tree);
  search_tidyup(&stringmatch_tree);
  exit(EXIT_SUCCESS);
  }


/* Handle the case of address test mode - if no arguments are given, read
addresses from stdin. Set debug_level to at least 1 to get full output. */

if (address_test_mode)
  {
  if (debug_level <= 0) debug_level = 1;
  if (recipients_arg < argc)
    {
    while (recipients_arg < argc)
      (void) verify_address(string_copy(argv[recipients_arg++]), TRUE, TRUE,
        stdout, NULL, NULL, FALSE, TRUE);  
    }  
  else 
    {
    for (;;)
      {
      char buffer[256];
      printf("> ");
      if (fgets(buffer, 256, stdin) == NULL) break;
      (void) verify_address(string_copy(buffer), TRUE, TRUE, stdout, NULL, NULL,
        FALSE, TRUE);
      }
    printf("\n");
    } 
  direct_tidyup();
  search_tidyup(&expand_tree);
  search_tidyup(&stringmatch_tree);
  exit(EXIT_SUCCESS);
  }



/* If only a single queue run is requested, without SMTP listening, we can just
turn into a queue runner, with an optional starting message id. */

if (queue_interval == 0 && !daemon_listen)
  {
  DEBUG(1) debug_printf("Single queue run%s%s%s%s\n",
    (start_queue_run_id == NULL)? "" : " starting at ",
    (start_queue_run_id == NULL)? "" : start_queue_run_id,
    (stop_queue_run_id == NULL)?  "" : " stopping at ",
    (stop_queue_run_id == NULL)?  "" : stop_queue_run_id);
  set_process_info("running the queue (single queue run)");
  queue_run(start_queue_run_id, stop_queue_run_id);
  exit(EXIT_SUCCESS);
  }


/* Find the login name of the real user running this process. This is always
needed, because it is written into the spool file. It may also be used to
construct a from: or a sender: header, and in this case we need the user's full
name as well, so save a copy of it, checked for RFC822 syntax and munged if
necessary, if it hasn't previously been set by the -F argument. We try to get
the passwd entry more than once, in case NIS or other delays are in evidence. */

user_login == NULL;
for (i = 1; i <= 10; i++)
  {
  if ((pw = getpwuid(real_uid)) != NULL)
    {
    user_login = string_copy(pw->pw_name);
    
    /* If user name has not been set by -F, set it from the passwd entry
    unless -f has been used to set the sender address by a trusted user. */
      
    if (user_name == NULL)
      {
      if (sender_address == NULL || (!trusted_caller && filter_test == NULL))
        {
        if (gecos_pattern == NULL || gecos_name == NULL)
          user_name = string_copy(parse_fix_phrase(pw->pw_gecos));
          
        /* If a pattern for matching the gecos field was supplied, apply
        it and then expand the name string .*/
          
        else   
          {    
          char *name = pw->pw_gecos; 
          regexp *re;
          regexp_compiling = gecos_pattern;
          re = regcomp(gecos_pattern); 
          if (regexec(re, name)) 
            {
            char *new_name; 
            expand_nmax = 0; 
            for (; expand_nmax < NSUBEXP; expand_nmax++)
              {
              expand_nstring[expand_nmax] = re->startp[expand_nmax];
              expand_nlength[expand_nmax] = re->endp[expand_nmax] - 
                expand_nstring[expand_nmax];
              }                             
            expand_nmax--;
            new_name = expand_string(gecos_name);
            expand_nmax = -1;
            if (new_name != NULL) name = new_name; 
            }   
          user_name = string_copy(parse_fix_phrase(name)); 
          }
        }   
         
      /* A trusted caller has used -f but not -F */   
       
      else user_name = "";
      }    
      
    /* Break the retry loop */
      
    break;
    }
  sleep(1);
  }

/* If we cannot get a user login, log the incident and give up, unless the
configuration specifies something to use. */

if (user_login == NULL)
  {
  if (unknown_login != NULL)
    {
    user_login = expand_string(unknown_login);  
    if (user_name == NULL) 
      user_name = (unknown_username != NULL)?
        expand_string(unknown_username) : "";
    }
  else log_write(LOG_PANIC_DIE, "Failed to get user name for uid %d", real_uid);
  } 

/* When operating on spool files, the following are the variables that
are used, so set them here so the standard code can be used to write
a header file. */

originator_login = user_login;
originator_uid = real_uid;
originator_gid = real_gid;

/* Run in daemon and/or queue-running mode. The function daemon_go() never
returns. */

if (daemon_listen || queue_interval > 0) daemon_go();


/* Arrange for message reception if recipients or SMTP were specified;
otherwise complain unless a version print happened or this is a filter
verification test. */

if (recipients_arg >= argc && !extract_recipients && !smtp_input)
  {
  if (version_printed) return EXIT_SUCCESS;
  if (filter_test == NULL) 
    { 
    fprintf(stderr, "exim: neither flags nor addresses given\n");
    return EXIT_FAILURE;
    } 
  }


/* Accept one or more new messages on the standard input; accept_msg
returns TRUE if there are more messages to be read (SMTP input), or
FALSE otherwise (not SMTP, or SMTP channel collapsed).

When a message has been read, its id is returned in message_id[]. If
doing immediate delivery, we fork a delivery process for each received
message, except for the last one, where we can save a process switch. */

/* If the sender ident has not been set (by a trusted caller) set it to
the caller. This will get overwritten below for an inetd call. */

if (sender_ident == NULL)
  sender_ident = (user_login != NULL)? user_login : 
    string_sprintf("uid%d", real_uid);

/* It is only in this mode that error_handling is allowed to be changed from
its default of ERRORS_SENDER by argument. (Idle thought: are any of the
sendmail error modes other than -oem ever actually used?) */

if (!smtp_input) error_handling = arg_error_handling;

/* If smtp input has been requested, this may just be a local call, or it may
be that exim has been fired up by inetd. We can distinguish between the two
cases by tring to find the peer name of the standard input. If it is not a
socket, then this is not an inetd call. If it is an inetd call, set up the host
address and make an identd call to get the sender_ident. This is all relevant
only if called by root or the exim uid. */

else if (real_uid == root_uid || real_uid == exim_uid)
  {
  struct sockaddr sock;
  struct sockaddr_in *sin = (struct sockaddr_in *)(&sock);
  int size = sizeof(sock);
  int rc;

  if ((rc = getpeername(0, &sock, &size) == 0))
    {
    sender_host_address = string_copy(inet_ntoa(sin->sin_addr));
    sender_fullhost = string_sprintf("[%s]", sender_host_address);
    set_process_info("handling incoming connection from [%s] via inetd",
      sender_host_address);
    verify_get_ident(0);
    }
    
  /* Seems to give "invalid argument" when not a socket on SunOS5,
  so don't bother. */
  /*****  
  else if (rc != ENOTSOCK)
    {
    fprintf(stderr, "exim: getpeername error: %s\n", strerror(errno));
    exit(EXIT_FAILURE);
    }
  *****/   
  }


/* If stdout does not exist, then dup stdin to stdout. This can happen
if exim is started from inetd. In this case fd 0 will be set to the socket,
but fd 1 will not be set. */

if (fstat(1, &statbuf) < 0) dup2(0, 1);


/* If the sender host address has been set, build sender_fullhost. */

if (sender_host_address != NULL) 
  sender_fullhost = string_sprintf("%s%s[%s]",
    (sender_host_name != NULL)? sender_host_name : "",
    (sender_host_name != NULL)? "" : "",
     sender_host_address);

/* Otherwise, set the sender host as unknown. This prevents host checking in 
the case of -bs not from inetd and also for -bS. */

else sender_host_unknown = TRUE;


/* If a non-trusted user supplies a sender address of "<>", then this has the 
effect of setting the user_null_sender flag, which affects the MAIL FROM 
command on outgoing SMTP, handling of the MAIL command in filters, and setting 
of the Return_Path header in delivered messages, but is otherwise ignored. */

if (sender_address != NULL && sender_address[0] == 0 && !trusted_caller)
  user_null_sender = TRUE;

/* If the user running this process is not trusted, or if a trusted caller did
not supply a sender address with a -f argument for the non-smtp interface, then
the sender is local and the address is the user running the process. However,
allow anybody to use -f when testing filter files. */

if ((sender_address == NULL && !smtp_input) ||
    (!trusted_caller && filter_test == NULL))
  {
  sender_local = TRUE;
  sender_address = user_login;
  sender_address_domain = 0;
  }

/* If the user running this process is trusted and has supplied a sender
address with a -f argument, or is passing in the message via SMTP and the
sending host address has been set (inetd invocation or otherwise), the sender
is not considered to be local. */

else sender_local = FALSE;

/* Ensure that the sender address is fully qualified unless it is the empty
address, which indicates an error message, or doesn't exist (root caller, smtp
interface, no -f argument). */

if (sender_address != NULL && sender_address_domain == 0 &&
    sender_address[0] != 0 && sender_address[0] != '@')
  {
  char *new_sender_address =
    store_malloc((int)strlen(sender_address) + 
      (int)strlen(qualify_domain_sender) + 2);
  sprintf(new_sender_address, "%s@%s", sender_address, qualify_domain_sender);
  sender_address = new_sender_address;
  }

/* Set up the incoming protocol name and the state of the program. Root
is allowed to force received protocol via the -oMr option above, and if we are
in a non-local SMTP state it means we have come via inetd and the process info
has already been set up. */

if (smtp_input)
  {
  if (received_protocol == NULL) received_protocol = "local-smtp";
  if (sender_local) set_process_info("accepting a local SMTP message from <%s>",
    sender_address);
  }
else
  {
  if (received_protocol == NULL) received_protocol = "local";
  set_process_info("accepting a local non-SMTP message from <%s>",
    sender_address);
  }

/* Loop for several messages when reading SMTP input. */

while (more)
  {
  /* In the SMTP case, we have to handle the initial SMTP input and
  build the recipients list, before calling accept_msg to read the
  message proper. Whatever sender address is actually given in the
  SMTP transaction is actually ignored for local senders - we use
  the actual sender, which is either the underlying user running this
  process or a -f argument provided by a root caller. This is indicated
  by the existence of sender_address, which will be NULL only for a trusted
  caller when no -f was given. The function smtp_setup_msg() throws away
  any previous recipients list that may be lying around from a previous
  message. */

  if (smtp_input)
    {
    int rc;
    char *real_sender_address = sender_address;
    if ((rc = smtp_setup_msg(stdin, stdout, smtp_first)) > 0)
      {
      smtp_first = FALSE;

      if (real_sender_address != NULL)
        {
        store_free(sender_address);
        sender_address = real_sender_address;
        }
      more = accept_msg(stdin, stdout, extract_recipients);
      if (real_sender_address == NULL)
        {
        store_free(sender_address);
        sender_address = real_sender_address;
        }

      if (message_id[0] == 0)
        {
        if (more) continue;
        exit(EXIT_FAILURE);
        }
      }
    else exit((rc == 0)? EXIT_SUCCESS : EXIT_FAILURE);
    }

  /* In the non-SMTP case, we have all the information from the command
  line, but must process it in case it is in the more general RFC822
  format, and in any case, to detect syntax errors. Also, it appears that
  the use of comma-separated lists as single arguments is common, so we
  had better support them. */

  else
    {
    int i;
    int count = argc - recipients_arg;
    char **list = argv + recipients_arg;
    
    /* Loop for each argument */
      
    for (i = 0; i < count; i++)
      {
      int start, end, domain;
      char *errmess;
      char *s = list[i];
    
      /* Loop for each comma-separated address */
       
      while (*s != 0)
        {
        BOOL finished = FALSE; 
        char *ss, *receiver; 
         
        for (ss = s; *ss != 0 && *ss != ','; ss++)
          {
          if (*ss == '\"')
            {
            while (*(++ss) != 0 && *ss != '\"')
              if (*ss == '\\' && ss[1] != 0) ss++;
            if (*ss == 0) break;
            }
          }
        if (*ss == ',') *ss = 0; else finished = TRUE;
 
        receiver =
          parse_extract_address(s, &errmess, &start, &end, &domain, FALSE);
        if (receiver == NULL)
          {
          if (error_handling == ERRORS_STDERR)
            fprintf(stderr, "exim: bad address \"%s\": %s\n", 
              string_printing(list[i], FALSE), errmess);
          else
            {
            error_block eblock;
            eblock.next = NULL;
            eblock.text1 = string_printing(list[i], FALSE);
            eblock.text2 = errmess;
            moan_to_sender(ERRMESS_BADARGADDRESS, &eblock, NULL, stdin);
            }
          return EXIT_FAILURE;
          }
        accept_add_recipient(receiver);
        s = ss;
        if (!finished) while (*(++s) != 0 && *s == ','); 
        } 
      }

    /* Show the recipients when debugging */
    
    DEBUG(9)
      {
      int i;
      if (sender_address != NULL) debug_printf("Sender: %s\n", sender_address);
      if (recipients_list != NULL)
        {
        debug_printf("Recipients:\n");
        for (i = 0; i < recipients_count; i++)
          debug_printf("  %s\n", recipients_list[i]);
        }
      }

    /* Read the data for the message. If filter_test is true, this will
    just read the headers for the message, and not write anything onto
    the spool. */

    more = accept_msg(stdin, stdout, extract_recipients);
     
    /* more is always FALSE here (not SMTP message) */
     
    if (message_id[0] == 0) return EXIT_FAILURE;
    }
    
  /* If this is a filter testing run, there are headers in store, but
  no message on the spool. Run the filtering code in testing mode, setting
  the domain to the qualify domain, the local part to the current user, and
  the return path to the sender unless it has already been set from a 
  return-path header in the message. */
  
  if (filter_test != NULL) 
    {
    deliver_domain = qualify_domain_recipient;
    deliver_localpart = user_login;
    deliver_home = getenv("HOME"); 
    if (return_path == NULL) return_path = string_copy(sender_address); 
    exit(filter_runtest(filter_fd)? EXIT_SUCCESS : EXIT_FAILURE);
    } 

  /* Else act on the result of message reception. We should not get here unless
  message_id[0] is non-zero. Unless queue_only is set, arrange for delivery
  of the message to happen. */ 

  else if (!queue_only)
    {
    /* If running as root, or if we can regain root by seteuid, call the
    delivery function directly for synchronous delivery, or fork a clone of 
    this process to deliver in the background. A fork failure is not a 
    disaster, as the delivery will eventually happen on a subsequent queue 
    run. */ 
    
    if (security_level <= 1)
      { 
      if (synchronous_delivery) (void)deliver_message(message_id, FALSE, FALSE);
      else if(fork() == 0)
        {
        (void)deliver_message(message_id, FALSE, FALSE);
        _exit(EXIT_SUCCESS);
        }
      }
      
    /* Otherwise, we have to re-exec exim in order to regain root for the
    delivery of the message. */ 
    
    else
      {
      int pid; 
      if ((pid = vfork()) == 0)
        {
        int i = 0; 
        char *argv[6];
        argv[i++] = exim_path;
        if (debug_level > 0)
          argv[i++] = string_sprintf("-d%d", debug_level);  
        if (dont_deliver) argv[i++] = "-N";   
        argv[i++] = "-Mc";
        argv[i++] = message_id;
        argv[i++] = NULL;       
        
        DEBUG(4) debug_printf("fork %s %s %s %s %s\n", argv[0], argv[1],
          argv[2], 
          (argv[3] == NULL)? "" : argv[3],
          (argv[4] == NULL)? "" : argv[4]); 
        
        execv(argv[0], argv);
        _exit(errno);
        }  
         
      /* In the parent, wait if synchronous delivery is required. */
      
      if (synchronous_delivery && pid > 0)
        {
        int status;
        while (wait(&status) != pid);  
        }   
      }   
    }

  /* The loop will repeat if more is TRUE. */
  }

return EXIT_SUCCESS;
}

/* End of exim.c */
