/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */


/* General functions concerned with directing, and generic director options. */


#include "exim.h"



/* Generic options for directors, all of which live inside director_instance
data blocks. */

optionlist optionlist_directors[] = {
  { "domains",            opt_stringptr,
                 (void *)offsetof(director_instance, domains) },
  { "driver",             opt_stringptr,
                 (void *)offsetof(director_instance, driver_name) },
  { "fail_verify",        opt_bool,
                 (void *)offsetof(director_instance, fail_verify) },
  { "more",               opt_bool,
                 (void *)offsetof(director_instance, more) },
  { "prefix",             opt_stringptr,
                 (void *)offsetof(director_instance, prefix) },
  { "prefix_optional",    opt_bool,
                 (void *)offsetof(director_instance, prefix_optional) },
  { "require_files",      opt_stringptr,
                 (void *)offsetof(director_instance, require_files) },
  { "suffix",             opt_stringptr,
                 (void *)offsetof(director_instance, suffix) },
  { "suffix_optional",    opt_bool,
                 (void *)offsetof(director_instance, suffix_optional) },
  { "transport",          opt_transportptr,
                 (void *)offsetof(director_instance, transport) },
  { "unseen",             opt_bool,
                 (void *)offsetof(director_instance, unseen) },                 
  { "verify",             opt_bool,
                 (void *)offsetof(director_instance, verify) }
};

int optionlist_directors_size =
  sizeof(optionlist_directors)/sizeof(optionlist);



/*************************************************
*           Find a local user                    *
*************************************************/

/* Try several times (if configured) to find a local user, in case delays in
NIS or NFS whatever cause an incorrect refusal. It's a pity that getpwnam()
doesn't have some kind of indication as to why it has failed. If the string
given consists entirely of digits, and the second argument is not NULL, assume
the string is the numerical value of the uid. Otherwise it is looked up using
getpwname(). The uid is passed back via return_uid, if not NULL, and function's
yield is the pointer to a passwd structure, if found.

Because this may be called several times in succession for the same user for
different directors, cache the result of the previous call so that it can be
re-used. Note that we can't just copy the structure, as the store it points to 
can get trashed. */

static struct passwd pwcopy;
static struct passwd *lastpw = NULL;
static char lastname[48] = { 0 };
static char lastdir[128];

struct passwd *direct_finduser(char *s, int *return_uid)
{
if (strcmp(lastname, s) != 0)
  {
  int i = 0;
  int digitlength = strspn(s, "0123456789");
  BOOL numerical = s[digitlength] == 0;

  if (numerical && return_uid != NULL)
    {
    *return_uid = atoi(s);
    return NULL;
    }

  strncpy(lastname, s, sizeof(lastname));

  for (;;)
    {
    if ((lastpw = getpwnam(s)) != NULL) break;
    if (++i > finduser_retries) break;
    sleep(1);
    }

  if (lastpw != NULL)
    {
    pwcopy.pw_uid = lastpw->pw_uid;
    pwcopy.pw_gid = lastpw->pw_gid;
    strncpy(lastdir, lastpw->pw_dir, sizeof(lastdir));  
    pwcopy.pw_name = lastname;
    pwcopy.pw_dir = lastdir;
    lastpw = &pwcopy;
    }
  }

if (return_uid != NULL)
  *return_uid = (lastpw == NULL)? (-1) : lastpw->pw_uid;
return lastpw;
}




/*************************************************
*           Find a local group                   *
*************************************************/

struct group *direct_findgroup(char *s, int *return_gid)
{
int i = 0;
struct group *gr;
int digitlength = strspn(s, "0123456789");
BOOL numerical = s[digitlength] == 0;

if (numerical && return_gid != NULL)
  {
  *return_gid = atoi(s);
  return NULL;
  }

for (;;)
  {
  if ((gr = getgrnam(s)) != NULL)
    {
    if (return_gid != NULL) *return_gid = gr->gr_gid;
    return gr;
    }
  if (++i > finduser_retries) break;
  sleep(1);
  }

if (return_gid != NULL) *return_gid = -1;
return NULL;
}




/*************************************************
*          Find user by expanding string         *
*************************************************/

struct passwd *direct_find_expanded_user(char *string, char *driver_name,
  char *driver_type, int *uid)
{
struct passwd *pw;
char *user = expand_string(string);
if (user == NULL)
  log_write(LOG_PANIC_DIE, "Failed to expand user string \"%s\" from the "
    "%s %s", string, driver_name, driver_type);
pw = direct_finduser(user, uid);
if (*uid < 0)
  log_write(LOG_PANIC_DIE, "Failed to find user \"%s\" from expanded string "
    "\"%s\" from the %s %s", user, string, driver_name, driver_type);
store_free(user);
return pw;
}



/*************************************************
*          Find group by expanding string        *
*************************************************/

struct group *direct_find_expanded_group(char *string, char *driver_name,
  char *driver_type, int *gid)
{
struct group *gr;
char *group = expand_string(string);
if (group == NULL)
  log_write(LOG_PANIC_DIE, "Failed to expand group string \"%s\" from the "
    "%s %s", string, driver_name, driver_type);
gr = direct_findgroup(group, gid);
if (*gid < 0)
  log_write(LOG_PANIC_DIE, "Failed to find group \"%s\" from expanded string "
    "\"%s\" from the %s %s", group, string, driver_name, driver_type);
store_free(group);
return gr;
}



/*************************************************
*             Initialize director list           *
*************************************************/

/* Read the directors configuration file, and set up a chain of director
instances according to its contents. Each director has generic options and may
also have its own private options. This function is only ever called when
directors == NULL. We use generic code in readconf to do the work. */

void direct_init(void)
{
director_instance *d;
readconf_driver_init("director",
  (driver_instance **)(&directors),     /* chain anchor */
  (driver_info *)directors_available,   /* available drivers */
  sizeof(director_info),                /* size of info blocks */
  &director_defaults,                   /* default values for generic options */
  sizeof(director_instance),            /* size of instance block */
  optionlist_directors,                 /* generic options */
  optionlist_directors_size);

/* A director may not have more=FALSE and unseen=TRUE */
  
for (d = directors; d != NULL; d = d->next)
  if (d->unseen && !d->more )
    log_write(LOG_PANIC_DIE, "Exim configuration error:\n  "
      "the combination of \"unseen\" and \"no_more\" is not permitted on "
      "a director,\n  but was set for the %s director", d->name);   
}




/*************************************************
*           Tidy up after directing              *
*************************************************/

/* Directors are entitled to keep hold of certain resources in their instance
blocks so as to save setting them up each time. An example is the open file for
the aliasfile director. Such directors must provide a tidyup entry point which
is called when all directing is finished via this function. */

void direct_tidyup(void)
{
director_instance *d;
for (d = directors; d != NULL; d = d->next)
  if (d->info->tidyup != NULL) (d->info->tidyup)(d);
}





/*************************************************
*                 Direct one address             *
*************************************************/

/* This function is passed in one address item, for processing by the
directors. It has been determined that the address is for one of the local
domains. The action can be one of:

  . adding the address to the chain on addr_local, for local delivery, and
    returning OK;
  . adding the address to the chain on addr_remote, for remote delivery, and
    returning OK;
  . returning FAIL;
  . returning DEFER;
  . adding one or more new addresses, with this one as parent, to the chain
    on addr_new, for reprocessing ab initio, and returning OK.

The verify flag is set if this is being called for verification rather than
delivery. If the director doesn't have its "verify" flag set, it is skipped.
Otherwise, the flag is passed to the director in case it needs to know.

If the director has the "more" flag set false (i.e. "no_more" has been
specified) then, if it fails to match a name, no further directors are tried.
This includes the case when the director would have been called, but for the
verify flag's being FALSE.

The generic options "domain", "prefix", and "suffix" are handled at this top
level. */


int direct_address(address_item *addr,
  address_item **addr_local,
  address_item **addr_remote,
  address_item **addr_new,
  BOOL verify)
{
director_instance *d;
address_item *parent;
BOOL more = TRUE;

DEBUG(9) debug_printf("Directing %s\n", addr->orig);

/* Loop through all director instances. */

for (d = directors; more && d != NULL; d = d->next)
  {
  int yield;
  int suffixchar;
  int suffixend = -1;
  char *oldlocal_part = NULL;
  BOOL loop_detected = FALSE;

  /* Skip this director if domain mismatch */

  if (d->domains != NULL &&
       !match_isinlist(addr->domain, d->domains, &(d->re_domains)))
    {
    DEBUG(9) debug_printf("%s director skipped: domain mismatch\n",
      d->name);
    continue;
    }

  /* If this director's "more" flag is FALSE, arrange that no subsequent
  directors are called. */

  more = d->more;
  DEBUG(9)
    {
    if (!more) debug_printf("%s director has more set FALSE\n", d->name);
    }

  /* Skip this director if verifying and it hasn't got the verify flag */

  if (verify && !d->verify)
    {
    DEBUG(9) debug_printf("%s director skipped: no_verify set\n",
      d->name);
    continue;
    }

  /* Handle any configured prefix by replacing the local_part address,
  saving it for restoration if the director fails. Skip the
  director if the prefix doesn't match, unless the prefix is optional. */

  if (d->prefix != NULL)
    {
    int plen = (int)strlen(d->prefix);
    if (strncmpic(d->prefix, addr->local_part, plen) == 0)
      {
      oldlocal_part = addr->local_part;
      addr->local_part += plen;
      DEBUG(9) debug_printf("stripped prefix %s\n", d->prefix);
      }
    else if (!d->prefix_optional)
      {
      DEBUG(9) debug_printf("%s director skipped: prefix mismatch\n",
        d->name);
      continue;
      }
    }

  /* Handle any configured suffix likewise, but for this we have to
  fudge the end of the address; save the character so it can be put
  back if the director fails. If we fail to match the suffix, remember
  to restore any change the prefix handler has made. */

  if (d->suffix != NULL)
    {
    int slen = (int)strlen(d->suffix);
    int alen = (int)strlen(addr->local_part);
    if (alen > slen &&
        strncmpic(d->suffix, addr->local_part + alen - slen, slen) == 0)
      {
      suffixend = alen - slen;
      suffixchar = addr->local_part[suffixend];
      addr->local_part[suffixend] = 0;
      DEBUG(9) debug_printf("stripped suffix %s\n", d->suffix);
      }
    else if (!d->suffix_optional)
      {
      DEBUG(9) debug_printf("%s director skipped: suffix mismatch\n",
        d->name);
      if (oldlocal_part != NULL) addr->local_part = oldlocal_part;
      continue;
      }
    }

  /* Loop protection: If this address has a parent with the same address that
  was directed by this director, we skip this director. This prevents a variety
  of possibly looping states, and saves us having to do anything special for
  the forwardfile director. */

  for (parent = addr->parent; parent != NULL; parent = parent->parent)
    {
    if (strcmpic(parent->local_part, addr->local_part) == 0 &&
        parent->director == d)
      {
      DEBUG(9) debug_printf("%s director skipped: previously directed %s\n",
        d->name, addr->local_part);
      loop_detected = TRUE;
      break;
      }
    }
  if (loop_detected) continue;


  /* Prefix and suffix (if any) have been stripped; nearly ready to call the
  director, first setting up values that may be used in string expansions. */

  deliver_localpart = addr->local_part;
  deliver_domain = addr->domain;

  /* Now we can call the director */

  yield = (d->info->code)(d, addr, addr_local, addr_remote, addr_new, verify);
  deliver_localpart = NULL;
  deliver_domain = NULL;
  deliver_home = NULL;    /* gets set by localuser */ 

  /* Success or deferral means we are finished with this address, as does an 
  internal or configuration failure. If we succeed on a director that has 
  "fail_verify" set, convert the result into a fail. If we succeed on a 
  director that has "unseen" set on it, we must make a copy of the address to
  hand on to the subsequent directors. Actually, we can't pass it on directly; 
  we have to put it on the new queue, but since it has the same address as
  the current one, it will be passed by all previous directors and also the
  one that has just succeeded, by the loop-avoidance code. */

  if (yield == OK || yield == DEFER || yield == ERROR)
    {
    addr->director = d;

    if (yield == OK)
      {
      if (verify && d->fail_verify)
        {
        yield = FAIL;
        addr->message = string_sprintf("%s director forced verify failure",
          d->name);
        }
      else if (d->unseen)
        {
        char *new_address = 
          string_sprintf("%s@%s", addr->local_part, addr->domain);
        address_item *new = deliver_make_addr(new_address);
        new->parent = addr;
        new->ignore_error |= addr->ignore_error; 
        new->errors_address = addr->errors_address; 
        addr->child_count++; 
        new->next = *addr_new;
        *addr_new = new;
        }     
      }   

    DEBUG(2)
      {
      if (yield == OK)
        {
        debug_printf("%s director succeeded for %s\n", d->name,
          addr->local_part);
        debug_printf("  transport: %s\n", (addr->transport == NULL)?
          "<none>" : addr->transport->name);
        }
      else if (yield == DEFER)
        {
        debug_printf("%s director deferred %s\n", d->name, addr->local_part);
        debug_printf("  message: %s\n", (addr->message == NULL)?
          "<none>" : addr->message);
        }
      else
        {
        debug_printf("%s director: error for %s\n", d->name, addr->local_part);
        debug_printf("  message: %s\n", (addr->message == NULL)?
          "<none>" : addr->message);
        }
      }
    return yield;
    }

  /* Restore prefix & suffix for the next director. */

  if (suffixend >= 0) addr->local_part[suffixend] = suffixchar;
  if (oldlocal_part != NULL) addr->local_part = oldlocal_part;

  /* Break the loop if a director forced a failure. */

  if (yield == FORCEFAIL) break;
  }

/* No directors accepted this address; fail it. */

if (addr->message == NULL)
  addr->message = string_sprintf("unknown local-part \"%s\" in domain \"%s\"",
    addr->local_part, addr->domain);
return FAIL;
}

/* End of direct.c */
