/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* The main code for delivering a message. */


#include "exim.h"



/*************************************************
*            Local static variables              *
*************************************************/

static address_item *addr_defer = NULL;
static address_item *addr_direct = NULL;
static address_item *addr_duplicate = NULL;
static address_item *addr_failed = NULL;
static address_item *addr_local = NULL;
static address_item *addr_new = NULL;
static address_item *addr_remote = NULL;
static address_item *addr_route = NULL;
static address_item *addr_succeed = NULL;

static BOOL nonrecipients_changed;

static int return_count;


/* Table for turning base-62 numbers into binary */

static char tab62[] =
          {0,1,2,3,4,5,6,7,8,9,0,0,0,0,0,0,     /* 0-9 */
           0,10,11,12,13,14,15,16,17,18,19,20,  /* A-K */
          21,22,23,24,25,26,27,28,29,30,31,32,  /* L-W */
          33,34,35, 0, 0, 0, 0, 0,              /* X-Z */
           0,36,37,38,39,40,41,42,43,44,45,46,  /* a-k */
          47,48,49,50,51,52,53,54,55,56,57,58,  /* l-w */
          59,60,61};                            /* x-z */



/*************************************************
*             Make a new address item            *
*************************************************/

/* This function gets the store and initializes with default values. The
transport_return value defaults to DEFER, so that any unexpected failure to
deliver does not wipe out the message. */

address_item *deliver_make_addr(char *address)
{
address_item *addr = store_malloc(sizeof(address_item));
addr->next = NULL;
addr->parent = NULL;
addr->orig = address;
addr->unique = addr->orig;

addr->local_part = NULL;
addr->domain = NULL;
addr->route_domain = NULL;
addr->rewrite_headers = FALSE;
addr->errors_address = NULL;
addr->ignore_error = FALSE;
addr->local = FALSE;
addr->delivered = FALSE;

addr->director = NULL;
addr->router = NULL;
addr->transport = NULL;
addr->host_list = NULL;
addr->transported = NULL;
addr->routed_by_domain = TRUE;

addr->child_count = 0;
addr->uid = -1;
addr->gid = -1;
addr->home_dir = NULL;
addr->expand_pipe = FALSE;
addr->allow_pipe = FALSE;
addr->allow_file = FALSE;
addr->allow_reply = FALSE;

addr->transport_return = DEFER;
addr->errno = ERRNO_UNKNOWNERROR;
addr->more_errno = 0;
addr->message = NULL;
addr->reply = NULL;

addr->special_action = SPECIAL_NONE;
addr->ltp_next_try = 0;
addr->ltp_expired = FALSE;
addr->dr_retry_exists = FALSE;

addr->retry_skipped = FALSE;
addr->retry_timedout = FALSE;
addr->retries = FALSE;
addr->return_filename = NULL;
addr->return_file = -1;

return addr;
}




/*************************************************
*              Compare lists of hosts            *
*************************************************/

/* This function is given two pointers to chains of host items, and it yields
TRUE if the lists refer to the same hosts in the same order, except that 
multiple hosts with the same non-negative MX values are permitted to appear in
different orders. Round-robinning nameservers can cause this to happen.

This enables exim to use a single SMTP transaction for sending to two entirely
different domains that happen to end up pointing at the same hosts. For
identical domains, the two pointers may well be the same, as the routing is
normally just copied. */

static BOOL same_hosts(host_item *one, host_item *two)
{
if (one == two) return TRUE;

while (one != NULL && two != NULL)
  {
  if (strcmp(one->name, two->name) != 0) 
    { 
    int mx = one->mx; 
    host_item *end_one = one;
    host_item *end_two = two; 
    
    /* Batch up only if MX >= 0 */
     
    if (mx < 0) return FALSE;
 
    /* Find the ends of the shortest sequence of identical MX values */
     
    while (end_one->next != NULL && end_one->next->mx == mx &&
           end_two->next != NULL && end_two->next->mx == mx)
      {       
      end_one = end_one->next;
      end_two = end_two->next;
      }  
      
    /* If there aren't any duplicates, there's no match. */
    
    if (end_one == one) return FALSE; 

    /* For each host in the 'one' sequence, check that it appears in the 'two'
    sequence, returning FALSE if not. */
     
    for (;;)
      {
      host_item *hi;
      for (hi = two; hi != end_two->next; hi = hi->next)
        if (strcmp(one->name, hi->name) == 0) break;
      if (hi == end_two->next) return FALSE;
      if (one == end_one) break;
      one = one->next; 
      }  

    /* All the hosts in the 'one' sequence were found in the 'two' sequence.
    Ensure both are pointing at the last host, and carry on as for equality. */
    
    two = end_two;
    } 
    
  /* Hosts matched */
   
  one = one->next;
  two = two->next;
  }

/* True if both are NULL */

return (one == two);
}



/*************************************************
*          Determine locality of an address      *
*************************************************/

/* This function determines whether an address is local or not, and sets the
"local" field in the address accordingly. It also sets the "local_part" and
"domain" fields, and lowercases appropriate parts of the address. */

void deliver_setlocal(address_item *addr)
{
char *domain = parse_find_at(addr->orig);
char *active = addr->orig;

/* If this is a source-routed address, strip off any local domains
at the start, and point to the rest as the current active address. If
we can determine that delivery is not local, unset the local flag and
set up the local part and the domain. If route stripping gets us to the
final address (after ':') check that it has a domain attached. If it
doesn't, turn it into domain form by attaching the final domain removed. */

addr->local = TRUE;
while (*active == '@')
  {
  char *p = active;
  while (*(++p) != ',' && *p != ':');
  domain = string_copynlc(active+1, p-active-1);

  /* The source route points at a local domain */

  if (match_isinlist(domain, local_domains, &re_local_domains))
    {
    char *newdomain;
    if (*p == ':' && (newdomain = strchr(p+1, '@')) == NULL)
      {
      int n = (int)strlen(p+1);
      char *newactive =
        store_malloc((int)strlen(domain) + n + 2);
      sprintf(newactive, "%s@%s", p+1, domain);
      active = newactive;
      domain = active + n;
      }
    else
      {
      active = p+1;
      domain = newdomain;
      }
    }

  /* It's a remote, source-routed address. In this case the "local_part"
  is the remainder of the address, identifiable as such because it begins
  with a comma or a colon. */

  else
    {
    addr->local_part = p;
    addr->domain = domain;
    addr->local = FALSE;
    return;
    }
  }


/* Deal with a non-source-routed address (which might be the final component of
a local source-routed address). If it is local, we must be prepared to do the
"percent hack" for configured domains.

On reaching this point, "active" points to the operative part of the address,
and the '@' in the address is pointed to by "domain". Both these are pointers
inside the "orig" address, and so the data they point to must be copied
so that it can be modified if necessary. */

addr->domain = string_copylc(domain+1);

/* The address is a local domain */

if (match_isinlist(addr->domain, local_domains, &re_local_domains))
  {
  addr->local_part = string_copynlc(active, domain-active);

  /* If the configuration variable percent_hack_domains is NULL (unset)
  we do the % hack for all domains, as other MTAs do. Otherwise, we
  do it only for those domains specified. We need to loop to handle
  the case of multiple local %-hacks. */

  while (percent_hack_domains == NULL ||
      match_isinlist(addr->domain, percent_hack_domains,
        &re_percent_hack_domains))
    {
    char *pc = strrchr(addr->local_part, '%');
    if (pc == NULL) break;

    /* Handle a % hack by making a copy of the new domain and then removing
    it from the local_part field. If the new domain is remote, we are done;
    otherwise let the loop continue to check for more % hacks. */

    addr->domain = string_copy(pc+1);
    *pc = 0;

    if (!match_isinlist(addr->domain, local_domains, &re_local_domains))
      {
      addr->local = FALSE;
      break;
      }
    }
  }

/* The address refers to a remote domain. Don't mess with the case
of the local part. */

else
  {
  addr->local_part = string_copyn(active, domain-active);
  addr->local = FALSE;
  }
}



/*************************************************
*           Perform a local delivery             *
*************************************************/

/* Each local delivery is performed in a separate process which sets its
uid and gid as specified. This is a safer way than simply changing and
restoring using seteuid; there is a body of opinion that seteuid cannot be
used safely. Besides, not all systems have seteuid.

If the security level is 1 or 3 this function will be running sete{g,u}id to
the exim uid and gid, and must use sete{g,u}id to regain root privilege before
using set{g,u}id. Otherwise it is running as root.

If the uid/gid are specified in the transport_instance, they are used; the
transport initialization must ensure that either both or neither are set.
Otherwise, the values associated with the address are used. If neither are set,
it is a configuration error.

Using a separate process makes it more complicated to get error information
back. We use a pipe to pass the return code and also an error code and error
text string back to the parent process. */

static void deliver_local(address_item *addr, char *id)
{
int uid = -1;
int gid = -1;
int pid, status, len;
int pfd[2];
address_item *addr2;
transport_instance *tp = addr->transport;
char buffer[256];

/* If transport not set or not local, panic */

if (tp == NULL)
  log_write(LOG_PANIC_DIE, "No transport set for local delivery of %s",
    addr->orig);
if (!tp->info->local)
  log_write(LOG_PANIC_DIE, "Attempt local delivery with non-local transport "
    "for %s: transport %s", addr->orig, tp->name);

/* Set up variables that are relevant to a single delivery, for use
in various expansions and the environment of scripts. Need to set them as
early as this in case an expansion is required to get a uid or gid or home. */

deliver_domain = addr->domain;
deliver_localpart = addr->local_part;
deliver_host = (addr->host_list == NULL)? "" : addr->host_list->name;

/* If delivering to a pipe or file, or sending an autoreply, get the local part
from the parent if there is a parent. */

if ((deliver_localpart[0] == '|' || deliver_localpart[0] == '/' ||
     deliver_localpart[0] == '>') && addr->parent != NULL) 
  deliver_localpart = addr->parent->local_part;

/* Get the uid and gid to use. First see if the transport either set explicit
values, or set the deliver_as_creator flag. It should not be possible for both
to be set. */

/* First see if there's a gid on the transport. */

if (tp->gid >= 0) 
  gid = tp->gid; 
else if (tp->expand_gid != NULL)
  (void) direct_find_expanded_group(tp->expand_gid, tp->name, "transport",
    &gid);

/* Now pick up a uid from the transport if one is set. */

if (tp->uid >= 0) uid = tp->uid;

/* Try for an expandable uid field. If it ends up as a numeric it, it does
not provide a passwd value from which a gid can be taken. */

else if (tp->expand_uid != NULL)
  {
  struct passwd *pw = direct_find_expanded_user(tp->expand_uid,
    tp->name, "transport", &uid);
  if (gid < 0 && pw != NULL) gid = pw->pw_gid;
  }

/* Otherwise, test the deliver_creator flag. */

else if (tp->deliver_as_creator)
  {
  uid = originator_uid;
  if (gid < 0) gid = originator_gid;
  }

/* If the transport didn't specify anything, then the address must. Note
that the address may have come via a router, in which case the only place
a uid/gid can be set is on the transport. */

else
  {
  if (addr->uid < 0)
    { 
    if (addr->director != NULL) 
      log_write(LOG_PANIC_DIE, "Neither the %s director nor the %s transport "
        "set a uid for local delivery of %s", addr->director->name,
        tp->name, addr->orig);
    else
      log_write(LOG_PANIC_DIE, "The %s transport has not set a "
        "uid for local delivery of %s", tp->name, addr->orig);
    }   
  uid = addr->uid;
  if (gid < 0) gid = addr->gid;
  }
  
/* If no gid has been set by this time, it is a disaster. */

if (gid < 0)
  log_write(LOG_PANIC_DIE, "User set without group for %s transport",
    tp->name);
 
/* See if the uid is on the list of banned uids for local delivery, and if
so, substitute the nobody user if configured. If there is no nobody, default to
"nobody", and if that isn't available, give up. */

if (never_users != NULL)
  {
  int *nn = never_users;
  while (*nn >= 0) if (*nn++ == uid)
    {
    if (nobody_uid < 0)
      {
      struct passwd *pw = direct_finduser("nobody", NULL);
      if (pw != NULL)
        {
        nobody_uid = pw->pw_uid;
        nobody_gid = pw->pw_gid;
        }
      }
    if (nobody_uid >= 0)
      {
      uid = nobody_uid;
      gid = nobody_gid;
      }
    else
      log_write(LOG_PANIC_DIE, "Uid %d is not permitted to deliver, but "
        "there is no \"nobody\" user specified", uid);
    }
  }
  
/* See if either the transport or the address specifies a home (current) 
directory. Expand it if necessary. */

deliver_home = (tp->home_dir != NULL)? tp->home_dir :
               (addr->home_dir != NULL)? addr->home_dir : NULL;
  
if (deliver_home != NULL)
  {
  char *rawhome = deliver_home; 
  deliver_home = expand_string(rawhome);  
  if (deliver_home == NULL)
    {
    /*** remember these if ever not dying
    deliver_localpart = NULL;
    deliver_domain = NULL;
    deliver_host = NULL; 
    ***/
    
    log_write(LOG_PANIC_DIE, "home directory string %s failed to expand: %s", 
      rawhome, expand_string_message);   
    }   
     
  if (*deliver_home != '/')
    log_write(LOG_PANIC_DIE, "home directory path %s is not absolute", 
    deliver_home);
  }  

DEBUG(2) debug_printf("delivering %s as %s using %s:\n  uid=%d gid=%d home=%s\n",
  addr->orig, addr->local_part, tp->name, uid, gid,
  (deliver_home == NULL)? "null" : deliver_home);

/* If the return_output flag is set on the transport, create and open a file in
the message log directory for the transport to write its output onto. This is
mainly used by pipe transports. The file needs to be unique to the address. */

if (tp->return_output)
  {
  addr->return_filename =
    string_sprintf("%s/msglog/%s-%d-%d", spool_directory, id, getpid(),
      return_count++);
  addr->return_file =
    open(addr->return_filename, O_WRONLY|O_CREAT|O_APPEND, 0400);
  if (addr->return_file < 0)
    log_write(LOG_PANIC_DIE, "Unable to create file for %s transport to "
      "return message: %s", tp->name, strerror(errno));
  }

/* Create the pipe for inter-process communication. */

if (pipe(pfd) != 0)
  log_write(LOG_PANIC_DIE, "Creation of pipe failed: %s", strerror(errno));

/* Now fork the process to do the real work in the sub-process. */

if ((pid = fork()) == 0)
  {
  /* Ignore SIGINT and SIGTERM during delivery (SIGHUP is ignored throughout
  exim, except when being a daemon). */

  signal(SIGINT, SIG_IGN);
  signal(SIGTERM, SIG_IGN);

  /* Close the unwanted half of the pipe, and set the required gid/uid, first
  regaining root privilege if necessary. Note that mac_sete{g,u}id expands
  to a failure on systems without the sete{g,u}id functions, but on such
  systems we should always be running as root here.

  If debug_transport is not NULL, do not change uid and gid here, since
  we want to continue running as root or exim in order to write to the
  debugging output file. */

  close(pfd[pipe_read]);

  #ifdef TRANSPORT_DEBUG
  if (debug_transport == NULL)
  #endif

  if ((geteuid() != root_uid && mac_seteuid(root_uid) < 0) ||
    setgid(gid) < 0 || setuid(uid) < 0)
      log_write(LOG_PANIC_DIE, "Unable to set uid=%d or gid=%d for local "
        "delivery to %s", uid, gid, addr->local_part);
        
  if (deliver_home != NULL && chdir(deliver_home) < 0)
    { 
    addr->transport_return = DEFER;
    addr->errno = errno;
    addr->message = string_sprintf("failed to chdir to %s", deliver_home);
    }   

  else
    { 
    DEBUG(2) debug_printf("Local delivery process %d uid=%d gid=%d\n",
      getpid(), geteuid(), getegid());
    set_process_info("delivering %s to %s using %s", message_id,
     addr->local_part, addr->transport->name);
    
    /* Now call the transport, write any error codes, special action and message
    down the pipe, and terminate the process successfully. */
    
    #ifdef TRANSPORT_DEBUG
    if (debug_transport != NULL)
      (debug_transport->code)(addr->transport, addr);
    else
    #endif
    
    (addr->transport->info->code)(addr->transport, addr);
    }
    
  /* Pass the result back down the pipe. */  

  write(pfd[pipe_write], &(addr->transport_return), sizeof(int));
  write(pfd[pipe_write], &(addr->errno), sizeof(int));
  write(pfd[pipe_write], &(addr->more_errno), sizeof(int));
  write(pfd[pipe_write], &(addr->special_action), sizeof(int));
  if (addr->message != NULL)
    write(pfd[pipe_write], addr->message, (int)strlen(addr->message) + 1);
  close(pfd[pipe_write]);
  exit(EXIT_SUCCESS);
  }

/* Panic if the fork did not succeed. */

if (pid <= 0)
  log_write(LOG_MAIN|LOG_PANIC_DIE, "Fork failed for local delivery to %s", 
    addr->orig);

/* Read the pipe to get the delivery status codes and a possible error message.
Our copy of the writing end must be closed first, as otherwise read() won't
return zero on an empty pipe. Afterwards, close the reading end. */

close(pfd[pipe_write]);
len = read(pfd[pipe_read], &status, sizeof(int));
if (len > 0)
  {
  addr->transport_return = status;
  len = read(pfd[pipe_read], &(addr->errno), sizeof(int));
  len = read(pfd[pipe_read], &(addr->more_errno), sizeof(int));
  len = read(pfd[pipe_read], &(addr->special_action), sizeof(int));
  len = read(pfd[pipe_read], buffer, sizeof(buffer));
  if (len > 0) addr->message = string_copy(buffer);
  }
close(pfd[pipe_read]);

/* If more than one address was involved (batched SMTP), copy the fields into 
all the others. */

for (addr2 = addr->next; addr2 != NULL; addr2 = addr2->next)
  {
  addr2->transport_return = addr->transport_return; 
  addr2->errno = addr->errno;
  addr2->more_errno = addr->more_errno;
  addr2->special_action = addr->special_action;   
  addr2->message = addr->message; 
  } 

/* Reset variables that are relevant to a single delivery. */

deliver_localpart = NULL;
deliver_domain = NULL;
deliver_host = NULL;
deliver_home = NULL;

/* Wait for the process to finish. If it terminates with a non-zero
code, freeze the message. */

while (wait(&status) != pid);
if ((status & 0xFF00) != 0)
  {
  addr->transport_return = DEFER;
  addr->special_action = SPECIAL_FREEZE;
  addr->message =
    string_sprintf("%s transport process returned a non-zero code (%d)",
      addr->transport->driver_name, (status >> 8) & 255);
  }
}



/*************************************************
*      Decrease counts in parents and mark done  *
*************************************************/

/* This function is called when an address is complete. The chain of parents is
scanned, and the count of children of each parent is decremented. If it becomes
zero for any parent, that address is added to the non-recipients tree because
it is complete. The yield is the address item for the original ancestor. */

static address_item *child_done(address_item *addr, char *now)
{
int decrement = 1;
while (addr->parent != NULL)
  {
  addr = addr->parent;
  if ((addr->child_count -= decrement) <= 0)
    {
    tree_add_nonrecipient(addr->orig);
    nonrecipients_changed = TRUE;
    fprintf(message_log, "%s %s: children all complete\n", now,
      addr->orig);
    fflush(message_log);
    }
  else decrement = 0;
  }
return addr;
}




/*************************************************
*    Actions at the end of handling an address   *
*************************************************/

/* This is a function for processing a single address when all that can be done
with it has been done. */

static void post_process_one(address_item *addr, int result, int logflags,
  int driver_type, int logchar)
{
char *now = tod_stamp(tod_log);
char *driver_kind = "?";
char *driver_name = "";

DEBUG(9) debug_printf("post-process %s\n", addr->orig);

/* Set up driver kind and name for logging */

if (driver_type == DTYPE_TRANSPORT)
  {
  if (addr->transport != NULL)
    {
    driver_name = addr->transport->name;
    driver_kind = " transport";
    }
  else driver_kind = "transporting";
  }
else if (driver_type == DTYPE_DIRECTOR)
  {
  if (addr->director != NULL)
    {
    driver_name = addr->director->name;
    driver_kind = " director";
    }
  else driver_kind = "directing";
  }
else if (driver_type == DTYPE_ROUTER)
  {
  if (addr->router != NULL)
    {
    driver_name = addr->router->name;
    driver_kind = " router";
    }
  else driver_kind = "routing";
  }

/* If we used a transport that has the "return_output" option set, and if it
did in fact generate some output, then we treat the message as failed if
it was not already set that way, so that the output gets returned to the
sender. Otherwise we just unlink the file, and remove the name so that if the
delivery failed anyway, we don't try to send back an empty file.

In any case, we close the message file, because we cannot afford to leave a
file-descriptor for one address while processing (maybe very many) others. */

if (addr->return_file >= 0)
  {
  struct stat statbuf;
  fsync(addr->return_file);
  if (fstat(addr->return_file, &statbuf) == 0 && statbuf.st_size > 0)
    {
    result = FAIL;
    if (addr->errno == 0 && addr->message == NULL)
      addr->message = "return message generated";
    }
  else
    {
    unlink(addr->return_filename);
    addr->return_filename = NULL;
    }
  close(addr->return_file);
  }

/* The sucess case happens only after delivery by a transport. */

if (result == OK)
  {
  address_item *topaddr, *dup;
  addr->next = addr_succeed;
  addr_succeed = addr;

  /* Don't deliver to this address again. For local addresses, the unique
  field contains the lower cased form of the local part. However, we do need
  also to have the original form of the address updated in the spool file so
  that listings of the spool mark the address as delivered. */

  DEBUG(9) debug_printf("%s succeeded: adding to nonrecipients list\n",
    addr->orig);

  tree_add_nonrecipient(addr->unique);
  if (addr->local && addr->parent == NULL) tree_add_nonrecipient(addr->orig);

  /* Check the list of duplicate addresses and ensure they are now marked
  done as well. Also their parents. */

  for (dup = addr_duplicate; dup != NULL; dup = dup->next)
    {
    if (strcmp(addr->unique, dup->unique) == 0)
      {
      tree_add_nonrecipient(dup->orig);
      (void)child_done(dup, now);
      }
    }

  /* Ensure the header file gets re-written */

  nonrecipients_changed = TRUE;
  
  /* Update the message log */

  if (addr->parent == NULL)
    fprintf(message_log, "%s %s: %s%s succeeded\n", now, addr->orig,
      driver_name, driver_kind);
  else
    fprintf(message_log, "%s %s <%s>: %s%s succeeded\n", now, addr->orig,
      addr->parent->orig, driver_name, driver_kind);
  fflush(message_log);

  /* Crawl back up the parents chain, decreasing the counts, and handling
  any that are complete. */

  topaddr = child_done(addr, now);

  /* Log the delivery. Local deliveries must have come via a director, but code
  carefully to avoid crashing if data is missing. Remote deliveries can come
  via a director or via a router, and should always have a pointer to
  the host item that succeeded. */

  if (addr->transport->info->local)
    log_write(LOG_MAIN, "%c> %s <%s> D=%s T=%s",
      logchar, 
      addr->local_part,
      topaddr->orig,
     (addr->director == NULL)? "" : addr->director->name,
      addr->transport->name);

  else
    {
    char buffer[256];
    BOOL equals_orig;
    BOOL source_routed =
      addr->local_part[0] == ',' || addr->local_part[0] == ':';

    if (source_routed)
      sprintf(buffer, "@%s%s", addr->domain, addr->local_part);
        else sprintf(buffer, "%s@%s", addr->local_part, addr->domain);
    equals_orig = strcmp(buffer, topaddr->orig);

    log_write(LOG_MAIN, "%c> %s%s%s%s%s%s%s%s T=%s H=%s [%s]%s",
      logchar, 
      buffer,
      equals_orig? " <" : "",
      equals_orig? topaddr->orig : "",
      equals_orig? ">" : "",
     (addr->director == NULL)? "" : " D=",
     (addr->director == NULL)? "" : addr->director->name,
     (addr->router == NULL)? ""   : " R=",
     (addr->router == NULL)? ""   : addr->router->name,
      addr->transport->name,
     (addr->transported == NULL)? "" : addr->transported->name,
     (addr->transported == NULL)? "" : addr->transported->address,
     (continue_transport == NULL)? "" : "*");
    }
  }


/* Soft failure, or local delivery process failed */

else if (result == DEFER || result == PANIC)
  {
  char *parent = (addr->parent == NULL)? NULL : addr->parent->orig;
  if (result == PANIC) logflags |= LOG_PANIC_DIE;
  addr->next = addr_defer;
  addr_defer = addr;
  
  /* Log the deferment in the message log */ 

  fprintf(message_log, "%s %s: %s%s deferred: %s%s%s\n",
    now,
    addr->orig,
    driver_name,
    driver_kind,
    (addr->errno <= 0)? "" : strerror(addr->errno),
    (addr->errno <= 0 || addr->message == NULL)? "" : ": ",
    (addr->message != NULL)? addr->message :
      (addr->errno <= 0)? "unknown error" : "");
  fflush(message_log);

  /* Either driver_name contains something and driver_kind contains
  " director" or whatever (note the leading space), or driver_name is
  a null string and driver_kind contains "directing" etc, without the
  leading space, if all directing or routing has been deferred. */

  log_write(logflags, "== %s%s%s%s %c%s%s defer (%d): %s%s%s",
     addr->orig,
     (parent == NULL)? "" : " <",
     (parent == NULL)? "" : parent,
     (parent == NULL)? "" : ">",
     (driver_name[0] == 0)? driver_kind[0] : toupper(driver_kind[1]),
     (driver_name[0] == 0)? driver_kind+1  : "=",
     (driver_name[0] == 0)? "" : driver_name,
     addr->errno,
     (addr->errno <= 0)? "" : strerror(addr->errno),
     (addr->errno <= 0 || addr->message == NULL)? "" : ": ",
     (addr->message == NULL)? "" : addr->message);

  /* The only currently implemented special action is to freeze the
  message. */

  if (addr->special_action == SPECIAL_FREEZE)
    {
    deliver_freeze = TRUE;
    deliver_frozen_at = time(NULL); 
    header_changed = TRUE;
    fprintf(message_log, "*** Frozen ***\n");
    fflush(message_log);
    log_write(LOG_MAIN, "Frozen");
    }
  }


/* Hard failure. If there is an address to which an error message can be sent,
put this address on the failed list. If not, put it on the deferred list and
freeze the mail message for human attention. The latter action can also be
explicitly requested by a router/director/transport. */

else
  {
  fprintf(message_log, "%s %s%s%s%s: %s%s failed: %s%s%s\n",
    now,
    addr->orig,
    (addr->parent == NULL)? "" : " <",
    (addr->parent == NULL)? "" : addr->parent->orig,
    (addr->parent == NULL)? "" : ">",
    driver_name,
    driver_kind,
    (addr->errno <= 0)? "" : strerror(addr->errno),
    (addr->errno <= 0)? "" : ": ",
    (addr->message == NULL)? "" : addr->message);
  fflush(message_log);

  log_write(LOG_MAIN, "** %s%s%s%s%s%s%s%s%s%s%s%s%s%s%s: %s%s%s",
     addr->orig,
    (addr->parent == NULL)? "" : " <",
    (addr->parent == NULL)? "" : addr->parent->orig,
    (addr->parent == NULL)? "" : ">",
    (addr->director == NULL)? ""  : " D=",
    (addr->director == NULL)? ""  : addr->director->name,
    (addr->router == NULL)? ""    : " R=",
    (addr->router == NULL)? ""    : addr->router->name,
    (addr->transport == NULL)? "" : " T=",
    (addr->transport == NULL)? "" : addr->transport->name,
    (addr->transported == NULL)? "" : " H=",
    (addr->transported == NULL)? "" : addr->transported->name,
    (addr->transported == NULL)? "" : " [",
    (addr->transported == NULL)? "" : addr->transported->address,
    (addr->transported == NULL)? "" : "]",
    (addr->errno <= 0)? ""        : strerror(addr->errno),
    (addr->errno <= 0)? ""        : ": ",
    (addr->message == NULL)? ""   : addr->message);

  /* If this is a delivery error, or a message for which not replies are 
  wanted, and ignore_errmsg_errors is set, force the ignore_error flag. 
  This will cause the address to be discarded later (with a log entry). */   

  if ((user_null_sender || sender_address[0] == 0) && 
      ignore_errmsg_errors) addr->ignore_error = TRUE;

  /* Freeze the message if requested, or if this is a delivery
  error message or a message for which no replies are ever wanted.
  Exim used to test  
   
    (sender_address[0] == 0 && addr->errors_address == NULL)
       
  but this can lead to meltdown when delivery to the errors_address
  is failing. Don't freeze if errors are being ignored - the actual
  code to ignore occurs later, instead of sending a message. */ 
  
  if (!addr->ignore_error &&
      (addr->special_action == SPECIAL_FREEZE ||
      user_null_sender || sender_address[0] == 0))
    {
    char *info = (addr->special_action == SPECIAL_FREEZE)? "" : 
      user_null_sender? 
        " (message created with -f <>)" : " (delivery error message)";
    deliver_freeze = TRUE;
    deliver_frozen_at = time(NULL); 
    header_changed = TRUE;
    addr->next = addr_defer;
    addr_defer = addr;
    fprintf(message_log, "*** Frozen%s\n", info);
    fflush(message_log);
    log_write(LOG_MAIN, "Frozen%s", info);
    }

  /* Don't put the address on the nonrecipients tree yet; wait until an
  error message has been successfully sent. */

  else
    {
    addr->next = addr_failed;
    addr_failed = addr;
    }
  }
}



/*************************************************
*       Post-process an address on its own       *
*************************************************/

/* This is called when we have finished with a single local address, or when we
have finished with one remote address during routing. After remote delivery,
alternate code is used, since a number of addresses are handled in one go. We
might call this after failure or deferral in directing or routing, but we hope
that it is usually after local transporting. The "last" flag is set if this is
known to be the very last delivery for this message. It makes it possible to
save a header file re-write in the case when the message is completed. */

static void post_process_single(address_item *addr, int result, int logflags,
  BOOL last, int driver_type, int logchar)
{
nonrecipients_changed = FALSE;
post_process_one(addr, result, logflags, driver_type, logchar);

/* If the tree of non-recipients has just been changed, or if any header
lines have been rewritten during the processing of this message, we re-write
the header file now so that it is correct for any subsequent re-deliveries.
However, we can skip this if this was the last delivery for the message and
there are no deferred addresses, because that means the message is now finished
with. */

if ((nonrecipients_changed || header_changed) &&
    (!last || addr_defer != NULL))
  {
  if (spool_write_header(message_id) <= 0)
    log_write(LOG_MAIN|LOG_PANIC_DIE, "unable to update spool header file "
      " for %s", message_id); 
  header_changed = FALSE;
  }
}




/*************************************************
*              Deliver one message               *
*************************************************/

/* This is the function which is called when a message is to be delivered. It
is passed the id of the message. It is possible that the message no longer
exists, if some other process has delivered it, and it is also possible that
the message is being worked on by another process, in which case the data file
will be locked.

During a queue run, if deliver_selectstring is set, skip messages that do not
have a recipient that matches the selection string.

If no delivery is attempted for any of the above reasons, the function returns
FALSE, otherwise TRUE.

If the give_up flag is set true, do not attempt any deliveries, but instead
fail all outstanding addresses and return the message to the sender (or
whoever).

A delivery operation has a process all to itself; we never deliver more than
one message in the same process. Therefore we needn't worry about store
leakage.

This function is called while running as root unless the security level is
1 or 3, in which case root can be regained by calling mac_sete{g,u}id. */

BOOL deliver_message(char *id, BOOL forced, BOOL give_up)
{
int i, rc, fd;
time_t now = time(NULL);
address_item *addr_last;
EXIM_DB *dbm_file;
char spoolname[256];

DEBUG(1) debug_printf("delivering message %s\n", id);
set_process_info("delivering %s", id);

/* Make the forcing flag available for directors/routers/transports, set up
the global message id field, and initialize the count for returned files. */

deliver_force = forced;
strcpy(message_id, id);
return_count = 0;

/* Open and lock the message's data file. Exim locks on this one because the
header file may get replaced as it is re-written during the delivery process.
Any failures cause messages to be written to the log. */

if (!spool_open_datafile(id))
  {
  /* If this exim run was passed an open channel by a transport, we can't
  go on to make use of it, so attempt to close it down tidily. */

  if (continue_transport != NULL)
    {
    transport_instance *t;
    for (t = transports; t != NULL; t = t->next)
      {
      if (strcmp(t->name, continue_transport) == 0)
        {
        if (t->info->closedown != NULL) (t->info->closedown)(t);
        break;
        }
      }
    }

  /* Do no more */

  return FALSE;
  }

/* Now read the contents of the header file, which will set up the headers in
store, and also the list of recipients and the tree of non-recipients and
assorted flags. If there is a reading or format error, give up; if the message
has been around for sufficiently long, remove it. */

sprintf(spoolname, "%s-H", id);
if ((rc = spool_read_header(spoolname, TRUE)) != spool_read_OK)
  {
  if (errno == ERRNO_SPOOLFORMAT)
    {
    struct stat statbuf;
    sprintf(big_buffer, "%s/input/%s", spool_directory, spoolname);
    if (stat(big_buffer, &statbuf) == 0)
      log_write(LOG_MAIN, "Format error in spool file %s: size=%d", 
        spoolname, statbuf.st_size);
    else log_write(LOG_MAIN, "Format error in spool file %s", spoolname);
    } 
  else     
    log_write(LOG_MAIN, "Error reading spool file %s: %s", spoolname, 
      strerror(errno));
  
  /* If we managed to read the envelope data, received_time contains the
  time the message was received. Otherwise, we can calculate it from the 
  message id. */   
  
  if (rc != spool_read_hdrerror)
    {
    received_time = 0; 
    for (i = 0; i < 6; i++) 
      received_time = received_time * 62 + tab62[id[i] - '0'];
    }  

  if (now - received_time > keep_malformed)
    {
    sprintf(spoolname, "%s/msglog/%s", spool_directory, id);
    unlink(spoolname);
    sprintf(spoolname, "%s/input/%s-D", spool_directory, id);
    unlink(spoolname);
    sprintf(spoolname, "%s/input/%s-H", spool_directory, id);
    unlink(spoolname);
    log_write(LOG_MAIN, "Message removed because older than %s",
      readconf_printtime(keep_malformed)); 
    }  
 
  close(deliver_datafile);
  deliver_datafile = -1; 
  return FALSE;
  }

/* A null recipients list indicates some kind of disaster. */

if (recipients_list == NULL)
  {
  log_write(LOG_MAIN, "Spool error: no recipients for %s", spoolname);
  close(deliver_datafile);
  deliver_datafile = -1; 
  return FALSE;
  }

/* If the message is frozen, do not attempt delivery, unless sufficient
time has passed since the last freezing. */

if (deliver_freeze)
  {
  if (auto_thaw > 0 && now > deliver_frozen_at + auto_thaw)
    log_write(LOG_MAIN, "Auto-thawed"); 
  else
    {  
    log_write(LOG_MAIN, "Message is frozen");
    close(deliver_datafile);
    deliver_datafile = -1; 
    return FALSE;
    } 
  }
  
 


/* Scan the recipients list, and for every one that is not in the non-
recipients tree, add an addr item to the chain of new addresses. Duplicates are
handled later by a different tree structure; we can't just extend the
non-recipients tree, because that will be re-written to the spool if the
message is deferred, and in any case there are casing complications for local
addresses. */

for (i = 0; i < recipients_count; i++)
  {
  if (tree_search_addr(tree_nonrecipients, recipients_list[i]) == NULL)
    {
    address_item *new = deliver_make_addr(recipients_list[i]);
    if (addr_new == NULL) addr_new = new; else addr_last->next = new;
    addr_last = new;
    }
  }

DEBUG(7)
  {
  address_item *p = addr_new;
  debug_printf("Delivery address list:\n");
  while (p != NULL)
    {
    debug_printf("  %s\n", p->orig);
    p = p->next;
    }
  }


/* If deliver_selectstring is set, ensure that at least one address matches the
selection string, but if so, allow all to be processed. If we were to skip the
other addresses, any header rewriting their routing might cause would not
happen. */

if (deliver_selectstring != NULL)
  {
  BOOL found = FALSE;
  address_item *p;
  for (p = addr_new; p != NULL; p = p->next)
    {
    if (strstr(p->orig, deliver_selectstring) != NULL)
      {
      found = TRUE;
      break;
      }
    }

  /* If no addresses matched the string, give up. */

  if (!found)
    {
    DEBUG(9) debug_printf("no addresses matched selectstring\n");
    close(deliver_datafile);
    deliver_datafile = -1; 
    return FALSE;
    }
  }


/* Set up the buffers used for copying over the file when delivering. */

deliver_in_buffer = store_malloc(DELIVER_BUFFER_SIZE);
deliver_out_buffer = store_malloc(DELIVER_BUFFER_SIZE);


/* Open the message log file. This records details of deliveries, deferments,
and failures for the benefit of the mail administrator. The log is not used by
exim itself to track the progress of a message; that is done by rewriting the
header spool file.

Exim is running as root here, unless seteuid() has been used to reduce
privilege while directing and routing. The message log can therefore end up
being owned by root. However, if an exim uid is defined, the msglog directory
will be owned by exim, and so when the time comes to delete the file, the
ownership doesn't matter. So don't waste effort making exim own it. However,
ensure that the mode is the same as other spool files. */

sprintf(spoolname, "%s/msglog/%s", spool_directory, id);
fd = open(spoolname, O_WRONLY|O_APPEND|O_CREAT, SPOOL_MODE);

if (fd < 0)
  {
  if(errno == ENOENT)
    {
    directory_make(spool_directory, "msglog", MSGLOG_DIRECTORY_MODE);
    fd = open(spoolname, O_WRONLY|O_APPEND|O_CREAT, SPOOL_MODE);
    }
  if (fd < 0)
    {
    log_write(LOG_MAIN|LOG_PANIC, "Couldn't open message log %s: %s", spoolname,
      strerror(errno));
    return FALSE;
    }
  }

/* If the file has been opened, make sure the file's group is the Exim gid
if exim_uid exists (can't have exim_uid set without exim_gid), and double-check
the mode because the group setting doesn't always get set automatically. */

if (fd >= 0 && exim_uid > 0)
  {
  fchown(fd, exim_uid, exim_gid);
  fchmod(fd, SPOOL_MODE);
  }

/* Now make a C stream out of it. */

message_log = fdopen(fd, "a");
if (message_log == NULL)
  {
  log_write(LOG_MAIN|LOG_PANIC, "Couldn't fdopen message log %s: %s", spoolname,
    strerror(errno));
  return FALSE;
  }


/* If asked to give up on this message, simply put all the addresses on
the failed chain, and set an appropriate error message. If the message
has a null sender and an address has no errors address, send it to the global
errors address. */

if (give_up)
  {
  address_item *addr;
  addr_failed = addr_new;
  addr_new = NULL;
  for (addr = addr_failed; addr != NULL; addr = addr->next)
    {
    addr->message = "delivery cancelled by administrator";
    if (sender_address[0] == 0 && addr->errors_address == NULL)
      addr->errors_address = errors_address;
    }
  }


/* Until there are no more new addresses, handle each one as follows:

 . If this is a generated address (indicated by the presence of a parent
   pointer) then check to see whether it is a pipe or a file, and if so,
   handle it directly here. The director that produced the address will have
   set the allow flags into the address, and also set the uid/gid required.
   Having the directors generate new addresses and then checking them here at
   the outer level is tidier than making each director do its checking, and
   means that directors don't need access to the failed address queue.

 . Determine if it is a local address; this may involve removing one
   or more leading "@<local-domain>" strings, and/or handing the percent
   hack on the local user name. A subroutines handles all this, setting the
   "local", "local_part", and "domain" fields in the address.

.  If it is a local address that was generated by another director, determine
   if any of its parents have the same local name. If so, generate a different
   string for previous delivery checking, and do the check. Without this
   code, if the address spqr generates spqr via a forward or alias file,
   delivery of the generated spqr stops further attempts at the top level spqr,
   which is not what is wanted.

 . Local addresses get put on the addr_direct chain, while remote addresses
   get put on the addr_route chain. However, before doing so, we check up
   on the retry database to see if a delay is set for directing or routing
   the address. If so, the address gets put directly onto the addr_defer
   chain. For directed addresses, while we have the retry database open, we
   check for the existence of a transport retry record, and save the next_try
   time if one is found. This saves a later database access for normal
   deliveries to local users.

 . Now we run the local addresses through the directors. A director may put
   the address on the addr_local chain for local delivery, or put it on the
   addr_failed chain if it is undeliveable, or it may generate child addresses
   and put them on the addr_new chain, or it may defer an address. All these
   things are passed as arguments so that the directors can be called for
   verification purposes as well.

 . Because directing may produce new addresses, we have to have an outer loop
   to do this all again. The reason for not doing the directing as soon as
   an address is determined to be local is that directing can in principle take
   some appreciable amount of time, and we do not want to have the retry
   database open any longer than is necessary, nor do we want to open and close
   it for each local address.

.  When all the directing is done, run the remote addresses through the
   routers. It may turn out that some apparently remote addresses are in fact
   local ones that have been abbreviated and so not recognized as local until
   the router expands them. Such addresses get put onto the addr_new chain
   and the whole thing recycles yet again. Wheels within wheels...
*/


while (addr_new != NULL)           /* Loop again after routing */
  {
  while (addr_new != NULL)         /* Loop again after directing */
    {
    address_item *addr;
    dbm_file = db_open("retry", O_RDONLY);

    /* Loop for current batch of new addresses */

    while (addr_new != NULL)
      {
      addr = addr_new;
      addr_new = addr->next;

      /* Handle generated address that is a pipe or a file or an autoreply. 
      If two different users specify delivery to the same pipe or file, there 
      should be two different deliveries, so build a unique string that
      incorporates the original address, and use this for duplicate testing
      and recording delivery. */

      if (addr->parent != NULL &&
         (addr->orig[0] == '|' || addr->orig[0] == '/' || addr->orig[0] == '>'))
        {
        char *destination;
        db_retry *retry_record; 
 
        addr->unique = string_sprintf("%s:%s", addr->orig, addr->parent->orig); 
 
        if (tree_search(tree_nonrecipients, addr->unique) != NULL)
          {
          DEBUG(9) debug_printf("%s was previously delivered: discarded\n",
            addr->orig);
          (void)child_done(addr, tod_stamp(tod_log));
          continue;
          }

        if (tree_search(tree_duplicates, addr->unique) != NULL)
          {
          DEBUG(9) debug_printf("%s is a duplicate address: discarded\n",
            addr->orig);
          addr->next = addr_duplicate;
          addr_duplicate = addr;
          continue;
          }

        tree_add_duplicate(addr->unique);

        /* Handle a pipe */

        if (addr->orig[0] == '|')
          {
          if (!addr->allow_pipe)
            {
            addr->errno = ERRNO_FORBIDPIPE;
            addr->message = string_sprintf("delivery to pipe forbidden: "
              "address <%s> generated: %s", addr->parent->orig, addr->orig);
            addr->next = addr_failed;
            addr_failed = addr;
            DEBUG(2) debug_printf("<%s>: parent <%s>: pipe forbidden\n",
              addr->orig, addr->parent->orig);
            }
          else
            {
            addr->local_part = addr->orig;
            addr->domain = addr->parent->domain;
            addr->transport = transport_address_pipe;
            if (addr->transport == NULL)
              {
              addr->errno = ERRNO_MISSINGPIPE;
              addr->message = string_sprintf("<%s>: parent <%s>: %s "
                "transport not configured", addr->orig, addr->parent->orig,
                address_pipe_transport);
              addr->next = addr_failed;
              addr_failed = addr;
              addr->errors_address = errors_address;
              }
            else
              {
              addr->next = addr_local;
              addr_local = addr;
              }
            }
          }

        /* Handle a file */

        else if (addr->orig[0] == '/')
          {
          if (!addr->allow_file)
            {
            addr->errno = ERRNO_FORBIDFILE;
            addr->message = string_sprintf("delivery to file forbidden: "
              "address <%s> generated: %s", addr->parent->orig, addr->orig);
            addr->next = addr_failed;
            addr_failed = addr;
            DEBUG(2) debug_printf("<%s>: parent <%s>: file forbidden\n",
              addr->orig, addr->parent->orig);
            }
          else
            {
            addr->local_part = addr->orig;
            addr->domain = addr->parent->domain;
            addr->transport = transport_address_file;
            if (addr->transport == NULL)
              {
              addr->errno = ERRNO_MISSINGFILE;
              addr->message = string_sprintf("<%s>: parent <%s>: %s "
                "transport not configured", addr->orig, addr->parent->orig,
                address_file_transport);
              addr->next = addr_failed;
              addr_failed = addr;
              addr->errors_address = errors_address;
              }
            else
              {
              addr->next = addr_local;
              addr_local = addr;
              }
            }
          }

        /* Handle an auto-reply */
         
        else
          {
          if (!addr->allow_reply)
            {
            addr->errno = ERRNO_FORBIDREPLY;
            addr->message = string_sprintf("autoreply forbidden: "
              "address <%s> generated: %s", addr->parent->orig, addr->orig);
            addr->next = addr_failed;
            addr_failed = addr;
            DEBUG(2) debug_printf("<%s>: parent <%s>: autoreply forbidden\n",
              addr->orig, addr->parent->orig);
            }
          else
            {
            addr->local_part = addr->orig;
            addr->domain = addr->parent->domain;
            addr->transport = transport_address_reply;
            if (addr->transport == NULL)
              {
              addr->errno = ERRNO_MISSINGREPLY;
              addr->message = string_sprintf("<%s>: parent <%s>: %s "
                "transport not configured", addr->orig, addr->parent->orig,
                address_reply_transport);
              addr->next = addr_failed;
              addr_failed = addr;
              addr->errors_address = errors_address;
              }
            else
              {
              addr->next = addr_local;
              addr_local = addr;
              }
            }
          }
          
        /* Files, pipes, and autoreplies may have transport retry information
        set for them. As the local delivery code expects the transport retry
        time to be set in addr->ltp_next_try (which happens for other local
        addresses as a byproduct of checking on the directing retry time),
        we must set this up. */

        destination = string_sprintf("T:%s@%s", addr->local_part,
          addr->domain);
        retry_record =
          (dbm_file == NULL)? NULL : db_read(dbm_file, destination);
        if (retry_record != NULL)
          {
          addr->ltp_next_try = retry_record->next_try;
          addr->ltp_expired = retry_record->expired;
          }

        continue;   /* with the next new address */
        }


      /* All addresses should have been made fully qualified when the message
      was accepted or when a director generated a new address, so panic if we
      find an unqualified one. */

      if (parse_find_at(addr->orig) == NULL)
        log_write(LOG_PANIC_DIE, "Unqualified address \"%s\" found in %s", 
          addr->orig, id);

      /* Determine locality - this sets the "local_part", "domain", and "local"
      fields, and lowercases the domain and the local part if local. */

      deliver_setlocal(addr);

      /* Now we can check for duplicates and previously delivered addresses.
      We need to check local addresses using the lower cased form of the local
      part. If the address is local, we must first check for a matching address
      in its parents, and if so, generate a different string to use for
      duplication checking. Stick a \ on the front in the simple case; it *is*
      possible for there to be more than one level; use \n\ for subsequent
      cases. */
          
      if (addr->local)
        {
        address_item *parent;
        for (parent = addr->parent; parent != NULL; parent = parent->parent)
          {
          if (strcmpic(addr->local_part, parent->local_part) == 0 &&
              strcmpic(addr->domain, parent->domain) == 0)
            break;
          }
        if (parent != NULL)
          {
          if (parent->unique[0] == '\\') 
            {
            addr->unique =
              string_sprintf("\\%c\\%s@%s", 
              (parent->unique[2] == '\\')? parent->unique[1] + 1 : '1', 
              addr->local_part, addr->domain);
            } 
          else addr->unique =
            string_sprintf("\\%s@%s", addr->local_part, addr->domain);
          }  
        else addr->unique =
          string_sprintf("%s@%s", addr->local_part, addr->domain);
        }
        
      /* Remote addresses may occasionally have parents, if the "unseen"
      option is in use. In these cases, an alternative unique name is
      required. */
      
      else if (addr->parent != NULL)
        {
        address_item *parent;
        for (parent = addr->parent; parent != NULL; parent = parent->parent)
          {
          if (strcmp(addr->local_part, parent->local_part) == 0 &&
              strcmp(addr->domain, parent->domain) == 0)
            break;
          }
        if (parent != NULL)
          {
          if (parent->unique[0] == '\\') 
            {
            addr->unique =
              string_sprintf("\\%c\\%s", 
              (parent->unique[2] == '\\')? parent->unique[1] + 1 : '1', 
              addr->unique);
            } 
          else addr->unique = string_sprintf("\\%s", addr->unique);
          }
        }     
         
      DEBUG(9) debug_printf("unique=%s\n", addr->unique);     
 
      /* Check for previous delivery. This can happen if configuration files
      change or if things got out of step. Ensure the counts in any parents
      are updated. */

      if (tree_search_addr(tree_nonrecipients, addr->unique) != NULL)
        {
        DEBUG(9) debug_printf("%s was previously delivered: discarded\n",
          addr->unique);
        (void)child_done(addr, tod_stamp(tod_log));
        continue;
        }

      /* Check for duplication. Remember duplicated addresses so they can
      be marked "delivered" when the duplicate is delivered. */

      if (tree_search_addr(tree_duplicates, addr->unique) != NULL)
        {
        DEBUG(9) debug_printf("%s is a duplicate address: discarded\n",
          addr->unique);
        addr->next = addr_duplicate;
        addr_duplicate = addr;
        continue;
        }

      /* Remember the first, to check for subsequent duplicates. */

      tree_add_duplicate(addr->unique);

      /* If the address is local, check on directing retry status, and add
      either to the directing chain or the defer chain. */

      if (addr->local)
        {
        char *destination = string_sprintf("D:%s@%s", addr->local_part,
          addr->domain);
        db_retry *retry_record =
          (dbm_file == NULL)? NULL : db_read(dbm_file, destination);

        /* Defer directing unless no retry data or we've passed the next
        retry time, or this message is forced. However, if the retry time
        has expired, try just one more time. If this fails, subsequent
        processing of the retry data should cause the address to fail.
        This ensures that local addresses are always directed at least
        once before being rejected. */

        if (retry_record != NULL && now < retry_record->next_try &&
            !deliver_force && !retry_record->expired)
          {
          addr->message = "retry time not reached";
          post_process_single(addr, DEFER, LOG_MAIN, FALSE, 
            DTYPE_DIRECTOR, '=');
          }

        /* Otherwise set up for directing. */

        else
          {
          addr->next = addr_direct;
          addr_direct = addr;

          /* Remember whether there is a retry record or not, so a request
          for its updating can be forced when directing occurs. */

          addr->dr_retry_exists = (retry_record != NULL);

          /* As an optimisation, save any transport next_try time for
          this destination. */

          if (dbm_file != NULL)
            {
            destination[0] = 'T';
            retry_record = db_read(dbm_file, destination);
            if (retry_record != NULL)
              {
              addr->ltp_next_try = retry_record->next_try;
              addr->ltp_expired = retry_record->expired;
              }
            }

          DEBUG(7) debug_printf("%s: queued for directing: "
            "transport next try = %d (%d)\n", addr->orig, addr->ltp_next_try,
            addr->ltp_expired);
          }
        }

      /* If the address is not local, check on routing retry status, and
      add either to the remote chain or the defer chain. */

      else
        {
        char *destination = string_sprintf("R:%s", addr->domain);
        db_retry *retry_record =
          (dbm_file == NULL)? NULL : db_read(dbm_file, destination);

        /* Defer routing unless no retry data or we've passed the next
        retry time, or this message is forced. However, if the retry time
        has expired, allow the routing attempt. If it fails again, the
        address will be failed. This ensures that each address is routed
        at least once, even after long-term routing failures. */

        if (retry_record != NULL && now < retry_record->next_try &&
            !deliver_force && !retry_record->expired)
          {
          addr->message = "retry time not reached";
          post_process_single(addr, DEFER, LOG_MAIN, FALSE, 
            DTYPE_ROUTER, '=');
          }

        /* Queue for routing, remembering whether there is a retry record or
        not, so a request for its updating can be forced when routing occurs. */

        else
          {
          addr->dr_retry_exists = (retry_record != NULL);
          addr->next = addr_route;
          addr_route = addr;
          DEBUG(7) debug_printf("%s: queued for routing\n", addr->orig);
          }
        }
      }

    /* The database is closed while directing is happening. */

    if (dbm_file != NULL) db_close(dbm_file);

    /* Run the local addresses that are not already deferred through the
    directors. If a director defers an address, or if there is an existing
    retry record in the database, add a retry item. Note that a director is
    permitted to generate a remote delivery if it is set up with a non-local
    transport. Ensure return_path is available so that it can be referred to
    in filter files or elsewhere. */

    while ((addr = addr_direct) != NULL)
      {
      int rc;
      addr_direct = addr->next;
      return_path = (addr->errors_address != NULL)?
        addr->errors_address : sender_address;
      if ((rc = direct_address(addr, &addr_local, &addr_remote, &addr_new,
        FALSE)) == DEFER || addr->dr_retry_exists)
          retry_add_item(addr, "D", TRUE, NULL, rc != DEFER);

      /* Handle addresses that are finished with. */

      if (rc != OK)
        post_process_single(addr, rc, LOG_MAIN, FALSE, DTYPE_DIRECTOR, '=');
      }
    }       /* Loop to handle any new addresses created by the directors */


  /* DEBUG: verify what's happened after all the directing */

  DEBUG(7)
    {
    address_item *p = addr_local;
    debug_printf("After directing:\n  Local addresses:\n");
    while (p != NULL)
      {
      debug_printf("    %s\n", p->orig);
      p = p->next;
      }

    p = addr_remote;
    debug_printf("  Remote addresses:\n");
    while (p != NULL)
      {
      debug_printf("    %s\n", p->orig);
      p = p->next;
      }

    p = addr_failed;
    debug_printf("  Failed addresses:\n");
    while (p != NULL)
      {
      debug_printf("    %s\n", p->orig);
      p = p->next;
      }

    p = addr_route;
    debug_printf("  Addresses to be routed:\n");
    while (p != NULL)
      {
      debug_printf("    %s\n", p->orig);
      p = p->next;
      }

    p = addr_defer;
    debug_printf("  Deferred addresses:\n");
    while (p != NULL)
      {
      debug_printf("    %s\n", p->orig);
      p = p->next;
      }
    }


  /* Now route those remote addresses that are not deferred. Routing may take
  some time (DNS timeouts) and I originally planned to get the local deliveries
  done before the routing, but since routing may cause addresses to get
  re-written, this is not the best plan. */

  dbm_file = NULL;
  while (addr_route != NULL)
    {
    int rc;
    address_item *addr = addr_route;
    char *old_domain = addr->domain;
    addr_route = addr->next;

    /* Just in case some router parameter refers to it. */
     
    return_path = (addr->errors_address != NULL)?
      addr->errors_address : sender_address;

    /* If a router defers an address, or if there is an existing retry record
    in the database, add a retry item. */

    if ((rc = route_address(addr, &addr_local, &addr_remote, &addr_new, FALSE))
      == DEFER || addr->dr_retry_exists)
        retry_add_item(addr, "R", FALSE, NULL, rc != DEFER);

    /* If an address turned out to be local after all, put it back on the
    addr_new chain for re-directing, and build a new original address. In fact,
    we need to do this by creating a new address with the old one as parent,
    so that the original address gets marked done when the child is done. */  

    if (rc == ISLOCAL)
      {
      address_item *old = addr;
      char *new_address; 
       
      if (addr->local_part[0] == ',' || addr->local_part[0] == ':')
        new_address = string_sprintf("@%s%s", addr->domain, addr->local_part);
      else
        new_address = string_sprintf("%s@%s", addr->local_part, addr->domain);
 
      addr = deliver_make_addr(new_address);
      addr->parent = old;
      addr->ignore_error |= old->ignore_error; 
      addr->errors_address = old->errors_address; 
      old->child_count++; 
       
      addr->next = addr_new;
      addr_new = addr;
      }

    /* Handle addresses that are finished with. */

    else if (rc != OK)
      post_process_single(addr, rc, LOG_MAIN, FALSE, DTYPE_ROUTER, '=');

    /* Successful routing: look to see if there are any more addresses waiting
    to be routed that have the same domain as this one started out with, and
    if so, copy the results of this routing for them and put them on the
    remote or local delivery queue as appropriate. However, we must only do
    this when the router permits it to happen - essentially it can be done
    only if the routing does not depend on the local part, and only the 
    router can know. */

    else if (addr->routed_by_domain)
      {
      address_item **chain = &addr_route;
      BOOL local = addr->transport->info->local; 

      while (*chain != NULL)
        {
        address_item *addr2 = *chain;

        if (strcmp(addr2->domain, old_domain) != 0)
          {
          chain = &(addr2->next);
          continue;
          }

        *chain = addr2->next;
        
        if (local) 
          {
          addr2->next = addr_local;
          addr_local = addr2;
          }
        else
          {        
          addr2->next = addr_remote;
          addr_remote = addr2;
          } 

        addr2->domain = addr->domain;
        addr2->route_domain = addr->route_domain;
        addr2->router = addr->router;
        addr2->transport = addr->transport;
        addr2->host_list = addr->host_list;

        DEBUG(7) debug_printf("Routing for %s copied from %s\n",
          addr2->orig, addr->orig);
        }
      }
    }
  }     /* Restart entire process if any remotes became local */


/* Free any resources that were cached during directing and routing. It's
debatable as to whether direct_tidyup() should be called immediately after
directing rather than waiting till routing is done. The point is that routing
can cause further directing if a "remote" domain turns out to be local.
However, as the resources are typically just open files (e.g. aliases) I don't
think it's critical. */

search_tidyup(&expand_tree);
search_tidyup(&stringmatch_tree);
direct_tidyup();
route_tidyup();

/* Problems with res_init() have sometimes caused overwriting to occur while 
routing. Do a double check that message_id has not been overwritten, as this is 
a serious disaster. */

if (strcmp(message_id, id) != 0)
  log_write(LOG_MAIN|LOG_PANIC_DIE, "Panic: message_id %s has been overwritten "
    "with %s", id, message_id);

/* Because address rewriting can happen in the routers or directors, we should
not really do ANY deliveries until all addresses have been routed or directed,
so that all recipients of the message get the same headers. However, this is in
practice not always possible, since sometimes remote addresses give DNS
timeouts for days on end. The pragmatic approach is to deliver what we can now,
saving any rewritten headers so that at least the next lot of recipients
benefit from the rewriting that has already been done. */


/* Do local deliveries first, unless this is a run to continue deliveries
to an external channel that is already set up. In that case, just defer
any local deliveries. */

DEBUG(2) debug_printf(">>> Local deliveries >>>\n");

if (continue_transport != NULL)
  {
  if (addr_defer == NULL) addr_defer = addr_local; else
    {
    address_item *addr = addr_defer;
    while (addr->next != NULL) addr = addr->next;
    addr->next = addr_local;
    }
  addr_local = NULL;
  }

/* True local deliveries are always done one at a time. However, local 
deliveries that are the result of routing rather than directing can be batched 
up in some cases. Typically this is when writing batched SMTP output files for 
use by some external transport mechanism. */

while (addr_local != NULL)
  {
  int result = DEFER;
  int logflags = LOG_MAIN;
  int logchar = '='; 
  transport_instance *tp;
  address_item *addr = addr_local;
  addr_local = addr->next;
  addr->next = NULL; 

  /* An internal disaster if there is no transport. */
   
  if ((tp = addr->transport) == NULL)
    {
    logflags |= LOG_PANIC;
    addr->message = "No transport set by director";
    }
  
  /* Otherwise we can try a delivery, but first check for routed addresses
  and possible amalgamation. We can't amalgamate if the transport depends on 
  the local-part in any way. */
    
  else
    {  
    if (addr->router != NULL && tp->local_smtp > local_smtp_one &&
        !readconf_depends((driver_instance *)tp, "local_part"))
      { 
      BOOL tp_sets_uid = tp->uid >= 0 || tp->expand_uid != NULL ||
        tp->deliver_as_creator;
      int local_smtp = tp->local_smtp; 
      address_item **anchor = &addr_local;
      address_item *last = addr;
      address_item *next;
        
      /* If the transport's options depend on the domain, turn the batch
      option from "all" into "domain". */
      
      if (local_smtp == local_smtp_all && 
        readconf_depends((driver_instance *)tp, "domain")) 
          local_smtp = local_smtp_domain;   
        
      /* Pick off all routed addresses which have the same transport and 
      errors address and (first) host. If the transport has no uid/gid 
      setting, they must also have the same uid/gid. If the local_smtp option 
      is "domain" they must also have the same domain. */ 
      
      while ((next = *anchor) != NULL)
        {
        if (next->router != NULL &&
            next->transport == addr->transport &&
            next->errors_address == addr->errors_address &&
            (tp_sets_uid || 
              (next->uid == addr->uid && next->gid == addr->gid)) &&
            (local_smtp == local_smtp_all ||
              strcmp(next->domain, addr->domain) == 0) &&
            ((addr->host_list == NULL && next->host_list == NULL) ||
             (addr->host_list != NULL && next->host_list != NULL &&
              strcmp(addr->host_list->name, next->host_list->name) == 0))) 
          {
          *anchor = next->next;
          next->next = NULL;
          last->next = next;
          last = next;
          }
        else anchor = &(next->next);
        }   
      }  
 
    /* We are set to try a delivery, but first check to see if there is
    a retry time that we need to wait for. It was saved when checking
    for directing deferment. If the retry time has expired, try for
    delivery one more time. If this causes deferment, the message will
    get failed by the retry checking code later on. This means that a 
    local delivery will always be tried at least once, even when the
    recipient has been failing for a very long time. This is reasonable
    because trying a local delivery is relatively cheap. Contrast remote
    deliveries, where there options controlling what happens after a long
    outage.       
    
    If there was a next_try time, or if delivery was deferred, build a retry 
    item for setting a new retry time or deleting the old retry record from 
    the database. These items are handled all together after all addresses 
    have been handled (so the database is open just for a short time). */
    
    if (now >= addr->ltp_next_try || addr->ltp_expired || deliver_force)
      { 
      return_path = (addr->errors_address != NULL)?
        addr->errors_address : sender_address;
      deliver_local(addr, id);
      result = addr->transport_return;
      if (result == DEFER || addr->ltp_next_try > 0)
        retry_add_item(addr, "T", TRUE, NULL, result != DEFER);
      }
    
    /* Defer if retry time not reached (default setting is result = DEFER). */
    
    else addr->message = "Retry time not yet reached"; 
    }   

  /* Do what has to be done immediately after a delivery. Flag this address
  as "last" if it is the last local delivery, and there are no addresses
  queued for routing or remote delivery. Note that when handling several
  batched addresses we mustn't rely on the next field afterwards, as the
  post_processing function may put the address onto a chain. */

  while (addr != NULL)
    {
    address_item *next = addr->next;
    post_process_single(addr, result, logflags,
      (addr_local == NULL && addr_route == NULL && addr_remote == NULL),
        DTYPE_TRANSPORT, logchar);
    addr = next;     
    logchar = '-'; 
    }     
  }


/* There may have been expansion lookups and string matches during local 
deliveries. Free any cached resources so as not to hold them during remote 
deliveries. */  

search_tidyup(&expand_tree);
search_tidyup(&stringmatch_tree);


/* Once the local deliveries are done, we do not need to be root any longer,
so if a non-root uid has been specified, give up privileges for good at this
point, whatever the security level, using seteuid to recover root privilege
if it has been temporarily given up. */

if (exim_uid > 0)
  {
  if (geteuid() != root_uid) mac_seteuid(root_uid);
  setgid(exim_gid);
  setuid(exim_uid);
  }


/* The function that read the headers counted the number of Received: headers.
If there are too many, we must not make any remote deliveries. */

if (received_count > received_headers_max)
  {
  log_write(LOG_MAIN, "** Too many \"Received\" headers for remote delivery");
  DEBUG(2) debug_printf("Too many \"Received\" headers for remote delivery\n");
  while (addr_remote != NULL)
    {
    address_item *addr = addr_remote;
    addr_remote = addr->next;
    addr->message = "Too many \"Received\" headers for remote delivery";
    addr->next = addr_failed;
    addr_failed = addr;
    }
  }



/* Do remote deliveries. We must pick off the queue all addresses that have the
same transport, remote destination, and errors_address, and hand them to the
transport in one go. If this is a run to continue delivering to an existing
delivery channel, skip all but those addresses that can go to that channel. The
skipped addresses just get defered. */

DEBUG(2) debug_printf(">>> Remote deliveries >>>\n");

while (addr_remote != NULL)
  {
  int logflags = LOG_MAIN;
  int logchar = '='; 
  host_item *h;
  address_item *addr = addr_remote;
  addr_remote = addr->next;
  addr->next = NULL;

  if (addr->transport == NULL)
    {
    logflags |= LOG_PANIC;
    addr->message = "No transport set by router or director";
    addr->transport_return = DEFER;
    }

  else
    {
    BOOL multi_domain = addr->transport->multi_domain; 
    address_item **anchor = &addr_remote;
    address_item *last = addr;
    address_item *next;

    /* Pick off all addresses which have the same transport, errors address,
    and destination. In some cases they point to the same host list, but we
    also need to check for identical host lists generated from entirely
    different domains. The host list pointers can be NULL in the case where the
    hosts are defined in the transport. If all addresses have the same domain,
    we can set the $domain expansion variable - configurations can arrange this
    by using the "domain" option, and then being able to look up things by
    domain can be useful. In fact, there is now a flag which can be set on
    a transport that restricts it to handling one domain at a time. */

    deliver_domain = addr->domain;
    while ((next = *anchor) != NULL)
      {
      if ((multi_domain || strcmp(next->domain, addr->domain) == 0) &&
          next->transport == addr->transport &&
          next->errors_address == addr->errors_address &&
          same_hosts(next->host_list, addr->host_list))
        {
        *anchor = next->next;
        next->next = NULL;
        last->next = next;
        last = next;
        if (multi_domain && deliver_domain != NULL && 
          strcmp(next->domain, deliver_domain) != 0)
            deliver_domain = NULL; 
        }
      else anchor = &(next->next);
      }

    /* If this is a run to continue delivery down an already-established
    channel, check that this set of addresses matches the transport and
    the channel. If it does not, defer the addresses. If a host list exists,
    we must check that the continue host is on the list. Otherwise, the
    host is set in the transport, and must therefore of necessity be the
    same for all addresses. */

    if (continue_transport != NULL)
      {
      BOOL ok = strcmp(continue_transport, addr->transport->name) == 0;
       
      if (ok && addr->host_list != NULL)
        {
        ok = FALSE;
        for (next = addr; !ok && next != NULL; next = next->next)
          {
          host_item *h;
          for (h = next->host_list; h != NULL; h = h->next)
            {
            if (strcmp(h->name, continue_hostname) == 0)
              { ok = TRUE; break; }
            }
          }
        }

      /* Addresses not suitable; defer and skip to next address. */

      if (!ok)
        {
        next = addr;
        while (next->next != NULL) next = next->next;
        next->next = addr_defer;
        addr_defer = addr;
        continue;
        }
      }

    /* Compute the return path, ensure that the transport is indeed a
    remote one, and then hand it the chain of addresses. */

    return_path = (addr->errors_address != NULL)?
      addr->errors_address : sender_address;

    if (addr->transport->info->local)
      log_write(LOG_PANIC_DIE, "Attempt non-local delivery with local "
        "transport for %s: transport %s", addr->orig, addr->transport->name);

    /* The transports set up the process info themselves as they may connect
    to more than one remote machine. */

    #ifdef TRANSPORT_DEBUG
    if (debug_transport != NULL)
      (debug_transport->code)(addr->transport, addr);
    else
    #endif

    (addr->transport->info->code)(addr->transport, addr);
    }

  /* If any host addresses were found to be unusable, scan the addresses still
  awaiting delivery, and if any have the same hosts, ensure they are marked
  unusable to save trying them again in this delivery. */

  for (h = addr->host_list; h != NULL; h = h->next)
    {
    if (h->status >= hstatus_unusable)
      {
      address_item *addr2;
      for (addr2 = addr_remote; addr2 != NULL; addr2 = addr2->next)
        {
        host_item *hh;
        if (addr2->host_list == NULL ||
            addr2->host_list == addr->host_list) continue;
        for (hh = addr2->host_list; hh != NULL; hh = hh->next)
          if (strcmp(hh->name, h->name) == 0) 
            { hh->status = h->status; hh->why = h->why; }
        }
      }
    }

  /* Do what has to be done immediately after a delivery for each set of
  addresses, then re-write the spool if necessary. Note that post_process_one
  puts the address on an appropriate queue; hence we must fish off the next
  one first. */

  nonrecipients_changed = FALSE;
  while (addr != NULL)
    {
    address_item *next = addr->next;
    post_process_one(addr, addr->transport_return, logflags, DTYPE_TRANSPORT,
      logchar);
    logchar = '-';   
    addr = next;
    }

  if ((nonrecipients_changed || header_changed) &&
      (addr_remote != NULL || addr_defer != NULL))
    {
    if (spool_write_header(message_id) <= 0)
      log_write(LOG_MAIN|LOG_PANIC_DIE, "unable to update spool header file "
        " for %s", message_id); 
    header_changed = FALSE;
    }
  }

DEBUG(9) debug_printf("tidying up after delivering %s\n", message_id);
set_process_info("tidying up after delivering %s", message_id);

/* Finished with the message log. */

fclose(message_log);

/* Next we must update the retry database. We do this in one fell swoop at the
end in order not to keep opening and closing (and locking) the database. Other
things (such as writing the msglog, updating the header file) must be done as
soon as possible after an individual delivery, and so they are handled in the
post_process_one() function above. The code for handling retries is hived off
into a separate module for convenience. We pass it the addresses of the various
chains, because deferred addresses can get moved onto the failed chain if the
retry cutoff time has expired for all alternative destinations. */

retry_update(&addr_defer, &addr_failed, &addr_succeed);

/* If there are any deferred addresses, see if this message has been on the
queue for longer than the maximum retry timeout for any retry rule. This can
happen if there are lots of messages for one domain, one or two of which
occasionally get through, resetting the retry clock. Though individual 
addresses get failed if they get tried and the message has been around longer
than their timeout, if there are sufficiently many messsages on the queue, some 
may not get tried for a very long time, as they get skipped after a young 
message has been failed. As an ultimate backstop, fail any message that has 
been on the queue longer than the longest retry time of any retry rule. */

if (now - received_time > retry_maximum_timeout)
  {
  while (addr_defer != NULL)
    {
    address_item *addr = addr_defer;
    addr_defer = addr->next;
    addr->next = addr_failed;
    addr_failed = addr;
    addr->retry_timedout = TRUE;
    addr->message = (addr->message == NULL)? "message retry timeout exceeded" :
      string_sprintf("%s: message retry timeout exceeded", addr->message);
    log_write(LOG_MAIN, "** %s: message retry timeout exceeded", addr->orig);
    }
  } 

/* If any addresses failed, we must send a message to somebody, unless 
ignore_error is set, in which case no action is taken. Simple delivery
failures go back to the sender; other cases may go to mailmaster or to an
address owner. It is possible for several messages to get sent if there are
addresses with different requirements. */

while (addr_failed != NULL)
  {
  int pid, fd;
  header_line *headers = header_list;
  address_item *addr;
  address_item *handled_addr = NULL;
  address_item **paddr;
  address_item *msgchain = NULL;
  address_item **pmsgchain = &msgchain;
  
  char *recipient = (addr_failed->errors_address == NULL)?
    ((sender_address[0] == 0)? errors_address : sender_address) :
    addr_failed->errors_address;

  /* If this is an error delivery message for which there is no other
  errors address, ignore it. This case happens when a delivery error
  message times out after (auto) thawing, since the normal handling of
  failures for such messages is to defer and freeze them. */ 
  
  if (sender_address[0] == 0 && recipient == errors_address)
    addr_failed->ignore_error = TRUE;  

  /* If the first address on the list has ignore_error set, just remove
  it from the list, throw away any saved message file, log it, and
  mark the recipient done. */ 
  
  if (addr_failed->ignore_error)
    {
    addr = addr_failed;
    addr_failed = addr->next;
    if (addr->return_filename != NULL) unlink(addr->return_filename);
    log_write(LOG_MAIN, "%s: error ignored", addr->orig);
    tree_add_nonrecipient(addr->unique); 
    if (spool_write_header(message_id) <= 0)
      log_write(LOG_MAIN|LOG_PANIC_DIE, "unable to update spool header "
        "file for %s", message_id); 
    }
    
  /* Otherwise, handle the sending of a message. Find the error address for 
  the first address, then send a message that includes all failed addresses 
  that have the same error address. */
  
  else
    {  
  
    /* Make a subprocess to send a message */
  
    pid = child_open(mailer_argv, NULL, 077, -1, -1, &fd,
      (debug_file != NULL)? fileno(debug_file) : -1);
  
    /* Creation of child failed */
  
    if (pid < 0)
      log_write(LOG_MAIN|LOG_PANIC_DIE, "Failed to create child process to send "
        "failure message");
  
    /* Creation of child succeeded */
  
    else
      {
      int ch, rc;
      int filecount = 0;
      char *bcc; 
      FILE *f = fdopen(fd, "w");
      if (errors_reply_to != NULL)
        fprintf(f, "Reply-to: %s\n", errors_reply_to);
      fprintf(f, "From: Mail Delivery System <Mailer-Daemon@%s>\n", 
        qualify_domain_sender);
      fprintf(f, "To: %s\n", recipient);
      
      /* Quietly copy to configured additional addresses if required. */
       
      bcc = moan_check_errorcopy(recipient);
      if (bcc != NULL) 
        {
        fprintf(f, "Bcc: %s\n", bcc);  
        store_free(bcc);
        }  
  
      if (strcmpic(sender_address, recipient) == 0)
        fprintf(f, "Subject: Mail delivery failed: returning message to sender\n\n"
"This message was created automatically by mail delivery software.\n\n"
"A message that you sent could not be delivered to all of its recipients. The\n"
"following address(es) failed:\n\n");
      else
        fprintf(f, "Subject: Mail delivery failed\n\n"
"This message was created automatically by mail delivery software.\n\n"
"A message sent by\n\n  <%s>\n\n"
"could not be delivered to all of its recipients. The following address(es)\n"
"failed:\n\n", sender_address);
  
      /* Scan the addresses for all that have the same errors_address, removing
      them from the addr_failed chain, but putting them on the msgchain if
      they have a file name for a return message. (There has already been a 
      check in post_process_one() for the existence of data in the message 
      file.) */
  
      paddr = &addr_failed;
      for (addr = addr_failed; addr != NULL; addr = *paddr)
        {
        if (strcmp(recipient, (addr->errors_address == NULL)?
            ((sender_address[0] == 0)? errors_address : sender_address) :
              addr->errors_address) != 0 || addr->ignore_error)
          {
          paddr = &(addr->next);
          continue;
          }
        *paddr = addr->next;
  
        /* Output this address and its error message. The message field may
        be quite long in some cases. To cope with this we split and indent
        it at any colon characters. */
  
        fprintf(f, "  %s:\n    ", string_printing(
          ((addr->orig[0] == '|' || addr->orig[0] == '/') && 
            addr->parent != NULL)?
              addr->parent->orig : addr->orig, FALSE));
  
        if (addr->errno > 0)
          fprintf(f, "%s:\n    ", strerror(addr->errno));
  
        if (addr->message == NULL)
          { if (addr->errno <= 0) fprintf(f, "unknown error\n"); }
        else
          {
          char *s = addr->message;
          while (*s != 0)
            {
            fputc(*s, f);
            if (*s++ == ':' && isspace(*s)) fprintf(f, "\n   ");
            }
          fputc('\n', f);
          }
  
        /* Add to msgchain if there's a return file. */
  
        if (addr->return_filename != NULL)
          {
          *pmsgchain = addr;
          addr->next = NULL;
          pmsgchain = &(addr->next);
          filecount++; 
          }
  
        /* Else save so that we can tick off the recipient when the
        message is sent. */
  
        else
          {
          addr->next = handled_addr;
          handled_addr = addr;
          }
        }
  
      /* If there were any file messages passed by the local transports, include
      them in the message. Then put the address on the handled chain. */
  
      if (msgchain != NULL)
        {
        address_item *nextaddr;
        fprintf(f,
          "\nThe following text was generated during the delivery attempt%s:\n",
          (filecount > 1)? "s" : "");
        for (addr = msgchain; addr != NULL; addr = nextaddr)
          {
          FILE *fm = fopen(addr->return_filename, "r");
          int save_errno = errno; 
          BOOL nodomain = 
            (addr->local_part[0] == '|' || addr->local_part[0] == '/') &&
            addr->parent != NULL; 
          fprintf(f, "\n------ %s%s%s ------\n\n",
            string_printing(addr->local_part, FALSE),
           nodomain? "" : "@",
           nodomain? "" : addr->domain);
          if (fm == NULL)
            fprintf(f, "    +++ System error... failed to open text file: %s\n",
              strerror(save_errno));  
          else
            {      
            while ((ch = fgetc(fm)) != EOF) fputc(ch, f);
            fclose(fm);
            }
          unlink(addr->return_filename);
  
          /* Can now add to handled chain, first fishing off the next
          address on the msgchain. */
  
          nextaddr = addr->next;
          addr->next = handled_addr;
          handled_addr = addr;
          }
        }
  
      /* Now copy the message, trying to give an intelligible comment if
      it is too long for it all to be copied. The limit isn't strictly
      applied because of the buffering. */
  
      if (return_size_limit > 0)
        {
        struct stat statbuf;
        int max = (return_size_limit/DELIVER_BUFFER_SIZE + 1)*DELIVER_BUFFER_SIZE;
        if (fstat(deliver_datafile, &statbuf) == 0 && statbuf.st_size > max)
          { 
          fprintf(f, "\n"
"------ This is a copy of your message, including all the headers.\n"
"------ The body of your message is %d characters long; only the first\n"
"------ %d or so are included here.\n\n", statbuf.st_size, (max/1000)*1000);
          }
        else fprintf(f, "\n"
"------ This is a copy of your message, including all the headers. ------\n\n");
        }
      else fprintf(f, "\n"
"------ This is a copy of your message, including all the headers. ------\n\n");
        
      fflush(f);
      transport_write_message(fileno(f), FALSE, FALSE, FALSE, FALSE, FALSE, NULL,
        return_size_limit);
  
      /* Close the file, which should send an EOF to the child process
      that is receiving the message. Wait for it to finish. */
  
      fclose(f);
      rc = child_close(pid);     /* Waits for child to close */

      /* If the process failed, there was some disaster in setting up the
      error message. Ensure that addr_defer is non-null, which will have
      the effect of leaving the message on the spool. The failed addresses
      will get tried again next time. */
  
      if (rc != 0)
        {
        log_write(LOG_MAIN, "Process failed (%d) when writing error message "
          "to %s", rc, recipient);
        if (addr_defer == NULL) addr_defer = (address_item *)(+1);
        }
  
      /* The message succeeded. Ensure that the recipients that failed are
      now marked finished with on the spool. */
  
      else
        {
        log_write(LOG_MAIN, "Error message sent to %s", recipient);
        for (addr = handled_addr; addr != NULL; addr = addr->next)
          tree_add_nonrecipient(addr->unique);
        if (spool_write_header(message_id) <= 0)
          log_write(LOG_MAIN|LOG_PANIC_DIE, "unable to update spool header "
            "file for %s", message_id); 
        } 
      }
    }
  }

/* If delivery was frozen and the mailmaster wants to be told, generate
an appropriate message, unless the message is a local error message - to
prevent loops - or any message that is addressed to the local mailmaster. */

if (deliver_freeze && freeze_tell_mailmaster && !local_error_message)
  {
  BOOL moan = TRUE;
  int i;
  for (i = 0; i < recipients_count; i++)
    {
    if (strcmp(recipients_list[i], errors_address) == 0)
      {
      moan = FALSE;
      break;
      }
    }

  if (moan) moan_tell_mailmaster("Message frozen",
    "Message %s has been frozen.\n", message_id);

  /***** This specific code can't be used in the single-message case. Long-
  term aim: provide more info in the message.

  if (moan) moan_tell_mailmaster("Message frozen",
    "Message %s has been frozen after a delivery attempt for <%s>.\n"
    "The error code was %d: %s%s%s\n",
     message_id,
     addr->orig,
     addr->errno,
    (addr->errno <= 0)? "" : strerror(addr->errno),
    (addr->errno <= 0)? "" : ": ",
    (addr->message == NULL)? "" : addr->message);
  *****/
  }


/* If there are now no deferred addresses, we are done. Preserve the
message log if so configured. */

if (addr_defer == NULL)
  {
  int rc; 
  sprintf(spoolname, "%s/msglog/%s", spool_directory, id);
  if (preserve_message_logs)
    {
    sprintf(big_buffer, "%s/msglog.OLD/%s", spool_directory, id);
    if ((rc = rename(spoolname, big_buffer)) < 0)
      {  
      directory_make(spool_directory, "msglog.OLD", MSGLOG_DIRECTORY_MODE);
      rc = rename(spoolname, big_buffer); 
      } 
    if (rc < 0)
      log_write(LOG_PANIC_DIE, "failed to move %s to the msglog.OLD "
        "directory", spoolname); 
    }
  else
    {      
    if (unlink(spoolname) < 0)
      log_write(LOG_PANIC_DIE, "failed to unlink %s", spoolname);
    }   

  /* Remove the two message files. */
   
  sprintf(spoolname, "%s/input/%s-D", spool_directory, id);
  if (unlink(spoolname) < 0)
    log_write(LOG_PANIC_DIE, "failed to unlink %s", spoolname);
  sprintf(spoolname, "%s/input/%s-H", spool_directory, id);
  if (unlink(spoolname) < 0)
    log_write(LOG_PANIC_DIE, "failed to unlink %s", spoolname);
  log_write(LOG_MAIN, "Completed");
  }

/* Otherwise, lose any temporary files that were catching output from pipes for 
any of the deferred addresses, and see if the message has been on the queue for
so long that it is time to send a warning message to the sender, unless it is a
mailer-daemon. If we can't make a process to send the message, don't worry. */

else 
  {
  address_item *addr;
  for (addr = addr_defer; addr != NULL; addr = addr->next)
    { 
    if (addr->return_filename != NULL) unlink(addr->return_filename);
    }   
  
  /* Now the warning stuff */
 
  if (delay_warning > 0 && sender_address[0] != 0)
    {
    int queue_time = time(NULL) - received_time;
    int count = queue_time/delay_warning;
  
    if (warning_count < count)
      {
      header_line *h;
      int fd;
      int pid = child_open(mailer_argv, NULL, 077, -1, -1, &fd,
        (debug_file != NULL)? fileno(debug_file) : -1);
  
      if (pid > 0)
        {
        FILE *f = fdopen(fd, "w");
        if (errors_reply_to != NULL)
          fprintf(f, "Reply-to: %s\n", errors_reply_to);
        fprintf(f, "From: Mail Delivery System <Mailer-Daemon@%s>\n",
          qualify_domain_sender);
        fprintf(f, "To: %s\n", sender_address);
        fprintf(f, "Subject: Warning: message %s delayed\n\n", message_id);
        fprintf(f,
"This message was created automatically by mail delivery software.\n\n"
"A message that you sent has not yet been delivered to all its recipients\n"
"after more than ");

        queue_time = count * delay_warning;
        if (queue_time < 120*60) fprintf(f, "%d minutes", queue_time/60);
          else fprintf(f, "%d hours", queue_time/3600);
        fprintf(f, " on the queue on %s.\n\n", primary_hostname);
  
        fprintf(f, "The message identifier is:     %s\n", message_id);
  
        for (h = header_list; h != NULL; h = h->next)
          {
          if (strncmpic(h->text, "Subject:", 8) == 0)
            fprintf(f, "The subject of the message is: %s", h->text + 9);
          else if (strncmpic(h->text, "Date:", 5) == 0)
            fprintf(f, "The date of the message is:    %s", h->text + 6);
          }
        fprintf(f, "\n");
  
        fprintf(f, "The address%s to which the message has not yet been "
          "delivered %s:\n\n",
          (addr_defer->next == NULL)? "" : "es",
          (addr_defer->next == NULL)? "is": "are");
  
        while (addr_defer != NULL)
          {
          address_item *addr = addr_defer;
          addr_defer = addr->next;
          fprintf(f, "  %s\n", string_printing(addr->orig, FALSE));
          }
  
        fprintf(f,
"\nNo action is required on your part. Delivery attempts will continue for\n"
"some time, and this message will be repeated at intervals if the message\n"
"remains undelivered. Eventually the mail delivery software will give up,\n"
"and when that happens, the message will be returned to you.\n");

        /* Close and wait for child process to complete. If there's an
        error, don't update the count. Failure to update the header file
        isn't as serious here as it is when recording a delivery. */
  
        fclose(f);
        if (child_close(pid) == 0)
          {
          warning_count = count;
          if (spool_write_header(message_id) <= 0)
            log_write(LOG_MAIN, "failed to write header file for %s",
              message_id);  
          }
        }
      }
    }   
  }


/* Closing the data file frees the lock; if the file has been unlinked it
will go away. Otherwise the message becomes available for another process
to try delivery. */

close(deliver_datafile);
deliver_datafile = -1; 
DEBUG(2) debug_printf("end delivery of %s\n", id);
return TRUE;
}

/* End of deliver.c */
