(herald ffix (syntax-table (env-syntax-table orbit-env)))

(define (create-variable name)
  (let ((var (obtain-from-pool variable-pool)))
    (set (variable-name       var) name)
    (set (variable-id         var) *variable-id*)
    (set (variable-binder     var) nil)
    (set (variable-definition var) nil)
    (set (variable-refs       var) '())
    (set (variable-type       var) type/top)
    (set (variable-rep        var) 'rep/pointer)       
    (set (variable-flag       var) nil)
    (set (variable-flags      var) '())
    (set *variable-id* (fx+ 1 *variable-id*))
    var))

(define (expand-object-form form)
  (destructure (((proc . clauses) form))
    (let ((op (generate-symbol 'op)))
      `(*object ,proc (lambda (,op)
                        (select ,op
                          ,@(map construct-method clauses)
                          (else nil)))))))

(define (construct-method clause)
  (cond ((pair? (car clause))     
         (destructure ((((op state . vars) . body) clause))
           (if (atom? state)        ; old form
               `((,op) (lambda (,state #f ,@vars)
                           (ignorable ,state)
                           ,@body))
               (destructure (((self obj) state))
                 `((,op) (lambda (,self ,obj ,@vars)
                             ,@body))))))
        (else
         `((,(car clause)) ,@(cdr clause)))))
           


(define (primop-argument-type node)
  (let* ((proc (call-proc (node-parent node)))
         (type (primop.type (primop-value proc) (node-parent node))))
    (if (and type (proc-type? type))
        (vref (proc-type-args type) (relation-index (node-role node)))
        type/top)))

(define (primop-result-type var)
  (let ((type (primop-argument-type (variable-binder var))))
    (cond ((and type (proc-type? type))
           (vref (proc-type-args type)
                 (fx- (variable-number var) 1)))
          (else type/top))))



(lset *n-ary->binary-arg-limit* '2)   ; Limit code explosion

(define (n-ary->binary call proc)
  (let ((args (cdr (call-args call)))
        (var (get-system-variable proc)))
    (cond ((or (null? args) (null? (cdr args)))
           (bug "not enough arguments in ~S for N-ARY->BINARY" call))
          ((null? (cddr args))
           (replace (call-proc call) (create-reference-node var))
           '#t)
          ((fx< *n-ary->binary-arg-limit* (length args))
           '#f)
          (else
           (let ((top (node-parent call))
                 (cont (detach ((call-arg 1) call))))
             (iterate loop ((args args) (cont cont))
               (cond ((null? (cddr args))
                      (let-nodes ((c1 ((* var) 1 cont
                                       (! (detach (car args)))
                                       (! (detach (cadr args))))))
                        (replace call c1))
                      '#t)
                     (else
                      (let-nodes ((l1 (#f v) ((* var) 1 cont 
                                              (! (detach (car args)))
                                              (* v))))
                        (loop (cdr args) l1))))))))))

(define (presimplify-to-funny-conditional node count)
  (destructure (((pred cont arg1 arg2) (call-proc+args node)))
    (construct-funny-conditional node pred cont arg1 arg2 count)))

(define (construct-funny-conditional node pred cont arg1 arg2 count)
  (let ((primop (if (primop-node? pred)
                    pred
                    (create-primop-node (known-primop pred)))))
    (walk detach (call-proc+args node))
    (if (reference-node? pred) (erase pred))
    (let-nodes ((c1 ((^ l1) 1 cont))
                 (l1 (#f v1)
                     (($ primop/conditional) 2
                      (! (construct-conditional-cont count v1 '#t))
                      (! (construct-conditional-cont count v1 '#f))
                      primop arg1 arg2)))
      (replace node c1))))

(define (construct-conditional-cont arg-count var value)
  (let* ((vars (make-vars arg-count))
         (args (map create-reference-node vars)))
    (let-nodes ((l1 (#f . vars) ((* var) 0 'value . args)))
      l1)))

(define (make-vars count)
  (do ((i 0 (fx+ i 1))
       (v '() (cons (create-variable 'v) v)))
      ((fx>= i count) v)))

(define (orbit-m68-init . directory)
  (orbit-m68-setup (if directory (car directory) '#f))
  (orbit-init 'base
              'constants
              'primops
              'arith
              'locations
              'low
              'predicates
              'open
              'aliases
              'carcdr
	      'genarith))

(define (orbit-setup directory)
  (set (table-entry *modules* 'base)       `(,directory base))
  (set (table-entry *modules* 'locations)  `(,directory locations))
  (set (table-entry *modules* 'carcdr)     `(,directory carcdr))
  (set (table-entry *modules* 'predicates) `(,directory predicates))
  (set (table-entry *modules* 'open)       `(,directory open))
  (set (table-entry *modules* 'aliases)    `(,directory aliases))
  (set (table-entry *modules* 'genarith)    `(,directory genarith))
  t)

(define (substitute-arguments lambda-proc call-node)
  (walk (lambda (var val)
          (cond ((not (used? var)) nil)  ; VAL may be *EMPTY* if VAR is unused
                ((reference-node? val)
                 (partial-substitute-variable var (reference-variable val)))
                ((not (lambda-node? val))
                 nil)
                ((substitute-lambda? var)
                 (substitute var val t))
                ((partial-substitute-lambda? val)
                 (partial-substitute var val))
                (else
                 (substitute-known-args var val))))
        (lambda-variables lambda-proc)
        (call-args call-node)))

(define (substitute-known-args var val)
  (cond ((and (variable? (lambda-rest-var val))
              (null? (variable-refs (lambda-rest-var val))))
         (let ((len (length (lambda-variables val))))
           (walk-refs-safely (lambda (n)
                               (if (eq? call-proc (node-role n))
                                   (shorten-call-args (node-parent n) len)))
                             var)))))

(define (shorten-call-args call count)
  (if (fx> (length (call-args call)) count)
      (let ((rest (nthcdr (call-proc+args call) count)))
        (walk erase-all (cdr rest))
        (set (cdr rest) '()))))

(define (partial-substitute-lambda? val)
  (and (or (not (lambda-rest-var val)) 
           (null? (variable-refs (lambda-rest-var val))))
       (every? (lambda (n)
                 (leaf-node? n))
               (call-proc+args (lambda-body val)))
       (primop-node? (call-proc (lambda-body val)))))

(define (check-continuation-var var val)
  (cond ((any? (lambda (ref)
                 (eq? (node-role ref) call-proc))
               (variable-refs var))
         (walk-refs-safely (lambda (ref)
                             (if (call-exit? ref)
                                 (fix-exit-reference var ref val)))
                           var))
        ((fxn= 2 (variable-number var))
         (walk-refs-safely (lambda (ref)
                             (if (and (call-exit? ref)
                                      (not (primop-node?
                                            (call-proc (node-parent ref)))))
                                 (fix-exit-reference var ref val)))
                           var))))

(define (fix-exit-reference var node value)
  (let ((proc (call-proc (node-parent node))))
    (cond ((eq? node proc)
           (return))
          ((not (primop-node? proc))
           (introduce-exit-lambda var node value '#t))
          ((eq? primop/y (primop-value proc))
           (introduce-exit-lambda var node value '#f))
          (else
           (replace-with-lambda
            node
            (primop.values-returned
             (primop-value (call-proc (node-parent node)))))))))

(define (introduce-exit-lambda var node value args?)
  (if (and args? (used? (lambda-rest-var value)))
      (bug '"don't know how to fixup n-ary exit ~S" value))
  (let* ((new-vars (free-map (lambda (var)
                               (if var
                                   (create-variable (variable-name var))
                                   nil))
                             (lambda-rest+variables value)))
         (cont (create-lambda-node 'c new-vars))
         (args (if (not args?)
                   '()
                   (map (lambda (v) (if v
                                        (create-reference-node v)
                                        (create-literal-node '#f)))
                        (cdr new-vars))))
         (call (create-call-node (fx+ '1 (length args)) '0)))
    (relate call-proc call (create-reference-node var))
    (relate-call-args call args)
    (relate lambda-body cont call)
    (replace node cont)))

(define (replace-with-lambda node count)
  (let* ((vars (do ((i 0 (fx+ i 1))
                    (v '() (cons (create-variable 'v) v)))
                   ((fx>= i count) v)))
         (l-node (create-lambda-node 'x `(#f . ,vars)))
         (c-node (create-call-node (fx+ 1 count) 0)))
    (move node
          (lambda (node)
            (relate lambda-body l-node c-node)
            (relate call-proc c-node node)
            (relate-call-args c-node (map create-reference-node vars))
            l-node))))

(define (fixup-call-node node)
  (walk fixup-value-node (call-proc+args node))
  (let ((proc (call-proc node)))
    (cond ((lambda-node? proc)
           (walk (lambda (var val)
                   (if (lambda-node? val)
                       (check-continuation-var var val)))
                 (lambda-variables proc)
                 (call-args node)))
          ((primop-node? proc)
           (select (primop-value proc)
             ((primop/conditional)
              (if (reference-node? ((call-arg 1) node))
                  (replace-with-lambda ((call-arg 1) node) 0))
              (if (reference-node? ((call-arg 2) node))
                  (replace-with-lambda ((call-arg 2) node) 0)))
             ((primop/undefined-effect)
              (simplify-undefined-effect node))
             ((primop/y)
              (fixup-y node)))))))
