(herald (assembler n32ams t 66)
        (env tsys (assembler as_open))
;        (syntax-table
;         (block (*require 'as_syntax '(assembler as_syntax) (repl-env))
;                (env-syntax-table (repl-env))))
        )

;;; ---------------- Operands

(define-structure-type n32-operand
  mode index-byte extension
  (;handler  
    ((pretty-print self stream)
     (pretty-print-n32-operand self stream))))

(define (pretty-print-n32-operand o port)
  (let ((mode (n32-operand-mode o)))
    (cond ((eq? mode n32-immediate-mode-bits)
           (format port "#~g" (n32-operand-extension o)))
          ((eq? mode n32-external-mode-bits)
           (format port "EXT~g" (n32-operand-extension o)))
          (else
           (format port "~g" (n32-operand-mode o))))))

(define (cons-n32-indexed-operand m ib e)
  (let ((o (make-n32-operand)))
    (set (n32-operand-mode o) m)
    (set (n32-operand-index-byte o) ib)
    (set (n32-operand-extension o) e)
    o))

(define (cons-n32-operand m e)
  (let ((o (make-n32-operand)))
    (set (n32-operand-mode o) m)
    (set (n32-operand-index-byte o) n32-null-fg)
    (set (n32-operand-extension o) e)
    o))

(define-integrable (n32-indexed? o)
  (neq? (n32-operand-index-byte o) n32-null-fg))

(define (n32-index-bytes src dst)
  (cond ((n32-indexed? src)
         (cond ((n32-indexed? dst)
                (n32-index-both src dst))
               (else
                (n32-index-src src))))
        ((n32-indexed? dst)
         (n32-index-dst src dst))
        (else
         n32-null-fg)))

(define-fg (n32-index-both src dst)
  (fields
   (subfield (n32-operand-index-byte src))
   (subfield (n32-operand-index-byte dst))))

(define-fg (n32-index-src src)
  (fields
   (subfield (n32-operand-index-byte src))))

(define-fg (n32-index-dst src dst)
  (fields
   (subfield (n32-operand-index-byte dst))))

;;; ----------------

(define n32-null-fg ((fg-template (n32-null-fg) (printer "<nullfg>"))))

;;; ---------------- Registers

(define-op n32 (r regnum) 0
  (vref n32-register-fgs regnum))

(define-w/?-fg (n32-r regnum)
  (printer "R~s" regnum)
  (fields
   (0 0) (fixed 3 regnum)))

(define (n32-regnum reg)
  (cond ((fixnum? reg) reg)
        ((and (n32-operand? reg) (n32-r-fg? (n32-operand-mode reg)))
         (fg-argref (n32-operand-mode reg) 0))
        (else
         (error "n32-regnum expecting a register fg or register number"))))

(define (n32-special-regnum reg)
  (cond ((fixnum? reg) reg)
        ((and (n32-operand? reg) (n32-special-r-fg? (n32-operand-mode reg)))
         (fg-argref (n32-operand-mode reg) 0))
        (else
         (error "n32-special-regnum expecting a special reg fg or special reg number"))))

(define n32-register-numbers '#(0 1 2 3 4 5 6 7 8 9 10 11))
(lset n32-register-fgs (make-vector 12))

(do ((i 0 (fx+ i 1)))
    ((fx> i 7) 'done)
  (set (vref n32-register-fgs i) (cons-n32-operand (n32-r i) n32-null-fg)))

(define-w/?-fg (n32-special-r regnum)
  (printer "~a" (vref special-reg-name regnum))
  (fields
   (fixed 2 regnum)))

(define special-reg-names '#("FP" "SP" "SB" "PC"))

(set (vref n32-register-fgs  8)
     (cons-n32-operand (n32-special-r 0) n32-null-fg))
(set (vref n32-register-fgs  9)
     (cons-n32-operand (n32-special-r 1) n32-null-fg))
(set (vref n32-register-fgs 10)
     (cons-n32-operand (n32-special-r 2) n32-null-fg))
(set (vref n32-register-fgs 11)
     (cons-n32-operand (n32-special-r 3) n32-null-fg))

;;; ---------------- Register relative, (and memory space)

(define-op n32 (d@r reg displ) 1
  (cons-n32-operand (cond ((fx< reg 8) (register-relative-mode reg displ))
                           (else (memory-space-mode (fx- reg 8) displ)))
                     (cond ((fx= reg 11)
                            (n32-pc-relative-displacement displ))
                           (else (n32-displacement displ)))))

(define-fg (register-relative-mode reg displ)
  (printer "~s(R~s)" displ reg)
  (fields (0 1) (fixed 3 reg)))

(define-fg (memory-space-mode reg displ)
  (printer "~g(~a)" displ (vref special-reg-names reg))
  (fields (1 1 0) (fixed 2 reg)))

(define (n32/label section label)
  ((machine-operation n32 d@r) 11 (as-data-tag section label)))

(define (n32-displacement displ)
  (cond ((7bit? displ) (7-bit-displacement displ))
        ((14bit? displ) (14-bit-displacement displ))
        ((30bit? displ) (30-bit-displacement displ))
        (else (error "displacement too large"))))

;;; (We count on BITS code to mask off high bits where necessary.)

(define-fg (7-bit-displacement displ)
  (fields
   (fixed 8 (fixnum-logand #x7F displ))))

(define-fg (14-bit-displacement displ)
  (fields ; will get the low order bits only
   (fixed 8 (fixnum-logior #x80 (fixnum-logand #x3F (fixnum-ashr displ 8))))
   (fixed 8 displ)))   

(define-fg (30-bit-displacement displ)
  (fields
   (fixed 8 (fixnum-logior #xC0 (fixnum-ashr displ 24)))
   (fixed 8 (fixnum-ashr displ 16))
   (fixed 8 (fixnum-ashr displ 8))
   (fixed 8 displ)))

(define-fg (n32-pc-relative-displacement tag)
  (context (general size start type))
  (fields
   (subfield (n32-minimum-displacement start tag))))

;;; ---------------- Memory relative

(define-op n32 (d@d@r reg disp1 disp2) 3
  (cons-n32-operand (memory-relative-mode (fx- reg 8) disp1 disp2)
                     (two-displacements disp1 disp2)))

(define-fg (memory-relative-mode reg disp1 disp2)
  (printer "~s(~s(~a))" disp2 disp1 (vref special-reg-names reg))
  (fields
   (1 0 0) (fixed 2 reg)))

(define-fg (two-displacements disp1 disp2)
  (printer "(~s)+~s" disp1 disp2)
  (fields
   (subfield (n32-displacement disp1))
   (subfield (n32-displacement disp2))))

;;; ---------------- Immediate

(define-op n32 ($ val) 4
  (cons-n32-operand n32-immediate-mode-bits
                     (n32-immediate-value val)))

(define-fg (n32-immediate-value val)
  (printer "~s" val)
  (context (general size start type))
  (fields 
   (subfield (choose-n32-immediate val size type))))

(define n32-immediate-mode-bits
  ((fg-template (immediate-mode-fg)
                (fields (1 0 1 0 0)))))

(define (choose-n32-immediate val size type)
  (cond ((eq? type 'float)
         (error "immediate floats NYI"))  ;; size=1 -> 32bit, size=0  -> 64bit
        ((equal? type ''format-6')        ;; temporary hack for ashi etc.
         (n32-immediate-integer-fg-8 val))
        (else
         (case size
           ((0) (n32-immediate-integer-fg-8 val))
           ((1) (n32-immediate-integer-fg-16 val))
           ((3) (n32-immediate-integer-fg-32 val))
           (else (error "bad n32 integer size: ~s" size))))))

(define-fg (n32-immediate-integer-fg-8 val)
  (fields (fixed 8 val)))

(define-fg (n32-immediate-integer-fg-16 val)
  (fields (fixed 8 (l-bit-field-fx val 8 8))
          (fixed 8 (l-bit-field-fx val 0 8))))

(define-fg (n32-immediate-integer-fg-32 val)
  (fields (fixed 8 (l-bit-field-fx val 24 8))
          (fixed 8 (l-bit-field-fx val 16 8))
          (fixed 8 (l-bit-field-fx val 8  8))
          (fixed 8 (l-bit-field-fx val 0  8))))

(define (n32-imm? x) 
  (and (n32-operand? x) (eq? (n32-operand-mode x) n32-immediate-mode-bits)))

(define (n32-quick x)
  (cond ((fixnum? x)
         x)
        ((n32-imm? x)
         (fg-argref (n32-operand-extension x) 0))
        ((and (n32-operand? x) (n32-special-r-fg? (n32-operand-mode x)))
         (fx+ 8 (fg-argref (n32-operand-mode x) 0)))
        (else 
         (error "n32-quick expected a number, immediate, or special register"))))

;;; ---------------- Absolute  (NYI)

;;; ---------------- External

(define-op n32 (ext disp1 disp2) 5
  (cons-n32-operand n32-external-mode-bits
                     (two-displacements disp1 disp2)))

(define n32-external-mode-bits
  ((fg-template (external-mode-fg)
        (printer "EXT")
        (fields (1 0 1 1 0)))))

;;; ---------------- Top of stack

(define-op n32 (tos) 6
  tos-operand)
  
(define-fg (tos-fg)
  (printer "TOS")
  (fields (1 0 1 1 1)))

(define tos-operand (cons-n32-operand (tos-fg) n32-null-fg))

;;; ---------------- Memory space (see register relative)

;;; ---------------- Scaled indexing

(define-op n32 (index base reg scale) 7
  (cons-n32-indexed-operand (n32-indexed-mode base reg scale)
                             (n32-index-byte (n32-operand-mode base) reg)
                             (n32-operand-extension base)))

(define-fg (n32-indexed-mode base reg scale)
  (printer "~g[r~s]~s" base reg scale)
  (fields
   (1 1 1)
   (fixed 2 (case scale ((:b) 0) ((:w) 1) ((:d) 2) ((:q) 3)
              (else (error "base scale factor: ~s" scale))))))

(define-fg (n32-index-byte base-mode (n32-regnum reg))
  (fields
   (subfield base-mode)
   (fixed 3 reg)))
