/* Scheme implementation intended for JACAL.
   Copyright (C) 1989, 1990 Aubrey Jaffer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

The author can be reached at jaffer@ai.mit.edu or
Aubrey Jaffer, 84 Pleasant St., Wakefield MA 01880
*/

#include <stdio.h>
#include <ctype.h>
#include <setjmp.h>
#include "scm.h"

char *charnames[]={"nul","soh","stx","etx","eot","enq","ack","bel"
		,"backspace","tab","linefeed","vt","page","return","so","si"
		,"dle","dc1","dc2","dc3","dc4","nak","syn","etb"
		,"can","em","call","altmode","fs","gs","rs","backnext"
		,"space","rubout","newline"};
char charnums[]=
"\000\001\002\003\004\005\006\007\
\010\011\012\013\014\015\016\017\
\020\021\022\023\024\025\026\027\
\030\031\032\033\034\035\036\037\
\040\177\012";

char *isymnames[]={"=>","and","begin","case","cond","define",""
		,"do","else","if","lambda","let","let*","letrec"
		,"or","quasiquote","quote","set!","unquote","unquote-splicing"
		,"."};

unsigned tok_buf_len = 30;
char *tok_buf;
SCM *stack_start_ptr;

SCM s_make_vector;
SCM s_call_with_input_file, s_call_with_output_file;
SCM s_input_portp, s_output_portp;
SCM s_open_input_file, s_open_output_file, s_close_port;
SCM s_read, s_read_char, s_peek_char, s_char_readyp;
SCM s_write, s_display, s_newline, s_write_char;

SCM lreadr(),lreadparen(),read_integer();
int read_token();

iprint(n,radix,f)
long n;
int radix;
FILE *f;
{
	register char d,ld;
	register long x;
	switch (radix) {
	case 2:
	case 8:
	case 10:
		if (n<0) {
			putc('-',f);
			n = -n;
		}
		ld = n % radix;
		n /= radix;
		for (x = 1;x <= n;x *= radix);
		x /= radix;
		while (x > 0) {
			d = n / x;
			n %= x;
			x /= radix;
			putc(d + '0',f);
			}
		putc(ld + '0',f);
		break;
	case 16:
		if (n<0) {
			putc('-',f);
			n = -n;
		}
		ld = n % radix;
		n /= radix;
		for (x = 1;x <= n;x *= radix);
		x /= radix;
		while (x > 0) {
			d = n / x;
			n %= x;
			x /= radix;
			putc(d + ((d > 9) ? 'a'-10 : '0'),f);
			}
		putc(ld + ((ld > 9) ? 'a'-10 : '0'),f);
		break;
	default:
		err("invalid radix",MAKINUM((long)radix));
	}
}
lprin1f(exp,f,writing)
SCM exp;
FILE *f;
int writing;
{
	register SCM tmp;
	register long i;
	if IMP(exp) {
		if INUMP(exp) iprint(INUM(exp),10,f);
		else if ICHRP(exp) {
			i = ICHR(exp);
			if (writing) fputs("#\\",f);
			if (!writing) putc((int)i,f);
			else if (i<=' ') fputs(charnames[i],f);
			else if (i=='\177') fputs(charnames[' '+1],f);
			else putc((int)i,f);
		}
		else if ISYMP(exp) fputs(ISYMCHARS(exp),f);
		else switch(exp) {
		case BOOL_F:
		  fputs("#f",f);
		  break;
		case BOOL_T:
		  fputs("#t",f);
		  break;
		case EOL:
		  fputs("()",f);
		  break;
		case EOF_VAL:
		  fputs("#[eof]",f);
		  break;
		case UNDEFINED:
		  fputs("#[undefined]",f);
		  break;
		case UNSPECIFIED:
		  fputs("#[unspecified]",f);
		  break;
		default:
		  fputs("#[unknown immediate 0x",f);
		  iprint(exp,16,f);
		  putc(']',f);
		}
	}
	else switch TYP6(exp) {
	case tcs_cons_imcar:
	case tcs_cons_nimcar:
		putc('(',f);
		CHECK_SIGINT;
		lprin1f(CAR(exp),f,writing);
		for(tmp=CDR(exp);NIMP(tmp);tmp=CDR(tmp)) {
			if NCONSP(tmp) break;
			putc(' ',f);
			CHECK_SIGINT;
			lprin1f(CAR(tmp),f,writing);
		}
		if NNULLP(tmp) {
			fputs(" . ",f);
			lprin1f(tmp,f,writing);
		}
		putc(')',f);
		break;
	case tcs_closures:
		fputs("#[CLOSURE ",f);
		tmp=CODE(exp);
		lprin1f(CAR(tmp),f,writing);
		for (tmp=CDR(tmp);NNULLP(tmp);tmp=CDR(tmp)) {
			putc(' ',f);
			CHECK_SIGINT;
			lprin1f(CAR(tmp),f,writing);
		}
		putc(']',f);
		break;
	case tc6_string:
		if (writing) {
			putc('\"',f);
			for(i=0;i<LENGTH(exp);++i) switch (CHARS(exp)[i]) {
			case '"':
			case '\\':
				putc('\\',f);
			default:
				putc(CHARS(exp)[i], f);
			}
			putc('\"',f);
		}
		else for(i=0;i<LENGTH(exp);++i) putc(CHARS(exp)[i], f);
		break;
	case tc6_vector:
		fputs("#(",f);
		for(i=0;i<(LENGTH(exp)-1);++i) {
			CHECK_SIGINT;
			lprin1f(VELTS(exp)[i],f,writing);
			putc(' ',f);
		}
		if (i<LENGTH(exp)) {
			CHECK_SIGINT;
			lprin1f(VELTS(exp)[i],f,writing);
		      }
		putc(')',f);
		break;
	case tcs_symbols:
		for(i = 0;i < LENGTH(NAMESTR(exp));i++)
			putc(CHARS(NAMESTR(exp))[i],f);
		break;
	case tcs_subrs:
		fputs("#[primitve-procedure ",f);
		fputs(CHARS(SNAME(exp)),f);
		putc(']',f);
		break;
	case tc6_contin:
		fputs("#[continuation ",f);
		iprint(LENGTH(exp),10,f);
		fputs(" @ ",f);
		iprint(JMPBUF(exp),16,f);
		putc(']',f);
		break;
	case tc6_smob:
		switch CAR(exp) {
		case tc_inport:
			fputs("#[input-port ",f);
			goto portprint;
		case tc_outport:
			fputs("#[output-port ",f);
portprint:
			if CLOSEDP(exp) fputs("closed",f);
			else iprint((long)fileno(STREAM(exp)),10,f);
			putc(']',f);
			return;
		}
	default:
		fputs("#[unknown-type ",f);
		iprint(TYP6(exp),10,f);
		fputs(" 0x",f);
		iprint(exp,16,f);
		putc(']',f);
	}
}

char *grow_tok_buf();

int flush_ws(f,eoferr)
FILE *f;
char *eoferr;
{
	register int c;
	while(1) switch (c = getc(f)) {
	case EOF:
goteof:
		if (eoferr) wta((SCM)eoferr,ENDFILE,s_read);
		else return c;
	case ';':
lp:
		switch (c = getc(f)) {
		case EOF:
			goto goteof;
		case LINE_INCREMENTORS:
			line_num++;
			break;
		default:
			goto lp;
		}
		break;
	case LINE_INCREMENTORS:
		line_num++;
	case WHITE_SPACES:
		break;
	default:
		return c;
	}
}
SCM lread(port)
SCM port;
{
	FILE *f;
	int c;
	if UNBNDP(port) port = cur_inp;
	else ASSERT(NIMP(port) && INPORTP(port) && OPENP(port),port,ARG1,s_read);
	f = STREAM(port);
	c = flush_ws(f,(char *)NULL);
	if (c == EOF) return EOF_VAL;
	ungetc(c,f);
	return lreadr(f);
}
SCM lreadr(f)
FILE *f;
{
	register int c,j;
	SCM p;
tryagain:
	c = flush_ws(f,"read");
	switch (c) {
	case '(':
		return lreadparen(f,"list");
	case ')':
		DEFER_SIGINT;
		err_head();
		fputs("WARNING: unexpected \")\"",stderr);
		fflush(stderr);
		ALLOW_SIGINT;
		goto tryagain;
	case '\'':
		return cons(s_quote,cons(lreadr(f),EOL));
	case '`':
		return cons(s_quasiquote,cons(lreadr(f),EOL));
	case ',':
		c = getc(f);
		if (c == '@') p = s_unquote_splicing;
		else {
			ungetc(c,f);
			p = s_unquote;
		}
		return cons(p,cons(lreadr(f),EOL));
	case '#':
		c = getc(f);
		switch(c) {
		case '(':
			return vector(lreadparen(f,"vector"));
		case 't':
		case 'T':
			return BOOL_T;
		case 'f':
		case 'F':
			return BOOL_F;
		case 'b':
		case 'B':
			return read_integer(2,0,f);
		case 'o':
		case 'O':
			return read_integer(8,0,f);
		case 'd':
		case 'D':
			return read_integer(10,0,f);
		case 'x':
		case 'X':
			return read_integer(16,0,f);
		case '\\':
			c = getc(f);
			j = read_token(c,f);
			if (j==1) return MAKICHR(c);
			for (c=0;c<sizeof charnames/sizeof(char *);c++)
				if (0==strcmp(charnames[c],tok_buf))
					return MAKICHR(charnums[c]);
			err("unknown # object",BOOL_F);
		default:
			err("unknown # object",MAKICHR(c));
		}
	case '\"':
		j = 0;
		while ((c = getc(f)) != '\"') {
			ASSERT(c != EOF,c,"eof in string",s_read);
			if (j+1 >= tok_buf_len) grow_tok_buf("string");
			if (c == '\\') c = getc(f);
			tok_buf[j] = c;
			++j;
		}
		if (j == 0) return nullstr;
		tok_buf[j] = 0;
		return makfromstr(tok_buf,(long)j);
	case DIGITS:
		ungetc(c,f);
		return read_integer(10,0,f);
	case '-':
		if (isdigit (c=getc(f))) {
			ungetc(c,f);
			return read_integer(10,1,f);
		}
		ungetc(c,f);
		c = '-';
		goto tok;
	case '+':
		if (isdigit (c=getc(f))) {
			ungetc(c,f);
			return read_integer(10,0,f);
		}
		ungetc(c,f);
		c = '+';
	default:
tok:
		return intern(tok_buf,(long)read_token(c,f));
	}
}
SCM read_integer(radix,sgn,f)
int radix,sgn;
FILE *f;
{
	register int c;
	register long n = 0,res;
	c = getc(f);
	if (c == '-') sgn = 1;
	else ungetc(c,f);
	while((c = getc(f)) != EOF) switch(c) {
	case DIGITS:
		c -= '0';
		goto accumulate;
	case 'A':
	case 'B':
	case 'C':
	case 'D':
	case 'E':
	case 'F':
		c = c-'A'+10;
		goto accumulate;
	case 'a':
	case 'b':
	case 'c':
	case 'd':
	case 'e':
	case 'f':
		c = c-'a'+10;
accumulate:
		ASSERT(((c>=0)&(c<radix)),MAKINUM((long)n),"wrong radix digit in number",s_read);
		res = n;
		n = n * radix - c;
		ASSERT((n + c)/radix == res,MAKINUM((long)res),OVERFLOW,s_read);
		continue;
	case '(':
	case ')':
	case '\'':
	case '`':
	case ',':
	case ';':
	case '#':
	case '\"':
	ungetc(c,f);
	case WHITE_SPACES:
		goto xit;
	case LINE_INCREMENTORS:
		line_num++;
		goto xit;
	default:
		err("bad char in number",MAKICHR(c));
	}
xit:
	if (!sgn) n = - n;
	res = MAKINUM((long)n);
	ASSERT(INUM(res) == n,res,OVERFLOW,s_read);
	return res;
}
int read_token(ic,f)
int ic;
FILE *f;
{
	register unsigned j = 1;
	register int c = ic;
	register char *p = tok_buf;
	p[0] = downcase[c];
	while(1) {
		if (j+1 >= tok_buf_len) p = grow_tok_buf("symbol");
		switch (c = getc(f)) {
		case '(':
		case ')':
		case '\'':
		case '`':
		case ',':
		case ';':
		case '#':
		case '\"':
			ungetc(c,f);
		case EOF:
		case WHITE_SPACES:
getout:
			p[j] = 0;
			return j;
		case LINE_INCREMENTORS:
			line_num++;
			goto getout;
		default:
			p[j++] = downcase[c];
		}
	}
}
SCM lreadparen(f,name)
FILE *f;
char *name;
{
	SCM tmp;
	register int c;
	c = flush_ws(f,name);
	if (c == ')') return EOL;
	ungetc(c,f);
	tmp = lreadr(f);
	if (tmp != s_dot) return cons(tmp,lreadparen(f,name));
	tmp = lreadr(f);
	c = flush_ws(f,name);
	if (c != ')') err("missing close paren",BOOL_F);
	return tmp;
}
SCM open_input_file(filename)
SCM filename;
{
	FILE *f;
	ASSERT(NIMP(filename) && STRINGP(filename),filename,ARG1,s_open_input_file);
	f = fopen(CHARS(filename),"r");
	if (!f) wta(CHARS(filename),NOFILE,s_open_input_file);
	return makport(f,tc_inport);
}
SCM open_output_file(filename)
SCM filename;
{
	FILE *f;
	ASSERT(NIMP(filename) && STRINGP(filename),filename,ARG1,s_open_output_file);
	f = fopen(CHARS(filename),"w");
	if (!f) wta(CHARS(filename),NOFILE,s_open_output_file);
	return makport(f,tc_outport);
}
SCM close_port(f)
SCM f;
{
	ASSERT(NIMP(f) && PORTP(f),f,ARG1,s_close_port);
	if CLOSEDP(f) return UNSPECIFIED;
	DEFER_SIGINT;
	fclose(STREAM(f));
	SETSTREAM(f,0);
	ALLOW_SIGINT;
	return UNSPECIFIED;
}
SCM call_with_input_file(str,proc)
SCM str,proc;
{
	SCM file,res;
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_call_with_input_file);
	ASSERT(NFALSEP(procedurep(proc)),proc,ARG2,s_call_with_input_file);
	file = open_input_file(str);
	res = apply(proc,file,listofnull);
	close_port(file);
	return res;
}
SCM call_with_output_file(str,proc)
SCM str,proc;
{
	SCM file,res;
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_call_with_output_file);
	ASSERT(NFALSEP(procedurep(proc)),proc,ARG2,s_call_with_output_file);
	file = open_output_file(str);
	res = apply(proc,file,listofnull);
	close_port(file);
	return res;
}
SCM input_portp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return (CAR(x) == tc_inport) ? BOOL_T : BOOL_F;
}
SCM output_portp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return (CAR(x) == tc_outport) ? BOOL_T : BOOL_F;
}
SCM current_input_port()
{
	return cur_inp;
}
SCM current_output_port()
{
	return cur_outp;
}

int f_getc(f)
FILE *f;
{
	int c;
#ifdef vms
	long old_sig_deferred;
	DEFER_SIGINT;
	old_sig_deferred = sig_deferred;
	c = getc(f);
	if ((old_sig_deferred == 0) && sig_deferred && (f == stdin))
		while(c && (c != EOF)) c = getc(f);
#else
	DEFER_SIGINT;
	c = getc(f);
#endif
	ALLOW_SIGINT;
	return c;
}
SCM read_char(port)
SCM port;
{
	int c;
	if UNBNDP(port) port = cur_inp;
	else ASSERT(NIMP(port) && INPORTP(port) && OPENP(port),port,ARG1,s_read_char);
	c = f_getc(STREAM(port));
	if (c == EOF) return EOF_VAL;
	return MAKICHR(c);
}
SCM peek_char(port)
SCM port;
{
	FILE *f;
	int c;
	if UNBNDP(port) port = cur_inp;
	else ASSERT(NIMP(port) && INPORTP(port) && OPENP(port),port,ARG1,s_peek_char);
	f = STREAM(port);
	c = f_getc(f);
	if (c == EOF) return EOF_VAL;
	ungetc(c,f);
	return MAKICHR(c);
}
SCM eof_objectp(x)
SCM x;
{
	return (EOF_VAL == x) ? BOOL_T : BOOL_F;
}
SCM lwrite(obj,port)
SCM obj,port;
{
	if UNBNDP(port) port = cur_outp;
	else ASSERT(NIMP(port) && OUTPORTP(port) && OPENP(port),port,ARG2,s_write);
	DEFER_SIGINT;
	lprin1f(obj,STREAM(port),1);
	ALLOW_SIGINT;
	return UNSPECIFIED;
}
SCM display(obj,port)
SCM obj,port;
{
	if UNBNDP(port) port = cur_outp;
	else ASSERT(NIMP(port) && OUTPORTP(port) && OPENP(port),port,ARG2,s_display);
	DEFER_SIGINT;
	lprin1f(obj,STREAM(port),0);
	ALLOW_SIGINT;
	return UNSPECIFIED;
}
SCM newline(port)
SCM port;
{
	if UNBNDP(port) port = cur_outp;
	else ASSERT(NIMP(port) && OUTPORTP(port) && OPENP(port),port,ARG1,s_newline);
	DEFER_SIGINT;
	putc('\n',STREAM(port));
	ALLOW_SIGINT;
	return UNSPECIFIED;
}
SCM write_char(chr,port)
SCM chr,port;
{
	if UNBNDP(port) port = cur_outp;
	else ASSERT(NIMP(port) && OUTPORTP(port) && OPENP(port),port,ARG2,s_write_char);
	ASSERT(ICHRP(chr),chr,ARG1,s_write_char);
	DEFER_SIGINT;
	putc((int)ICHR(chr),STREAM(port));
	ALLOW_SIGINT;
	return UNSPECIFIED;
}

init_io(){
	init_subr("gc",tc6_subr_0,gc);
	init_subr("call-with-current-continuation",tc6_subr_1,call_cc);

	s_call_with_input_file=init_subr("call-with-input-file",tc6_subr_2,call_with_input_file);
	s_call_with_output_file=init_subr("call-with-output-file",tc6_subr_2,call_with_output_file);
	s_input_portp=init_subr("input-port?",tc6_subr_1,input_portp);
	s_output_portp=init_subr("output-port?",tc6_subr_1,output_portp);
	init_subr("current-input-port",tc6_subr_0,current_input_port);
	init_subr("current-output-port",tc6_subr_0,current_output_port);
	s_open_input_file=init_subr("open-input-file",tc6_subr_1,open_input_file);
	s_open_output_file=init_subr("open-output-file",tc6_subr_1,open_output_file);
	s_close_port=init_subr("close-input-port",tc6_subr_1,close_port);
	init_subr("close-output-port",tc6_subr_1,close_port);
	s_read=init_subr("read",tc6_subr_1,lread);
	s_read_char=init_subr("read-char",tc6_subr_1,read_char);
	s_peek_char=init_subr("peek-char",tc6_subr_1,peek_char);
	init_subr("eof-object?",tc6_subr_1,eof_objectp);
	s_write=init_subr("write",tc6_subr_2,lwrite);
	s_display=init_subr("display",tc6_subr_2,display);
	s_newline=init_subr("newline",tc6_subr_1,newline);
	s_write_char=init_subr("write-char",tc6_subr_2,write_char);
}

#define CELL_UP(p) (cell *)((((long)(p))+7)&~7)
#define CELL_DN(p) (cell *)(((((long)(p))-7)|7)+1)
unsigned heap_lim_ind = 0,num_heap_segs = 1;
unsigned long heap_size = 0;
cell *heap_org,**heap_lims;
SCM freelist = EOL;
#ifdef unix
	char *malloc();
#else
#ifdef TURBO
#include <alloc.h>
#else
#include <malloc.h>
#endif
#endif

char *must_malloc(len,what)
unsigned long len;
char *what;
{
	char *ptr;
	unsigned int size = len;
	if (len != size) goto malerr;
	ptr = malloc(size);
	if (ptr != NULL) return ptr;
	gc();
	ptr = malloc(size);
	if (ptr != NULL) return ptr;
malerr:
	wta(MAKINUM((long)len),NALLOC,(SCM)what);
}
#define NEWCELL(_into) {if IMP(freelist) _into = gc_for_newcell();\
	else {_into = freelist;freelist = CDR(freelist);++gc_cells_allocated;}}
int symhash_dim = NUM_HASH_BUCKETS;

init_isyms()
{
	int hash,i = NUM_ISYMS,n = symhash_dim;
	char *cname,c;
	while (0 <= --i) {
		hash = 0;
		cname = isymnames[i];
		while(c = *cname++) hash = ((hash * 17) ^ c) % n;
		VELTS(symhash)[hash] =
			cons(MAKISYM((long)i),VELTS(symhash)[hash]);
	}
}
/* if length is negative, use the given string directly if possible */
SCM intern(name,len)
char *name;
long len;
{
	SCM lsym;
	long alen = (len < 0) ? -len:len;
	register char *tmp = name;
	register int i = alen, hash = 0, n = symhash_dim;
	while(0 < i--) hash = ((hash * 17) ^ *tmp++) % n;
	if(len>=0) for(lsym=VELTS(symhash)[hash];NIMP(lsym);lsym=CDR(lsym)) {
		if ISYMP(CAR(lsym)) {
			tmp = ISYMCHARS(CAR(lsym));
			for(i = 0;i < alen;i++) {
				if (tmp[i] == 0) goto trynext;
				if (name[i] != tmp[i]) goto trynext;
				}
			if (tmp[i] == 0) return CAR(lsym);
		}
		else {
			tmp = CHARS(NAMESTR(CAR(lsym)));
			if (alen != LENGTH(NAMESTR(CAR(lsym)))) goto trynext;
			for(i = alen;0 < --i;)
				if (name[i] != tmp[i]) goto trynext;
			return CAR(lsym);
		}
trynext:
		;
	}
	DEFER_SIGINT;
	if ((len < 0) && ((long)name & 1 == 0)) {
		NEWCELL(lsym);
		SETLENGTH(lsym,alen,tc6_string);
		SETCHARS(lsym,name);
	}
	else lsym = makfromstr(name, alen);
	{
		SCM z = lsym;
		NEWCELL(lsym);
		SETNAMESTR(lsym,z);
		VCELL(lsym) = UNDEFINED;
	}
	VELTS(symhash)[hash] = cons(lsym,VELTS(symhash)[hash]);
	ALLOW_SIGINT;
	return lsym;
}
SCM cons(x,y)
SCM x,y;
{
	register SCM z;
	NEWCELL(z);
	CAR(z) = x;
	CDR(z) = y;
	return z;
}
SCM makstr(length)
long length;
{
	register SCM s;
	char *str = must_malloc(length+1,"string");
	str[length] = 0;
	NEWCELL(s);
	DEFER_SIGINT;
	SETLENGTH(s,length,tc6_string);
	SETCHARS(s,str);
	ALLOW_SIGINT;
	return s;
}
SCM makfromstr(src, len)
char *src;
long len;
{
	SCM s;
	register char *dst;
	s = makstr(len);
	dst = CHARS(s);
	while (0 <= --len) *dst++ = *src++;
	return s;
}
SCM make_vector(k,fill)
SCM k,fill;
{
	register SCM v;
	register long i;
	register char *str;
	ASSERT(INUMP(k),k,ARG1,s_make_vector);
	i = INUM(k);
	if (i == 0) return nullvect;
	str = must_malloc(i*sizeof(SCM),"vector");
	NEWCELL(v);
	DEFER_SIGINT;
	SETLENGTH(v,i,tc6_vector);
	SETCHARS(v,str);
	for(;--i>=0;) ((SCM *)str)[i] = fill;
	ALLOW_SIGINT;
	return v;
}
SCM init_subr(name,type,fcn)
char *name;
int type;
SCM (*fcn)();
{
	SCM sym = intern(name,-(long)strlen(name));
	register SCM z;
	NEWCELL(z);
	SETSNAME(z,NAMESTR(sym),type);
	SUBRF(z) = fcn;
	return VCELL(sym) = z;
}
SCM closure(code,env)
SCM code,env;
{
	register SCM z;
	ASSERT(ilength(code)>1,code,ARG1,s_lambda);
	NEWCELL(z);
	SETCODE(z,code);
	ENV(z) = env;
	return z;
}
long stack_size(start)
SCM *start;
{
	long stack;
#ifdef STACK_GROWS_UP
	stack = &stack - start;
#else
	stack = start - &stack;
#endif
	return stack;
}
SCM throwval = UNDEFINED;
SCM call_cc(proc)
SCM proc;
{
	long j;
	SCM cont;
#ifdef CHEAP_CONTINUATIONS
	NEWCELL(cont);
	DEFER_SIGINT;
	SETLENGTH(cont,0,tc6_contin);
	SETJMPBUF(cont,must_malloc((long)sizeof(jmp_buf),"continuation"));
	ALLOW_SIGINT;
#else
	register SCM *src,*dst;
	NEWCELL(cont);
	DEFER_SIGINT;
	SETLENGTH(cont,stack_size(stack_start_ptr),tc6_contin);
	SETJMPBUF(cont,must_malloc(sizeof(jmp_buf)+LENGTH(cont)*sizeof(SCM *)
					,"continuation"));
	ALLOW_SIGINT;
	src = stack_start_ptr;
#ifndef STACK_GROWS_UP
	src -= LENGTH(cont);
#endif
	dst = (SCM *)(CHARS(cont)+sizeof(jmp_buf));
	for (j = LENGTH(cont);0 <= --j;) *dst++ = *src++;
#endif
	if (setjmp(JMPBUF(cont))) return throwval;
	return apply(proc,cont,listofnull);
}
throw(cont,val)
SCM cont,val;
{
#ifndef CHEAP_CONTINUATIONS
	register long j;
	register SCM *src,*dst;
	dst = stack_start_ptr;
#ifdef STACK_GROWS_UP
	if (dst + LENGTH(cont) >= &cont) grow_throw(cont,val);
#else
	dst -= LENGTH(cont);
	if (dst <= &cont) grow_throw(cont,val);
#endif
	src = (SCM *)(CHARS(cont)+sizeof(jmp_buf));
	for (j = LENGTH(cont);0 <= --j;) *dst++ = *src++;
#endif
	throwval = val;
	longjmp(JMPBUF(cont),1);
}
#ifndef CHEAP_CONTINUATIONS
grow_throw(cont,val)
SCM cont,val;
{
	long growth[100];
	throw(cont,val);
}
#endif
SCM makport(stream,type)
FILE *stream;
int type;
{
	register SCM z;
	DEFER_SIGINT;
	NEWCELL(z);
	SETLENGTH(z,0L,type);
	SETSTREAM(z,stream);
	ALLOW_SIGINT;
	return z;
}

fixconfig(s1,s2)
char *s1, *s2;
{
	fputs(s1,stdout);
	puts(s2);
	puts("in config.h and recompile scm");
	exit(1);
}
int init_heap_seg(base,size)
cell *base;
long size;
{
	register cell *ptr = base, *seg_org, *seg_end;
	unsigned int len = size;
	int i = 0,ni = heap_lim_ind;
	if (base == NULL) return 0;
	seg_org = CELL_UP(ptr);
	seg_end = CELL_DN((char *)ptr + len);
	while((i < ni) && (heap_lims[i] <= seg_org)) i++;
	ni = i;
	heap_lim_ind += 2;
	for(i+=2;i < heap_lim_ind;i++) heap_lims[i] = heap_lims[i-2];
	heap_lims[ni++] = seg_org;
	heap_lims[ni++] = seg_end;
	for (ptr=seg_org;ptr<seg_end;ptr++) {
		CAR(ptr) = (SCM)tc_free_cell;
		CDR(ptr) = (SCM)(ptr+1);
	}
	CDR(--ptr) = freelist;
	freelist = (SCM)seg_org;
	heap_size += seg_end - seg_org;
	growth_mon("heap",heap_size,"bytes");
	return len;
}
alloc_some_heap()
{
	register unsigned int len = num_heap_segs*2;
	register cell *ptr;
	if (heap_lim_ind >= len)
		if(!(heap_lims = (cell **)realloc(heap_lims,
				(num_heap_segs = len)*2*sizeof(cell *))))
			err("could not realloc heap_lims",BOOL_F);
		else growth_mon("number of heaps",(long)num_heap_segs,"segments");
	len = HEAP_SEG_SIZE;
	if (len != HEAP_SEG_SIZE)
		fixconfig("reduce","size of HEAP_SEG_SIZE");
trya:	
	if (len < MIN_HEAP_SEG_SIZE) return;
	ptr = (cell *) malloc(len);
	if (ptr == NULL) {
		len /= 2;
		goto trya;
	}
	init_heap_seg(ptr,len);
}
char *grow_tok_buf(typstr)
char *typstr;
{
	register unsigned len = tok_buf_len;
	register long llen = len;
	llen += len / 2;
	len = llen;
	if(len != llen) goto dontdo;
	tok_buf = (char *)realloc(tok_buf,len);
	if (tok_buf == NULL) {
		tok_buf = (char *)malloc(tok_buf_len);
		if(tok_buf == NULL) errjmp_ok = 0;
dontdo:
		wta(MAKINUM((long)llen),NALLOC,(SCM)typstr);
	}
	else tok_buf_len = len;
	growth_mon("tok_buf",(long)tok_buf_len,"bytes");
	return tok_buf;
}
init_storage()
{
	SCM p;
	unsigned j;
	stack_start_ptr = &p; /* this is set again in repl */
 	if (stack_size(&p) < 0)
		fixconfig(
#ifdef STACK_GROWS_UP		
			"add\n#"
#else
			"remove\n#"
#endif
			,"define STACK_GROWS_UP");

	tok_buf = must_malloc((long)tok_buf_len,"tok_buf");

	heap_lims = (cell **)
		must_malloc((long)num_heap_segs*2*sizeof(cell *),"heap_lims");
	j = INIT_HEAP_SIZE;
	if ((j != INIT_HEAP_SIZE) || !init_heap_seg((cell *) malloc(j),j))
		alloc_some_heap();
	heap_org = heap_lims[0];
		/* heap_lims[0] can change. do not remove heap_org */


	cur_inp=makport(stdin,tc_inport);
	cur_outp=makport(stdout,tc_outport);
	listofnull = cons(EOL,EOL);
	undefineds = cons(UNDEFINED,EOL);
	CDR(undefineds) = undefineds;
	nullstr = makstr(0L);
	NEWCELL(nullvect);
	SETLENGTH(nullvect,0L,tc6_vector);
	SETCHARS(nullvect,NULL);

	symhash = make_vector(MAKINUM((long)symhash_dim),EOL);
	init_isyms();
}
/* The way of garbage collecting which allows us to use the cstack is due to*/
/* Scheme In One Defun, but in C this time.
 
 *                        COPYRIGHT (c) 1989 BY                             *
 *        PARADIGM ASSOCIATES INCORPORATED, CAMBRIDGE, MASSACHUSETTS.       *
 *			   ALL RIGHTS RESERVED                              *

Permission to use, copy, modify, distribute and sell this software
and its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all copies
and that both that copyright notice and this permission notice appear
in supporting documentation, and that the name of Paradigm Associates
Inc not be used in advertising or publicity pertaining to distribution
of the software without specific, written prior permission.

PARADIGM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
PARADIGM BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

gjc@paradigm.com

Paradigm Associates Inc          Phone: 617-492-6079
29 Putnam Ave, Suite 6
Cambridge, MA 02138
*/

SCM gc_for_newcell()
{
	SCM fl;
	gc();
	if ((gc_cells_collected < MIN_GC_YIELD) || IMP(freelist))
		alloc_some_heap();
	++gc_cells_allocated;
	fl = freelist;
	freelist = CDR(fl);
	return fl;
}

SCM sys_protects[NUM_PROTECTS];
jmp_buf save_regs_gc_mark;

SCM gc()
{
	long j = NUM_PROTECTS;
	DEFER_SIGINT;
	errjmp_ok = 0;
	gc_ms_stats_start();
	while(0 <= --j) gc_mark(sys_protects[j]);
	/* This assumes that all registers are saved into the jmp_buf */
	setjmp(save_regs_gc_mark);
	mark_locations((SCM *) save_regs_gc_mark,
			(long)sizeof(save_regs_gc_mark)/sizeof(SCM *));
#ifdef STACK_GROWS_UP
	mark_locations(stack_start_ptr,&j-stack_start_ptr);
#else
	mark_locations(&j,stack_start_ptr-&j);
#endif
#ifdef SHORT_ALIGN
	mark_locations((SCM *) (((char *)save_regs_gc_mark)+2),
			(long)sizeof(save_regs_gc_mark)/sizeof(SCM *));
#ifdef STACK_GROWS_UP
	mark_locations((SCM *)(((char *)stack_start_ptr)+2),
		&j-stack_start_ptr);
#else
	mark_locations((SCM *)(((char *)&j)+2),
		stack_start_ptr-&j);
#endif
#endif
	gc_sweep();
	gc_ms_stats_end();
	errjmp_ok = 1;
	ALLOW_SIGINT;
	return UNSPECIFIED;
}

gc_mark(p)
SCM p;
{
	register long i;	
	register SCM ptr = p;
gc_mark_loop:
	if IMP(ptr) return;
	if (NCELLP(ptr)
#ifndef RECKLESS
	    || (heap_org > (cell *)ptr)
	    || ((cell *)ptr >= heap_lims[heap_lim_ind-1])
#endif
		) err("rogue pointer in heap",ptr);
	if GCMARKP(ptr) return;
	switch TYP6(ptr) {
	case tcs_cons_nimcar:
		SETGCMARK(ptr);
		gc_mark(CAR(ptr));
		ptr = GCCDR(ptr);
		goto gc_mark_loop;
	case tcs_cons_imcar:
		SETGCMARK(ptr);
		ptr = GCCDR(ptr);
		goto gc_mark_loop;
	case tcs_symbols:
		SETGCMARK(ptr);
		gc_mark(NAMESTR(ptr));
		ptr = GCCDR(ptr);
		goto gc_mark_loop;
	case tcs_closures:
		SETGCMARK(ptr);
		gc_mark(CODE(ptr));
		ptr = GCCDR(ptr);
		goto gc_mark_loop;
	case tc6_vector:
		SETGCMARK(ptr);
		i=LENGTH(ptr);
		if (i == 0) return;
		while(--i>0) gc_mark(GCVELTS(ptr)[i]);
		ptr = GCVELTS(ptr)[0];
		goto gc_mark_loop;
	case tc6_contin:
		SETGCMARK(ptr);
		mark_locations(GCVELTS(ptr),
			LENGTH(ptr) + sizeof(jmp_buf)/sizeof(SCM *));
		return;
	case tc6_string:
		SETGCMARK(ptr);
	case tcs_subrs:
		return;
	case tc6_smob:
		switch CAR(ptr) {
		case tc_inport:
		case tc_outport:
			SETGCMARK(ptr);
			return;
		default:
			;
		}
	default:	
		err("unknown data type in gc_mark",ptr);
	}
}

mark_locations(x,n)
SCM x[];
long n;
{
	register long m = n;
	register int i,j;
	register SCM ptr;
	while(0 <= --m) if CELLP(ptr = x[m]) {
		i=0;
		j=heap_lim_ind;
		do {
			if (heap_lims[i++] > (cell *)ptr) break;
			if (heap_lims[--j] <= (cell *)ptr) break;
			if ((i != j) &&
				(heap_lims[i++] <= (cell *)ptr) &&
				(heap_lims[--j] > (cell *)ptr)) continue;
			if NFREEP(ptr) gc_mark(ptr);
			break;
		} while(i<j);
	}
}

gc_sweep()
{
	register cell *ptr, *heap_hi;
	register SCM nfreelist = freelist;
	register long n=0,m=0;
	int i=0;
	while (i<heap_lim_ind)
	   for(ptr=heap_lims[i++],heap_hi=heap_lims[i++];ptr<heap_hi;++ptr) {
		switch TYP6(ptr) {
		case tcs_cons_imcar:
		case tcs_cons_nimcar:
		case tcs_closures:
			if GCMARKP(ptr) goto cmrkcontinue;
			break;
		case tc6_vector:
/*		case tc6_bignum: */
			if GCMARKP(ptr) goto cmrkcontinue;
			m += (LENGTH(ptr)*sizeof(SCM));
			free(CHARS(ptr));
			break;
		case tc6_string:
			if GCMARKP(ptr) goto cmrkcontinue;
			m += LENGTH(ptr)+1;
			free(CHARS(ptr));
			break;
		case tc6_contin:
			if GCMARKP(ptr) goto cmrkcontinue;
			m += LENGTH(ptr) + sizeof(jmp_buf);
			free(CHARS(ptr));
			break;
		case tcs_symbols:
			goto cmrkcontinue;
		case tc6_smob:
			switch CAR(ptr) {
			case tc_inport:
			case tc_outport:
				if GCMARKP(ptr) goto cmrkcontinue;
				else if (!CLOSEDP(ptr)) {
				  gc_ports_collected++;
				  close_port(ptr);
				}
			default:
				;
			}
		case tcs_subrs:
			continue;
		default:
			DEFER_SIGINT;
			putc('(',stdout);
			iprint(CAR(ptr),16,stdout);
			fputs(" . ",stdout);
			iprint(CDR(ptr),16,stdout);
			putc(')',stdout);
			fputs(" @ ",stdout);
			iprint(ptr,16,stdout);
			err("unknown type in gc_sweep",ptr);
		}
		++n;
		CAR(ptr) = (SCM)tc_free_cell;
		CDR(ptr) = nfreelist;
		nfreelist = (SCM)ptr;
		continue;
cmrkcontinue:
		CLRGCMARK(ptr);
	}
	gc_cells_collected = n;
	gc_malloc_collected = m;
	freelist = nfreelist;
}
