#| -*-Scheme-*-

$Header: unxpar.scm,v 14.2 88/06/13 11:59:40 GMT cph Rel $

Copyright (c) 1988 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Pathname Parsing -- Unix
;;; package: (runtime pathname-parser)

(declare (usual-integrations))

;;; A note about parsing of filename strings: the standard syntax for
;;; a filename string is "<name>.<version>.<type>".  Since the Unix
;;; file system treats "." just like any other character, it is
;;; possible to give files strange names like "foo.bar.baz.mum".  In
;;; this case, the resulting name would be "foo.bar.baz", and the
;;; resulting type would be "mum".  In general, degenerate filenames
;;; (including names with non-numeric versions) are parsed such that
;;; the characters following the final "." become the type, while the
;;; characters preceding the final "." become the name.

(define (parse-pathname string receiver)
  (let ((end (string-length string)))
    (parse-host string 0 end
      (lambda (host start)
	(parse-device string start end
	  (lambda (device start)
	    (let ((components
		   (let ((components
			  (substring-components string start end #\/)))
		     (append (expand-directory-prefixes (car components))
			     (cdr components)))))
	      (parse-name (car (last-pair components))
		(lambda (name type version)
		  (receiver host
			    device
			    (simplify-directory
			     (parse-directory-components
			      (except-last-pair components)))
			    name
			    type
			    version))))))))))

(define (pathname-as-directory pathname)
  (make-pathname
   (pathname-host pathname)
   (pathname-device pathname)
   (let ((directory (pathname-directory pathname)))
     (let ((file (pathname-unparse-name (pathname-name pathname)
					(pathname-type pathname)
					(pathname-version pathname))))
       (if (string-null? file)
	   directory
	   (simplify-directory
	    (let ((file-components (list (parse-directory-component file))))
	      (cond ((or (null? directory) (eq? directory 'UNSPECIFIC))
		     file-components)
		    ((pair? directory)
		     (append directory file-components))
		    (else (error "Illegal pathname directory" directory))))))))
   false
   false
   false))

(define (parse-host string start end receiver)
  (let ((index (substring-find-next-char string start end #\:)))
    (if (and index (substring-prefix? "::" 0 2 string index end))
	(receiver (substring string start index) (+ index 2))
	(receiver false start))))

(define (parse-device string start end receiver)
  (let ((index (substring-find-next-char string start end #\:)))
    (if index
	(receiver (substring string start index) (1+ index))
	(receiver false start))))

(define (parse-directory-components components)
  (if (null? components)
      '()
      (cons (if (string-null? (car components))
		'ROOT
		(parse-directory-component (car components)))
	    (map parse-directory-component (cdr components)))))

(define (parse-directory-component component)
  (cond ((string=? "*" component) 'WILD)
	((string=? "." component) 'SELF)
	((string=? ".." component) 'UP)
	(else component)))

(define (expand-directory-prefixes string)
  (if (string-null? string)
      (list string)
      (case (string-ref string 0)
	((#\$)
	 (string-components
	  (get-environment-variable
	   (substring string 1 (string-length string)))
	  #\/))
	((#\~)
	 (let ((user-name (substring string 1 (string-length string))))
	   (string-components
	    (if (string-null? user-name)
		(get-environment-variable "HOME")
		(get-user-home-directory user-name))
	    #\/)))
	(else (list string)))))

(define (parse-name string receiver)
  (let ((start 0)
	(end (string-length string)))
    (define (find-next-dot start)
      (substring-find-next-char string start end #\.))

    (define (find-previous-dot start)
      (substring-find-previous-char string start end #\.))

    (define (parse-version start)
      (cond ((= start end) "")
	    ((substring=? string start end "*" 0 1) 'WILD)
	    ((substring-find-next-char string start end #\*)
	     (substring string start end))
	    (else
	     (let ((n (digits->number (reverse! (substring->list string start
								 end))
				      1 0)))
	       (if (and n (>= n 0))
		   (if (= n 0) 'NEWEST n)
		   (substring string start end))))))

    (if (= start end)
	(receiver false false false)
	(let ((index (find-next-dot start)))
	  (if index
	      (let ((start* (1+ index))
		    (name (wildify string start index)))
		(if (= start* end)
		    (receiver name "" false)
		    (or (let ((index (find-next-dot start*)))
			  (and index
			       (let ((version (parse-version (1+ index))))
				 (and (not (string? version))
				      (receiver name
						(wildify string start* index)
						version)))))
			(let ((index (find-previous-dot start)))
			  (receiver (wildify string start index)
				    (wildify string (1+ index) end)
				    false)))))
	      (receiver (wildify string start end) false false))))))

(define (wildify string start end)
  (if (substring=? string start end "*" 0 1)
      'WILD
      (substring string start end)))

(define (string-components string delimiter)
  (substring-components string 0 (string-length string) delimiter))

(define (substring-components string start end delimiter)
  (define (loop start)
    (let ((index (substring-find-next-char string start end delimiter)))
      (if index
	  (cons (substring string start index)
		(loop (1+ index)))
	  (list (substring string start end)))))
  (loop start))

(define (digits->number digits weight accumulator)
  (if (null? digits)
      accumulator
      (let ((value (char->digit (car digits) 10)))
	(and value
	     (digits->number (cdr digits)
			     (* weight 10)
			     (+ (* weight value) accumulator))))))