;;; -*- Mode: Emacs-Lisp;  -*-
;;; File: x-mouse-base.el
;;; Author: Heinz Schmidt (hws@ICSI.Berkeley.EDU)
;;; Copyright (C) International Computer Science Institute, 1991
;;;
;;; COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY.
;;; It is subject to the terms of the GNU EMACS GENERAL PUBLIC LICENSE
;;; described in a file COPYING in the GNU EMACS distribution or to be obtained
;;; from Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: X-mouse dependent realization of mouse primitives for sky-mouse
;;;* 
;;;* RELATED PACKAGES: sky-mouse.el,
;;;*                   rebinds mouse-keys and set C-h z to mouse help.
;;;*
;;;* HISTORY:
;;;* Last edited: Mar  8 14:39 1992 (hws)
;;;* Created: Wed Nov 28 1990 (hws)
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(provide 'x-mouse-base)

(require 'x-mouse)

;;;----------------------------------------------------------------------------
;;; Selection methods

(defmacro save-excursion-info ()
  '(setq mouse-save-excursion-info
	(list (mark) (point) (current-buffer) (selected-window))))

(defun mouse-set-point (arg)
  "Select window mouse is on, move point and mark to mouse position, i.e.
make the current region empty.
Programmers use this at the begin of Mouse Point Commands, mouse commands that
let the user select a point to operate from there. (Free-hand) Mouse Region
Commands, that require visual feedback to make a reliable selection are started
with the companion commands mouse-drag-point or mouse-extend-drag-point.

Novice mouse command programmers often try to use dragging for the wrong
purposes. Mouse-set-point is also used if the region between mouse and point is
affected and needs not be selected upon command completion, or simply if the
point and mouse cursors provide sufficient feedback. For instance
warping the mouse to point or killing the text between point and mouse
do not require that region to be marked."
  (interactive)
  (save-excursion-info)
  (x-mouse-set-point arg)
  (push-mark (point))
  (sky-state-reset))

(defun mouse-extend-drag-point (arg)
  "Extends the current region to the point under mouse."
  (interactive)
  (x-mouse-set-point arg)
  (exchange-point-and-mark) (sit-for 1)
  (mark-region (mark) (point)) (sit-for 1)
  (exchange-point-and-mark))

(defun mouse-end-drag-point (arg)
  "Extends the current region to the point under mouse."
  (x-mouse-set-point arg)
  (exchange-point-and-mark)
  (mark-region (mark) (point))
  (exchange-point-and-mark))

;;; A simulation no feedback in between
(fset 'mouse-drag-point 'mouse-set-point)
(fset 'mouse-warp-to-point 'scroll-point-to-mouse)

;;;----------------------------------------------------------------------------
;;; Feedback

(defun mark-region (from to)
  "Mark the current region (by flashing) and put the region contents into
the cut buffer."
  (store-cut-buffer from to)
  (exchange-point-and-mark) (sit-for 1) 
  (exchange-point-and-mark))

(defun forget-region ())
(defun cursor-glyph (symbol))

;;;----------------------------------------------------------------------------
;;; (No) Screens under Emacs

(defun redisplay-screen (&rest ignore))

;;;----------------------------------------------------------------------------
;;; Mouse maps

(defvar mouse::global-map (make-keymap)
  "The global mouse map. Used to interpret mouse buttons in buffers that do
not have a local mosue map. Commands are installed in this map by means of
defmouse.")
(defvar mouse-handler-down-hook nil 
  "* A hook to run after processing button down events.")
(defvar mouse-handler-up-hook nil 
  "* A hook to run after processing button up events.")
(defvar helpful-mouse-handling-p nil)

(defun handle-mouse-button (arg)
  "Finds the buffer under the mouse and dispatches to the command possibly
using the buffer local command table."
  (let ((downp (zerop (logand x-mouse-item 4)))
	map handler)
    (save-window-excursion
      (x-mouse-select arg)
      (set-buffer (window-buffer (selected-window)))
      (setq map (or (and (boundp 'mouse::local-map) mouse::local-map)
		    mouse::global-map)))
    ;; No multi-click hint for now, time would be in sec, mouse in char
    ;; giving rise to confusion on boundaries.
    ;;(if (and (> multi-click-timeout 0) downp)
    ;;    (provide-multi-click-hint x-mouse-item (the-time) x-mouse-pos))
    ;; handle command or provide help
    (setq handler (aref map x-mouse-item))
    (if helpful-mouse-handling-p
	(describe-x-mouse-event handler)
      (funcall handler arg))
    ;; execute hooks / decode only if there is a hook
    (if downp
	(if mouse-handler-down-hook (funcall mouse-handler-down-hook arg))
      (if mouse-handler-up-hook (funcall mouse-handler-up-hook arg)))))

(defun create-mouse-map (&optional map)
  "If MAP is non-nil, copies MAP. Otherwise creates an empty mouse map beeping
on button down and ignoring button up."
  (cond (map (copy-keymap map))
	(t (let ((new (make-keymap)))
	     (dolist (c all-down-buttons-decoded)
	       (defmouse new c 'mouse-beep 'mouse-ignore))
	     new))))

(defun mouse-use-local-map (map &optional buffer)
  "Use MAP as the local mouse map in BUFFER (current buffer if omitted).
If MAP is nil, the global mouse map will be used instead."
  (when (and map (not (vectorp map))) (error "Invalid mouse map"))
  (cond ((bufferp buffer)
	 (save-excursion
	   (set-buffer buffer)
	   (if (not (boundp 'mouse::local-map))
	       (make-local-variable 'mouse::local-map))      
	   (setq mouse::local-map map)))
	(t
	 (if (not (boundp 'mouse::local-map))
	     (make-local-variable 'mouse::local-map))
	 (setq mouse::local-map map))))

(defun mouse-local-map () (and (boundp 'mouse::local-map) mouse::local-map))

(defun init-x-mouse-button-handler ()
  (dolist (c all-down-buttons-decoded)
	  (defmouse mouse-map c 'handle-mouse-button 'handle-mouse-button)))

;;;----------------------------------------------------------------------------
;;; decoding x-mouse buttons

(defun encode-button (repr &optional up)
  (symbol-value 
   (intern (if up 
	       (concat "x-button-" (downcase repr) "-up")
	     (concat "x-button-" (downcase repr))))))

(defun decode-button-code (char-num)
  "CHAR-NUM is a button code such as the value of x-mouse-item.
The function returns the corresponding decoded button as a list
(downp btn controlp metap shiftp)."
  (list (zerop (logand char-num 4))     ; downp
	(- 2 (logand char-num 3))	; 0-1-2
	(/= 0 (logand char-num 64))     ; controlp
	(/= 0 (logand char-num 32))     ; metap
	(/= 0 (logand char-num 16))     ; shiftp
  ))
        
;;; (decode-button-code (aref (encode-button "C-M-Right" t) 0))
;;; (encode-button "C-M-Right" t)
;;; (decode-button-code x-mouse-item)

;;;----------------------------------------------------------------------------
;;; HELP

(defun help-with-mouse (&rest ignore)
  "Describes what the mouse feels like under the current key bindings. Successive
mouse actions are described as they occur. Type RET to end the command."
  (interactive)
  (setq helpful-mouse-handling-p t)
  (unwind-protect
      (read-input "Play Mouse (type RET to exit).")
    (setq helpful-mouse-handling-p nil)))

(defun describe-x-mouse-event (handler)
  (apply (function describe-mouse-event)
	 handler (decode-button-code x-mouse-item)))

;;;----------------------------------------------------------------------------

(defun mouse-define-internal (down-command up-command code feedback map)
  ;; x-button-c-m-middle x-button-m-s-right-up x-button-c-m-s-right
  (setq map
	(cond ((eq map ':global) mouse::global-map)
	      ((eq map ':local)
	       (or (and (boundp 'mouse::local-map) mouse::local-map)
		   (mouse-use-local-map
		    (create-mouse-map mouse::global-map))))
	      (t map)))
  (let* ((down-code (encode-button code))
	 (up-code (encode-button code t)))
    (define-key map down-code down-command)
    (define-key map up-code up-command)
    ))

(defvar all-buttons '("Left" "Middle" "Right"))
(defvar all-modifiers '("" "C-" "M-" "S-" "C-M-" "C-S-" "M-S-" "C-M-S-"))
(defun all-down-buttons-decoded ()
  (let (buttons)
    (dolist (m all-modifiers)
      (dolist (b all-buttons)
	(push (concat m b) buttons)))
    (reverse buttons)))
(defvar all-down-buttons-decoded (all-down-buttons-decoded))

(defun mouse-get-button-spec-and-install-command ()
  (cond ((fboundp 'x-popup-menu)
	 (let ((down (nth 0 *mouse-installation-spec*))
	       (up (nth 1 *mouse-installation-spec*))
	       (code
		(x-popup-menu 
		 nil
		 (list "Select A Button"
		       (cons "Select A Button" 
			     (mapcar (function (lambda (b) (cons b b)))
				     all-down-buttons-decoded))))))	   
	   (when code (mouse-define-internal down up code nil ':global)
		 (message "...installed %s/%s on %s/-Up"down up code))))
	(t (beep)
	   (message "x-menus are not available. Can't get button spec."))))

;;;----------------------------------------------------------------------------
;;; X cut buffer interface

(fset 'cut-buffer (function x-get-cut-buffer))
(fset 'mouse-paste-cut-buffer (function x-paste-text))

(defun store-cut-buffer (from to)
  (x-store-cut-buffer (buffer-substring from to)))

;;;----------------------------------------------------------------------------
;;;
;;; Focus, ignore for emacs, is fix anyway
;;;

;;; Set-focus, i.e. kbp input stays with current screen (or window in
;;; plain emacs). Requires epoch-mouse.el to respect current-focus-screen.

(defvar current-focus-screen nil "The currently focused screen.")
(defun set-kbd-focus (screen))
(defun release-kbd-focus ())

;;;----------------------------------------------------------------------------
;;;
;;; Global Mouse Installation

(defmouse :global  "Left" 'mouse-drag-point 'mouse-end-drag-point)
(defmouse :global  "Middle" 'mouse-paste-cut-buffer 'mouse-ignore)
(defmouse :global  "Right" 'mouse-extend-drag-point 'mouse-end-drag-point)

;;; Exchange Left and Right only if we run under Openlook (olwm).
;;; Middle is the ADJUST/EXTEND key of choice there.

(cond
 ((equal (getenv "WINMGR") "olwm")
  (defmouse :global  "Middle" 'mouse-extend-drag-point 'mouse-end-drag-point)
  (defmouse :global  "Right" 'mouse-paste-cut-buffer 'mouse-ignore)
  ))

;;; CONSTRAINED marks/kills : Shift
;;; Make it easy to change your mind about what is marked.
;;;  toggle between freehand drag-extend and mark-thing by shift up-down.
;;;  use same modifier chord (here shift only) for save/kill/yank.

(defmouse :global  "S-Left" 'mouse-mark-thing 'mouse-ignore)
(defmouse :global  "S-Middle" 'mouse-save-kill-yank 'mouse-ignore)

  ;;; right reserved for visual kill-ring in separate screen
  ;;;               "more-above" scrolls up, 
  ;;;              ( one-line-per-kill yanks or yank-pops )*
  ;;;               "more-below" scrolls down

;;; FROM-MOUSE-TO-FOCUS: Meta
;;; Make the std sequence easy: meta-down,Left,Middle*,Right,Left,meta-up


(defmouse :global  "M-Left" 'mouse-toggle-focus-screen 'mouse-ignore)
(defmouse :global  "M-Middle" 'mouse-yank-thing-to-point 'mouse-ignore)
(defmouse :global  "M-Right" 'mouse-warp-to-point 'mouse-ignore)

;;; MORE MICE POWER: control

(defmouse :global  "C-Left" 'mouse-set-point-force-kbd 'mouse-ignore)
(defmouse :global  "C-Middle" 'mouse-extend-drag-point 'mouse-kill-region)

;;; don't do it the other way round else up transition is interpreted in other map
(defmouse :global  "C-Right" 'mouse-set-point 'mouse-scroll-mode)

;;;
;;; EXTENDED MOUSE: control-shift (sort of control-x of the mouse)
;;;
;;; requires habit since with most keys irrelevant,
;;; use for rare things.

(defmouse :global  "C-S-Left" 'x-buffer-menu 'mouse-ignore)
(defmouse :global  "C-S-Middle" 'x-help 'mouse-ignore)
;; right: most recent commands?

;;;
;;; TEMPORARY MOUSE: meta-shift
;;;

(defmouse :global  "M-S-Left" 'mouse-set-point 'mouse-execute-kbd-macro)
(defmouse :global  "M-S-Middle" 'mouse-set-point 'mouse-execute-at-point)
(defmouse :global  "M-S-Right" 'mouse-drag-point 'mouse-execute-on-region)

;;; Mode MOUSE: meta-control
;;; Why comes meta-control Mouse to mind for mark-thing and yank-thing?!
;;; Make sure it works or does not do any damage if hit accidentally.

(defmouse :global  "C-M-Left" 'mouse-mark-thing 'mouse-ignore)
(defmouse :global  "C-M-Middle" 'mouse-yank-thing-to-point 'mouse-ignore)
(defmouse :global  "C-M-Right" 'mouse-drag-point 'mouse-fill-or-indent)

;;;
;;; INSANE MOUSE: meta-control-shift
;;;

;; Left describe mouse?
(defmouse :global "C-M-S-Middle" 'mouse-set-point 'help-with-mouse-tutorial)
(defmouse :global "C-M-S-Right" 'mouse-message 'mouse-message)


;;;----------------------------------------------------------------------------
;;; Dependent upon...
;;; Scrolling is in a separate module now

(autoload 'mouse-scroll-mode "scrollm" "Starts mouse scrolling."  t)

(init-x-mouse-button-handler)

