;;; -*- Mode: Emacs-Lisp;  -*-
;;; File: sdb-mode.el
;;; Author: Heinz Schmidt (hws@ICSI.Berkeley.EDU)
;;; Copyright (C) International Computer Science Institute, 1991
;;;
;;; COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY.
;;; It is subject to the terms of the GNU EMACS GENERAL PUBLIC LICENSE
;;; described in a file COPYING in the GNU EMACS distribution or to be obtained
;;; from Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: SDB mode, a wrap around gdb for the Sather debugger.
;;;*
;;;* RELATED PACKAGES: gdb-mode.el
;;;*
;;;* HISTORY:
;;;* Last edited: Mar  8 14:43 1992 (hws)
;;;*  Sep  6 13:20 1991 (hws): extend visit this to include stack frames
;;;*  Jun  2 19:47 1991 (hws): adapt to changed sdb output format.
;;;*  May 13 00:28 1991 (hws): allow for 1 line difference between sdb and
;;;*                           gdb break; make C-x SP use sdb-break in Sather
;;;*  Apr  7 17:22 1991 (hws): added mouse interface to follow attributes
;;;* Created: Sun Apr  7 10:46:01 1991 (hws)
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(load-library "gdb")

;;; do we have mouse support?
(defvar sdb-mouse-p
  (and (featurep 'sky-mouse) (or (featurep 'x-mouse) (featurep 'epoch-mouse-base))))

(defvar sdb-prompt-pattern "^(.*sdb) *"
  "A regexp to recognize the prompt for sdb.")

(defvar sdb-mode-map nil
  "Keymap for sdb-mode.")

(defvar sdb-mouse-map (if sdb-mouse-p (create-mouse-map mouse::global-map))
  "Local mouse map for sdb, inherits from global mouse-map.")

(if sdb-mode-map
   nil
  (setq sdb-mode-map (copy-keymap gdb-mode-map))
  (define-key sdb-mode-map "\C-ca" 'sdb-show-attributes)
  (define-key sdb-mode-map "\C-cl" 'sdb-show-location)
  (define-key sdb-mode-map "\C-cr" 'sdb-show-routines)
  (define-key sdb-mode-map "\C-cv" 'sdb-show-variables))

(defun sdb-mode ()
  "Major mode for interacting with an inferior sdb process.
Sdb is gdb (the GNU debugger) with Sather extensions. 
The following commands are available:

\\{sdb-mode-map}

\\[gdb-display-frame] displays in the other window
the last line referred to in the sdb buffer.

\\[gdb-step],\\[gdb-next], and \\[gdb-nexti] in the sdb window,
call sdb to step,next or nexti and then update the other window
with the current file and position.

If you are in a source file, you may select a point to break
at, by doing \\[gdb-break].

Commands:
Many commands are inherited from shell mode. 
Additionally we have:

\\[gdb-display-frame] display frames file in other window
\\[gdb-step] advance one line in program
\\[gdb-next] advance one line in program (skip over calls).
\\[send-gdb-command] used for special printing of an arg at the current point.
C-x SPACE sets break point at current line.

If the Sky-mouse package is loaded, the following mouse functions are 
available:
 M-S-Left    sdb-mouse-visit-thing (attribute or array index)
 M-S-Middle  sdb-mouse-show-attributes (of self)
 M-S-Right   sdb-mouse-leave."
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'sdb-mode)
  (setq mode-name "Inferior Sdb")
  (setq mode-line-process '(": %s"))
  (use-local-map sdb-mode-map)
  (make-local-variable 'last-input-start)
  (if sdb-mouse-p (mouse-use-local-map sdb-mouse-map))
  (setq last-input-start (make-marker))
  (make-local-variable 'last-input-end)
  (setq last-input-end (make-marker))
  (make-local-variable 'gdb-last-frame)
  (setq gdb-last-frame nil)
  (make-local-variable 'gdb-last-frame-displayed-p)
  (setq gdb-last-frame-displayed-p t)
  (make-local-variable 'gdb-delete-prompt-marker)
  (setq gdb-delete-prompt-marker nil)
  (make-local-variable 'gdb-filter-accumulator)
  (setq gdb-filter-accumulator nil)
  (make-local-variable 'shell-prompt-pattern)
  (setq shell-prompt-pattern sdb-prompt-pattern)
  (run-hooks 'shell-mode-hook 'gdb-mode-hook))

(defvar sdb-command-name "sdb" "Pathname for executing sdb.")

(defun sdb (path)
  "Run sdb on program FILE in buffer *sdb-FILE*.
The directory containing FILE becomes the initial working directory
and source-file directory for SDB.  If you wish to change this, use
the SDB commands `cd DIR' and `directory'."
  (interactive "FRun sdb on file: ")
  (setq path (expand-file-name path))
  (let ((file (file-name-nondirectory path)))
    (switch-to-buffer (concat "*sdb-" file "*"))
    (setq default-directory (file-name-directory path))
    (or (bolp) (newline))
    (insert "Current directory is " default-directory "\n")	
    (make-shell (concat "sdb-" file) sdb-command-name nil 
		"-fullname" "-cd" default-directory 
		file)
    (sdb-mode)
    (set-process-filter (get-buffer-process (current-buffer)) 'gdb-filter)
    (set-process-sentinel (get-buffer-process (current-buffer)) 'gdb-sentinel)
    (gdb-set-buffer)))

(defun gdb-set-buffer ()
  (cond ((or (eq major-mode 'gdb-mode) (eq major-mode 'sdb-mode))
	 (setq current-gdb-buffer (current-buffer)))))

; Seems no longer necessary, gdb-break is just fine; but keep until this is final.
;
;(defun sdb-break ()
;  "Set SDB breakpoint at this source line."
;  (interactive)
;  (let ((file-name (file-name-nondirectory buffer-file-name))
;	(line (save-restriction
;		(widen)
;		(count-lines 1 (point))))) ; here the logic differs from gdb, which
;					; breaks after the line, is that what we want?
;    (send-string (get-buffer-process current-gdb-buffer)
;		 (concat "break " file-name ":" line "\n"))))
;
;;;; make the sdb-break available ONLY on sather files 
;
;(if (not (featurep 'sather)) (require 'sather))
;
;(define-key sather-mode-map "\C-x " 'sdb-break)

(defun sdb-send-cmd (string)
  (end-of-buffer)
  (let ((obuf (current-buffer)))	; save-excursion is not ok with process-buffers
    (set-buffer current-gdb-buffer)
    (end-of-buffer)
    (insert string)
    (shell-send-input)
    (set-buffer obuf)))
;; (send-string (get-buffer-process current-gdb-buffer) string)

(defun sdb-show-attributes ()
  "Show the attributes of self."
  (interactive)
  (sdb-send-cmd "show"))

(defun sdb-show-routines ()
  "Show the routines of self."
  (interactive)
  (sdb-send-cmd "rshow"))

(defun sdb-show-location ()
  "Show the current location in visit stack."
  (interactive)
  (sdb-send-cmd "location"))

(defun sdb-show-variables ()
  "List the state of sather debugging."
  (interactive)
  (sdb-send-cmd "sdvars"))


;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Mousing around

(cond (sdb-mouse-p
       (defmouse sdb-mouse-map "M-S-Left" 'sdb-mouse-visit-thing 'mouse-ignore)
       (defmouse sdb-mouse-map "M-S-Middle" 
	 '(lambda (point) "Show attributes." (interactive)
	    (sdb-show-attributes))
	 'mouse-ignore)
       (defmouse sdb-mouse-map "M-S-Right" 
	 '(lambda (point) "Leave current object." (interactive)
	    (sdb-send-cmd "leave"))
	 'mouse-ignore)))

(defun wait-and-to-last-output ()
  (sit-for 1)
  (backward-sexp) (re-search-backward "(sdb)" nil t) (forward-sexp))

(defun sdb-mouse-visit-thing (point)
  "Visit the thing pointed to. Clicking to the index of an array element
visits that element. Clicking to the index of a stack frame visits that frame."
  (interactive)
  (mouse-mark-thing point)
  (let (thing 
	nump framep
	(beg (region-beginning))
	(end (region-end)))
    (save-excursion
      (goto-char beg)
      (cond ((looking-at "\#") (setq framep t beg (1+ beg)))
	    ((looking-at "[0-9]") 
	     (if (save-excursion (backward-char 1) (looking-at "\#"))
		 (setq framep t)
	       (setq nump t))))
      (save-restriction
	(narrow-to-region beg end)
	(setq thing 
	      (buffer-substring
	       (point-min)
	       (if (re-search-forward "\\:" nil t)
		   (progn (backward-char 1) (mark-region (point-min) (point)) (point))
		 (point-max))))))
    (end-of-buffer)
    (cond (framep (sdb-send-cmd (concat "frame " thing)))
	  (t 
	   (sdb-send-cmd (concat "visit " (if nump "[" "") thing (if nump "]" "")))
	   ;; show it
	   (wait-and-to-last-output)
	   (cond ((re-search-forward "Can't" nil t)
		  (end-of-buffer))
		 (t 
		  (sdb-send-cmd "show")
		  (wait-and-to-last-output)
		  (sit-for 0)
		  (if (re-search-forward "Array(" nil t)
		      (progn (message "seen") (sdb-send-cmd "array[*]")))))
	   (end-of-buffer)
	   ))))

