;;; -*- Mode: Emacs-Lisp;  -*-
;;; File: poor-mans-mouse.el
;;; Author: Heinz Schmidt (hws@ICSI.Berkeley.EDU)
;;; Copyright (C) International Computer Science Institute, 1991
;;;
;;; COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY.
;;; It is subject to the terms of the GNU EMACS GENERAL PUBLIC LICENSE
;;; described in a file COPYING in the GNU EMACS distribution or to be obtained
;;; from Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: A poor substitute for some of the mouse commands provided by 
;;;*           sky-mouse. Sky-mouse loads this when nothing else is around
;;;*           for instance when you login from modem.
;;;* 
;;;* RELATED PACKAGES: sky-mouse.el
;;;*
;;;* HISTORY:
;;;* Last edited: May 24 00:55 1991 (hws)
;;;* Created:  Wed Nov 28 1990 (hws)
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(provide 'poor-mans-mouse)

;;; Cut buffer, allow multiple Emacs windows under MacTerminal to communicate
;;; (without moving the hand to the move.-:)

(defun paste-cut-buffer (arg)
  (interactive "p")
  (insert-file "~/.poor-mans-cut-buffer")
  )

(defun store-cut-buffer (from to)
  (write-region from to "~/.poor-mans-cut-buffer" nil))

(defun copy-to-cut-buffer ()
  "Copies the current region to the cut-buffer."
  (interactive)
  (when (/= (mark) (point))
    (store-cut-buffer (region-beginning) (region-end))))

;;; A simple version of yanking to focus:
;;; make two window use one as target and the other as source
;;; where you switch buffers and mark and yank to other window.
(defun insert-other-window (&optional text) 
  "Put text in other window, if NIL insert region."
  (interactive)
  (if (null text) (setq text (buffer-substring (region-beginning)
					       (region-end))))
  (other-window 1)
  (insert text)
  (other-window 1))

(defun insert-sp-other-window ()
  "Insert a Space in other window."
  (interactive)
  (insert-other-window " "))

(defun insert-nl-other-window ()
  "Insert a Newline in other window."
  (interactive)
  (insert-other-window "\n"))

;;; Don't try to be smart with whitespace. Unlike mouse-yank-to-point
;;; the cursor is moved over whitespace and determines a region while
;;; mouse must extend an unexact point. While collecting interesting
;;; things point includes whitespace. User may specify explicit spacing
;;; if necessary.
(defun yank-to-other-window ()
"Yank the current region to other window. Point will return
to here so it can continue to mark and yank."
  (interactive)
  (let ((white-space (looking-at " \t\n")))
    ;; don't concat to previous yank
    (setq last-command 'anything-but-yank
	  this-command 'mouse-yank-thing-to-point) 
    (copy-region-as-kill (mark) (point))
    (insert-other-window)
    (push-mark (point))))

(defun mark-thing-no-mark (arg)
  "Mark thing at point."
  (interactive "d")
  (mark-thing arg t) ; no marking 
  )

;;; Make sure user-defined mouse commands work with M-x.
;;; Provide dummy versions of the basic functions called.

(defun mark-region (from to)
  "Mark the current region (by flashing) and put the region contents into
the cut buffer."
  (store-cut-buffer from to)
  (exchange-point-and-mark) (sit-for 1) 
  (exchange-point-and-mark)  )
(defun forget-region (&rest ignore))
(defun mouse-end-drag-point (&rest ignore) )
(defun mouse-set-point (&rest ignore))
(defun mouse-drag-point (&rest ignore))

;;; Provide the key-bindings for a minor mode so we do not
;;; affect existing bindings automatically.
;;; do not provide mark-thing & save/kill/yank -- std keys work just fine

(defvar 
  *poor-mouse-key-bindings*
  '(("\C-c\C-w" . copy-to-cut-buffer)
    ("\C-c\C-y" . paste-cut-buffer)
	
    ("\C-c\C-@" . mark-thing-no-mark)

    ("\C-c\C-c" . yank-to-other-window)
    ("\C-c "    . insert-sp-other-window)
    ("\C-c
"   . insert-nl-other-window)
    ("\C-c\n"   . insert-nl-other-window)

    ("\C-c\C-q"  . fill-or-indent)
    ("\C-c%"    . to-percent-linewise)

    )
  "* Alist of key bindings (key . function) for Poor-Mouse minor mode.")

(defvar *poor-mouse-old-key-bindings*)

(defvar poor-mouse-mode nil)
(make-variable-buffer-local 'poor-mouse-mode)
(setq-default poor-mouse-mode nil)

(defvar poor-mouse-mode-hooks nil 
  "*Hooks run when switching to poor-mouse mode.")


(push '(poor-mouse-mode " Poor-Mouse") minor-mode-alist)

(defun poor-mouse-mode ()
  "Minor mode for simulating mouse commands by key hits.
C-c C-w  --  copies to region to cut buffer (across Emacs processes)
C-c C-y  --  yanks cut buffer to point

C-c C-@  --  mark-thing
C-c C-c  --  copy (yanks) over to other window
C-c RET  --  insert Newling over in other window
C-c SP   --  insert Space over in other window

C-c C-q  --  fill-or-indent
"
  (interactive)
  (cond (poor-mouse-mode
	 (local-set-keys-from-alist *poor-mouse-old-key-bindings*)
	 (setq poor-mouse-mode nil))
	(t 
	 (setq *poor-mouse-old-key-bindings*
	       (local-set-keys-from-alist *poor-mouse-key-bindings*))
	 (setq poor-mouse-mode t)))
  (run-hooks poor-mouse-mode-hooks))

(defun local-set-keys-from-alist (bindings-alist)
  "Bind keys locally according to alist (key . fun) and return
old bindings."
  (let (old-bindings)
    (dolist (item bindings-alist)
      (let* ((key (car item))
	     (sym (cdr item))
	     (old-binding (local-key-binding key)))
	(if old-binding
	    (push (cons key old-binding) old-bindings))
	(local-set-key key sym)))
    old-bindings))
