;;;
;;; minbuf.el
;;;
;;; make the minibuffer screen stay attached to the bottom
;;; of a particular window. the minibuffer takes on the same
;;; width and font as the screen it's attached to; it's height
;;; is unaltered.
;;;
;;; the function "anchor-minibuffer-to-selected-screen" can be used
;;; interactively to attach the minibuffer to the current screen
;;;
;;; if the variable *minibuffer-sticks-to-selected-screen-p* is
;;; non-null, the minibuffer follows the focus (i.e. it attaches
;;; itself to whichever screen has the input focus).
;;;
;;; this file should be loaded after epoch.el, as it hacks the
;;; event-alist and handle-focus-change
;;;
;;; Original author unknown.
;;; Contributed by Dan L. Pierson, pierson@encore.com
;;; Reworked by Alan Carroll and Chris Love
;;; Additions by Heinz W. Schmidt (hws@ICSI.Berkeley.EDU)
;;;* HISTORY:
;;;* Last edited: Jan 16 14:43 1992 (hws)
;;;*  Jun 30 17:50 1991 (hws): remove misleading docu and unused defvar, avoid
;;;*                           multi positioning at startup (move,resize,select)
;;;*  May 26 22:48 1991 (hws): anchor resize invisibly
;;;*  May 20 09:05 1991 (hws): reset-minibuffer on first anchor action
;;;*  May  7 11:51 1991 (hws): reset-minibuffer command added
;;;                             and auto-deiconify minibuffer

(require 'mini-cl) 

(defvar *minibuffer-anchor-screen* (current-screen)
  "The screen that the minibuffer sticks to.")

(defvar *minibuffer-sticks-to-selected-screen-p* t
  "If non-null, causes the minibuffer to follow the input focus.")

(defvar *minibuffer-anchor-inhibit* nil)

;;; When this file is loaded the current screen may not be exposed.
;;; So let's postpone the proper setting until the first time, the minibuffer
;;; really is to move to the right place.

(defvar first-time-after-startup t)

(defun handle-move-event (type event screen)
  (setq *minibuffer-anchor-screen* (current-screen))
  (if (and *minibuffer-sticks-to-selected-screen-p*
	   (not first-time-after-startup))
    (cond
      ((eq screen *minibuffer-anchor-screen*)
	(move-minibuffer-screen-with-anchor-screen)
	(setq *minibuffer-anchor-inhibit* nil)
      )
      ((and (not *minibuffer-anchor-inhibit*) (eq screen (minibuf-screen)))
	(move-minibuffer-screen-with-anchor-screen)
	(setq *minibuffer-anchor-inhibit* t)
      )
      (t (setq *minibuffer-anchor-inhibit* nil)
      )))
  (on-event::handler type event screen))

(push-event 'move 'handle-move-event)

(defun handle-resize-event (type event screen)
  (setq *minibuffer-anchor-screen* (current-screen))
  (and *minibuffer-sticks-to-selected-screen-p*
       (not first-time-after-startup)
       (or (eq screen *minibuffer-anchor-screen*)
	   (eq screen (minibuf-screen)))
       (resize-minibuffer-screen-with-anchor-screen))
  (on-event::handler type event screen))

(push-event 'resize 'handle-resize-event)

;;; --------------------------------------------------------------------------
(defvar minibuf-delta-x 0)
(defvar minibuf-delta-y 0)

(defun backing-screen-info (&optional win)
"Return the screen information for the backing X-window of the given screen"
  (setq win (or win (current-screen)))
  (let ((parent nil)
	(root   nil)
	(wlist  nil))
    (while (and (setq wlist (query-tree win))
		(setq parent (cadr wlist))
		(setq root   (car wlist))
		(not (equal root parent)))
      (setq win parent))
    (and parent (screen-information win))))

(defun wm-location-delta (&optional win)
"Given optional SCREEN-OR-XWIN, this function returns a list of (DX DY)
for how much the window manager changes move-screen requests."
  (setq win (or win (current-screen)))
  (let
    (
      (start-info (screen-information win))
      x y dx dy info
    )
    (move-screen 0 0 win)		;move it to a fixed location
    (setq info (screen-information win)) ; where did it actually end up?
    (setq dx (- (car info)))
    (setq dy (- (cadr info)))
    (move-screen (+ (car start-info) dx) (+ (cadr start-info) dy) win)
    (setq start-wm-loc-delta (list dx dy))
  ))

;;; Allright, calculate the actual deltas
;;; Things to account for -
;;; 1. Size of WM borders
;;; 2. Size of X window borders
;;; 3. WM fudging of move requests
;;;

(defun reset-minibuffer ()
  "Resets the minibuffer state. Recalculates the window manager
specific information like title-bar offsets etc. Also
deiconizes the minibuffer and puts it back in place."
  (interactive)
  (requires-screen-visible (minibuf-screen))
  (let*
      ((minfo (screen-information (minibuf-screen)))
       (mbinfo (backing-screen-info (minibuf-screen)))
       ;; When this is called the minibuffer may be
       ;; the selected screen, but then 
       (screen (if (eq (current-screen) (minibuf-screen))
		   (car (screen-list))
		 (current-screen)))
       (einfo (screen-information screen))
       (ebinfo (backing-screen-info screen))
       (wm-fudge (wm-location-delta (minibuf-screen))))
    (setq minibuf-delta-x
	  (+
	   ;; for now, just line up the edge of the edit screen with the edge of
	   ;; the minibuffer screen, so only account for the WM fudge factor.
	   (car wm-fudge)
	   ))
    (setq minibuf-delta-y
	  (+
	   (cadr wm-fudge)
	   ;; size of the minibuffer title
	   (- (nth 1 minfo) (nth 1 mbinfo))
	   ;; the size of the bottom bar on the edit screen
	   (- (+ (nth 1 ebinfo) (nth 3 ebinfo)) (+ (nth 1 einfo) (nth 3 einfo)))
	   (nth 4 mbinfo)		; borders
	   (nth 4 ebinfo)
	   ))))

;;; --------------------------------------------------------------------------

(defun requires-screen-visible (win)
  (if (not (screen-mapped-p win)) (epoch::iconify-screen win)))

(defun move-minibuffer-screen-with-anchor-screen ()
  (requires-screen-visible (minibuf-screen))
  (let* ((mbscreen (minibuf-screen))
	 (info (screen-information *minibuffer-anchor-screen*))
	 )
    (move-screen
     (+ (car info) minibuf-delta-x)
     (+ (nth 1 info) (nth 3 info) minibuf-delta-y)
     mbscreen)
    ))

(defun resize-minibuffer-screen-with-anchor-screen ()
  (requires-screen-visible (minibuf-screen))
  (let* ((char-width (epoch::screen-width *minibuffer-anchor-screen*))
	 (mbscreen (minibuf-screen))
	 (font (car (font nil *minibuffer-anchor-screen*)))
	 (mbfont (car (font nil mbscreen)))
	 )
    (move-minibuffer-screen-with-anchor-screen)
    ;; make the minibuffer have the same font as the screen it's attached to
    (when (and font mbfont (not (string= font mbfont)))
      (font font mbscreen))
    (change-screen-size char-width nil mbscreen)
    ))

(defvar *unmap-while-anchor-minibuffer* nil
  "Under some window managers unmapping the minibuffer while recomputing
its position makes it a visually nicer atomic operation without visible
intermediate steps (e.g. olwm). Some may behave frantic in this 
situation.")

(defun anchor-minibuffer-to-selected-screen ()
  "Make the minibuffer stick to the current screen."
  (interactive)
  (requires-screen-visible (minibuf-screen))
  (cond (first-time-after-startup
	 (reset-minibuffer)
	 (resize-minibuffer-screen-with-anchor-screen)
	 (setq first-time-after-startup nil)))
  (cond ((not (eq *minibuffer-anchor-screen* (current-screen)))
	 (setq *minibuffer-anchor-screen* (current-screen))
	 (and *minibuffer-sticks-to-selected-screen-p*
	      (progn (if *unmap-while-anchor-minibuffer* (unmap-screen (minibuf-screen)))
		     (resize-minibuffer-screen-with-anchor-screen)
		     (if *unmap-while-anchor-minibuffer* (map-screen (minibuf-screen))))))))

;;Avoid this, there are other hooks around, too
;;(setq *select-screen-hook* 'anchor-minibuffer-to-selected-screen)
(if (not (memq 'anchor-minibuffer-to-selected-screen
	       *select-screen-hook*))
    (push 'anchor-minibuffer-to-selected-screen *select-screen-hook*))

(defun iconify-emacs ()
  "Iconifies all screens of this emacs process."
  (interactive)
  (dolist (s (screen-list))
    (iconify-screen s))
  (iconify-screen (minibuf-screen)))

;;; Avoid widow minibuffers when an epoch screen is iconified.
;;;
;;; Redefinition of iconify-screen does not work, it is not called when
;;; window is iconified from within WMGR.
 
(push-event 'map 'handle-map-event)
(defun handle-map-event (type event screen)
  (when (eq screen *minibuffer-anchor-screen*)
    ;; at this point the screen is already (un)mapped.
    (cond ((not (screen-mapped-p screen))
	   (iconify-screen (minibuf-screen)))
	  (t 
	   ;; warp mouse close to screen cursor, but don't loose it
	   (let ((coords (epoch::query-cursor)))
	     (epoch::warp-mouse (1+ (car coords)) (1+ (cdr coords)) ;char line screen
				(current-screen))))))
  (on-event::handler type event screen))
