#!/usr/bin/env perl
use 5.012005;
use ExtUtils::MakeMaker;
use ExtUtils::Manifest;
use IPC::Cmd qw/can_run/;
use Cwd;
use File::Copy;
use Carp;

# Since MANIFEST.SKIP(generated by make-manifest-skip.pl) is being ignored by 
# ExtUtils::Manifest::manicheck which checks all files, we replace the 
# manicheck-related functions here.
# (this is for EUMM < 7.02 . this problem was confirmed on EUMM 6.98)
*{ExtUtils::Manifest::manicheck}    = sub {};
*{ExtUtils::Manifest::_check_files} = sub {};
$ExtUtils::Manifest::Debug = 1;

my $SLEUTHKIT_BASE    = "";
my $SLEUTHKIT_HEADERS = "";
my $SLEUTHKIT_LIBS    = "";
my $LIBEWF_BASE       = "";
my $AFFLIB_BASE       = "";
my $AFFLIB_DEPDIR     = "";
my $INC_PATH          = "";
my $CC                = "";
my $CCFLAGS           = "";
my $LDDFLAGS          = "";

## Make program. There are at least 3 choices here: make, nmake, dmake
## make - GNU make
## nmake - Visual Studio's make
## dmake - Strawberry Perl's make
my $MAKE_PROG         = "";


## OS check, currently only Linux is supported
if($^O =~ /^(linux|freebsd|netbsd|darwin)$/ ) {
    print "$^O OS detected, continuing ... \n";

    ## Check if certain binaries are present
    for my $bin (qw/autoconf automake libtool libtoolize gcc g++ pkg-config/) {
        croak "[ERROR] $bin not found" if !can_run($bin);
    };
    ## Check if libfuse is present
    my $libfuse_version = `pkg-config --modversion fuse 2> /dev/null`;
    croak "[ERROR] libfuse needs to be installed" if !$libfuse_version;

    $CC                = "g++";
    $CCFLAGS           = "-Wall";
    $SLEUTHKIT_BASE    = getcwd()."/sleuthkit";
    $SLEUTHKIT_HEADERS = "$SLEUTHKIT_BASE/";
    $SLEUTHKIT_LIBS    = "$SLEUTHKIT_BASE/tsk/.libs";
    $LIBEWF_BASE       = "$ENV{PWD}/libewf";
    $AFFLIB_BASE       = "$ENV{PWD}/AFFLIBv3";
    $AFFLIB_DEPDIR     = "$ENV{PWD}/AFFLIBv3/sandbox";
    $INC_PATH          = "-I$SLEUTHKIT_HEADERS -I.";
    $MAKE_PROG         = "make";
    $LDDFLAGS          = " -Wl,--no-as-needed -shared  -L$SLEUTHKIT_LIBS -ltsk -lstdc++";
} elsif($^O =~ /^(cygwin|msys)$/) {
    croak <<_ERRMSG;
[ERROR] Currently, on cygwin/msys, the XS code compiles, but linking 
        to libtsk.a is not working.
        
        If you're interested in fixing this, please fork and send a pull
        request on Github
        https://github.com/wsdookadr/Tsk-XS
        or send a patch via the CPAN issue tracker
        https://rt.cpan.org/Public/Dist/Display.html?Name=Tsk
_ERRMSG

    ## Check if certain binaries are present
    for my $bin (qw/autoconf automake libtool libtoolize gcc g++/) {
        croak "[ERROR] $bin not found" if !can_run($bin);
    };

    my $CWD = getcwd();
    my $uname_output = `uname -a`;
    if($uname_output =~ /x86_64 Cygwin/ && $^O eq "cygwin") {
        system("patch sleuthkit/tsk/auto/sqlite3.c < patch/cygwin64-sqlite.patch");
    }
    $CWD =~ s{\/}{\\}g;
    $CC                = "g++";
    #$CCFLAGS           = "-Wall";
    $SLEUTHKIT_BASE    = "$CWD\\sleuthkit";
    $SLEUTHKIT_HEADERS = "$SLEUTHKIT_BASE";
    $SLEUTHKIT_LIBS    = "$CWD\\win_binaries";
    $INC_PATH          = " \"-iquote$SLEUTHKIT_HEADERS\" \"-iquote$CWD\" ";
    ## should be discarded(line that copies header)
    #copy("patch\\tsk_incs.h","sleuthkit\\tsk\\tsk_incs.h");
    $MAKE_PROG         = "make";
    $LDDFLAGS          = " -shared  -L$SLEUTHKIT_LIBS \"$SLEUTHKIT_LIBS\\libtsk.lib\" -lstdc++";
} elsif($^O eq "MSWin32") {
    ## Note: autotools are not needed here because it's expected that
    ## sleuthkit will be compiled separately using MSYS2.
    ## The custom Makefile targets are also not intended for this particular
    ## environment as they are the ones meant to build sleuthkit using
    ## autotools. 
    croak <<_ERRMSG;
[ERROR] Currently, on Windows, using Strawberry Perl, the XS code compiles but
        linking to libtsk.a is not working. This was tested
        with Strawberry Perl 5.18.4(on 64 and 32 bit) and sleuthkit was
        built using MSYS(for 32-bit) and MSYS2(for 64-bit).

        If you're interested in fixing this, please fork 
        https://github.com/wsdookadr/Tsk-XS and send
        a pull request on github, or send a patch via the CPAN issue
        tracker https://rt.cpan.org/Public/Dist/Display.html?Name=Tsk 

_ERRMSG

    my $CWD = getcwd();
    $CC                = "g++";
    $CCFLAGS           = "-v -Wall";
    $CCFLAGS           = "";
    $SLEUTHKIT_BASE    = "c:\\MinGW\\msys\\1.0\\home\\u\\sleuthkit";
    $SLEUTHKIT_HEADERS = "$SLEUTHKIT_BASE";
    $SLEUTHKIT_LIBS    = "$SLEUTHKIT_BASE\\tsk\\.libs\\";
    $INC_PATH          = " \"-I$SLEUTHKIT_HEADERS\" ";
    $MAKE_PROG         = "dmake";
    $LDDFLAGS          = " -shared  \"-L$SLEUTHKIT_LIBS\" -ltsk -lstdc++";
    if(!(-f "$SLEUTHKIT_LIBS\\libtsk.a")) {
        croak <<_ERRMSG;
libtsk.a not found. sleuthkit should be compiled using MSYS2.
_ERRMSG
    }
} else {
    croak "[ERROR] Tsk is not yet supported on $^O";
}

## Paths to dependencies


WriteMakefile(
    INST_ARCHLIB      => './blib/arch',
    INST_LIB          => './blib/lib',
    NAME              => 'Tsk',
    VERSION_FROM      => 'lib/Tsk.pm', 
    LICENSE           => "perl",
    PREREQ_PM         => {}, 
    ($] >= 5.005 ?     
      (ABSTRACT_FROM  => 'lib/Tsk.pm',
       AUTHOR         => 'Stefan Petrea <stefan@garage-coding.com>') : ()),
    #LIBS              => ["-L$SLEUTHKIT_LIBS -L$PBREWPATH -ltsk -lstdc++"], # e.g., '-lm'
    LIBS              => "-lstdc++ -ltsk", # e.g., '-lm'
    #LINKTYPE          => 'static',
    #XSOPT             => "-C++ -nolinenumbers",
    XSOPT             => "-C++",
    DEFINE            => '-DDEBUG', # e.g., '-DHAVE_SOMETHING',
    OPTIMIZE          => '-O0 -g',
    CC                => $CC,
    CCFLAGS           => $CCFLAGS,
    MAKE              => $MAKE_PROG,
    #LDLOADLIBS	      => "-lstdc++",
    LDDLFLAGS 	      => $LDDFLAGS,
    #MYEXTLIB          => "sleuthkit/tsk/.libs/libtsk.so.10",
    #LDDLFLAGS          => "-DDEBUG -O0 -g -shared -Wl,-rpath $SLEUTHKIT_LIBS",
    #LDDLFLAGS          => "-DDEBUG -O0 -g -shared",
    BUILD_REQUIRES    => {
            'ExtUtils::MakeMaker' => 6.48,
            'Devel::Symdump'      => 2.11,
            'Digest::MD5'         => 2.53,
            #'Test::LeakTrace'     => 0.14,
            #'Devel::Peek'         => undef,
    },
    NORECURS          => 1,
    XS                => {
        "Main.xs"            => "Main.c",
        "TskFsName.xs"       => "TskFsName.c",
        "TskFsMeta.xs"       => "TskFsMeta.c",
        "TskFsBlock.xs"      => "TskFsBlock.c",
        "TskFsFile.xs"       => "TskFsFile.c",
        "TskFsInfo.xs"       => "TskFsInfo.c",
        "TskFsDir.xs"        => "TskFsDir.c",
        "TskVsInfo.xs"       => "TskVsInfo.c",
        "TskVsPartInfo.xs"   => "TskVsPartInfo.c",
        "TskImgInfo.xs"      => "TskImgInfo.c",
        "TskStack.xs"        => "TskStack.c",
        "TskFsAttribute.xs"  => "TskFsAttribute.c",
    },
    INC               => $INC_PATH, # e.g., '-I. -I/usr/include/other'
    OBJECT            => '$(O_FILES)', 
    dist	      => {
     COMPRESS => 'gzip -9f',
     SUFFIX   => '.gz'
    },
    TYPEMAPS          => ['typemap'],

    ## add rpaths for both the install path and for the current path(for development)
    dynamic_lib => { OTHERLDFLAGS => '-Wl,-rpath,$(INSTALLSITEARCH)/auto/$(FULLEXT) -Wl,-rpath,sleuthkit/tsk/.libs/'},
);

package MY;

############################################
# Custom Makefile targets for dependencies #
############################################

sub postamble {
    return "" if $^O eq "MSWin32";
    package MY;
    my $str = <<"_POSTAMBLE";

libewf/config.status:
	cd $LIBEWF_BASE ; \\
	./configure;      \\
	make;             \\
	ln -s libewf/.libs lib || true;

AFFLIBv3/config.status:
	cd $AFFLIB_BASE;            \\
	./bootstrap.sh;             \\
	./configure;                \\
	make;                       \\
        mkdir sandbox;              \\
        cd sandbox;                 \\
	ln -f -s ../lib/* . ;       \\
	ln -f -s ../lib/.libs/* . ; \\
        mkdir lib ;                 \\
        cp ../lib/.libs/* lib/ ;    \\
        ln -f -s ../lib/.libs . ;   \\
        mkdir include ;             \\
        cd include;                 \\
	ln -f -s ../../lib afflib;  \\
        cd ..;                      \\
        cd ..;

sleuthkit/config.status: AFFLIBv3/config.status libewf/config.status
	echo "sleuthkit_custom runs";                                                       \\
	cd sleuthkit;                                                                       \\
	./bootstrap;                                                                        \\
	./configure --disable-java --with-libewf=$LIBEWF_BASE --with-afflib=$AFFLIB_DEPDIR; \\
	make;

custom.sleuthkit: sleuthkit/config.status AFFLIBv3/config.status libewf/config.status
_POSTAMBLE

    return $str;
};


sub top_targets {
    package MY;
    my $str = shift->SUPER::top_targets(@_);
    return $str if $^O eq "MSWin32";
    $str =~ s/^all :: (.*)/all :: custom.sleuthkit $1/;
    return $str;
};


## make sure that LD_LIBRARY_PATH includes the path to libtsk.so
sub test {
    package MY;
    my $self = shift;
    my $str = $self->MM::test(@_);
    return $str if $^O eq "MSWin32";
    $str =~ s/^test\s::\s(.*)$/test :: custom.install $1/xms;
    $str =~ s{PERL_DL_NONLAZY=1}{PERL_DL_NONLAZY=1 LD_LIBRARY_PATH=./blib/arch/auto/Tsk/}m;
    return $str;
};


sub install {
    package MY;
    my $self = shift;
    my $str = $self->MM::install(@_);
    return $str if $^O eq "MSWin32";
    $str =~ s/^install\s::\s(.*)$/install :: custom.install $1/xms;
    my $custom_install=<<"_INSTALL";

custom.install: custom.sleuthkit
	cp $SLEUTHKIT_LIBS/libtsk.so.10.2.0 \$(INST_ARCHAUTODIR)/
	cd \$(INST_ARCHAUTODIR) ; ln -f -s libtsk.so.10.2.0 libtsk.so.10 ; ln -f -s libtsk.so.10  libtsk.so;
	true

_INSTALL

    $str = "$custom_install $str";
    return $str;
}
