package Bank::RMD;

use 5.006001;
use strict;
use warnings;
use DateTime;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
  calculate
	
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
	
);

our $VERSION = '0.20';

sub new {
  
  my ( $package, %params ) = @_;
  my $param;
  $param->{taxyear} = $params{taxyear} || '2005'; #unused 
  bless $param, $package;

}

sub calculate() {
  my ( $package, %params ) = @_;

  return undef unless 
    defined $params{ownerAge} ||
    defined $params{ownerBirthDay};

  return undef unless 
    $params{balance};

  if ( $params{ownerBirthday} ) {
    my $dt = DateTime->now;
    my $age = $dt - $params{ownerBirthday};
    $params{ownerAge} = $age->delta_days;
  }

  if ( $params{beneficiaryBirthday} ) {
    my $dt = DateTime->now;
    my $age = $dt - $params{beneficiaryBirthday};
    $params{beneficiaryAge} = $age->delta_days;
  }

  $params{beneficiaryAge} = $params{beneficiaryAge} || $params{ownerAge};

  if ( abs($params{ownerAge} - $params{beneficiaryAge}) le 10 ) {
    my $lifeexpectancy = $package->getexpectancy( { type => 'universal', ownerAge => $params{ownerAge} } );
    $package->{rmd} = sprintf('%.2f', $params{balance} / $lifeexpectancy );
    $package->{divisor} = $lifeexpectancy;
    return { divisor => $lifeexpectancy, rmd => sprintf('%.2f', $params{balance} / $lifeexpectancy ) };
  }
  else {
    my $lifeexpectancy = $package->getexpectancy( {type => 'joint', ownerAge => $params{ownerAge}, beneficiaryAge => $params{beneficiaryAge}} );
    $package->{rmd} = sprintf('%.2f', $params{balance} / $lifeexpectancy );
    $package->{divisor} = $lifeexpectancy;
    return { divisor => $lifeexpectancy, rmd => sprintf('%.2f', $params{balance} / $lifeexpectancy ) };
  }
 
}


sub getexpectancy {

  my ( $package, $params ) = @_;

  if ( $params->{type} =~ /single|universal/ ) {
    my $values = $package->universal;
    return $values->{$params->{ownerAge}};
  }
  else {
    my $values = $package->jointtable;
    return $values->{$params->{ownerAge}}->{$params->{beneficiaryAge}};
  }
  return undef;

}

sub rmd { 
  my ( $package ) = @_;
  return $package->{rmd};
}

sub divisor { 
  my ( $package ) = @_;
  return $package->{divisor};
}

sub jointtable { 
  return {
          '33' => {
                    '35' => '56.2', '33' => '57.2', '32' => '57.7', '21' => '64.8', '26' => '61.3',
                    '22' => '64.1', '30' => '58.8', '23' => '63.3', '29' => '59.4', '25' => '61.9',
                    '27' => '60.6', '39' => '54.7', '28' => '60.0', '36' => '55.8', '20' => '65.6',
                    '38' => '55.0', '34' => '56.7', '37' => '55.4', '24' => '62.6', '31' => '58.2'
                  },
          '32' => {
                    '35' => '56.8', '33' => '57.7', '32' => '58.2', '21' => '65.1', '26' => '61.6',
                    '22' => '64.3', '30' => '59.2', '23' => '63.6', '29' => '59.8', '25' => '62.2',
                    '27' => '61.0', '39' => '55.3', '28' => '60.4', '36' => '56.4', '20' => '65.8',
                    '38' => '55.6', '34' => '57.2', '37' => '56.0', '24' => '62.9', '31' => '58.7'
                  },
          '90' => {
                    '33' => '50.4', '32' => '51.4', '90' => '7.8', '63' => '22.9', '21' => '62.1',
                    '71' => '16.8', '80' => '11.4', '26' => '57.2', '99' => '6.4', '72' => '16.1',
                    '44' => '39.8', '55' => '29.8', '84' => '9.7', '74' => '14.8', '27' => '56.2',
                    '95' => '6.9', '57' => '28.0', '61' => '24.6', '20' => '63.0', '92' => '7.4',
                    '89' => '8.1', '31' => '52.4', '35' => '48.5', '91' => '7.6', '78' => '12.4',
                    '48' => '36.1', '87' => '8.6', '93' => '7.2', '77' => '13.0', '65' => '21.3',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '22.1', '97' => '6.6',
                    '58' => '27.1', '41' => '42.7', '81' => '10.9', '52' => '32.4', '60' => '25.4',
                    '56' => '28.9', '73' => '15.4', '66' => '20.5', '45' => '38.9', '86' => '9.0',
                    '76' => '13.6', '62' => '23.7', '54' => '30.6', '67' => '19.7', '70' => '17.5',
                    '68' => '19.0', '88' => '8.3', '30' => '53.3', '82' => '10.5', '25' => '58.2',
                    '28' => '55.3', '40' => '43.7', '75' => '14.2', '83' => '10.1', '59' => '26.3',
                    '69' => '18.2', '49' => '35.2', '24' => '59.1', '53' => '31.5', '79' => '11.9',
                    '42' => '41.7', '22' => '61.1', '46' => '38.0', '23' => '60.1', '96' => '6.8',
                    '85' => '9.3', '36' => '47.5', '94' => '7.1', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '6.5', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '63' => {
                    '67' => '26.4', '33' => '50.9', '32' => '51.8', '63' => '28.1', '21' => '62.3',
                    '68' => '26.1', '26' => '57.5', '30' => '53.7', '44' => '41.0', '55' => '32.6',
                    '25' => '58.4', '27' => '56.5', '28' => '55.6', '57' => '31.3', '40' => '44.5',
                    '61' => '29.0', '20' => '63.2', '69' => '25.7', '59' => '30.1', '49' => '36.9',
                    '24' => '59.4', '31' => '52.7', '35' => '49.0', '53' => '33.9', '48' => '37.7',
                    '22' => '61.3', '42' => '42.7', '46' => '39.3', '23' => '60.3', '65' => '27.2',
                    '29' => '54.6', '50' => '36.1', '39' => '45.3', '64' => '27.6', '36' => '48.1',
                    '51' => '35.4', '58' => '30.7', '41' => '43.6', '47' => '38.5', '38' => '46.3',
                    '52' => '34.6', '60' => '29.6', '34' => '49.9', '56' => '31.9', '66' => '26.8',
                    '37' => '47.2', '45' => '40.1', '43' => '41.8', '62' => '28.5', '54' => '33.2'
                  },
          '21' => {
                    '27' => '66.6', '25' => '67.3', '28' => '66.2', '21' => '69.1', '26' => '66.9',
                    '20' => '69.6', '22' => '68.6', '24' => '67.7', '23' => '68.2', '29' => '65.9'
                  },
          '71' => {
                    '67' => '22.8', '33' => '50.6', '32' => '51.6', '63' => '25.2', '21' => '62.1',
                    '71' => '20.9', '70' => '21.3', '68' => '22.3', '26' => '57.3', '72' => '20.5',
                    '30' => '53.5', '44' => '40.3', '55' => '30.9', '74' => '19.7', '25' => '58.3',
                    '27' => '56.4', '28' => '55.4', '75' => '19.4', '57' => '29.4', '40' => '44.0',
                    '61' => '26.5', '20' => '63.1', '69' => '21.8', '59' => '27.9', '49' => '35.9',
                    '24' => '59.2', '31' => '52.5', '35' => '48.7', '53' => '32.5', '78' => '18.5',
                    '79' => '18.3', '48' => '36.7', '22' => '61.2', '42' => '42.1', '77' => '18.8',
                    '46' => '38.5', '23' => '60.2', '65' => '23.9', '29' => '54.4', '50' => '35.0',
                    '39' => '44.9', '64' => '24.5', '36' => '47.7', '51' => '34.2', '58' => '28.6',
                    '41' => '43.0', '47' => '37.6', '38' => '45.9', '52' => '33.3', '60' => '27.2',
                    '34' => '49.6', '56' => '30.1', '73' => '20.1', '66' => '23.4', '37' => '46.8',
                    '45' => '39.4', '76' => '19.1', '43' => '41.2', '62' => '25.8', '54' => '31.7'
                  },
          '102' => {
                     '33' => '50.4', '32' => '51.4', '90' => '6.1', '63' => '22.7', '21' => '62.1',
                     '71' => '16.4', '102' => '3.6', '80' => '10.5', '26' => '57.2', '99' => '4.1',
                     '72' => '15.6', '44' => '39.8', '55' => '29.7', '84' => '8.4', '74' => '14.2',
                     '27' => '56.2', '95' => '4.8', '57' => '27.9', '61' => '24.4', '108' => '2.9',
                     '20' => '63.0', '109' => '2.8', '92' => '5.5', '103' => '3.4', '89' => '6.4',
                     '31' => '52.4', '35' => '48.5', '91' => '5.8', '78' => '11.6', '107' => '3.0',
                     '48' => '36.0', '87' => '7.1', '93' => '5.3', '77' => '12.2', '106' => '3.1',
                     '65' => '21.1', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9',
                     '97' => '4.4', '58' => '27.0', '41' => '42.7', '81' => '9.9', '52' => '32.4',
                     '60' => '25.3', '56' => '28.8', '101' => '3.7', '73' => '14.9', '66' => '20.2',
                     '45' => '38.9', '86' => '7.5', '76' => '12.9', '62' => '23.6', '54' => '30.5',
                     '67' => '19.4', '70' => '17.1', '68' => '18.6', '88' => '6.8', '30' => '53.3',
                     '100' => '3.9', '82' => '9.4', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.5', '83' => '8.9', '59' => '26.1', '69' => '17.9', '49' => '35.1',
                     '24' => '59.1', '104' => '3.3', '53' => '31.4', '79' => '11.0', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '3.2', '96' => '4.6',
                     '85' => '8.0', '36' => '47.5', '94' => '5.1', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '4.3', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                   },
          '80' => {
                    '33' => '50.5', '32' => '51.4', '63' => '23.6', '21' => '62.1', '71' => '18.1',
                    '80' => '13.8', '26' => '57.2', '72' => '17.5', '44' => '40.0', '55' => '30.1',
                    '84' => '12.6', '74' => '16.4', '27' => '56.3', '57' => '28.4', '61' => '25.1',
                    '20' => '63.1', '89' => '11.5', '31' => '52.4', '35' => '48.5', '78' => '14.5',
                    '48' => '36.3', '87' => '11.9', '77' => '15.0', '65' => '22.1', '29' => '54.3',
                    '50' => '34.5', '39' => '44.7', '64' => '22.8', '58' => '27.5', '41' => '42.8',
                    '81' => '13.4', '52' => '32.7', '60' => '25.9', '56' => '29.2', '73' => '16.9',
                    '66' => '21.3', '45' => '39.0', '86' => '12.1', '76' => '15.4', '62' => '24.3',
                    '54' => '30.9', '67' => '20.6', '70' => '18.7', '68' => '20.0', '88' => '11.7',
                    '30' => '53.4', '82' => '13.1', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '15.9', '83' => '12.8', '59' => '26.7', '69' => '19.3', '49' => '35.4',
                    '24' => '59.2', '53' => '31.8', '79' => '14.1', '42' => '41.8', '22' => '61.1',
                    '46' => '38.1', '23' => '60.1', '85' => '12.3', '36' => '47.6', '51' => '33.6',
                    '47' => '37.2', '38' => '45.7', '34' => '49.5', '37' => '46.6', '43' => '40.9'
                  },
          '26' => {
                    '27' => '63.6', '25' => '64.6', '28' => '63.2', '21' => '66.9', '26' => '64.1',
                    '20' => '67.5', '22' => '66.3', '24' => '65.2', '23' => '65.7', '29' => '62.8'
                  },
          '99' => {
                    '33' => '50.4', '32' => '51.4', '90' => '6.4', '63' => '22.7', '21' => '62.1',
                    '71' => '16.4', '80' => '10.6', '26' => '57.2', '99' => '4.5', '72' => '15.7',
                    '44' => '39.8', '55' => '29.7', '84' => '8.6', '74' => '14.3', '27' => '56.2',
                    '95' => '5.2', '57' => '27.9', '61' => '24.4', '20' => '63.0', '92' => '5.9',
                    '89' => '6.7', '31' => '52.4', '35' => '48.5', '91' => '6.1', '78' => '11.8',
                    '48' => '36.0', '87' => '7.4', '93' => '5.6', '77' => '12.4', '65' => '21.1',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9', '97' => '4.9',
                    '58' => '27.0', '41' => '42.7', '81' => '10.1', '52' => '32.4', '60' => '25.3',
                    '56' => '28.8', '73' => '15.0', '66' => '20.3', '45' => '38.9', '86' => '7.8',
                    '76' => '13.0', '62' => '23.6', '54' => '30.6', '67' => '19.5', '70' => '17.2',
                    '68' => '18.7', '88' => '7.0', '30' => '53.3', '82' => '9.6', '25' => '58.2',
                    '28' => '55.3', '40' => '43.6', '75' => '13.6', '83' => '9.1', '59' => '26.2',
                    '69' => '17.9', '49' => '35.1', '24' => '59.1', '53' => '31.5', '79' => '11.2',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '5.0',
                    '85' => '8.2', '36' => '47.5', '94' => '5.4', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '4.7', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                  },
          '72' => {
                    '67' => '22.5', '33' => '50.6', '32' => '51.5', '63' => '24.9', '21' => '62.1',
                    '71' => '20.5', '70' => '20.9', '68' => '22.0', '26' => '57.3', '72' => '20.0',
                    '30' => '53.5', '44' => '40.2', '55' => '30.8', '74' => '19.3', '25' => '58.3',
                    '27' => '56.3', '28' => '55.4', '75' => '18.9', '57' => '29.2', '40' => '43.9',
                    '61' => '26.3', '20' => '63.1', '69' => '21.4', '59' => '27.7', '49' => '35.8',
                    '24' => '59.2', '31' => '52.5', '35' => '48.7', '53' => '32.4', '78' => '18.0',
                    '79' => '17.7', '48' => '36.6', '22' => '61.2', '42' => '42.1', '77' => '18.3',
                    '46' => '38.4', '23' => '60.2', '65' => '23.7', '29' => '54.4', '50' => '34.9',
                    '39' => '44.9', '64' => '24.3', '36' => '47.7', '51' => '34.1', '58' => '28.4',
                    '41' => '43.0', '47' => '37.5', '38' => '45.8', '52' => '33.2', '60' => '27.0',
                    '34' => '49.6', '56' => '30.0', '73' => '19.6', '66' => '23.1', '37' => '46.8',
                    '45' => '39.3', '76' => '18.6', '43' => '41.1', '62' => '25.6', '54' => '31.6'
                  },
          '44' => {
                    '35' => '51.7', '33' => '53.2', '32' => '53.9', '21' => '63.1', '26' => '58.8',
                    '48' => '44.5', '42' => '47.3', '22' => '62.2', '46' => '45.4', '30' => '55.5',
                    '23' => '61.4', '44' => '46.3', '29' => '56.3', '25' => '59.6', '27' => '57.9',
                    '39' => '49.1', '28' => '57.1', '40' => '48.5', '36' => '51.0', '41' => '47.9',
                    '47' => '44.9', '20' => '64.0', '38' => '49.7', '34' => '52.4', '49' => '44.2',
                    '45' => '45.8', '37' => '50.4', '24' => '60.5', '43' => '46.8', '31' => '54.7'
                  },
          '55' => {
                    '33' => '51.4', '32' => '52.3', '21' => '62.5', '26' => '57.8', '30' => '54.1',
                    '44' => '42.4', '55' => '35.6', '25' => '58.7', '27' => '56.8', '28' => '55.9',
                    '57' => '34.7', '40' => '45.5', '20' => '63.4', '59' => '33.9', '49' => '38.9',
                    '24' => '59.6', '31' => '53.2', '35' => '49.7', '53' => '36.6', '48' => '39.6',
                    '22' => '61.5', '42' => '43.9', '46' => '40.9', '23' => '60.6', '29' => '55.0',
                    '50' => '38.3', '39' => '46.3', '36' => '48.8', '51' => '37.7', '58' => '34.3',
                    '41' => '44.7', '47' => '40.2', '38' => '47.1', '52' => '37.2', '34' => '50.5',
                    '56' => '35.1', '37' => '47.9', '45' => '41.6', '43' => '43.1', '54' => '36.1'
                  },
          '84' => {
                    '33' => '50.5', '32' => '51.4', '63' => '23.2', '21' => '62.1', '71' => '17.4',
                    '80' => '12.6', '26' => '57.2', '72' => '16.7', '44' => '39.9', '55' => '29.9',
                    '84' => '11.1', '74' => '15.5', '27' => '56.3', '57' => '28.2', '61' => '24.8',
                    '20' => '63.0', '89' => '9.9', '31' => '52.4', '35' => '48.5', '78' => '13.4',
                    '48' => '36.2', '87' => '10.3', '77' => '13.9', '65' => '21.6', '29' => '54.3',
                    '50' => '34.3', '39' => '44.7', '64' => '22.4', '58' => '27.3', '41' => '42.7',
                    '81' => '12.2', '52' => '32.5', '60' => '25.6', '56' => '29.0', '73' => '16.1',
                    '66' => '20.9', '45' => '39.0', '86' => '10.5', '76' => '14.4', '62' => '24.0',
                    '54' => '30.8', '67' => '20.1', '70' => '18.0', '68' => '19.4', '88' => '10.1',
                    '30' => '53.4', '82' => '11.8', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '15.0', '83' => '11.4', '59' => '26.5', '69' => '18.7', '49' => '35.3',
                    '24' => '59.2', '53' => '31.7', '79' => '13.0', '42' => '41.8', '22' => '61.1',
                    '46' => '38.0', '23' => '60.1', '85' => '10.8', '36' => '47.6', '51' => '33.4',
                    '47' => '37.1', '38' => '45.6', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '74' => {
                    '67' => '21.9', '33' => '50.5', '32' => '51.5', '63' => '24.5', '21' => '62.1',
                    '71' => '19.7', '70' => '20.2', '68' => '21.3', '26' => '57.3', '72' => '19.3',
                    '30' => '53.4', '44' => '40.1', '55' => '30.5', '74' => '18.4', '25' => '58.2',
                    '27' => '56.3', '28' => '55.4', '75' => '18.0', '57' => '28.9', '40' => '43.9',
                    '61' => '25.9', '20' => '63.1', '69' => '20.8', '59' => '27.4', '49' => '35.6',
                    '24' => '59.2', '31' => '52.5', '35' => '48.6', '53' => '32.2', '78' => '17.0',
                    '79' => '16.7', '48' => '36.5', '22' => '61.2', '42' => '42.0', '77' => '17.3',
                    '46' => '38.3', '23' => '60.2', '65' => '23.1', '29' => '54.4', '50' => '34.8',
                    '39' => '44.8', '64' => '23.8', '36' => '47.7', '51' => '33.9', '58' => '28.1',
                    '41' => '42.9', '47' => '37.4', '38' => '45.8', '52' => '33.0', '60' => '26.6',
                    '34' => '49.6', '56' => '29.7', '73' => '18.8', '66' => '22.5', '37' => '46.7',
                    '45' => '39.2', '76' => '17.6', '43' => '41.1', '62' => '25.2', '54' => '31.4'
                  },
          '27' => {
                    '27' => '63.1', '25' => '64.2', '28' => '62.7', '21' => '66.6', '26' => '63.6',
                    '20' => '67.2', '22' => '65.9', '24' => '64.7', '23' => '65.3', '29' => '62.2'
                  },
          '95' => {
                    '33' => '50.4', '32' => '51.4', '90' => '6.9', '63' => '22.8', '21' => '62.1',
                    '71' => '16.5', '80' => '10.9', '26' => '57.2', '99' => '5.2', '72' => '15.8',
                    '44' => '39.8', '55' => '29.7', '84' => '9.0', '74' => '14.5', '27' => '56.2',
                    '95' => '5.8', '57' => '27.9', '61' => '24.5', '20' => '63.0', '92' => '6.4',
                    '89' => '7.2', '31' => '52.4', '35' => '48.5', '91' => '6.7', '78' => '12.0',
                    '48' => '36.1', '87' => '7.8', '93' => '6.2', '77' => '12.6', '65' => '21.1',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '22.0', '97' => '5.5',
                    '58' => '27.1', '41' => '42.7', '81' => '10.4', '52' => '32.4', '60' => '25.3',
                    '56' => '28.8', '73' => '15.1', '66' => '20.3', '45' => '38.9', '86' => '8.2',
                    '76' => '13.2', '62' => '23.6', '54' => '30.6', '67' => '19.6', '70' => '17.3',
                    '68' => '18.8', '88' => '7.5', '30' => '53.3', '82' => '9.9', '25' => '58.2',
                    '28' => '55.3', '40' => '43.6', '75' => '13.8', '83' => '9.4', '59' => '26.2',
                    '69' => '18.0', '49' => '35.1', '24' => '59.1', '53' => '31.5', '79' => '11.4',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '5.7',
                    '85' => '8.6', '36' => '47.5', '94' => '6.0', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '5.4', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                  },
          '57' => {
                    '33' => '51.2', '32' => '52.1', '21' => '62.4', '26' => '57.7', '30' => '54.0',
                    '44' => '41.9', '55' => '34.7', '25' => '58.6', '27' => '56.7', '28' => '55.8',
                    '57' => '33.7', '40' => '45.1', '20' => '63.4', '59' => '32.8', '49' => '38.3',
                    '24' => '59.6', '31' => '53.0', '35' => '49.4', '53' => '35.8', '48' => '39.0',
                    '22' => '61.5', '42' => '43.5', '46' => '40.4', '23' => '60.5', '29' => '54.9',
                    '50' => '37.6', '39' => '46.0', '36' => '48.6', '51' => '37.0', '58' => '33.2',
                    '41' => '44.3', '47' => '39.7', '38' => '46.8', '52' => '36.4', '34' => '50.3',
                    '56' => '34.2', '37' => '47.7', '45' => '41.2', '43' => '42.7', '54' => '35.2'
                  },
          '61' => {
                    '67' => '27.6', '33' => '51.0', '32' => '51.9', '63' => '29.0', '21' => '62.3',
                    '68' => '27.3', '26' => '57.5', '30' => '53.8', '44' => '41.2', '55' => '33.2',
                    '25' => '58.5', '27' => '56.6', '28' => '55.6', '57' => '32.0', '40' => '44.6',
                    '61' => '29.9', '20' => '63.3', '69' => '27.0', '59' => '30.9', '49' => '37.3',
                    '24' => '59.4', '31' => '52.8', '35' => '49.1', '53' => '34.5', '48' => '38.0',
                    '22' => '61.3', '42' => '42.9', '46' => '39.6', '23' => '60.4', '65' => '28.3',
                    '29' => '54.7', '50' => '36.6', '39' => '45.5', '64' => '28.6', '36' => '48.2',
                    '51' => '35.8', '58' => '31.4', '41' => '43.8', '47' => '38.8', '38' => '46.4',
                    '52' => '35.1', '60' => '30.4', '34' => '50.0', '56' => '32.6', '66' => '27.9',
                    '37' => '47.3', '45' => '40.4', '43' => '42.1', '62' => '29.5', '54' => '33.8'
                  },
          '108' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.7', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '2.9', '80' => '10.3', '26' => '57.2', '99' => '3.5',
                     '72' => '15.5', '44' => '39.8', '55' => '29.6', '84' => '8.2', '74' => '14.1',
                     '27' => '56.2', '95' => '4.3', '57' => '27.9', '61' => '24.4', '108' => '1.9',
                     '20' => '63.0', '109' => '1.8', '92' => '5.1', '103' => '2.7', '89' => '6.1',
                     '31' => '52.4', '35' => '48.5', '91' => '5.4', '78' => '11.5', '107' => '2.1',
                     '48' => '36.0', '87' => '6.8', '93' => '4.8', '77' => '12.1', '106' => '2.2',
                     '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.8',
                     '97' => '3.9', '58' => '27.0', '41' => '42.7', '81' => '9.7', '52' => '32.3',
                     '60' => '25.2', '56' => '28.8', '101' => '3.1', '73' => '14.8', '66' => '20.2',
                     '45' => '38.8', '86' => '7.3', '76' => '12.8', '62' => '23.5', '54' => '30.5',
                     '67' => '19.4', '70' => '17.0', '68' => '18.6', '88' => '6.4', '30' => '53.3',
                     '100' => '3.3', '82' => '9.2', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.4', '83' => '8.7', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.5', '53' => '31.4', '79' => '10.9', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '2.4', '96' => '4.1',
                     '85' => '7.7', '36' => '47.5', '94' => '4.6', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '3.7', '34' => '49.4', '37' => '46.5', '43' => '40.8'
                   },
          '115' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.5', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '2.5', '80' => '10.2', '26' => '57.2', '99' => '3.1',
                     '72' => '15.5', '44' => '39.8', '55' => '29.6', '84' => '8.1', '74' => '14.1',
                     '27' => '56.2', '95' => '4.1', '57' => '27.9', '61' => '24.4', '108' => '1.4',
                     '20' => '63.0', '109' => '1.2', '92' => '4.9', '103' => '2.3', '89' => '5.9',
                     '31' => '52.4', '113' => '1.0', '35' => '48.5', '91' => '5.2', '78' => '11.4',
                     '107' => '1.5', '48' => '36.0', '87' => '6.7', '93' => '4.6', '77' => '12.1',
                     '106' => '1.7', '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6',
                     '64' => '21.8', '97' => '3.6', '114' => '1.0', '58' => '27.0', '41' => '42.7',
                     '81' => '9.7', '52' => '32.3', '60' => '25.2', '56' => '28.7', '101' => '2.7',
                     '73' => '14.8', '66' => '20.2', '45' => '38.8', '86' => '7.1', '76' => '12.7',
                     '62' => '23.5', '54' => '30.5', '67' => '19.4', '70' => '17.0', '68' => '18.6',
                     '115' => '1.0', '88' => '6.3', '30' => '53.3', '100' => '2.9', '110' => '1.1',
                     '82' => '9.1', '25' => '58.2', '28' => '55.3', '40' => '43.6', '75' => '13.4',
                     '83' => '8.6', '112' => '1.0', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.1', '53' => '31.4', '79' => '10.8', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '1.9', '96' => '3.8',
                     '85' => '7.6', '36' => '47.5', '94' => '4.3', '111' => '1.0', '51' => '33.3',
                     '47' => '37.0', '38' => '45.6', '98' => '3.4', '34' => '49.4', '37' => '46.5',
                     '43' => '40.7'
                   },
          '20' => {
                    '27' => '67.2', '25' => '67.9', '28' => '66.9', '21' => '69.6', '26' => '67.5',
                    '20' => '70.1', '22' => '69.1', '24' => '68.3', '23' => '68.7', '29' => '66.6'
                  },
          '109' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.7', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '2.8', '80' => '10.3', '26' => '57.2', '99' => '3.4',
                     '72' => '15.5', '44' => '39.8', '55' => '29.6', '84' => '8.2', '74' => '14.1',
                     '27' => '56.2', '95' => '4.3', '57' => '27.9', '61' => '24.4', '108' => '1.8',
                     '20' => '63.0', '109' => '1.7', '92' => '5.0', '103' => '2.6', '89' => '6.0',
                     '31' => '52.4', '35' => '48.5', '91' => '5.3', '78' => '11.5', '107' => '2.0',
                     '48' => '36.0', '87' => '6.8', '93' => '4.8', '77' => '12.1', '106' => '2.1',
                     '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.8',
                     '97' => '3.8', '58' => '27.0', '41' => '42.7', '81' => '9.7', '52' => '32.3',
                     '60' => '25.2', '56' => '28.7', '101' => '3.0', '73' => '14.8', '66' => '20.2',
                     '45' => '38.8', '86' => '7.2', '76' => '12.8', '62' => '23.5', '54' => '30.5',
                     '67' => '19.4', '70' => '17.0', '68' => '18.6', '88' => '6.4', '30' => '53.3',
                     '100' => '3.2', '82' => '9.2', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.4', '83' => '8.7', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.4', '53' => '31.4', '79' => '10.9', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '2.3', '96' => '4.0',
                     '85' => '7.7', '36' => '47.5', '94' => '4.5', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '3.6', '34' => '49.4', '37' => '46.5', '43' => '40.7'
                   },
          '92' => {
                    '33' => '50.4', '32' => '51.4', '90' => '7.4', '63' => '22.9', '21' => '62.1',
                    '71' => '16.7', '80' => '11.2', '26' => '57.2', '99' => '5.9', '72' => '16.0',
                    '44' => '39.8', '55' => '29.7', '84' => '9.3', '74' => '14.6', '27' => '56.2',
                    '95' => '6.4', '57' => '28.0', '61' => '24.5', '20' => '63.0', '92' => '7.0',
                    '89' => '7.7', '31' => '52.4', '35' => '48.5', '91' => '7.2', '78' => '12.2',
                    '48' => '36.1', '87' => '8.3', '93' => '6.8', '77' => '12.8', '65' => '21.2',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '22.0', '97' => '6.1',
                    '58' => '27.1', '41' => '42.7', '81' => '10.7', '52' => '32.4', '60' => '25.4',
                    '56' => '28.8', '73' => '15.3', '66' => '20.4', '45' => '38.9', '86' => '8.6',
                    '76' => '13.4', '62' => '23.7', '54' => '30.6', '67' => '19.6', '70' => '17.4',
                    '68' => '18.9', '88' => '8.0', '30' => '53.3', '82' => '10.2', '25' => '58.2',
                    '28' => '55.3', '40' => '43.7', '75' => '14.0', '83' => '9.8', '59' => '26.2',
                    '69' => '18.1', '49' => '35.1', '24' => '59.1', '53' => '31.5', '79' => '11.7',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '6.3',
                    '85' => '9.0', '36' => '47.5', '94' => '6.6', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '6.0', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '103' => {
                     '33' => '50.4', '32' => '51.4', '90' => '6.0', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '3.4', '80' => '10.4', '26' => '57.2', '99' => '4.0',
                     '72' => '15.6', '44' => '39.8', '55' => '29.7', '84' => '8.4', '74' => '14.2',
                     '27' => '56.2', '95' => '4.7', '57' => '27.9', '61' => '24.4', '108' => '2.7',
                     '20' => '63.0', '109' => '2.6', '92' => '5.4', '103' => '3.3', '89' => '6.3',
                     '31' => '52.4', '35' => '48.5', '91' => '5.7', '78' => '11.6', '107' => '2.8',
                     '48' => '36.0', '87' => '7.1', '93' => '5.2', '77' => '12.2', '106' => '2.9',
                     '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9',
                     '97' => '4.3', '58' => '27.0', '41' => '42.7', '81' => '9.9', '52' => '32.4',
                     '60' => '25.3', '56' => '28.8', '101' => '3.6', '73' => '14.9', '66' => '20.2',
                     '45' => '38.9', '86' => '7.5', '76' => '12.9', '62' => '23.6', '54' => '30.5',
                     '67' => '19.4', '70' => '17.1', '68' => '18.6', '88' => '6.7', '30' => '53.3',
                     '100' => '3.8', '82' => '9.4', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.5', '83' => '8.8', '59' => '26.1', '69' => '17.9', '49' => '35.1',
                     '24' => '59.1', '104' => '3.2', '53' => '31.4', '79' => '11.0', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '3.0', '96' => '4.5',
                     '85' => '7.9', '36' => '47.5', '94' => '5.0', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '4.1', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                   },
          '89' => {
                    '33' => '50.4', '32' => '51.4', '63' => '22.9', '21' => '62.1', '71' => '16.9',
                    '80' => '11.5', '26' => '57.2', '72' => '16.2', '44' => '39.8', '55' => '29.8',
                    '84' => '9.9', '74' => '14.9', '27' => '56.2', '57' => '28.0', '61' => '24.6',
                    '20' => '63.0', '89' => '8.3', '31' => '52.4', '35' => '48.5', '78' => '12.6',
                    '48' => '36.1', '87' => '8.9', '77' => '13.1', '65' => '21.3', '29' => '54.3',
                    '50' => '34.3', '39' => '44.6', '64' => '22.1', '58' => '27.2', '41' => '42.7',
                    '81' => '11.1', '52' => '32.4', '60' => '25.4', '56' => '28.9', '73' => '15.5',
                    '66' => '20.5', '45' => '38.9', '86' => '9.2', '76' => '13.7', '62' => '23.8',
                    '54' => '30.7', '67' => '19.8', '70' => '17.6', '68' => '19.0', '88' => '8.6',
                    '30' => '53.3', '82' => '10.6', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '14.3', '83' => '10.2', '59' => '26.3', '69' => '18.3', '49' => '35.2',
                    '24' => '59.1', '53' => '31.5', '79' => '12.0', '42' => '41.7', '22' => '61.1',
                    '46' => '38.0', '23' => '60.1', '85' => '9.5', '36' => '47.5', '51' => '33.3',
                    '47' => '37.0', '38' => '45.6', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '31' => {
                    '35' => '57.4', '33' => '58.2', '32' => '58.7', '21' => '65.3', '26' => '62.0',
                    '22' => '64.6', '30' => '59.7', '23' => '63.9', '29' => '60.2', '25' => '62.6',
                    '27' => '61.4', '39' => '56.0', '28' => '60.8', '36' => '57.0', '20' => '66.1',
                    '38' => '56.3', '34' => '57.8', '37' => '56.6', '24' => '63.2', '31' => '59.2'
                  },
          '113' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.6', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '2.6', '80' => '10.2', '26' => '57.2', '99' => '3.2',
                     '72' => '15.5', '44' => '39.8', '55' => '29.6', '84' => '8.1', '74' => '14.1',
                     '27' => '56.2', '95' => '4.1', '57' => '27.9', '61' => '24.4', '108' => '1.5',
                     '20' => '63.0', '109' => '1.3', '92' => '4.9', '103' => '2.4', '89' => '5.9',
                     '31' => '52.4', '113' => '1.0', '35' => '48.5', '91' => '5.2', '78' => '11.4',
                     '107' => '1.6', '48' => '36.0', '87' => '6.7', '93' => '4.6', '77' => '12.1',
                     '106' => '1.8', '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6',
                     '64' => '21.8', '97' => '3.6', '114' => '1.0', '58' => '27.0', '41' => '42.7',
                     '81' => '9.7', '52' => '32.3', '60' => '25.2', '56' => '28.7', '101' => '2.8',
                     '73' => '14.8', '66' => '20.2', '45' => '38.8', '86' => '7.2', '76' => '12.7',
                     '62' => '23.5', '54' => '30.5', '67' => '19.4', '70' => '17.0', '68' => '18.6',
                     '115' => '1.0', '88' => '6.3', '30' => '53.3', '100' => '3.0', '110' => '1.2',
                     '82' => '9.1', '25' => '58.2', '28' => '55.3', '40' => '43.6', '75' => '13.4',
                     '83' => '8.6', '112' => '1.0', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.2', '53' => '31.4', '79' => '10.8', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '2.0', '96' => '3.9',
                     '85' => '7.6', '36' => '47.5', '94' => '4.4', '111' => '1.1', '51' => '33.3',
                     '47' => '37.0', '38' => '45.6', '98' => '3.4', '34' => '49.4', '37' => '46.5',
                     '43' => '40.7'
                   },
          '35' => {
                    '35' => '55.2', '33' => '56.2', '32' => '56.8', '21' => '64.4', '26' => '60.6',
                    '22' => '63.6', '30' => '58.0', '23' => '62.8', '29' => '58.6', '25' => '61.4',
                    '27' => '59.9', '39' => '53.4', '28' => '59.3', '36' => '54.7', '20' => '65.2',
                    '38' => '53.8', '34' => '55.7', '37' => '54.3', '24' => '62.1', '31' => '57.4'
                  },
          '91' => {
                    '33' => '50.4', '32' => '51.4', '90' => '7.6', '63' => '22.9', '21' => '62.1',
                    '71' => '16.7', '80' => '11.3', '26' => '57.2', '99' => '6.1', '72' => '16.0',
                    '44' => '39.8', '55' => '29.7', '84' => '9.5', '74' => '14.7', '27' => '56.2',
                    '95' => '6.7', '57' => '28.0', '61' => '24.5', '20' => '63.0', '92' => '7.2',
                    '89' => '7.9', '31' => '52.4', '35' => '48.5', '91' => '7.4', '78' => '12.3',
                    '48' => '36.1', '87' => '8.4', '93' => '7.0', '77' => '12.9', '65' => '21.3',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '22.1', '97' => '6.4',
                    '58' => '27.1', '41' => '42.7', '81' => '10.8', '52' => '32.4', '60' => '25.4',
                    '56' => '28.9', '73' => '15.4', '66' => '20.5', '45' => '38.9', '86' => '8.8',
                    '76' => '13.5', '62' => '23.7', '54' => '30.6', '67' => '19.7', '70' => '17.4',
                    '68' => '18.9', '88' => '8.1', '30' => '53.3', '82' => '10.3', '25' => '58.2',
                    '28' => '55.3', '40' => '43.7', '75' => '14.1', '83' => '9.9', '59' => '26.3',
                    '69' => '18.2', '49' => '35.2', '24' => '59.1', '53' => '31.5', '79' => '11.8',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '6.5',
                    '85' => '9.1', '36' => '47.5', '94' => '6.8', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '6.3', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '78' => {
                    '67' => '21.0', '33' => '50.5', '32' => '51.5', '63' => '23.8', '21' => '62.1',
                    '71' => '18.5', '70' => '19.1', '68' => '20.3', '26' => '57.3', '72' => '18.0',
                    '30' => '53.4', '44' => '40.0', '55' => '30.2', '74' => '17.0', '25' => '58.2',
                    '27' => '56.3', '28' => '55.3', '75' => '16.5', '57' => '28.5', '40' => '43.8',
                    '61' => '25.3', '20' => '63.1', '69' => '19.7', '59' => '26.9', '49' => '35.4',
                    '24' => '59.2', '31' => '52.4', '35' => '48.6', '53' => '31.9', '78' => '15.2',
                    '79' => '14.9', '48' => '36.3', '22' => '61.1', '42' => '41.9', '77' => '15.6',
                    '46' => '38.2', '23' => '60.2', '65' => '22.4', '29' => '54.4', '50' => '34.5',
                    '39' => '44.7', '64' => '23.1', '36' => '47.6', '51' => '33.6', '58' => '27.7',
                    '41' => '42.8', '47' => '37.2', '38' => '45.7', '52' => '32.8', '60' => '26.1',
                    '34' => '49.5', '56' => '29.3', '73' => '17.5', '66' => '21.7', '37' => '46.6',
                    '45' => '39.1', '76' => '16.0', '43' => '40.9', '62' => '24.6', '54' => '31.0'
                  },
          '107' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.8', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '3.0', '80' => '10.3', '26' => '57.2', '99' => '3.5',
                     '72' => '15.6', '44' => '39.8', '55' => '29.6', '84' => '8.2', '74' => '14.1',
                     '27' => '56.2', '95' => '4.4', '57' => '27.9', '61' => '24.4', '108' => '2.1',
                     '20' => '63.0', '109' => '2.0', '92' => '5.1', '103' => '2.8', '89' => '6.1',
                     '31' => '52.4', '35' => '48.5', '91' => '5.4', '78' => '11.5', '107' => '2.2',
                     '48' => '36.0', '87' => '6.9', '93' => '4.9', '77' => '12.1', '106' => '2.3',
                     '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.8',
                     '97' => '3.9', '58' => '27.0', '41' => '42.7', '81' => '9.8', '52' => '32.3',
                     '60' => '25.2', '56' => '28.8', '101' => '3.1', '73' => '14.8', '66' => '20.2',
                     '45' => '38.8', '86' => '7.3', '76' => '12.8', '62' => '23.5', '54' => '30.5',
                     '67' => '19.4', '70' => '17.0', '68' => '18.6', '88' => '6.5', '30' => '53.3',
                     '100' => '3.3', '82' => '9.2', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.4', '83' => '8.7', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.6', '53' => '31.4', '79' => '10.9', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '2.5', '96' => '4.2',
                     '85' => '7.7', '36' => '47.5', '94' => '4.6', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '3.7', '34' => '49.4', '37' => '46.5', '43' => '40.8'
                   },
          '48' => {
                    '35' => '50.8', '33' => '52.3', '32' => '53.2', '21' => '62.8', '26' => '58.3',
                    '48' => '42.4', '42' => '45.8', '22' => '61.9', '46' => '43.4', '30' => '54.8',
                    '23' => '61.0', '44' => '44.5', '29' => '55.7', '25' => '59.2', '27' => '57.4',
                    '39' => '47.8', '28' => '56.5', '40' => '47.1', '36' => '50.0', '41' => '46.4',
                    '47' => '42.9', '20' => '63.7', '38' => '48.5', '34' => '51.5', '49' => '41.9',
                    '45' => '44.0', '37' => '49.2', '24' => '60.1', '43' => '45.1', '31' => '54.0'
                  },
          '87' => {
                    '33' => '50.4', '32' => '51.4', '63' => '23.0', '21' => '62.1', '71' => '17.0',
                    '80' => '11.9', '26' => '57.2', '72' => '16.4', '44' => '39.9', '55' => '29.8',
                    '84' => '10.3', '74' => '15.1', '27' => '56.2', '57' => '28.1', '61' => '24.7',
                    '20' => '63.0', '89' => '8.9', '31' => '52.4', '35' => '48.5', '78' => '12.9',
                    '48' => '36.1', '87' => '9.4', '77' => '13.4', '65' => '21.4', '29' => '54.3',
                    '50' => '34.3', '39' => '44.6', '64' => '22.2', '58' => '27.2', '41' => '42.7',
                    '81' => '11.4', '52' => '32.5', '60' => '25.5', '56' => '28.9', '73' => '15.7',
                    '66' => '20.7', '45' => '38.9', '86' => '9.6', '76' => '13.9', '62' => '23.8',
                    '54' => '30.7', '67' => '19.9', '70' => '17.7', '68' => '19.2', '88' => '9.1',
                    '30' => '53.3', '82' => '11.0', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '14.5', '83' => '10.6', '59' => '26.4', '69' => '18.4', '49' => '35.2',
                    '24' => '59.2', '53' => '31.6', '79' => '12.4', '42' => '41.8', '22' => '61.1',
                    '46' => '38.0', '23' => '60.1', '85' => '9.9', '36' => '47.5', '51' => '33.4',
                    '47' => '37.0', '38' => '45.6', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '93' => {
                    '33' => '50.4', '32' => '51.4', '90' => '7.2', '63' => '22.8', '21' => '62.1',
                    '71' => '16.6', '80' => '11.1', '26' => '57.2', '99' => '5.6', '72' => '15.9',
                    '44' => '39.8', '55' => '29.7', '84' => '9.2', '74' => '14.6', '27' => '56.2',
                    '95' => '6.2', '57' => '28.0', '61' => '24.5', '20' => '63.0', '92' => '6.8',
                    '89' => '7.5', '31' => '52.4', '35' => '48.5', '91' => '7.0', '78' => '12.1',
                    '48' => '36.1', '87' => '8.1', '93' => '6.6', '77' => '12.7', '65' => '21.2',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '22.0', '97' => '5.9',
                    '58' => '27.1', '41' => '42.7', '81' => '10.6', '52' => '32.4', '60' => '25.4',
                    '56' => '28.8', '73' => '15.2', '66' => '20.4', '45' => '38.9', '86' => '8.5',
                    '76' => '13.3', '62' => '23.7', '54' => '30.6', '67' => '19.6', '70' => '17.3',
                    '68' => '18.8', '88' => '7.8', '30' => '53.3', '82' => '10.1', '25' => '58.2',
                    '28' => '55.3', '40' => '43.7', '75' => '13.9', '83' => '9.6', '59' => '26.2',
                    '69' => '18.1', '49' => '35.1', '24' => '59.1', '53' => '31.5', '79' => '11.6',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '6.1',
                    '85' => '8.8', '36' => '47.5', '94' => '6.4', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '5.8', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '77' => {
                    '67' => '21.2', '33' => '50.5', '32' => '51.5', '63' => '23.9', '21' => '62.1',
                    '71' => '18.8', '70' => '19.4', '68' => '20.6', '26' => '57.3', '72' => '18.3',
                    '30' => '53.4', '44' => '40.0', '55' => '30.3', '74' => '17.3', '25' => '58.2',
                    '27' => '56.3', '28' => '55.3', '75' => '16.8', '57' => '28.6', '40' => '43.8',
                    '61' => '25.4', '20' => '63.1', '69' => '19.9', '59' => '27.0', '49' => '35.5',
                    '24' => '59.2', '31' => '52.4', '35' => '48.6', '53' => '32.0', '78' => '15.6',
                    '79' => '15.3', '48' => '36.4', '22' => '61.1', '42' => '41.9', '77' => '16.0',
                    '46' => '38.2', '23' => '60.2', '65' => '22.5', '29' => '54.4', '50' => '34.6',
                    '39' => '44.8', '64' => '23.2', '36' => '47.6', '51' => '33.7', '58' => '27.8',
                    '41' => '42.9', '47' => '37.3', '38' => '45.7', '52' => '32.8', '60' => '26.2',
                    '34' => '49.5', '56' => '29.4', '73' => '17.8', '66' => '21.8', '37' => '46.7',
                    '45' => '39.1', '76' => '16.4', '43' => '41.0', '62' => '24.7', '54' => '31.1'
                  },
          '106' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.8', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '3.1', '80' => '10.3', '26' => '57.2', '99' => '3.6',
                     '72' => '15.6', '44' => '39.8', '55' => '29.6', '84' => '8.2', '74' => '14.1',
                     '27' => '56.2', '95' => '4.5', '57' => '27.9', '61' => '24.4', '108' => '2.2',
                     '20' => '63.0', '109' => '2.1', '92' => '5.2', '103' => '2.9', '89' => '6.2',
                     '31' => '52.4', '35' => '48.5', '91' => '5.5', '78' => '11.5', '107' => '2.3',
                     '48' => '36.0', '87' => '6.9', '93' => '4.9', '77' => '12.2', '106' => '2.4',
                     '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9',
                     '97' => '4.0', '58' => '27.0', '41' => '42.7', '81' => '9.8', '52' => '32.3',
                     '60' => '25.3', '56' => '28.8', '101' => '3.2', '73' => '14.8', '66' => '20.2',
                     '45' => '38.8', '86' => '7.3', '76' => '12.8', '62' => '23.5', '54' => '30.5',
                     '67' => '19.4', '70' => '17.1', '68' => '18.6', '88' => '6.5', '30' => '53.3',
                     '100' => '3.4', '82' => '9.2', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.5', '83' => '8.7', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.7', '53' => '31.4', '79' => '10.9', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '2.6', '96' => '4.2',
                     '85' => '7.8', '36' => '47.5', '94' => '4.7', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '3.8', '34' => '49.4', '37' => '46.5', '43' => '40.8'
                   },
          '65' => {
                    '67' => '25.4', '33' => '50.8', '32' => '51.7', '63' => '27.2', '21' => '62.2',
                    '68' => '25.0', '26' => '57.4', '30' => '53.6', '44' => '40.7', '55' => '32.0',
                    '25' => '58.4', '27' => '56.5', '28' => '55.5', '57' => '30.7', '40' => '44.3',
                    '61' => '28.3', '20' => '63.2', '69' => '24.6', '59' => '29.4', '49' => '36.6',
                    '24' => '59.3', '31' => '52.7', '35' => '48.9', '53' => '33.5', '48' => '37.4',
                    '22' => '61.3', '42' => '42.5', '46' => '39.0', '23' => '60.3', '65' => '26.2',
                    '29' => '54.6', '50' => '35.8', '39' => '45.2', '64' => '26.7', '36' => '48.0',
                    '51' => '35.0', '58' => '30.0', '41' => '43.4', '47' => '38.2', '38' => '46.1',
                    '52' => '34.2', '60' => '28.8', '34' => '49.8', '56' => '31.4', '66' => '25.8',
                    '37' => '47.0', '45' => '39.9', '43' => '41.6', '62' => '27.7', '54' => '32.7'
                  },
          '29' => {
                    '27' => '62.2', '25' => '63.3', '28' => '61.7', '21' => '65.9', '26' => '62.8',
                    '20' => '66.6', '22' => '65.2', '24' => '63.9', '23' => '64.6', '29' => '61.2'
                  },
          '50' => {
                    '33' => '52.0', '32' => '52.9', '21' => '62.7', '26' => '58.1', '30' => '54.6',
                    '44' => '43.8', '55' => '38.3', '25' => '59.0', '27' => '57.2', '28' => '56.3',
                    '57' => '37.6', '40' => '46.5', '20' => '63.6', '59' => '37.1', '49' => '40.9',
                    '24' => '59.9', '31' => '53.7', '35' => '50.4', '53' => '39.1', '48' => '41.5',
                    '22' => '61.8', '42' => '45.1', '46' => '42.6', '23' => '60.8', '29' => '55.4',
                    '50' => '40.4', '39' => '47.3', '36' => '49.6', '51' => '40.0', '58' => '37.3',
                    '41' => '45.8', '47' => '42.0', '38' => '48.0', '52' => '39.5', '34' => '51.2',
                    '56' => '38.0', '37' => '48.8', '45' => '43.2', '43' => '44.4', '54' => '38.7'
                  },
          '39' => {
                    '35' => '53.4', '33' => '54.7', '32' => '55.3', '21' => '63.7', '26' => '59.6',
                    '22' => '62.9', '30' => '56.7', '23' => '62.1', '29' => '57.4', '25' => '60.4',
                    '27' => '58.9', '39' => '51.2', '28' => '58.1', '36' => '52.8', '20' => '64.6',
                    '38' => '51.7', '34' => '54.0', '37' => '52.3', '24' => '61.2', '31' => '56.0'
                  },
          '64' => {
                    '67' => '25.9', '33' => '50.8', '32' => '51.8', '63' => '27.6', '21' => '62.2',
                    '68' => '25.5', '26' => '57.4', '30' => '53.6', '44' => '40.8', '55' => '32.3',
                    '25' => '58.4', '27' => '56.5', '28' => '55.5', '57' => '31.0', '40' => '44.4',
                    '61' => '28.6', '20' => '63.2', '69' => '25.2', '59' => '29.8', '49' => '36.7',
                    '24' => '59.4', '31' => '52.7', '35' => '48.9', '53' => '33.7', '48' => '37.5',
                    '22' => '61.3', '42' => '42.6', '46' => '39.2', '23' => '60.3', '65' => '26.7',
                    '29' => '54.6', '50' => '35.9', '39' => '45.3', '64' => '27.1', '36' => '48.0',
                    '51' => '35.2', '58' => '30.4', '41' => '43.5', '47' => '38.3', '38' => '46.2',
                    '52' => '34.4', '60' => '29.2', '34' => '49.9', '56' => '31.6', '66' => '26.3',
                    '37' => '47.1', '45' => '40.0', '43' => '41.7', '62' => '28.1', '54' => '33.0'
                  },
          '97' => {
                    '33' => '50.4', '32' => '51.4', '90' => '6.6', '63' => '22.8', '21' => '62.1',
                    '71' => '16.5', '80' => '10.7', '26' => '57.2', '99' => '4.9', '72' => '15.8',
                    '44' => '39.8', '55' => '29.7', '84' => '8.8', '74' => '14.4', '27' => '56.2',
                    '95' => '5.5', '57' => '27.9', '61' => '24.5', '20' => '63.0', '92' => '6.1',
                    '89' => '6.9', '31' => '52.4', '35' => '48.5', '91' => '6.4', '78' => '11.9',
                    '48' => '36.1', '87' => '7.6', '93' => '5.9', '77' => '12.5', '65' => '21.1',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9', '97' => '5.2',
                    '58' => '27.0', '41' => '42.7', '81' => '10.2', '52' => '32.4', '60' => '25.3',
                    '56' => '28.8', '73' => '15.1', '66' => '20.3', '45' => '38.9', '86' => '8.0',
                    '76' => '13.1', '62' => '23.6', '54' => '30.6', '67' => '19.5', '70' => '17.2',
                    '68' => '18.7', '88' => '7.3', '30' => '53.3', '82' => '9.7', '25' => '58.2',
                    '28' => '55.3', '40' => '43.6', '75' => '13.7', '83' => '9.2', '59' => '26.2',
                    '69' => '18.0', '49' => '35.1', '24' => '59.1', '53' => '31.5', '79' => '11.3',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '5.3',
                    '85' => '8.4', '36' => '47.5', '94' => '5.7', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '5.0', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                  },
          '114' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.6', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '2.5', '80' => '10.2', '26' => '57.2', '99' => '3.2',
                     '72' => '15.5', '44' => '39.8', '55' => '29.6', '84' => '8.1', '74' => '14.1',
                     '27' => '56.2', '95' => '4.1', '57' => '27.9', '61' => '24.4', '108' => '1.4',
                     '20' => '63.0', '109' => '1.3', '92' => '4.9', '103' => '2.3', '89' => '5.9',
                     '31' => '52.4', '113' => '1.0', '35' => '48.5', '91' => '5.2', '78' => '11.4',
                     '107' => '1.6', '48' => '36.0', '87' => '6.7', '93' => '4.6', '77' => '12.1',
                     '106' => '1.8', '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6',
                     '64' => '21.8', '97' => '3.6', '114' => '1.0', '58' => '27.0', '41' => '42.7',
                     '81' => '9.7', '52' => '32.3', '60' => '25.2', '56' => '28.7', '101' => '2.7',
                     '73' => '14.8', '66' => '20.2', '45' => '38.8', '86' => '7.1', '76' => '12.7',
                     '62' => '23.5', '54' => '30.5', '67' => '19.4', '70' => '17.0', '68' => '18.6',
                     '115+' => '1.0', '88' => '6.3', '30' => '53.3', '100' => '3.0', '110' => '1.1',
                     '82' => '9.1', '25' => '58.2', '28' => '55.3', '40' => '43.6', '75' => '13.4',
                     '83' => '8.6', '112' => '1.0', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.1', '53' => '31.4', '79' => '10.8', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '1.9', '96' => '3.9',
                     '85' => '7.6', '36' => '47.5', '94' => '4.3', '111' => '1.0', '51' => '33.3',
                     '47' => '37.0', '38' => '45.6', '98' => '3.4', '34' => '49.4', '37' => '46.5',
                     '43' => '40.7'
                   },
          '58' => {
                    '33' => '51.2', '32' => '52.1', '21' => '62.4', '26' => '57.6', '30' => '53.9',
                    '44' => '41.7', '55' => '34.3', '25' => '58.6', '27' => '56.7', '28' => '55.8',
                    '57' => '33.2', '40' => '45.0', '20' => '63.3', '59' => '32.3', '49' => '38.0',
                    '24' => '59.5', '31' => '53.0', '35' => '49.4', '53' => '35.4', '48' => '38.7',
                    '22' => '61.4', '42' => '43.3', '46' => '40.2', '23' => '60.5', '29' => '54.8',
                    '50' => '37.3', '39' => '45.8', '36' => '48.5', '51' => '36.7', '58' => '32.8',
                    '41' => '44.2', '47' => '39.4', '38' => '46.7', '52' => '36.0', '34' => '50.3',
                    '56' => '33.7', '37' => '47.6', '45' => '40.9', '43' => '42.5', '54' => '34.8'
                  },
          '41' => {
                    '35' => '52.7', '33' => '54.0', '32' => '54.7', '21' => '63.5', '26' => '59.3',
                    '48' => '46.4', '42' => '48.8', '22' => '62.6', '46' => '47.1', '30' => '56.1',
                    '23' => '61.7', '44' => '47.9', '29' => '56.9', '25' => '60.1', '27' => '58.5',
                    '39' => '50.3', '28' => '57.7', '40' => '49.8', '36' => '52.0', '41' => '49.3',
                    '47' => '46.7', '20' => '64.3', '38' => '50.9', '34' => '53.3', '49' => '46.1',
                    '45' => '47.5', '37' => '51.4', '24' => '60.9', '43' => '48.3', '31' => '55.4'
                  },
          '81' => {
                    '33' => '50.5', '32' => '51.4', '63' => '23.4', '21' => '62.1', '71' => '17.9',
                    '80' => '13.4', '26' => '57.2', '72' => '17.3', '44' => '39.9', '55' => '30.0',
                    '84' => '12.2', '74' => '16.2', '27' => '56.3', '57' => '28.3', '61' => '25.0',
                    '20' => '63.1', '89' => '11.1', '31' => '52.4', '35' => '48.5', '78' => '14.2',
                    '48' => '36.2', '87' => '11.4', '77' => '14.7', '65' => '21.9', '29' => '54.3',
                    '50' => '34.4', '39' => '44.7', '64' => '22.7', '58' => '27.5', '41' => '42.8',
                    '81' => '13.1', '52' => '32.6', '60' => '25.8', '56' => '29.2', '73' => '16.7',
                    '66' => '21.2', '45' => '39.0', '86' => '11.7', '76' => '15.1', '62' => '24.2',
                    '54' => '30.9', '67' => '20.5', '70' => '18.5', '68' => '19.8', '88' => '11.3',
                    '30' => '53.4', '82' => '12.7', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '15.6', '83' => '12.4', '59' => '26.6', '69' => '19.1', '49' => '35.3',
                    '24' => '59.2', '53' => '31.8', '79' => '13.8', '42' => '41.8', '22' => '61.1',
                    '46' => '38.1', '23' => '60.1', '85' => '11.9', '36' => '47.6', '51' => '33.5',
                    '47' => '37.2', '38' => '45.7', '34' => '49.5', '37' => '46.6', '43' => '40.9'
                  },
          '52' => {
                    '33' => '51.7', '32' => '52.6', '21' => '62.6', '26' => '58.0', '30' => '54.4',
                    '44' => '43.2', '55' => '37.2', '25' => '58.9', '27' => '57.1', '28' => '56.1',
                    '57' => '36.4', '40' => '46.0', '20' => '63.5', '59' => '35.7', '49' => '40.1',
                    '24' => '59.8', '31' => '53.5', '35' => '50.0', '53' => '38.0', '48' => '40.6',
                    '22' => '61.7', '42' => '44.6', '46' => '41.8', '23' => '60.7', '29' => '55.2',
                    '50' => '39.5', '39' => '46.8', '36' => '49.2', '51' => '39.0', '58' => '36.0',
                    '41' => '45.3', '47' => '41.2', '38' => '47.6', '52' => '38.5', '34' => '50.9',
                    '56' => '36.8', '37' => '48.4', '45' => '42.5', '43' => '43.8', '54' => '37.6'
                  },
          '60' => {
                    '67' => '28.2', '33' => '51.0', '32' => '51.9', '63' => '29.6', '21' => '62.3',
                    '68' => '27.9', '26' => '57.6', '30' => '53.8', '44' => '41.4', '55' => '33.5',
                    '25' => '58.5', '27' => '56.6', '28' => '55.7', '57' => '32.4', '40' => '44.7',
                    '61' => '30.4', '20' => '63.3', '69' => '27.6', '59' => '31.3', '49' => '37.5',
                    '24' => '59.5', '31' => '52.9', '35' => '49.2', '53' => '34.8', '48' => '38.2',
                    '22' => '61.4', '42' => '43.0', '46' => '39.8', '23' => '60.4', '65' => '28.8',
                    '29' => '54.7', '50' => '36.8', '39' => '45.6', '64' => '29.2', '36' => '48.3',
                    '51' => '36.1', '58' => '31.9', '41' => '43.9', '47' => '39.0', '38' => '46.5',
                    '52' => '35.4', '60' => '30.9', '34' => '50.1', '56' => '32.9', '66' => '28.5',
                    '37' => '47.4', '45' => '40.6', '43' => '42.2', '62' => '30.0', '54' => '34.1'
                  },
          '56' => {
                    '33' => '51.3', '32' => '52.2', '21' => '62.4', '26' => '57.7', '30' => '54.0',
                    '44' => '42.1', '55' => '35.1', '25' => '58.7', '27' => '56.8', '28' => '55.9',
                    '57' => '34.2', '40' => '45.3', '20' => '63.4', '59' => '33.3', '49' => '38.6',
                    '24' => '59.6', '31' => '53.1', '35' => '49.5', '53' => '36.2', '48' => '39.3',
                    '22' => '61.5', '42' => '43.7', '46' => '40.7', '23' => '60.5', '29' => '54.9',
                    '50' => '38.0', '39' => '46.1', '36' => '48.7', '51' => '37.4', '58' => '33.7',
                    '41' => '44.5', '47' => '40.0', '38' => '47.0', '52' => '36.8', '34' => '50.4',
                    '56' => '34.7', '37' => '47.8', '45' => '41.4', '43' => '42.9', '54' => '35.7'
                  },
          '101' => {
                     '33' => '50.4', '32' => '51.4', '90' => '6.2', '63' => '22.7', '21' => '62.1',
                     '71' => '16.4', '102' => '3.7', '80' => '10.5', '26' => '57.2', '99' => '4.2',
                     '72' => '15.6', '44' => '39.8', '55' => '29.7', '84' => '8.5', '74' => '14.2',
                     '27' => '56.2', '95' => '5.0', '57' => '27.9', '61' => '24.4', '108' => '3.1',
                     '20' => '63.0', '109' => '3.0', '92' => '5.6', '103' => '3.6', '89' => '6.5',
                     '31' => '52.4', '35' => '48.5', '91' => '5.9', '78' => '11.7', '107' => '3.1',
                     '48' => '36.0', '87' => '7.2', '93' => '5.4', '77' => '12.3', '106' => '3.2',
                     '65' => '21.1', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9',
                     '97' => '4.6', '58' => '27.0', '41' => '42.7', '81' => '10.0', '52' => '32.4',
                     '60' => '25.3', '56' => '28.8', '101' => '3.9', '73' => '14.9', '66' => '20.2',
                     '45' => '38.9', '86' => '7.6', '76' => '12.9', '62' => '23.6', '54' => '30.6',
                     '67' => '19.4', '70' => '17.1', '68' => '18.7', '88' => '6.9', '30' => '53.3',
                     '100' => '4.1', '82' => '9.4', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.6', '83' => '9.0', '59' => '26.1', '69' => '17.9', '49' => '35.1',
                     '24' => '59.1', '104' => '3.5', '53' => '31.5', '79' => '11.1', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '3.4', '96' => '4.8',
                     '85' => '8.0', '36' => '47.5', '94' => '5.2', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '4.4', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                   },
          '73' => {
                    '67' => '22.2', '33' => '50.6', '32' => '51.5', '63' => '24.7', '21' => '62.1',
                    '71' => '20.1', '70' => '20.6', '68' => '21.6', '26' => '57.3', '72' => '19.6',
                    '30' => '53.4', '44' => '40.2', '55' => '30.6', '74' => '18.8', '25' => '58.3',
                    '27' => '56.3', '28' => '55.4', '75' => '18.4', '57' => '29.1', '40' => '43.9',
                    '61' => '26.1', '20' => '63.1', '69' => '21.1', '59' => '27.5', '49' => '35.7',
                    '24' => '59.2', '31' => '52.5', '35' => '48.6', '53' => '32.3', '78' => '17.5',
                    '79' => '17.2', '48' => '36.6', '22' => '61.2', '42' => '42.0', '77' => '17.8',
                    '46' => '38.4', '23' => '60.2', '65' => '23.4', '29' => '54.4', '50' => '34.8',
                    '39' => '44.8', '64' => '24.0', '36' => '47.7', '51' => '34.0', '58' => '28.3',
                    '41' => '43.0', '47' => '37.5', '38' => '45.8', '52' => '33.1', '60' => '26.8',
                    '34' => '49.6', '56' => '29.8', '73' => '19.2', '66' => '22.8', '37' => '46.7',
                    '45' => '39.3', '76' => '18.1', '43' => '41.1', '62' => '25.4', '54' => '31.5'
                  },
          '66' => {
                    '67' => '24.9', '33' => '50.7', '32' => '51.7', '63' => '26.8', '21' => '62.2',
                    '68' => '24.5', '26' => '57.4', '30' => '53.6', '44' => '40.6', '55' => '31.8',
                    '25' => '58.4', '27' => '56.4', '28' => '55.5', '57' => '30.4', '40' => '44.2',
                    '61' => '27.9', '20' => '63.2', '69' => '24.1', '59' => '29.1', '49' => '36.4',
                    '24' => '59.3', '31' => '52.6', '35' => '48.9', '53' => '33.3', '48' => '37.2',
                    '22' => '61.2', '42' => '42.4', '46' => '38.9', '23' => '60.3', '65' => '25.8',
                    '29' => '54.5', '50' => '35.6', '39' => '45.1', '64' => '26.3', '36' => '47.9',
                    '51' => '34.8', '58' => '29.8', '41' => '43.3', '47' => '38.1', '38' => '46.1',
                    '52' => '34.0', '60' => '28.5', '34' => '49.8', '56' => '31.1', '66' => '25.3',
                    '37' => '47.0', '45' => '39.8', '43' => '41.5', '62' => '27.3', '54' => '32.5'
                  },
          '45' => {
                    '35' => '51.5', '33' => '52.9', '32' => '53.7', '21' => '63.0', '26' => '58.6',
                    '48' => '44.0', '42' => '46.9', '22' => '62.2', '46' => '44.8', '30' => '55.3',
                    '23' => '61.3', '44' => '45.8', '29' => '56.1', '25' => '59.5', '27' => '57.8',
                    '39' => '48.7', '28' => '56.9', '40' => '48.1', '36' => '50.7', '41' => '47.5',
                    '47' => '44.4', '20' => '64.0', '38' => '49.4', '34' => '52.2', '49' => '43.6',
                    '45' => '45.3', '37' => '50.0', '24' => '60.4', '43' => '46.3', '31' => '54.5'
                  },
          '86' => {
                    '33' => '50.4', '32' => '51.4', '63' => '23.1', '21' => '62.1', '71' => '17.1',
                    '80' => '12.1', '26' => '57.2', '72' => '16.5', '44' => '39.9', '55' => '29.8',
                    '84' => '10.5', '74' => '15.2', '27' => '56.2', '57' => '28.1', '61' => '24.7',
                    '20' => '63.0', '89' => '9.2', '31' => '52.4', '35' => '48.5', '78' => '13.0',
                    '48' => '36.1', '87' => '9.6', '77' => '13.5', '65' => '21.5', '29' => '54.3',
                    '50' => '34.3', '39' => '44.6', '64' => '22.3', '58' => '27.2', '41' => '42.7',
                    '81' => '11.7', '52' => '32.5', '60' => '25.5', '56' => '29.0', '73' => '15.8',
                    '66' => '20.7', '45' => '38.9', '86' => '9.9', '76' => '14.1', '62' => '23.9',
                    '54' => '30.7', '67' => '20.0', '70' => '17.8', '68' => '19.2', '88' => '9.4',
                    '30' => '53.3', '82' => '11.3', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '14.6', '83' => '10.9', '59' => '26.4', '69' => '18.5', '49' => '35.2',
                    '24' => '59.2', '53' => '31.6', '79' => '12.5', '42' => '41.8', '22' => '61.1',
                    '46' => '38.0', '23' => '60.1', '85' => '10.2', '36' => '47.5', '51' => '33.4',
                    '47' => '37.1', '38' => '45.6', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '76' => {
                    '67' => '21.4', '33' => '50.5', '32' => '51.5', '63' => '24.1', '21' => '62.1',
                    '71' => '19.1', '70' => '19.6', '68' => '20.8', '26' => '57.3', '72' => '18.6',
                    '30' => '53.4', '44' => '40.1', '55' => '30.3', '74' => '17.6', '25' => '58.2',
                    '27' => '56.3', '28' => '55.3', '75' => '17.2', '57' => '28.7', '40' => '43.8',
                    '61' => '25.6', '20' => '63.1', '69' => '20.2', '59' => '27.1', '49' => '35.5',
                    '24' => '59.2', '31' => '52.4', '35' => '48.6', '53' => '32.0', '78' => '16.0',
                    '79' => '15.7', '48' => '36.4', '22' => '61.1', '42' => '41.9', '77' => '16.4',
                    '46' => '38.2', '23' => '60.2', '65' => '22.7', '29' => '54.4', '50' => '34.6',
                    '39' => '44.8', '64' => '23.4', '36' => '47.6', '51' => '33.8', '58' => '27.9',
                    '41' => '42.9', '47' => '37.3', '38' => '45.7', '52' => '32.9', '60' => '26.3',
                    '34' => '49.6', '56' => '29.5', '73' => '18.1', '66' => '22.0', '37' => '46.7',
                    '45' => '39.1', '76' => '16.8', '43' => '41.0', '62' => '24.8', '54' => '31.2'
                  },
          '62' => {
                    '67' => '27.0', '33' => '50.9', '32' => '51.8', '63' => '28.5', '21' => '62.3',
                    '68' => '26.7', '26' => '57.5', '30' => '53.7', '44' => '41.1', '55' => '32.9',
                    '25' => '58.4', '27' => '56.5', '28' => '55.6', '57' => '31.6', '40' => '44.5',
                    '61' => '29.5', '20' => '63.2', '69' => '26.4', '59' => '30.5', '49' => '37.1',
                    '24' => '59.4', '31' => '52.8', '35' => '49.1', '53' => '34.2', '48' => '37.8',
                    '22' => '61.3', '42' => '42.8', '46' => '39.4', '23' => '60.4', '65' => '27.7',
                    '29' => '54.7', '50' => '36.3', '39' => '45.4', '64' => '28.1', '36' => '48.1',
                    '51' => '35.6', '58' => '31.1', '41' => '43.7', '47' => '38.6', '38' => '46.3',
                    '52' => '34.9', '60' => '30.0', '34' => '50.0', '56' => '32.2', '66' => '27.3',
                    '37' => '47.2', '45' => '40.3', '43' => '41.9', '62' => '29.0', '54' => '33.5'
                  },
          '54' => {
                    '33' => '51.5', '32' => '52.4', '21' => '62.5', '26' => '57.8', '30' => '54.2',
                    '44' => '42.6', '55' => '36.1', '25' => '58.8', '27' => '56.9', '28' => '56.0',
                    '57' => '35.2', '40' => '45.6', '20' => '63.5', '59' => '34.5', '49' => '39.3',
                    '24' => '59.7', '31' => '53.3', '35' => '49.8', '53' => '37.1', '48' => '39.9',
                    '22' => '61.6', '42' => '44.1', '46' => '41.2', '23' => '60.6', '29' => '55.1',
                    '50' => '38.7', '39' => '46.4', '36' => '48.9', '51' => '38.1', '58' => '34.8',
                    '41' => '44.8', '47' => '40.5', '38' => '47.2', '52' => '37.6', '34' => '50.6',
                    '56' => '35.7', '37' => '48.1', '45' => '41.9', '43' => '43.3', '54' => '36.6'
                  },
          '67' => {
                    '67' => '24.4', '33' => '50.7', '32' => '51.7', '63' => '26.4', '21' => '62.2',
                    '68' => '24.0', '26' => '57.4', '30' => '53.6', '44' => '40.6', '55' => '31.6',
                    '25' => '58.3', '27' => '56.4', '28' => '55.5', '57' => '30.2', '40' => '44.2',
                    '61' => '27.6', '20' => '63.2', '69' => '23.6', '59' => '28.8', '49' => '36.3',
                    '24' => '59.3', '31' => '52.6', '35' => '48.8', '53' => '33.1', '48' => '37.1',
                    '22' => '61.2', '42' => '42.3', '46' => '38.8', '23' => '60.3', '65' => '25.4',
                    '29' => '54.5', '50' => '35.5', '39' => '45.1', '64' => '25.9', '36' => '47.9',
                    '51' => '34.7', '58' => '29.5', '41' => '43.3', '47' => '38.0', '38' => '46.0',
                    '52' => '33.9', '60' => '28.2', '34' => '49.8', '56' => '30.9', '66' => '24.9',
                    '37' => '46.9', '45' => '39.7', '43' => '41.4', '62' => '27.0', '54' => '32.3'
                  },
          '70' => {
                    '67' => '23.2', '33' => '50.6', '32' => '51.6', '63' => '25.4', '21' => '62.2',
                    '71' => '21.3', '70' => '21.8', '68' => '22.7', '26' => '57.3', '72' => '20.9',
                    '30' => '53.5', '44' => '40.3', '55' => '31.1', '74' => '20.2', '25' => '58.3',
                    '27' => '56.4', '28' => '55.4', '75' => '19.9', '57' => '29.5', '40' => '44.0',
                    '61' => '26.7', '20' => '63.1', '69' => '22.2', '59' => '28.1', '49' => '35.9',
                    '24' => '59.3', '31' => '52.5', '35' => '48.7', '53' => '32.6', '78' => '19.1',
                    '79' => '18.9', '48' => '36.8', '22' => '61.2', '42' => '42.2', '77' => '19.4',
                    '46' => '38.6', '23' => '60.2', '65' => '24.3', '29' => '54.4', '50' => '35.1',
                    '39' => '44.9', '64' => '24.8', '36' => '47.8', '51' => '34.3', '58' => '28.8',
                    '41' => '43.1', '47' => '37.7', '38' => '45.9', '52' => '33.4', '60' => '27.4',
                    '34' => '49.7', '56' => '30.3', '73' => '20.6', '66' => '23.7', '37' => '46.8',
                    '45' => '39.4', '76' => '19.6', '43' => '41.3', '62' => '26.1', '54' => '31.8'
                  },
          '68' => {
                    '67' => '24.0', '33' => '50.7', '32' => '51.6', '63' => '26.1', '21' => '62.2',
                    '68' => '23.5', '26' => '57.4', '30' => '53.5', '44' => '40.5', '55' => '31.4',
                    '25' => '58.3', '27' => '56.4', '28' => '55.4', '57' => '29.9', '40' => '44.1',
                    '61' => '27.3', '20' => '63.1', '69' => '23.1', '59' => '28.6', '49' => '36.2',
                    '24' => '59.3', '31' => '52.6', '35' => '48.8', '53' => '32.9', '48' => '37.0',
                    '22' => '61.2', '42' => '42.3', '46' => '38.7', '23' => '60.2', '65' => '25.0',
                    '29' => '54.5', '50' => '35.3', '39' => '45.0', '64' => '25.5', '36' => '47.8',
                    '51' => '34.5', '58' => '29.2', '41' => '43.2', '47' => '37.9', '38' => '46.0',
                    '52' => '33.7', '60' => '27.9', '34' => '49.7', '56' => '30.7', '66' => '24.5',
                    '37' => '46.9', '45' => '39.6', '43' => '41.4', '62' => '26.7', '54' => '32.1'
                  },
          '88' => {
                    '33' => '50.4', '32' => '51.4', '63' => '23.0', '21' => '62.1', '71' => '16.9',
                    '80' => '11.7', '26' => '57.2', '72' => '16.3', '44' => '39.9', '55' => '29.8',
                    '84' => '10.1', '74' => '15.0', '27' => '56.2', '57' => '28.0', '61' => '24.6',
                    '20' => '63.0', '89' => '8.6', '31' => '52.4', '35' => '48.5', '78' => '12.7',
                    '48' => '36.1', '87' => '9.1', '77' => '13.2', '65' => '21.4', '29' => '54.3',
                    '50' => '34.3', '39' => '44.6', '64' => '22.2', '58' => '27.2', '41' => '42.7',
                    '81' => '11.3', '52' => '32.5', '60' => '25.5', '56' => '28.9', '73' => '15.6',
                    '66' => '20.6', '45' => '38.9', '86' => '9.4', '76' => '13.8', '62' => '23.8',
                    '54' => '30.7', '67' => '19.8', '70' => '17.6', '68' => '19.1', '88' => '8.8',
                    '30' => '53.3', '82' => '10.8', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '14.4', '83' => '10.4', '59' => '26.3', '69' => '18.3', '49' => '35.2',
                    '24' => '59.2', '53' => '31.6', '79' => '12.2', '42' => '41.8', '22' => '61.1',
                    '46' => '38.0', '23' => '60.1', '85' => '9.7', '36' => '47.5', '51' => '33.4',
                    '47' => '37.0', '38' => '45.6', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '30' => {
                    '35' => '58.0', '33' => '58.8', '32' => '59.2', '21' => '65.6', '26' => '62.3',
                    '22' => '64.9', '30' => '60.2', '23' => '64.2', '29' => '60.7', '25' => '62.9',
                    '27' => '61.8', '39' => '56.7', '28' => '61.2', '36' => '57.6', '20' => '66.3',
                    '38' => '57.0', '34' => '58.4', '37' => '57.3', '24' => '63.6', '31' => '59.7'
                  },
          '100' => {
                     '33' => '50.4', '32' => '51.4', '90' => '6.3', '63' => '22.7', '21' => '62.1',
                     '71' => '16.4', '102' => '3.9', '80' => '10.6', '26' => '57.2', '99' => '4.4',
                     '72' => '15.7', '44' => '39.8', '55' => '29.7', '84' => '8.5', '74' => '14.3',
                     '27' => '56.2', '95' => '5.1', '57' => '27.9', '61' => '24.4', '108' => '3.3',
                     '20' => '63.0', '109' => '3.2', '92' => '5.8', '103' => '3.8', '89' => '6.6',
                     '31' => '52.4', '35' => '48.5', '91' => '6.0', '78' => '11.7', '107' => '3.3',
                     '48' => '36.0', '87' => '7.3', '93' => '5.5', '77' => '12.3', '106' => '3.4',
                     '65' => '21.1', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9',
                     '97' => '4.7', '58' => '27.0', '41' => '42.7', '81' => '10.0', '52' => '32.4',
                     '60' => '25.3', '56' => '28.8', '101' => '4.1', '73' => '15.0', '66' => '20.3',
                     '45' => '38.9', '86' => '7.7', '76' => '12.9', '62' => '23.6', '54' => '30.6',
                     '67' => '19.5', '70' => '17.1', '68' => '18.7', '88' => '6.9', '30' => '53.3',
                     '100' => '4.2', '82' => '9.5', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.6', '83' => '9.0', '59' => '26.1', '69' => '17.9', '49' => '35.1',
                     '24' => '59.1', '104' => '3.7', '53' => '31.5', '79' => '11.1', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '3.5', '96' => '4.9',
                     '85' => '8.1', '36' => '47.5', '94' => '5.3', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '4.5', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                   },
          '110' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.6', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '2.7', '80' => '10.3', '26' => '57.2', '99' => '3.3',
                     '72' => '15.5', '44' => '39.8', '55' => '29.6', '84' => '8.1', '74' => '14.1',
                     '27' => '56.2', '95' => '4.2', '57' => '27.9', '61' => '24.4', '108' => '1.7',
                     '20' => '63.0', '109' => '1.6', '92' => '5.0', '103' => '2.5', '89' => '6.0',
                     '31' => '52.4', '113' => '1.2', '35' => '48.5', '91' => '5.3', '78' => '11.5',
                     '107' => '1.9', '48' => '36.0', '87' => '6.8', '93' => '4.7', '77' => '12.1',
                     '106' => '2.0', '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6',
                     '64' => '21.8', '97' => '3.8', '114' => '1.1', '58' => '27.0', '41' => '42.7',
                     '81' => '9.7', '52' => '32.3', '60' => '25.2', '56' => '28.7', '101' => '2.9',
                     '73' => '14.8', '66' => '20.2', '45' => '38.8', '86' => '7.2', '76' => '12.7',
                     '62' => '23.5', '54' => '30.5', '67' => '19.4', '70' => '17.0', '68' => '18.6',
                     '115' => '1.1', '88' => '6.4', '30' => '53.3', '100' => '3.1', '110' => '1.5',
                     '82' => '9.2', '25' => '58.2', '28' => '55.3', '40' => '43.6', '75' => '13.4',
                     '83' => '8.6', '112' => '1.3', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.3', '53' => '31.4', '79' => '10.9', '42' => '41.7',
                     '22' => '61.1', '' => '', '46' => '37.9', '23' => '60.1', '105' => '2.2',
                     '96' => '4.0', '85' => '7.7', '36' => '47.5', '94' => '4.5', '111' => '1.4',
                     '51' => '33.3', '47' => '37.0', '38' => '45.6', '98' => '3.5', '34' => '49.4',
                     '37' => '46.5', '43' => '40.7'
                   },
          '82' => {
                    '33' => '50.5', '32' => '51.4', '63' => '23.4', '21' => '62.1', '71' => '17.7',
                    '80' => '13.1', '26' => '57.2', '72' => '17.1', '44' => '39.9', '55' => '30.0',
                    '84' => '11.8', '74' => '15.9', '27' => '56.3', '57' => '28.3', '61' => '24.9',
                    '20' => '63.1', '89' => '10.6', '31' => '52.4', '35' => '48.5', '78' => '13.9',
                    '48' => '36.2', '87' => '11.0', '77' => '14.4', '65' => '21.8', '29' => '54.3',
                    '50' => '34.4', '39' => '44.7', '64' => '22.6', '58' => '27.4', '41' => '42.8',
                    '81' => '12.7', '52' => '32.6', '60' => '25.8', '56' => '29.1', '73' => '16.5',
                    '66' => '21.1', '45' => '39.0', '86' => '11.3', '76' => '14.9', '62' => '24.1',
                    '54' => '30.8', '67' => '20.4', '70' => '18.3', '68' => '19.7', '88' => '10.8',
                    '30' => '53.4', '82' => '12.4', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '15.4', '83' => '12.1', '59' => '26.6', '69' => '19.0', '49' => '35.3',
                    '24' => '59.2', '53' => '31.7', '79' => '13.5', '42' => '41.8', '22' => '61.1',
                    '46' => '38.1', '23' => '60.1', '85' => '11.5', '36' => '47.6', '51' => '33.5',
                    '47' => '37.1', '38' => '45.6', '34' => '49.5', '37' => '46.6', '43' => '40.9'
                  },
          '25' => {
                    '27' => '64.2', '25' => '65.1', '28' => '63.7', '21' => '67.3', '26' => '64.6',
                    '20' => '67.9', '22' => '66.7', '24' => '65.6', '23' => '66.2', '29' => '63.3'
                  },
          '28' => {
                    '27' => '62.7', '25' => '63.7', '28' => '62.1', '21' => '66.2', '26' => '63.2',
                    '20' => '66.9', '22' => '65.6', '24' => '64.3', '23' => '64.9', '29' => '61.7'
                  },
          '40' => {
                    '35' => '53.0', '33' => '54.3', '32' => '55.0', '21' => '63.6', '26' => '59.4',
                    '48' => '47.1', '42' => '49.3', '22' => '62.7', '46' => '47.7', '30' => '56.4',
                    '23' => '61.9', '44' => '48.5', '29' => '57.1', '25' => '60.2', '27' => '58.7',
                    '39' => '50.8', '28' => '57.9', '40' => '50.2', '36' => '52.4', '41' => '49.8',
                    '47' => '47.4', '20' => '64.4', '38' => '51.3', '34' => '53.7', '49' => '46.8',
                    '45' => '48.1', '37' => '51.8', '24' => '61.1', '43' => '48.9', '31' => '55.7'
                  },
          '75' => {
                    '67' => '21.6', '33' => '50.5', '32' => '51.5', '63' => '24.3', '21' => '62.1',
                    '71' => '19.4', '70' => '19.9', '68' => '21.0', '26' => '57.3', '72' => '18.9',
                    '30' => '53.4', '44' => '40.1', '55' => '30.4', '74' => '18.0', '25' => '58.2',
                    '27' => '56.3', '28' => '55.3', '75' => '17.6', '57' => '28.8', '40' => '43.8',
                    '61' => '25.7', '20' => '63.1', '69' => '20.5', '59' => '27.2', '49' => '35.6',
                    '24' => '59.2', '31' => '52.5', '35' => '48.6', '53' => '32.1', '78' => '16.5',
                    '79' => '16.2', '48' => '36.5', '22' => '61.1', '42' => '42.0', '77' => '16.8',
                    '46' => '38.3', '23' => '60.2', '65' => '22.9', '29' => '54.4', '50' => '34.7',
                    '39' => '44.8', '64' => '23.6', '36' => '47.7', '51' => '33.8', '58' => '28.0',
                    '41' => '42.9', '47' => '37.4', '38' => '45.7', '52' => '33.0', '60' => '26.5',
                    '34' => '49.6', '56' => '29.6', '73' => '18.4', '66' => '22.3', '37' => '46.7',
                    '45' => '39.2', '76' => '17.2', '43' => '41.0', '62' => '25.0', '54' => '31.3'
                  },
          '83' => {
                    '33' => '50.5', '32' => '51.4', '63' => '23.3', '21' => '62.1', '71' => '17.5',
                    '80' => '12.8', '26' => '57.2', '72' => '16.9', '44' => '39.9', '55' => '29.9',
                    '84' => '11.4', '74' => '15.7', '27' => '56.3', '57' => '28.2', '61' => '24.9',
                    '20' => '63.1', '89' => '10.2', '31' => '52.4', '35' => '48.5', '78' => '13.7',
                    '48' => '36.2', '87' => '10.6', '77' => '14.2', '65' => '21.7', '29' => '54.3',
                    '50' => '34.4', '39' => '44.7', '64' => '22.5', '58' => '27.4', '41' => '42.8',
                    '81' => '12.4', '52' => '32.6', '60' => '25.7', '56' => '29.1', '73' => '16.3',
                    '66' => '21.0', '45' => '39.0', '86' => '10.9', '76' => '14.7', '62' => '24.1',
                    '54' => '30.8', '67' => '20.2', '70' => '18.2', '68' => '19.5', '88' => '10.4',
                    '30' => '53.4', '82' => '12.1', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '15.2', '83' => '11.7', '59' => '26.5', '69' => '18.8', '49' => '35.3',
                    '24' => '59.2', '53' => '31.7', '79' => '13.2', '42' => '41.8', '22' => '61.1',
                    '46' => '38.0', '23' => '60.1', '85' => '11.1', '36' => '47.6', '51' => '33.5',
                    '47' => '37.1', '38' => '45.6', '34' => '49.5', '37' => '46.6', '43' => '40.9'
                  },
          '112' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.6', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '2.6', '80' => '10.2', '26' => '57.2', '99' => '3.2',
                     '72' => '15.5', '44' => '39.8', '55' => '29.6', '84' => '8.1', '74' => '14.1',
                     '27' => '56.2', '95' => '4.1', '57' => '27.9', '61' => '24.4', '108' => '1.5',
                     '20' => '63.0', '109' => '1.4', '92' => '4.9', '103' => '2.4', '89' => '5.9',
                     '31' => '52.4', '113' => '1.0', '35' => '48.5', '91' => '5.3', '78' => '11.5',
                     '107' => '1.7', '48' => '36.0', '87' => '6.7', '93' => '4.7', '77' => '12.1',
                     '106' => '1.9', '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6',
                     '64' => '21.8', '97' => '3.7', '114' => '1.0', '58' => '27.0', '41' => '42.7',
                     '81' => '9.7', '52' => '32.3', '60' => '25.2', '56' => '28.7', '101' => '2.8',
                     '73' => '14.8', '66' => '20.2', '45' => '38.8', '86' => '7.2', '76' => '12.7',
                     '62' => '23.5', '54' => '30.5', '67' => '19.4', '70' => '17.0', '68' => '18.6',
                     '115+' => '1.0', '88' => '6.3', '30' => '53.3', '100' => '3.0', '110' => '1.3',
                     '82' => '9.1', '25' => '58.2', '28' => '55.3', '40' => '43.6', '75' => '13.4',
                     '83' => '8.6', '112' => '1.0', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.2', '53' => '31.4', '79' => '10.8', '42' => '41.7',
                     '22' => '61.1', '' => '', '46' => '37.9', '23' => '60.1', '105' => '2.0',
                     '96' => '3.9', '85' => '7.6', '36' => '47.5', '94' => '4.4', '111' => '1.1',
                     '51' => '33.3', '47' => '37.0', '38' => '45.6', '98' => '3.5', '34' => '49.4',
                     '37' => '46.5',
                     '43' => '40.7'
                   },
          '59' => {
                    '33' => '51.1', '32' => '52.0', '21' => '62.3', '26' => '57.6', '30' => '53.8',
                    '44' => '41.5', '55' => '33.9', '25' => '58.5', '27' => '56.7', '28' => '55.7',
                    '57' => '32.8', '40' => '44.9', '20' => '63.3', '59' => '31.8', '49' => '37.8',
                    '24' => '59.5', '31' => '52.9', '35' => '49.3', '53' => '35.1', '48' => '38.5',
                    '22' => '61.4', '42' => '43.2', '46' => '40.0', '23' => '60.4', '29' => '54.8',
                    '50' => '37.1', '39' => '45.7', '36' => '48.4', '51' => '36.4', '58' => '32.3',
                    '41' => '44.0', '47' => '39.2', '38' => '46.6', '52' => '35.7', '34' => '50.2',
                    '56' => '33.3', '37' => '47.5', '45' => '40.7', '43' => '42.4', '54' => '34.5'
                  },
          '69' => {
                    '67' => '23.6', '33' => '50.6', '32' => '51.6', '63' => '25.7', '21' => '62.2',
                    '68' => '23.1', '26' => '57.3', '30' => '53.5', '44' => '40.4', '55' => '31.2',
                    '25' => '58.3', '27' => '56.4', '28' => '55.4', '57' => '29.7', '40' => '44.1',
                    '61' => '27.0', '20' => '63.1', '69' => '22.6', '59' => '28.3', '49' => '36.0',
                    '24' => '59.3', '31' => '52.6', '35' => '48.7', '53' => '32.8', '48' => '36.9',
                    '22' => '61.2', '42' => '42.2', '46' => '38.6', '23' => '60.2', '65' => '24.6',
                    '29' => '54.5', '50' => '35.2', '39' => '45.0', '64' => '25.2', '36' => '47.8',
                    '51' => '34.4', '58' => '29.0', '41' => '43.1', '47' => '37.8', '38' => '45.9',
                    '52' => '33.6', '60' => '27.6', '34' => '49.7', '56' => '30.5', '66' => '24.1',
                    '37' => '46.9', '45' => '39.5', '43' => '41.3', '62' => '26.4', '54' => '32.0'
                  },
          '49' => {
                    '35' => '50.6', '33' => '52.2', '32' => '53.0', '21' => '62.8', '26' => '58.2',
                    '48' => '41.9', '42' => '45.4', '22' => '61.8', '46' => '43.0', '30' => '54.7',
                    '23' => '60.9', '44' => '44.2', '29' => '55.6', '25' => '59.1', '27' => '57.3',
                    '39' => '47.5', '28' => '56.4', '40' => '46.8', '36' => '49.8', '41' => '46.1',
                    '47' => '42.4', '20' => '63.7', '38' => '48.2', '34' => '51.4', '49' => '41.4',
                    '45' => '43.6', '37' => '49.0', '24' => '60.0', '43' => '44.8', '31' => '53.8'
                  },
          '24' => {
                    '27' => '64.7', '25' => '65.6', '28' => '64.3', '21' => '67.7', '26' => '65.2',
                    '20' => '68.3', '22' => '67.2', '24' => '66.1', '23' => '66.6', '29' => '63.9'
                  },
          '104' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.9', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '3.3', '80' => '10.4', '26' => '57.2', '99' => '3.8',
                     '72' => '15.6', '44' => '39.8', '55' => '29.6', '84' => '8.3', '74' => '14.2',
                     '27' => '56.2', '95' => '4.6', '57' => '27.9', '61' => '24.4', '108' => '2.5',
                     '20' => '63.0', '109' => '2.4', '92' => '5.4', '103' => '3.2', '89' => '6.3',
                     '31' => '52.4', '35' => '48.5', '91' => '5.6', '78' => '11.6', '107' => '2.6',
                     '48' => '36.0', '87' => '7.0', '93' => '5.1', '77' => '12.2', '106' => '2.7',
                     '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9',
                     '97' => '4.2', '58' => '27.0', '41' => '42.7', '81' => '9.8', '52' => '32.4',
                     '60' => '25.3', '56' => '28.8', '101' => '3.5', '73' => '14.9', '66' => '20.2',
                     '45' => '38.8', '86' => '7.4', '76' => '12.8', '62' => '23.5', '54' => '30.5',
                     '67' => '19.4', '70' => '17.1', '68' => '18.6', '88' => '6.6', '30' => '53.3',
                     '100' => '3.7', '82' => '9.3', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.5', '83' => '8.8', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '3.0', '53' => '31.4', '79' => '11.0', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '2.9', '96' => '4.4',
                     '85' => '7.9', '36' => '47.5', '94' => '4.9', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '4.0', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                   },
          '53' => {
                    '33' => '51.6', '32' => '52.5', '21' => '62.5', '26' => '57.9', '30' => '54.3',
                    '44' => '42.9', '55' => '36.6', '25' => '58.8', '27' => '57.0', '28' => '56.1',
                    '57' => '35.8', '40' => '45.8', '20' => '63.5', '59' => '35.1', '49' => '39.7',
                    '24' => '59.7', '31' => '53.4', '35' => '49.9', '53' => '37.5', '48' => '40.3',
                    '22' => '61.6', '42' => '44.3', '46' => '41.5', '23' => '60.7', '29' => '55.2',
                    '50' => '39.1', '39' => '46.6', '36' => '49.1', '51' => '38.5', '58' => '35.4',
                    '41' => '45.1', '47' => '40.9', '38' => '47.4', '52' => '38.0', '34' => '50.8',
                    '56' => '36.2', '37' => '48.2', '45' => '42.2', '43' => '43.6', '54' => '37.1'
                  },
          '79' => {
                    '67' => '20.8', '33' => '50.5', '32' => '51.5', '63' => '23.7', '21' => '62.1',
                    '71' => '18.3', '70' => '18.9', '68' => '20.1', '26' => '57.2', '72' => '17.7',
                    '30' => '53.4', '44' => '40.0', '55' => '30.1', '74' => '16.7', '25' => '58.2',
                    '27' => '56.3', '28' => '55.3', '75' => '16.2', '57' => '28.4', '40' => '43.8',
                    '61' => '25.2', '20' => '63.1', '69' => '19.5', '59' => '26.8', '49' => '35.4',
                    '24' => '59.2', '31' => '52.4', '35' => '48.6', '53' => '31.8', '78' => '14.9',
                    '79' => '14.5', '48' => '36.3', '22' => '61.1', '42' => '41.9', '77' => '15.3',
                    '46' => '38.1', '23' => '60.2', '65' => '22.2', '29' => '54.3', '50' => '34.5',
                    '39' => '44.7', '64' => '22.9', '36' => '47.6', '51' => '33.6', '58' => '27.6',
                    '41' => '42.8', '47' => '37.2', '38' => '45.7', '52' => '32.7', '60' => '26.0',
                    '34' => '49.5', '56' => '29.3', '73' => '17.2', '66' => '21.5', '37' => '46.6',
                    '45' => '39.1', '76' => '15.7', '43' => '40.9', '62' => '24.4', '54' => '31.0'
                  },
          '42' => {
                    '35' => '52.3', '33' => '53.7', '32' => '54.4', '21' => '63.3', '26' => '59.1',
                    '48' => '45.8', '42' => '48.3', '22' => '62.5', '46' => '46.5', '30' => '55.9',
                    '23' => '61.6', '44' => '47.3', '29' => '56.7', '25' => '59.9', '27' => '58.3',
                    '39' => '49.9', '28' => '57.5', '40' => '49.3', '36' => '51.7', '41' => '48.8',
                    '47' => '46.1', '20' => '64.2', '38' => '50.4', '34' => '53.0', '49' => '45.4',
                    '45' => '46.9', '37' => '51.1', '24' => '60.8', '43' => '47.8', '31' => '55.2'
                  },
          '22' => {
                    '27' => '65.9', '25' => '66.7', '28' => '65.6', '21' => '68.6', '26' => '66.3',
                    '20' => '69.1', '22' => '68.1', '24' => '67.2', '23' => '67.6', '29' => '65.2'
                  },
          '46' => {
                    '35' => '51.2', '33' => '52.7', '32' => '53.5', '21' => '63.0', '26' => '58.5',
                    '48' => '43.4', '42' => '46.5', '22' => '62.1', '46' => '44.3', '30' => '55.1',
                    '23' => '61.2', '44' => '45.4', '29' => '56.0', '25' => '59.4', '27' => '57.7',
                    '39' => '48.4', '28' => '56.8', '40' => '47.7', '36' => '50.5', '41' => '47.1',
                    '47' => '43.9', '20' => '63.9', '38' => '49.1', '34' => '52.0', '49' => '43.0',
                    '45' => '44.8', '37' => '49.8', '24' => '60.3', '43' => '45.9', '31' => '54.3'
                  },
          '23' => {
                    '27' => '65.3', '25' => '66.2', '28' => '64.9', '21' => '68.2', '26' => '65.7',
                    '20' => '68.7', '22' => '67.6', '24' => '66.6', '23' => '67.1', '29' => '64.6'
                  },
          '105' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.9', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '3.2', '80' => '10.4', '26' => '57.2', '99' => '3.7',
                     '72' => '15.6', '44' => '39.8', '55' => '29.6', '84' => '8.3', '74' => '14.2',
                     '27' => '56.2', '95' => '4.5', '57' => '27.9', '61' => '24.4', '108' => '2.4',
                     '20' => '63.0', '109' => '2.3', '92' => '5.3', '103' => '3.0', '89' => '6.2',
                     '31' => '52.4', '35' => '48.5', '91' => '5.6', '78' => '11.5', '107' => '2.5',
                     '48' => '36.0', '87' => '7.0', '93' => '5.0', '77' => '12.2', '106' => '2.6',
                     '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9',
                     '97' => '4.1', '58' => '27.0', '41' => '42.7', '81' => '9.8', '52' => '32.3',
                     '60' => '25.3', '56' => '28.8', '101' => '3.4', '73' => '14.9', '66' => '20.2',
                     '45' => '38.8', '86' => '7.4', '76' => '12.8', '62' => '23.5', '54' => '30.5',
                     '67' => '19.4', '70' => '17.1', '68' => '18.6', '88' => '6.6', '30' => '53.3',
                     '100' => '3.5', '82' => '9.3', '25' => '58.2', '28' => '55.3', '40' => '43.6',
                     '75' => '13.5', '83' => '8.8', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.9', '53' => '31.4', '79' => '10.9', '42' => '41.7',
                     '22' => '61.1', '46' => '37.9', '23' => '60.1', '105' => '2.7', '96' => '4.3',
                     '85' => '7.8', '36' => '47.5', '94' => '4.8', '51' => '33.3', '47' => '37.0',
                     '38' => '45.6', '98' => '3.9', '34' => '49.4', '37' => '46.5', '43' => '40.8'
                   },
          '96' => {
                    '33' => '50.4', '32' => '51.4', '90' => '6.8', '63' => '22.8', '21' => '62.1',
                    '71' => '16.5', '80' => '10.8', '26' => '57.2', '99' => '5.0', '72' => '15.8',
                    '44' => '39.8', '55' => '29.7', '84' => '8.9', '74' => '14.4', '27' => '56.2',
                    '95' => '5.7', '57' => '27.9', '61' => '24.5', '20' => '63.0', '92' => '6.3',
                    '89' => '7.1', '31' => '52.4', '35' => '48.5', '91' => '6.5', '78' => '11.9',
                    '48' => '36.1', '87' => '7.7', '93' => '6.1', '77' => '12.5', '65' => '21.1',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9', '97' => '5.3',
                    '58' => '27.0', '41' => '42.7', '81' => '10.3', '52' => '32.4', '60' => '25.3',
                    '56' => '28.8', '73' => '15.1', '66' => '20.3', '45' => '38.9', '86' => '8.1',
                    '76' => '13.1', '62' => '23.6', '54' => '30.6', '67' => '19.5', '70' => '17.2',
                    '68' => '18.8', '88' => '7.4', '30' => '53.3', '82' => '9.8', '25' => '58.2',
                    '28' => '55.3', '40' => '43.6', '75' => '13.8', '83' => '9.3', '59' => '26.2',
                    '69' => '18.0', '49' => '35.1', '24' => '59.1', '53' => '31.5', '79' => '11.3',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '5.5',
                    '85' => '8.5', '36' => '47.5', '94' => '5.9', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '5.2', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                  },
          '85' => {
                    '33' => '50.4', '32' => '51.4', '63' => '23.1', '21' => '62.1', '71' => '17.3',
                    '80' => '12.3', '26' => '57.2', '72' => '16.6', '44' => '39.9', '55' => '29.9',
                    '84' => '10.8', '74' => '15.4', '27' => '56.3', '57' => '28.1', '61' => '24.8',
                    '20' => '63.0', '89' => '9.5', '31' => '52.4', '35' => '48.5', '78' => '13.2',
                    '48' => '36.2', '87' => '9.9', '77' => '13.7', '65' => '21.6', '29' => '54.3',
                    '50' => '34.3', '39' => '44.7', '64' => '22.3', '58' => '27.3', '41' => '42.7',
                    '81' => '11.9', '52' => '32.5', '60' => '25.6', '56' => '29.0', '73' => '16.0',
                    '66' => '20.8', '45' => '38.9', '86' => '10.2', '76' => '14.3', '62' => '23.9',
                    '54' => '30.7', '67' => '20.1', '70' => '17.9', '68' => '19.3', '88' => '9.7',
                    '30' => '53.3', '82' => '11.5', '25' => '58.2', '28' => '55.3', '40' => '43.7',
                    '75' => '14.8', '83' => '11.1', '59' => '26.4', '69' => '18.6', '49' => '35.2',
                    '24' => '59.2', '53' => '31.6', '79' => '12.8', '42' => '41.8', '22' => '61.1',
                    '46' => '38.0', '23' => '60.1', '85' => '10.5', '36' => '47.5', '51' => '33.4',
                    '47' => '37.1', '38' => '45.6', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '36' => {
                    '35' => '54.7', '33' => '55.8', '32' => '56.4', '21' => '64.2', '26' => '60.4',
                    '22' => '63.4', '30' => '57.6', '23' => '62.6', '29' => '58.3', '25' => '61.1',
                    '27' => '59.6', '39' => '52.8', '28' => '59.0', '36' => '54.2', '20' => '65.0',
                    '38' => '53.3', '34' => '55.3', '37' => '53.7', '24' => '61.9', '31' => '57.0'
                  },
          '94' => {
                    '33' => '50.4', '32' => '51.4', '90' => '7.1', '63' => '22.8', '21' => '62.1',
                    '71' => '16.6', '80' => '11.0', '26' => '57.2', '99' => '5.4', '72' => '15.9',
                    '44' => '39.8', '55' => '29.7', '84' => '9.1', '74' => '14.5', '27' => '56.2',
                    '95' => '6.0', '57' => '27.9', '61' => '24.5', '20' => '63.0', '92' => '6.6',
                    '89' => '7.3', '31' => '52.4', '35' => '48.5', '91' => '6.8', '78' => '12.0',
                    '48' => '36.1', '87' => '8.0', '93' => '6.4', '77' => '12.6', '65' => '21.2',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '22.0', '97' => '5.7',
                    '58' => '27.1', '41' => '42.7', '81' => '10.5', '52' => '32.4', '60' => '25.3',
                    '56' => '28.8', '73' => '15.2', '66' => '20.4', '45' => '38.9', '86' => '8.3',
                    '76' => '13.2', '62' => '23.6', '54' => '30.6', '67' => '19.6', '70' => '17.3',
                    '68' => '18.8', '88' => '7.6', '30' => '53.3', '82' => '10.0', '25' => '58.2',
                    '28' => '55.3', '40' => '43.7', '75' => '13.9', '83' => '9.5', '59' => '26.2',
                    '69' => '18.0', '49' => '35.1', '24' => '59.1', '53' => '31.5', '79' => '11.5',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '5.9',
                    '85' => '8.7', '36' => '47.5', '94' => '6.2', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '5.6', '34' => '49.5', '37' => '46.6', '43' => '40.8'
                  },
          '111' => {
                     '33' => '50.4', '32' => '51.4', '90' => '5.6', '63' => '22.7', '21' => '62.1',
                     '71' => '16.3', '102' => '2.7', '80' => '10.3', '26' => '57.2', '99' => '3.3',
                     '72' => '15.5', '44' => '39.8', '55' => '29.6', '84' => '8.1', '74' => '14.1',
                     '27' => '56.2', '95' => '4.2', '57' => '27.9', '61' => '24.4', '108' => '1.6',
                     '20' => '63.0', '109' => '1.5', '92' => '5.0', '103' => '2.5', '89' => '6.0',
                     '31' => '52.4', '113' => '1.1', '35' => '48.5', '91' => '5.3', '78' => '11.5',
                     '107' => '1.8', '48' => '36.0', '87' => '6.8', '93' => '4.7', '77' => '12.1',
                     '106' => '1.9', '65' => '21.0', '29' => '54.3', '50' => '34.2', '39' => '44.6',
                     '64' => '21.8', '97' => '3.7', '114' => '1.0', '58' => '27.0', '41' => '42.7',
                     '81' => '9.7', '52' => '32.3', '60' => '25.2', '56' => '28.7', '101' => '2.9',
                     '73' => '14.8', '66' => '20.2', '45' => '38.8', '86' => '7.2', '76' => '12.7',
                     '62' => '23.5', '54' => '30.5', '67' => '19.4', '70' => '17.0', '68' => '18.6',
                     '115' => '1.0', '88' => '6.3', '30' => '53.3', '100' => '3.1', '110' => '1.4',
                     '82' => '9.1', '25' => '58.2', '28' => '55.3', '40' => '43.6', '75' => '13.4',
                     '83' => '8.6', '112' => '1.1', '59' => '26.1', '69' => '17.8', '49' => '35.1',
                     '24' => '59.1', '104' => '2.3', '53' => '31.4', '79' => '10.8', '42' => '41.7',
                     '22' => '61.1', '' => '', '46' => '37.9', '23' => '60.1', '105' => '2.1',
                     '96' => '3.9', '85' => '7.6', '36' => '47.5', '94' => '4.4', '111' => '1.2',
                     '51' => '33.3', '47' => '37.0', '38' => '45.6', '98' => '3.5', '34' => '49.4',
                     '37' => '46.5',
                     '43' => '40.7'
                   },
          '51' => {
                    '33' => '51.9', '32' => '52.7', '21' => '62.6', '26' => '58.0', '30' => '54.5',
                    '44' => '43.5', '55' => '37.7', '25' => '58.9', '27' => '57.1', '28' => '56.2',
                    '57' => '37.0', '40' => '46.3', '20' => '63.6', '59' => '36.4', '49' => '40.5',
                    '24' => '59.9', '31' => '53.6', '35' => '50.2', '53' => '38.5', '48' => '41.0',
                    '22' => '61.7', '42' => '44.8', '46' => '42.2', '23' => '60.8', '29' => '55.3',
                    '50' => '40.0', '39' => '47.0', '36' => '49.4', '51' => '39.5', '58' => '36.7',
                    '41' => '45.5', '47' => '41.6', '38' => '47.8', '52' => '39.0', '34' => '51.0',
                    '56' => '37.4', '37' => '48.6', '45' => '42.8', '43' => '44.1', '54' => '38.1'
                  },
          '47' => {
                    '35' => '51.0', '33' => '52.5', '32' => '53.3', '21' => '62.9', '26' => '58.4',
                    '48' => '42.9', '42' => '46.1', '22' => '62.0', '46' => '43.9', '30' => '55.0',
                    '23' => '61.1', '44' => '44.9', '29' => '55.8', '25' => '59.3', '27' => '57.5',
                    '39' => '48.1', '28' => '56.7', '40' => '47.4', '36' => '50.2', '41' => '46.7',
                    '47' => '43.4', '20' => '63.8', '38' => '48.8', '34' => '51.7', '49' => '42.4',
                    '45' => '44.4', '37' => '49.5', '24' => '60.2', '43' => '45.5', '31' => '54.1'
                  },
          '38' => {
                    '35' => '53.8', '33' => '55.0', '32' => '55.6', '21' => '63.9', '26' => '59.9',
                    '22' => '63.0', '30' => '57.0', '23' => '62.2', '29' => '57.7', '25' => '60.6',
                    '27' => '59.1', '39' => '51.7', '28' => '58.4', '36' => '53.3', '20' => '64.7',
                    '38' => '52.2', '34' => '54.4', '37' => '52.7', '24' => '61.4', '31' => '56.3'
                  },
          '98' => {
                    '33' => '50.4', '32' => '51.4', '90' => '6.5', '63' => '22.8', '21' => '62.1',
                    '71' => '16.4', '80' => '10.7', '26' => '57.2', '99' => '4.7', '72' => '15.7',
                    '44' => '39.8', '55' => '29.7', '84' => '8.7', '74' => '14.3', '27' => '56.2',
                    '95' => '5.4', '57' => '27.9', '61' => '24.4', '20' => '63.0', '92' => '6.0',
                    '89' => '6.8', '31' => '52.4', '35' => '48.5', '91' => '6.3', '78' => '11.8',
                    '48' => '36.0', '87' => '7.5', '93' => '5.8', '77' => '12.4', '65' => '21.1',
                    '29' => '54.3', '50' => '34.2', '39' => '44.6', '64' => '21.9', '97' => '5.0',
                    '58' => '27.0', '41' => '42.7', '81' => '10.1', '52' => '32.4', '60' => '25.3',
                    '56' => '28.8', '73' => '15.0', '66' => '20.3', '45' => '38.9', '86' => '7.9',
                    '76' => '13.0', '62' => '23.6', '54' => '30.6', '67' => '19.5', '70' => '17.2',
                    '68' => '18.7', '88' => '7.1', '30' => '53.3', '82' => '9.6', '25' => '58.2',
                    '28' => '55.3', '40' => '43.6', '75' => '13.7', '83' => '9.2', '59' => '26.2',
                    '69' => '17.9', '49' => '35.1', '24' => '59.1', '53' => '31.5', '79' => '11.2',
                    '42' => '41.7', '22' => '61.1', '46' => '37.9', '23' => '60.1', '96' => '5.2',
                    '85' => '8.3', '36' => '47.5', '94' => '5.6', '51' => '33.3', '47' => '37.0',
                    '38' => '45.6', '98' => '4.8', '34' => '49.5', '37' => '46.5', '43' => '40.8'
                  },
          '34' => {
                    '35' => '55.7', '33' => '56.7', '32' => '57.2', '21' => '64.6', '26' => '60.9',
                    '22' => '63.8', '30' => '58.4', '23' => '63.1', '29' => '59.0', '25' => '61.6',
                    '27' => '60.3', '39' => '54.0', '28' => '59.6', '36' => '55.3', '20' => '65.4',
                    '38' => '54.4', '34' => '56.2', '37' => '54.8', '24' => '62.3', '31' => '57.8'
                  },
          '37' => {
                    '35' => '54.3', '33' => '55.4', '32' => '56.0', '21' => '64.0', '26' => '60.1',
                    '22' => '63.2', '30' => '57.3', '23' => '62.4', '29' => '58.0', '25' => '60.9',
                    '27' => '59.4', '39' => '52.3', '28' => '58.7', '36' => '53.7', '20' => '64.9',
                    '38' => '52.7', '34' => '54.8', '37' => '53.2', '24' => '61.6', '31' => '56.6'
                  },
          '43' => {
                    '35' => '52.0', '33' => '53.4', '32' => '54.2', '21' => '63.2', '26' => '58.9',
                    '48' => '45.1', '42' => '47.8', '22' => '62.4', '46' => '45.9', '30' => '55.7',
                    '23' => '61.5', '44' => '46.8', '29' => '56.5', '25' => '59.8', '27' => '58.1',
                    '39' => '49.5', '28' => '57.3', '40' => '48.9', '36' => '51.3', '41' => '48.3',
                    '47' => '45.5', '20' => '64.1', '38' => '50.1', '34' => '52.7', '49' => '44.8',
                    '45' => '46.3', '37' => '50.7', '24' => '60.6', '43' => '47.3', '31' => '54.9'
                  }
        };
}

sub universal {
  return {
          '90' => '11.4', '71' => '26.5', '70' => '27.4', '102' => '5.5', '80' => '18.7', '99' => '6.7',
          '88' => '12.7', '72' => '25.6', '100' => '6.3', '110' => '3.1', '82' => '17.1', '84' => '15.5',
          '74' => '23.8', '83' => '16.3', '95' => '8.6', '75' => '22.9', '108' => '3.7', '115' => '1.9',
          '92' => '10.2', '112' => '2.6', '109' => '3.4', '103' => '5.2', '89' => '12.0', '113' => '2.4',
          '104' => '4.9', '91' => '10.8', '78' => '20.3', '79' => '19.5', '107' => '3.9', '87' => '13.4',
          '93' => '9.6', '77' => '21.2', '106' => '4.2', '96' => '8.1', '105' => '4.5', '85' => '14.8',
          '97' => '7.6', '94' => '9.1', '114' => '2.1', '111' => '2.9', '81' => '17.9', '98' => '7.1',
          '73' => '24.7', '101' => '5.9', '76' => '22.0', '86' => '14.1'
        };
}
1;
__END__

=head1 NAME

Bank::RMD - Perl extension for calculating the RMD for IRAs.

=head1 SYNOPSIS

  use warnings;
  use strict;
  use Bank::RMD;

  # Using OO

  my $calc = new Bank::RMD;
  
  my $rmd = $calc->calculate( 
    ownerAge => 75, 
    balance => 500000 ); # Will use universal table;
  
  .. OR ..

  my $rmd = $calc->calculate( 
    ownerAge => 79, 
    beneficiaryAge => 65, 
    balance => 500000 ); # Will use joint table because beneficiary is > 10 years younger.

  print $rmd->divisor; # Will show the divisor used
  print $rmd->rmd; # Will show the RMD calculated.

=head1 DESCRIPTION

After retirement age, the IRS requires IRA owners to begin taking a minimum
distribution. Called the Required Minimum Distribution, this table is big
and cumbersome to calculate on your own. 

=head1 SEE ALSO

IRS Publication 590 (2005) Individual Retirement Arrange,ents (IRAs)
URL: http://www.irs.gov/publications/p590/index.html

=head1 AUTHOR

Tyler Hardison, E<lt>thardison@bayfed.comE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 by Tyler Hardison

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.


=cut
