package WWW::ManageWP;
use strict;
use warnings;
use WWW::ManageWP::Remote;
use WWW::ManageWP::Simple;

our $VERSION = 0.05;

##########
## Init ##
##########

sub new {
	my $class  = shift;
	my $params = $class->_prep_args([qw/apiKEY apiURL/], @_);
	return bless($params, $class);
}

##########
## Core ##
##########

sub delete {return shift->remote->delete(@_)}
sub get    {return shift->remote->get(@_)}
sub post   {return shift->remote->post(@_)}

##############
## Autoload ##
##############

# autoload is used here to send simple methods off to WWW::ManageWP::Simple

sub AUTOLOAD {
	my $self = shift;
	our $AUTOLOAD;
	my ($key) = $AUTOLOAD =~ /.*::([\w_]+)/o;
	return if ($key eq 'DESTROY');
	{
		no strict 'refs';
		if (!defined(&{"WWW::ManageWP::Simple::$key"})) {
			die "ERROR! [$key] is not a valid method\n";
		}
	}
	return $self->simple->$key(@_);
}

##########
## Help ##
##########

sub apiKEY {return shift->{apiKEY}}
sub apiURL {return shift->{apiURL}}

sub simple {
	my $self = shift;
	if (!$self->{simple}
		|| ref($self->{simple} ne 'WWW::ManageWP::Simple')) {
			$self->{simple} = WWW::ManageWP::Simple->new(
				apiKEY => $self->apiKEY,
				apiURL => $self->apiURL,
			);
	}
	return $self->{simple};
}

sub remote {
	my $self = shift;
	if (!$self->{http} || ref($self->{http} ne 'WWW::ManageWP::Remote')) {
		$self->{http} = WWW::ManageWP::Remote->new(
			apiKEY => $self->apiKEY,
			apiURL => $self->apiURL,
		);
	}
	return $self->{http};
}

#############
## Private ##
#############

sub _prep_args {
	my $self     = shift;
	my $required = shift;
	my $params   = ref($_[0]) ? $_[0] : {@_};
	foreach my $required (@{$required}) {
		if (!defined($params->{$required})) {
			die "[$required] is a required parameter\n";
		}
	}
	return $params;
}

1;

=head1 NAME

WWW::ManageWP

=head1 DESCRIPTION

API Client for "ManageWP", a tool to manage Wordpress sites from one dashboard.

http://www.managewp.com

=head1 SYNOPSIS

This client is still in early stages, and there will be packages added in
the versions to come to add better wrapping around the ManageWP API.

API Documentation: http://managewp.com/docs/api/#documentation

Create the object:

	my $managewp = WWW::ManageWP->new(apiKEY => $apiKEY, apiURL => $apiURL);

Call methods manually:

	my $users = $managewp->get(method => 'users/list');

	my $plans = $managewp->get(method => 'plans/list');

Call simple methods:

	my $register_response
		= $managewp->register_user(user_email => 'test@example.com');

	my $autologin
		= $managewp->autologin_user(user_email => 'test@example.com');

	my $changeplan_response
		= $managewp->change_plan(
			user_email => 'test@example.com',
			plan_id    => $plan_id,
	);

	my $removal_response
		= $managewp->remove_user(user_email => 'test@example.com');

	my $users = $managewp->list_users;

=cut

