package Dist::Util::Debian;

our $DATE = '2017-01-13'; # DATE
our $VERSION = '0.001'; # VERSION

use 5.010001;
use strict;
use warnings;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(
                       dist_has_deb
               );

sub dist_has_deb {
    require HTTP::Tiny;

    my $dist = shift;
    my $deb = "lib" . lc($dist) . "-perl";

    my $url = "https://packages.debian.org/sid/$deb";
    my $res = HTTP::Tiny->new->get($url);
    unless ($res->{success}) {
        warn "Can't check $url: $res->{status} - $res->{reason}";
        return undef;
    }
    if ($res->{content} =~ /No such package/) {
        return 0;
    } elsif ($res->{content} =~ /Package: \Q$deb\E \(/) {
        return 1;
    } else {
        warn "Can't understand the content of $url, no indication of package exists or doesn't exist";
        return undef;
    }
}

1;
# ABSTRACT: Utilities related to Perl distribution and Debian

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Util::Debian - Utilities related to Perl distribution and Debian

=head1 VERSION

This document describes version 0.001 of Dist::Util::Debian (from Perl distribution Dist-Util-Debian), released on 2017-01-13.

=head1 SYNOPSIS

 use Dist::Util::Debian qw(
     dist_has_deb
 );

 say dist_has_deb("HTTP-Tiny"); # -> 1
 say dist_has_deb("Foo");       # -> 0

=head1 DESCRIPTION

=head1 FUNCTIONS

=head2 dist_has_deb($dist) => bool

Return true if distribution named C<$dist> has a corresponding Debian package.
Currently the way the routine checks this is rather naive: it converts C<$dist>
to Debian package name by turning it to lowercase and adds "lib" + "-perl"
prefix and suffix (a small percentage of distributions do not follow this rule).
Then it checks against this URL: L<https://packages.debian.org/sid/$package>.

Will warn and return undef on error, e.g. the URL cannot be checked or does not
contain negative/positive indicator of existence.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Dist-Util-Debian>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Dist-Util-Debian>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Dist-Util-Debian>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
