package Bencher::Scenario::LevenshteinModules;

our $DATE = '2016-01-20'; # DATE
our $VERSION = '0.10'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    modules => {
        'Text::Levenshtein' => 0.11,
    },
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 2, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 1, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
    ],
    on_result_failure => 'warn',
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.10 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenario-LevenshteinModules), released on 2016-01-20.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::XS> 0.502

L<Text::LevenshteinXS> 0.03

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code) [no_unicode_support]

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code) [no_unicode_support]

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=back

=head1 BENCHMARK DATASETS

=over

=item * a

=item * foo

=item * program

=item * reve [unicode]

=item * euro [unicode]

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m LevenshteinModules >>):

 +----------------------------------------+---------+------------+-----------+---------+---------+
 | participant                            | dataset | rate (/s)  | time (μs) | errors  | samples |
 +----------------------------------------+---------+------------+-----------+---------+---------+
 | Text::Levenshtein::fastdistance        | program | 1.19e+04   | 84.2      | 1.1e-07 | 20      |
 | PERLANCAR::Text::Levenshtein::editdist | program | 1.83e+04   | 54.7      | 2.5e-08 | 22      |
 | Text::Levenshtein::fastdistance        | reve    | 2.7e+04    | 37.1      | 5.3e-08 | 20      |
 | Text::Levenshtein::fastdistance        | euro    | 29238.7    | 34.2012   | 4.6e-11 | 20      |
 | Text::Levenshtein::fastdistance        | foo     | 5.46e+04   | 18.3      | 6.4e-09 | 22      |
 | PERLANCAR::Text::Levenshtein::editdist | foo     | 7.17e+04   | 13.9      | 5.8e-09 | 26      |
 | Text::Levenshtein::fastdistance        | a       | 1.58e+05   | 6.34      | 2.8e-09 | 28      |
 | PERLANCAR::Text::Levenshtein::editdist | a       | 2.77e+05   | 3.62      | 6.7e-09 | 20      |
 | Text::Levenshtein::XS::distance        | program | 4.5e+05    | 2.22      | 3.9e-09 | 23      |
 | Text::Levenshtein::XS::distance        | reve    | 5.58e+05   | 1.79      | 7.8e-10 | 23      |
 | Text::Levenshtein::XS::distance        | euro    | 5.9688e+05 | 1.6754    | 2.3e-11 | 20      |
 | Text::Levenshtein::XS::distance        | foo     | 6.5e+05    | 1.5       | 6.5e-09 | 21      |
 | Text::Levenshtein::XS::distance        | a       | 7.986e+05  | 1.252     | 3.8e-10 | 24      |
 | Text::LevenshteinXS::distance          | program | 2.9e+06    | 0.345     | 2.1e-10 | 20      |
 | Text::LevenshteinXS::distance          | foo     | 5.93e+06   | 0.169     | 4.1e-10 | 21      |
 | Text::LevenshteinXS::distance          | a       | 7.39e+06   | 0.135     | 1.6e-10 | 20      |
 +----------------------------------------+---------+------------+-----------+---------+---------+


Benchmark module startup overhead (C<< bencher -m LevenshteinModules --module-startup >>):

 +------------------------------+-----------+------------------------+---------+---------+
 | participant                  | time (ms) | mod_overhead_time (ms) | errors  | samples |
 +------------------------------+-----------+------------------------+---------+---------+
 | Text::LevenshteinXS          | 15        | 10.4                   | 0.00012 | 23      |
 | Text::Levenshtein            | 11.8      | 7.2                    | 2.3e-05 | 20      |
 | Text::Levenshtein::XS        | 8.53      | 3.93                   | 2.5e-05 | 20      |
 | PERLANCAR::Text::Levenshtein | 4.7       | 0.100000000000001      | 1.5e-05 | 20      |
 | perl -e1 (baseline)          | 4.6       | 0                      | 2.2e-05 | 20      |
 +------------------------------+-----------+------------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Bencher::Scenario::DamerauLevenshteinModules>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
