package Bencher::Scenario::LevenshteinModules;

our $DATE = '2016-01-11'; # DATE
our $VERSION = '0.06'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    modules => {
        'Text::Levenshtein' => 0.11,
    },
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)",
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 3, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 1, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
    ],
    on_result_failure => 'warn',
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.06 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenario-LevenshteinModules), released on 2016-01-11.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::Damerau::PP> 0.25

L<Text::Levenshtein::Damerau::XS> 3.0

L<Text::Levenshtein::XS> 0.502

L<Text::LevenshteinXS> 0.03

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code)

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code)

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::PP::pp_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::XS::xs_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default option:

 +-----+-------------------------------------------------------------------------------+-------------+------------+---------+---------+
 | seq | name                                                                          | rate        | time       | errors  | samples |
 +-----+-------------------------------------------------------------------------------+-------------+------------+---------+---------+
 | 5   | {dataset=>"program",participant=>"Text::Levenshtein::fastdistance"}           | 1.9e+04     | 52.6μs     | 2.7e-08 | 20      |
 | 2   | {dataset=>"program",participant=>"PERLANCAR::Text::Levenshtein::editdist"}    | 3.021e+04   | 33.1μs     | 9.9e-09 | 36      |
 | 7   | {dataset=>"euro",participant=>"Text::Levenshtein::fastdistance"}              | 5.07e+04    | 19.7μs     | 2.7e-08 | 20      |
 | 20  | {dataset=>"euro",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"} | 5.12e+04    | 19.5μs     | 1.6e-08 | 31      |
 | 4   | {dataset=>"foo",participant=>"Text::Levenshtein::fastdistance"}               | 8.893e+04   | 11.24μs    | 2.4e-09 | 38      |
 | 17  | {dataset=>"foo",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}  | 9.392e+04   | 10.65μs    | 2.6e-09 | 33      |
 | 1   | {dataset=>"foo",participant=>"PERLANCAR::Text::Levenshtein::editdist"}        | 1.1728e+05  | 8.5265μs   | 1e-10   | 30      |
 | 16  | {dataset=>"a",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}    | 2.8383e+05  | 3.5232μs   | 4.7e-11 | 20      |
 | 3   | {dataset=>"a",participant=>"Text::Levenshtein::fastdistance"}                 | 2.88e+05    | 3.48μs     | 1.7e-09 | 20      |
 | 0   | {dataset=>"a",participant=>"PERLANCAR::Text::Levenshtein::editdist"}          | 4.5e+05     | 2.22μs     | 8.3e-10 | 20      |
 | 10  | {dataset=>"program",participant=>"Text::Levenshtein::XS::distance"}           | 649762      | 1.53903μs  | 0       | 20      |
 | 25  | {dataset=>"euro",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"} | 8.11e+05    | 1.23μs     | 4.3e-10 | 20      |
 | 12  | {dataset=>"euro",participant=>"Text::Levenshtein::XS::distance"}              | 914273      | 1.09377μs  | 0       | 20      |
 | 22  | {dataset=>"foo",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}  | 941504      | 1.06213μs  | 0       | 20      |
 | 9   | {dataset=>"foo",participant=>"Text::Levenshtein::XS::distance"}               | 1.00903e+06 | 0.991049μs | 0       | 20      |
 | 8   | {dataset=>"a",participant=>"Text::Levenshtein::XS::distance"}                 | 1.28056e+06 | 0.780906μs | 0       | 20      |
 | 21  | {dataset=>"a",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}    | 1.2807e+06  | 0.780822μs | 0       | 20      |
 | 15  | {dataset=>"program",participant=>"Text::LevenshteinXS::distance"}             | 4.373e+06   | 0.2287μs   | 4.7e-11 | 29      |
 | 14  | {dataset=>"foo",participant=>"Text::LevenshteinXS::distance"}                 | 9.71e+06    | 0.103μs    | 7e-11   | 20      |
 | 13  | {dataset=>"a",participant=>"Text::LevenshteinXS::distance"}                   | 1.21e+07    | 0.0825μs   | 1.3e-10 | 27      |
 +-----+-------------------------------------------------------------------------------+-------------+------------+---------+---------+


Benchmark module startup overhead:

 +-----+--------------------------------+--------+-------------------+---------+---------+
 | seq | name                           | time   | mod_overhead_time | errors  | samples |
 +-----+--------------------------------+--------+-------------------+---------+---------+
 | 4   | Text::LevenshteinXS            | 8.53ms | 6.07ms            | 1.3e-05 | 21      |
 | 2   | Text::Levenshtein              | 7.1ms  | 4.6ms             | 2.3e-05 | 20      |
 | 3   | Text::Levenshtein::XS          | 4.98ms | 2.52ms            | 1e-05   | 20      |
 | 5   | Text::Levenshtein::Damerau::PP | 4.2ms  | 1.8ms             | 1.6e-05 | 21      |
 | 6   | Text::Levenshtein::Damerau::XS | 3.8ms  | 1.3ms             | 2e-05   | 20      |
 | 1   | PERLANCAR::Text::Levenshtein   | 2.7ms  | 0.22ms            | 1.2e-05 | 20      |
 | 0   | perl -e1 (baseline)            | 2.5ms  | 0ms               | 9.8e-06 | 20      |
 +-----+--------------------------------+--------+-------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
