package Bencher::Scenario::LevenshteinModules;

our $DATE = '2016-01-10'; # DATE
our $VERSION = '0.05'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)",
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 3, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 1, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
    ],
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.05 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenario-LevenshteinModules), released on 2016-01-10.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::XS> 0.502

L<Text::LevenshteinXS> 0.03

L<Text::Levenshtein::Damerau::PP> 0.25

L<Text::Levenshtein::Damerau::XS> 3.0

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code)

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code)

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::PP::pp_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::XS::xs_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default option:

 +-----+----------------------------------------------------------------------------------+-------------+-------------+---------+---------+
 | seq | name                                                                             | rate        | time        | errors  | samples |
 +-----+----------------------------------------------------------------------------------+-------------+-------------+---------+---------+
 | 7   | {dataset=>"program",participant=>"Text::Levenshtein::fastdistance"}              | 1.88e+04    | 53.3μs      | 9.5e-08 | 25      |
 | 20  | {dataset=>"program",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"} | 22787.3     | 43.884μs    | 0       | 31      |
 | 2   | {dataset=>"program",participant=>"PERLANCAR::Text::Levenshtein::editdist"}       | 3.025e+04   | 33.06μs     | 1e-08   | 35      |
 | 8   | {dataset=>"reve",participant=>"Text::Levenshtein::fastdistance"}                 | 4.344e+04   | 23.02μs     | 6.4e-09 | 22      |
 | 21  | {dataset=>"reve",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}    | 4.967e+04   | 20.13μs     | 5.7e-09 | 27      |
 | 9   | {dataset=>"euro",participant=>"Text::Levenshtein::fastdistance"}                 | 4.975e+04   | 20.1μs      | 6.2e-09 | 23      |
 | 22  | {dataset=>"euro",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}    | 5.19e+04    | 19.3μs      | 2.3e-08 | 27      |
 | 3   | {dataset=>"reve",participant=>"PERLANCAR::Text::Levenshtein::editdist"}          | 6.77e+04    | 14.8μs      | 5.3e-09 | 32      |
 | 4   | {dataset=>"euro",participant=>"PERLANCAR::Text::Levenshtein::editdist"}          | 8.23e+04    | 12.2μs      | 1.1e-08 | 27      |
 | 6   | {dataset=>"foo",participant=>"Text::Levenshtein::fastdistance"}                  | 8.609e+04   | 11.62μs     | 3.3e-09 | 20      |
 | 19  | {dataset=>"foo",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}     | 9.45e+04    | 10.6μs      | 1.1e-08 | 29      |
 | 1   | {dataset=>"foo",participant=>"PERLANCAR::Text::Levenshtein::editdist"}           | 1.17e+05    | 8.51μs      | 2.8e-09 | 28      |
 | 5   | {dataset=>"a",participant=>"Text::Levenshtein::fastdistance"}                    | 286965      | 3.48475μs   | 0       | 26      |
 | 18  | {dataset=>"a",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}       | 2.88e+05    | 3.48μs      | 1.6e-09 | 21      |
 | 0   | {dataset=>"a",participant=>"PERLANCAR::Text::Levenshtein::editdist"}             | 451950      | 2.21263μs   | 0       | 20      |
 | 25  | {dataset=>"program",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"} | 5.79e+05    | 1.73μs      | 2.5e-09 | 20      |
 | 12  | {dataset=>"program",participant=>"Text::Levenshtein::XS::distance"}              | 647850      | 1.54357μs   | 0       | 20      |
 | 26  | {dataset=>"reve",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}    | 7.6506e+05  | 1.3071μs    | 1.1e-11 | 32      |
 | 27  | {dataset=>"euro",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}    | 8.1391e+05  | 1.2286μs    | 1.1e-11 | 20      |
 | 13  | {dataset=>"reve",participant=>"Text::Levenshtein::XS::distance"}                 | 8.451e+05   | 1.183μs     | 4.4e-11 | 20      |
 | 14  | {dataset=>"euro",participant=>"Text::Levenshtein::XS::distance"}                 | 9.18e+05    | 1.089μs     | 4.6e-11 | 20      |
 | 24  | {dataset=>"foo",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}     | 9.4e+05     | 1.06μs      | 4.2e-10 | 20      |
 | 11  | {dataset=>"foo",participant=>"Text::Levenshtein::XS::distance"}                  | 1.0194e+06  | 0.98097μs   | 1.1e-11 | 20      |
 | 23  | {dataset=>"a",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}       | 1.27348e+06 | 0.785248μs  | 0       | 30      |
 | 10  | {dataset=>"a",participant=>"Text::Levenshtein::XS::distance"}                    | 1.28528e+06 | 0.778039μs  | 0       | 20      |
 | 17  | {dataset=>"program",participant=>"Text::LevenshteinXS::distance"}                | 4.42139e+06 | 0.226173μs  | 0       | 22      |
 | 16  | {dataset=>"foo",participant=>"Text::LevenshteinXS::distance"}                    | 9.29e+06    | 0.108μs     | 5.3e-11 | 20      |
 | 15  | {dataset=>"a",participant=>"Text::LevenshteinXS::distance"}                      | 1.15622e+07 | 0.0864889μs | 0       | 20      |
 +-----+----------------------------------------------------------------------------------+-------------+-------------+---------+---------+


Benchmark module startup overhead:

 +-----+--------------------------------+-------+-------------------+---------+---------+
 | seq | name                           | time  | mod_overhead_time | errors  | samples |
 +-----+--------------------------------+-------+-------------------+---------+---------+
 | 4   | Text::LevenshteinXS            | 9.1ms | 6.2ms             | 5.9e-05 | 20      |
 | 2   | Text::Levenshtein              | 7.6ms | 4.7ms             | 8.3e-05 | 20      |
 | 3   | Text::Levenshtein::XS          | 5.2ms | 2.3ms             | 4.6e-05 | 20      |
 | 5   | Text::Levenshtein::Damerau::PP | 4.3ms | 1.4ms             | 2.5e-05 | 21      |
 | 6   | Text::Levenshtein::Damerau::XS | 4ms   | 1.1ms             | 3.5e-05 | 20      |
 | 1   | PERLANCAR::Text::Levenshtein   | 3.2ms | 0.29ms            | 5e-05   | 20      |
 | 0   | perl -e1 (baseline)            | 2.9ms | 0ms               | 4.5e-05 | 21      |
 +-----+--------------------------------+-------+-------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
