package Font::TTF::OTTags;

=head1 NAME

Font::TTF::OTTags - Utilities for TrueType/OpenType tags

=head1 SYNOPSIS

  use Font::TTF::OTTags qw( %tttags %ttnames readtagsfile );
  
  # Look at built-in stuff:
  $script_tag = $tttags{'SCRIPT'}{'Cypriot Syllabary'};
  $lang_name = $ttnames{'LANGUAGE'}{'AFK '}
  
  # Read latest tags file to add to built-in definitions
  readtagsfile ("C:\\Program Files\\Microsoft VOLT\\TAGS.txt");
  
First-level keys to %tttags and %ttnames are:

=over

'SCRIPT' -- retrieve script tag or name

'LANGUAGE' -- retrieve language tag or name

'FEATURE' -- retrieve feature tag or name

=back

Built-in data has been derived from Microsoft's tag registry at
L<http://www.microsoft.com/typography/otspec/ttoreg.htm>, 
updated to draft v1.5 of the OpenType Spec.

=head1 METHODS

=cut

use strict;
use vars qw( %tttags %ttnames @EXPORT_OK );
use Exporter qw( import );
@EXPORT_OK = qw( %tttags %ttnames readtagsfile);


%tttags = (

# All data below derived from VOLT 1.3 TAGS.txt file.
# For conveninence of human checking, these are now sorted alphabetically.

'SCRIPT' => {
    "Arabic" => "arab",
    "Armenian" => "armn",
    "Balinese" => "bali",
    "Bengali v.2" => "bng2",
    "Bengali" => "beng",
    "Bopomofo" => "bopo",
    "Braille" => "brai",
    "Buginese" => "bugi",
    "Buhid" => "buhd",
    "Byzantine Music" => "byzm",
    "CJK Ideographic" => "hani",
    "Canadian Syllabics" => "cans",
    "Carian" => "cari",
    "Cham" => "cham",
    "Cherokee" => "cher",
    "Coptic" => "copt",
    "Cypriot Syllabary" => "cprt",
    "Cyrillic" => "cyrl",
    "Default" => "DFLT",
    "Deseret" => "dsrt",
    "Devanagari v.2" => "dev2",
    "Devanagari" => "deva",
    "Ethiopic" => "ethi",
    "Georgian" => "geor",
    "Glagolitic" => "glag",
    "Gothic" => "goth",
    "Greek" => "grek",
    "Gujarati v.2" => "gjr2",
    "Gujarati" => "gujr",
    "Gurmukhi v.2" => "gur2",
    "Gurmukhi" => "guru",
    "Hangul Jamo" => "jamo",
    "Hangul" => "hang",
    "Hanunoo" => "hano",
    "Hebrew" => "hebr",
    "Hiragana" => "kana",
    "Javanese" => "java",
    "Kannada v.2" => "knd2",
    "Kannada" => "knda",
    "Katakana" => "kana",
    "Kayah Li" => "kali",
    "Kharosthi" => "khar",
    "Khmer" => "khmr",
    "Lao" => "lao ",
    "Latin" => "latn",
    "Lepcha" => "lepc",
    "Limbu" => "limb",
    "Linear B" => "linb",
    "Lycian" => "lyci",
    "Lydian" => "lydi",
    "Malayalam v.2" => "mlm2",
    "Malayalam" => "mlym",
    "Mathematical Alphanumeric Symbols" => "math",
    "Mongolian" => "mong",
    "Musical Symbols" => "musc",
    "Myanmar" => "mymr",
    "N'Ko" => "nko ",
    "New Tai Lue" => "talu",
    "Ogham" => "ogam",
    "Ol Chiki" => "olck",
    "Old Italic" => "ital",
    "Old Persian Cuneiform" => "xpeo",
    "Oriya v.2" => "ory2",
    "Oriya" => "orya",
    "Osmanya" => "osma",
    "Phags-pa" => "phag",
    "Phoenician" => "phnx",
    "Rejang" => "rjng",
    "Runic" => "runr",
    "Saurashtra" => "saur",
    "Shavian" => "shaw",
    "Sinhala" => "sinh",
    "Sumero-Akkadian Cuneiform" => "xsux",
    "Sundanese" => "sund",
    "Syloti Nagri" => "sylo",
    "Syriac" => "syrc",
    "Tagalog" => "tglg",
    "Tagbanwa" => "tagb",
    "Tai Le" => "tale",
    "Tamil v.2" => "tml2",
    "Tamil" => "taml",
    "Telugu v.2" => "tel2",
    "Telugu" => "telu",
    "Thaana" => "thaa",
    "Thai" => "thai",
    "Tibetan" => "tibt",
    "Tifinagh" => "tfng",
    "Ugaritic Cuneiform" => "ugar",
    "Vai" => "vai ",
    "Yi" => "yi  ",
    }, 

'LANGUAGE' => {
    "Aari" => "ARI ",
    "Abaza" => "ABA ",
    "Abkhazian" => "ABK ",
    "Adyghe" => "ADY ",
    "Afar" => "AFR ",
    "Afrikaans" => "AFK ",
    "Agaw" => "AGW ",
    "Albanian" => "SQI ",
    "Alsatian" => "ALS ",
    "Altai" => "ALT ",
    "Amharic" => "AMH ",
    "Arabic" => "ARA ",
    "Arakanese" => "ARK ",
    "Armenian" => "HYE ",
    "Assamese" => "ASM ",
    "Athapaskan" => "ATH ",
    "Avar" => "AVR ",
    "Awadhi" => "AWA ",
    "Aymara" => "AYM ",
    "Azeri" => "AZE ",
    "Badaga" => "BAD ",
    "Baghelkhandi" => "BAG ",
    "Balante" => "BLN ",
    "Balkar" => "BAL ",
    "Balochi" => "BLI ",
    "Balti" => "BLT ",
    "Bambara" => "BMB ",
    "Bamileke" => "BML ",
    "Bashkir" => "BSH ",
    "Basque" => "EUQ ",
    "Baule" => "BAU ",
    "Belarussian" => "BEL ",
    "Bemba" => "BEM ",
    "Bench" => "BCH ",
    "Bengali" => "BEN ",
    "Berber" => "BBR ",
    "Beti" => "BTI ",
    "Bhili" => "BHI ",
    "Bhojpuri" => "BHO ",
    "Bible Cree" => "BCR ",
    "Bikol" => "BIK ",
    "Bilen" => "BIL ",
    "Blackfoot" => "BKF ",
    "Bosnian" => "BOS ",
    "Brahui" => "BRH ",
    "Braj Bhasha" => "BRI ",
    "Breton" => "BRE ",
    "Bulgarian" => "BGR ",
    "Burmese" => "BRM ",
    "Carrier" => "CRR ",
    "Catalan" => "CAT ",
    "Cebuano" => "CEB ",
    "Chaha Gurage" => "CHG ",
    "Chattisgarhi" => "CHH ",
    "Chechen" => "CHE ",
    "Cherokee" => "CHR ",
    "Chichewa" => "CHI ",
    "Chin" => "QIN ",
    "Chinese Hong Kong" => "ZHH ",
    "Chinese Phonetic" => "ZHP ",
    "Chinese Simplified" => "ZHS ",
    "Chinese Traditional" => "ZHT ",
    "Chipewyan" => "CHP ",
    "Chukchi" => "CHK ",
    "Church Slavonic" => "CSL ",
    "Chuvash" => "CHU ",
    "Comorian" => "CMR ",
    "Coptic" => "COP ",
    "Corsican" => "COS ",
    "Cree" => "CRE ",
    "Crimean Tatar" => "CRT ",
    "Croatian" => "HRV ",
    "Czech" => "CSY ",
    "Dangme" => "DNG ",
    "Danish" => "DAN ",
    "Dargwa" => "DAR ",
    "Dari" => "DRI ",
    "Default" => "dflt",
    "Dhivehi (OBSOLETE)" => "DHV ",
    "Dhivehi" => "DIV ",
    "Dinka" => "DNK ",
    "Djerma" => "DJR ",
    "Dogri" => "DGR ",
    "Dungan" => "DUN ",
    "Dutch" => "NLD ",
    "Dzongkha" => "DZN ",
    "Eastern Cree" => "ECR ",
    "Ebira" => "EBI ",
    "Edo" => "EDO ",
    "Efik" => "EFI ",
    "English" => "ENG ",
    "Erzya" => "ERZ ",
    "Esperanto" => "NTO ",
    "Estonian" => "ETI ",
    "Even" => "EVN ",
    "Evenki" => "EVK ",
    "Ewe" => "EWE ",
    "Faroese" => "FOS ",
    "Farsi" => "FAR ",
    "Fijian" => "FJI ",
    "Filipino" => "PIL ",
    "Finnish" => "FIN ",
    "Flemish" => "FLE ",
    "Fon" => "FON ",
    "Forest Nenets" => "FNE ",
    "French Antillean" => "FAN ",
    "French" => "FRA ",
    "Frisian" => "FRI ",
    "Friulian" => "FRL ",
    "Fulani" => "FUL ",
    "Futa" => "FTA ",
    "Ga" => "GAD ",
    "Gaelic" => "GAE ",
    "Gagauz" => "GAG ",
    "Galician" => "GAL ",
    "Garhwali" => "GAW ",
    "Garo" => "GRO ",
    "Garshuni" => "GAR ",
    "Ge'ez" => "GEZ ",
    "Georgian" => "KAT ",
    "German" => "DEU ",
    "Gilyak" => "GIL ",
    "Gondi" => "GON ",
    "Greek" => "ELL ",
    "Greenlandic" => "GRN ",
    "Guarani" => "GUA ",
    "Gujarati" => "GUJ ",
    "Gumuz" => "GMZ ",
    "Haitian" => "HAI ",
    "Halam" => "HAL ",
    "Hammer-Banna" => "HBN ",
    "Harari" => "HRI ",
    "Harauti" => "HAR ",
    "Hausa" => "HAU ",
    "Hawaiin" => "HAW ",
    "Hebrew" => "IWR ",
    "High Mari" => "HMA ",
    "Hiligaynon" => "HIL ",
    "Hindi" => "HIN ",
    "Hindko" => "HND ",
    "Ho" => "HO  ",
    "Hungarian" => "HUN ",
    "Icelandic" => "ISL ",
    "Igbo" => "IBO ",
    "Ijo" => "IJO ",
    "Ilokano" => "ILO ",
    "Inari Sami" => "ISM ",
    "Indonesian" => "IND ",
    "Ingush" => "ING ",
    "Inuktitut" => "INU ",
    "Irish Traditional" => "IRT ",
    "Irish" => "IRI ",
    "Italian" => "ITA ",
    "Japanese" => "JAN ",
    "Javanese" => "JAV ",
    "Judezmo" => "JUD ",
    "Jula" => "JUL ",
    "Kabardian" => "KAB ",
    "Kachchi" => "KAC ",
    "Kalenjin" => "KAL ",
    "Kalmyk" => "KLM ",
    "Kamba" => "KMB ",
    "Kannada" => "KAN ",
    "Kanuri" => "KNR ",
    "Karachay" => "KAR ",
    "Karaim" => "KRM ",
    "Karakalpak" => "KRK ",
    "Karelian" => "KRL ",
    "Karen" => "KRN ",
    "Kashmiri" => "KSH ",
    "Kazakh" => "KAZ ",
    "Kebena" => "KEB ",
    "Khakass" => "KHA ",
    "Khanty-Kazim" => "KHK ",
    "Khanty-Shurishkar" => "KHS ",
    "Khanty-Vakhi" => "KHV ",
    "Khasi" => "KSI ",
    "Khmer" => "KHM ",
    "Khowar" => "KHW ",
    "Khutsuri Georgian" => "KGE ",
    "Kikongo" => "KON ",
    "Kikuyu" => "KIK ",
    "Kildin Sami" => "KSM ",
    "Kirghiz" => "KIR ",
    "Kisii" => "KIS ",
    "Kodagu" => "KOD ",
    "Kokni" => "KKN ",
    "Komi-Permyak" => "KOP ",
    "Komi-Zyrian" => "KOZ ",
    "Komo" => "KMO ",
    "Komso" => "KMS ",
    "Konkani" => "KOK ",
    "Koorete" => "KRT ",
    "Korean Old Hangul" => "KOH ",
    "Korean" => "KOR ",
    "Koryak" => "KYK ",
    "Kpelle" => "KPL ",
    "Krio" => "KRI ",
    "Kui" => "KUI ",
    "Kulvi" => "KUL ",
    "Kumaoni" => "KMN ",
    "Kumyk" => "KUM ",
    "Kurdish" => "KUR ",
    "Kurukh" => "KUU ",
    "Kuy" => "KUY ",
    "L-Cree" => "LCR ",
    "Ladakhi" => "LDK ",
    "Ladin" => "LAD ",
    "Lahuli" => "LAH ",
    "Lak" => "LAK ",
    "Lambani" => "LAM ",
    "Lao" => "LAO ",
    "Latin" => "LAT ",
    "Latvian" => "LVI ",
    "Laz" => "LAZ ",
    "Lezgi" => "LEZ ",
    "Limbu" => "LMB ",
    "Lingala" => "LIN ",
    "Lithuanian" => "LTH ",
    "Lomwe" => "LMW ",
    "Low Mari" => "LMA ",
    "Lower Sorbian" => "LSB ",
    "Luba" => "LUB ",
    "Luganda" => "LUG ",
    "Luhya" => "LUH ",
    "Lule Sami" => "LSM ",
    "Luo" => "LUO ",
    "Luxembourgish" => "LTZ ",
    "Macedonian" => "MKD ",
    "Maithili" => "MTH ",
    "Majang" => "MAJ ",
    "Makua" => "MAK ",
    "Malagasy" => "MLG ",
    "Malay" => "MLY ",
    "Malayalam Reformed" => "MLR ",
    "Malayalam Traditional" => "MAL ",
    "Male" => "MLE ",
    "Malinke" => "MLN ",
    "Maltese" => "MTS ",
    "Manchu" => "MCH ",
    "Mandinka" => "MND ",
    "Maninka" => "MNK ",
    "Manipuri" => "MNI ",
    "Mansi" => "MAN ",
    "Manx Gaelic" => "MNX ",
    "Maori" => "MRI ",
    "Mapudungun" => "MAP ",
    "Marathi" => "MAR ",
    "Marwari" => "MAW ",
    "Mbundu" => "MBN ",
    "Me'en" => "MEN ",
    "Mende" => "MDE ",
    "Mizo" => "MIZ ",
    "Mohawk" => "MOH ",
    "Moksha" => "MOK ",
    "Moldavian" => "MOL ",
    "Mon" => "MON ",
    "Mongolian" => "MNG ",
    "Moose Cree" => "MCR ",
    "Moroccan" => "MOR ",
    "Mundari" => "MUN ",
    "N'Ko" => "NKO ",
    "N-Cree" => "NCR ",
    "Naga-Assamese" => "NAG ",
    "Nagari" => "NGR ",
    "Nanai" => "NAN ",
    "Naskapi" => "NAS ",
    "Ndebele" => "NDB ",
    "Ndonga" => "NDG ",
    "Nepali" => "NEP ",
    "Newari" => "NEW ",
    "Nisi" => "NIS ",
    "Niuean" => "NIU ",
    "Nkole" => "NKL ",
    "Nogai" => "NOG ",
    "Northern Sami" => "NSM ",
    "Northern Tai" => "NTA ",
    "Norway House Cree" => "NHC ",
    "Norwegian" => "NOR ",
    "Nynorsk" => "NYN ",
    "Occitan" => "OCI ",
    "Oji-Cree" => "OCR ",
    "Ojibway" => "OJB ",
    "Oriya" => "ORI ",
    "Oromo" => "ORO ",
    "Ossetian" => "OSS ",
    "Palaung" => "PLG ",
    "Palestinian Aramaic" => "PAA ",
    "Pali" => "PAL ",
    "Palpa" => "PAP ",
    "Pashto" => "PAS ",
    "Polish" => "PLK ",
    "Polytonic Greek" => "PGR ",
    "Portuguese" => "PTG ",
    "Provencal" => "PRO ",
    "Punjabi" => "PAN ",
    "R-Cree" => "RCR ",
    "Rajasthani" => "RAJ ",
    "Rhaeto-Romanic" => "RMS ",
    "Riang" => "RIA ",
    "Romanian" => "ROM ",
    "Romany" => "ROY ",
    "Ruanda" => "RUA ",
    "Russian Buriat" => "RBU ",
    "Russian" => "RUS ",
    "Rusyn" => "RSY ",
    "Sadri" => "SAD ",
    "Samoan" => "SMO ",
    "Sango" => "SGO ",
    "Sanskrit" => "SAN ",
    "Santali" => "SAT ",
    "Saraiki" => "SRK ",
    "Sayisi" => "SAY ",
    "Sekota" => "SEK ",
    "Selkup" => "SEL ",
    "Sena" => "SNA ",
    "Serbian" => "SRB ",
    "Serer" => "SRR ",
    "Shan" => "SHN ",
    "Sibe" => "SIB ",
    "Sidamo" => "SID ",
    "Silte Gurage" => "SIG ",
    "Sindhi" => "SND ",
    "Sinhalese" => "SNH ",
    "Skolt Sami" => "SKS ",
    "Slavey" => "SLA ",
    "Slovak" => "SKY ",
    "Slovenian" => "SLV ",
    "Sodo Gurage" => "SOG ",
    "Somali" => "SML ",
    "Soninke" => "SNK ",
    "Sotho" => "SOT ",
    "South Slavey" => "SSL ",
    "Southern Sami" => "SSM ",
    "Spanish" => "ESP ",
    "Suri" => "SUR ",
    "Sutu" => "SXT ",
    "Svan" => "SVA ",
    "Swadaya Aramaic" => "SWA ",
    "Swahili" => "SWK ",
    "Swazi" => "SWZ ",
    "Swedish" => "SVE ",
    "Syriac" => "SYR ",
    "TH-Cree" => "TCR ",
    "Tabasaran" => "TAB ",
    "Tahitian" => "THT ",
    "Tai Lue" => "XBD ",
    "Tajiki" => "TAJ ",
    "Tamil" => "TAM ",
    "Tatar" => "TAT ",
    "Telugu" => "TEL ",
    "Temne" => "TMN ",
    "Thai" => "THA ",
    "Tibetan" => "TIB ",
    "Tigre" => "TGR ",
    "Tigrinya" => "TGY ",
    "Todo" => "TOD ",
    "Tonga" => "TNG ",
    "Tongan" => "TGN ",
    "Tsonga" => "TSG ",
    "Tswana" => "TNA ",
    "Tulu" => "TUL ",
    "Tundra Nenets" => "TNE ",
    "Turkish" => "TRK ",
    "Turkmen" => "TKM ",
    "Turoyo Aramaic" => "TUA ",
    "Tuvin" => "TUV ",
    "Twi" => "TWI ",
    "Udmurt" => "UDM ",
    "Ukrainian" => "UKR ",
    "Upper Sorbian" => "USB ",
    "Urdu" => "URD ",
    "Uyghur" => "UYG ",
    "Uzbek" => "UZB ",
    "Venda" => "VEN ",
    "Vietnamese" => "VIT ",
    "Wa" => "WA  ",
    "Wagdi" => "WAG ",
    "Welsh" => "WEL ",
    "West-Cree" => "WCR ",
    "Wolof" => "WLF ",
    "Woods Cree" => "DCR ",
    "Xhosa" => "XHS ",
    "Y-Cree" => "YCR ",
    "Yakut" => "YAK ",
    "Yi Classic" => "YIC ",
    "Yi Modern" => "YIM ",
    "Yiddish" => "JII ",
    "Yoruba" => "YBA ",
    "Zande" => "ZND ",
    "Zulu" => "ZUL ",
    },

'FEATURE' => {
    "Above-Base Forms" => "abvf",
    "Above-Base Mark Positioning" => "abvm",
    "Above-Base Substitutions" => "abvs",
    "Access All Alternates" => "aalt",
    "Akhands" => "akhn",
    "Alternate Annotation Forms" => "nalt",
    "Alternate Half Widths" => "halt",
    "Alternate Vertical Half Metrics" => "vhal",
    "Alternate Vertical Metrics" => "valt",
    "Alternative Fractions" => "afrc",
    "Below-Base Forms" => "blwf",
    "Below-Base Mark Positioning" => "blwm",
    "Below-Base Substitutions" => "blws",
    "Capital Spacing" => "cpsp",
    "Case-Sensitive Forms" => "case",
    "Centered CJK Punctuation" => "cpct",
    "Conjunct Forms After Ro" => "cfar",
    "Conjunct Forms" => "cjct",
    "Contextual Alternates" => "calt",
    "Contextual Ligatures" => "clig",
    "Contextual Swash" => "cswh",
    "Cursive Positioning" => "curs",
    "Default Processing" => "dflt",
    "Denominators" => "dnom",
    "Diphthongs (OBSOLETE)" => "dpng",
    "Discretionary Ligatures" => "dlig",
    "Distances" => "dist",
    "Expert Forms" => "expt",
    "Final Glyph On Line Alternates" => "falt",
    "Fractions" => "frac",
    "Full Widths" => "fwid",
    "Glyph Composition/Decomposition" => "ccmp",
    "Halant Forms" => "haln",
    "Half Forms" => "half",
    "Half Widths" => "hwid",
    "Hangul" => "hngl",
    "Historical Forms" => "hist",
    "Historical Ligatures" => "hlig",
    "Hojo (JIS X 0212-1990) Kanji Forms" => "hojo",
    "Horizontal Kana Alternates" => "hkna",
    "Initial Forms" => "init",
    "Isolated Forms" => "isol",
    "Italics" => "ital",
    "JIS2004 Forms" => "jp04",
    "JIS78 Forms" => "jp78",
    "JIS83 Forms" => "jp83",
    "JIS90 Forms" => "jp90",
    "Justification Alternates" => "jalt",
    "Kerning" => "kern",
    "Leading Jamo Forms" => "ljmo",
    "Left Bounds" => "lfbd",
    "Lining Figures" => "lnum",
    "Localized Forms" => "locl",
    "Mark Positioning via Substitution" => "mset",
    "Mark Positioning" => "mark",
    "Mark to Mark Positioning" => "mkmk",
    "Mathematical Greek" => "mgrk",
    "Medial Forms #2" => "med2",
    "Medial Forms" => "medi",
    "NLC Kanji Forms" => "nlck",
    "Nukta Forms" => "nukt",
    "Numerators" => "numr",
    "Oldstyle Figures" => "onum",
    "Optical Bounds" => "opbd",
    "Optical Size" => "size",
    "Ordinals" => "ordn",
    "Ornaments" => "ornm",
    "Petite Capitals From Capitals" => "c2pc",
    "Petite Capitals" => "pcap",
    "Post-base Forms" => "pstf",
    "Post-base Substitutions" => "psts",
    "Pre-base Forms" => "pref",
    "Pre-base Substitutions" => "pres",
    "Proportional Alternate Vertical Metrics" => "vpal",
    "Proportional Alternate Widths" => "palt",
    "Proportional Figures" => "pnum",
    "Proportional Kana" => "pkna",
    "Proportional Widths" => "pwid",
    "Quarter Widths" => "qwid",
    "Rakar Forms" => "rkrf",
    "Randomize" => "rand",
    "Reph Forms" => "rphf",
    "Required Ligatures" => "rlig",
    "Right Bounds" => "rtbd",
    "Right-To-Left Alternates" => "rtla",
    "Ruby Notation Forms" => "ruby",
    "Scientific Inferiors" => "sinf",
    "Simplified Forms" => "smpl",
    "Slashed Zero" => "zero",
    "Small Capitals From Capitals" => "c2sc",
    "Small Capitals" => "smcp",
    "Standard Ligatures" => "liga",
    "Stylistic Alternates" => "salt",
    "Stylistic Set 1" => "ss01",
    "Stylistic Set 10" => "ss10",
    "Stylistic Set 11" => "ss11",
    "Stylistic Set 12" => "ss12",
    "Stylistic Set 13" => "ss13",
    "Stylistic Set 14" => "ss14",
    "Stylistic Set 15" => "ss15",
    "Stylistic Set 16" => "ss16",
    "Stylistic Set 17" => "ss17",
    "Stylistic Set 18" => "ss18",
    "Stylistic Set 19" => "ss19",
    "Stylistic Set 2" => "ss02",
    "Stylistic Set 20" => "ss20",
    "Stylistic Set 3" => "ss03",
    "Stylistic Set 4" => "ss04",
    "Stylistic Set 5" => "ss05",
    "Stylistic Set 6" => "ss06",
    "Stylistic Set 7" => "ss07",
    "Stylistic Set 8" => "ss08",
    "Stylistic Set 9" => "ss09",
    "Subscript" => "subs",
    "Superscript" => "sups",
    "Swash" => "swsh",
    "Tabular Figures" => "tnum",
    "Terminal Forms #2" => "fin2",
    "Terminal Forms #3" => "fin3",
    "Terminal Forms" => "fina",
    "Third Widths" => "twid",
    "Titling" => "titl",
    "Traditional Forms" => "trad",
    "Traditional Name Forms" => "tnam",
    "Trailing Jamo Forms" => "tjmo",
    "Unicase" => "unic",
    "Vattu Variants" => "vatu",
    "Vertical Alternates and Rotation" => "vrt2",
    "Vertical Kana Alternates" => "vkna",
    "Vertical Kerning" => "vkrn",
    "Vertical Writing" => "vert",
    "Vowel Jamo Forms" => "vjmo",
    }
);

{
    foreach my $s (qw ( SCRIPT LANGUAGE FEATURE ) )
    {
        map { $ttnames{$s}{$tttags{$s}{$_}} = $_ }  keys %{$tttags{$s}};
    }
}


=head2 readtagsfile ( filename )

Read a file in the syntax of Tags.txt (included with Microsoft VOLT) to obtain additional/replacement tag definitions.

Returns 0 if can't open the file; else 1.

=cut

sub readtagsfile
{
    my $fname = shift;
    open (TAGS, $fname) or return 0;
	my ($what, $name, $tag);
	while (<TAGS>)
	{
		($what, $name, $tag) = (m/"([^"]*)", "([^"]*)", "([^"]*)"/);  #"
		$ttnames{$what}{$tag} = $name;
		$tttags{$what}{$name} = $tag;
	}
	close TAGS;
    return 1;	
}



1;
