package # hide from PAUSE
App::YTDL::Arguments;

use warnings;
use strict;
use 5.010000;

use Exporter qw( import );
our @EXPORT_OK = qw( from_arguments_to_choices );

use Term::ANSIScreen       qw( :cursor :screen );
use Term::Choose           qw( choose );
use URI                    qw();
use URI::Escape            qw( uri_unescape );

use if $^O eq 'MSWin32', 'Win32::Console::ANSI';

use App::YTDL::ChooseVideos qw( choose_videos );
use App::YTDL::DataExtract  qw( add_entry_to_info add_vimeo_entry_to_info );
use App::YTDL::GetData      qw( get_yt_list_info get_vimeo_list_info get_download_info );



sub from_arguments_to_choices {
    my ( $opt, @ids ) = @_;
    my $info = {};
    my $invalid_char = $opt->{invalid_char};
    my $more = 0;
    for my $id ( @ids ) {
        if ( my $uploader_id = _youtube_user_id( $opt, $id ) ) {
            my $ex = 'youtube';
            my $tmp = _yt_list_info( $opt, 'CL', $uploader_id );
            next if ! defined $tmp;
            my ( $vi ) = keys %{$tmp->{$ex}};
            my $prompt = $tmp->{$ex}{$vi}{uploader};
            choose_videos( $opt, $info, $tmp, $ex, $prompt );
        }
        elsif ( my $playlist_id = _youtube_playlist_id( $opt, $id ) ) {
            my $ex = 'youtube';
            my $tmp = _yt_list_info( $opt, 'PL', $playlist_id );
            next if ! defined $tmp;
            my $prompt = $playlist_id;
            choose_videos( $opt, $info, $tmp, $ex, $prompt );
        }
        elsif ( my $more_ids = _youtube_more_ids( $opt, $id ) ) {
            my $ex = 'youtube';
            my $tmp = _yt_more_url_info( $opt, $more_ids );
            next if ! defined $tmp;
            my $prompt = $more_ids;
            choose_videos( $opt, $info, $tmp, $ex, $prompt );
        }
        elsif ( my $video_id = _youtube_video_id( $opt, $id )  ) {
            my $ex = 'youtube';
            $info->{$ex}{$video_id}{extractor} = 'youtube';
        }
        elsif ( my $vimeo_uploader_id = _vimeo_uploader_id( $opt, $id )  ) {
            my $ex = 'vimeo';
            my $tmp = _vimeo_list_info( $opt, $vimeo_uploader_id );
            next if ! defined $tmp;
            my ( $vi ) = keys %{$tmp->{$ex}};
            my $prompt = $tmp->{$ex}{$vi}{uploader} || $vimeo_uploader_id;
            choose_videos( $opt, $info, $tmp, $ex, $prompt );
        }
        else {
            _generic( $opt, $info, $id );
        }
    }
    return $info;
}


sub _generic {
    my ( $opt, $info, $id ) = @_;
    my $message = "Fetching download info: ";
    my $tmp = get_download_info( $opt, $id, $message );
    next if ! defined $tmp;
    my ( $ex ) = keys %$tmp;
    my @keys = keys %{$tmp->{$ex}};
    if ( @keys == 1 ) {
        my $video_id = $keys[0];
        $info->{$ex}{$video_id} = $tmp->{$ex}{$video_id};
    }
    else {
        my $prompt = $id;
        choose_videos( $opt, $info, $tmp, $ex, $prompt );
    }
}


sub _youtube_video_id {
    my ( $opt, $id ) = @_;
    my $invalid_char = $opt->{invalid_char};
    return     if ! $opt->{fast_list_youtube};
    return     if ! $id;
    return $id if   $id =~ m{^[\p{PerlWord}-]{11}\z};
    return     if   $id !~ $opt->{yt_regexp};
    return $1  if   $id =~ m{/.*?[?&;!](?:v|video_id)=([^$invalid_char]+)};
    return $1  if   $id =~ m{/(?:e|v|embed)/([^$invalid_char]+)};
    return $1  if   $id =~ m{#p/(?:u|search)/\d+/([^&?/]+)};
    return $1  if   $id =~ m{youtu.be/([^$invalid_char]+)};
    return;
}

sub _youtube_playlist_id {
    my ( $opt, $id ) = @_;
    my $invalid_char = $opt->{invalid_char};
    return    if ! $opt->{fast_list_youtube};
    return    if ! $id;
    return $1 if   $id =~ m{^p#(?:[FP]L)?([^$invalid_char]+)\z};
    return    if   $id !~ $opt->{yt_regexp};
    return $1 if   $id =~ m{/.*?[?&;!]list=([^$invalid_char]+)};
    return;
}

sub _youtube_user_id {
    my ( $opt, $id ) = @_;
    my $invalid_char = $opt->{invalid_char};
    return    if ! $opt->{fast_list_youtube};
    return    if ! $id;
    return $1 if   $id =~ m{^c#([^$invalid_char]+)\z};
    return    if   $id !~ $opt->{yt_regexp};
    return $1 if   $id =~ m{/user/([^$invalid_char]+)};
    return $1 if   $id =~ m{/channel/([^$invalid_char]+)}; # ?
    return;
}

sub _youtube_more_ids {
    my ( $opt, $id ) = @_;
    my $invalid_char = $opt->{invalid_char};
    return    if ! $opt->{fast_list_youtube};
    return    if ! $id;
    return    if   $id !~ $opt->{yt_regexp};
    return $1 if   uri_unescape( $id ) =~ m{youtu\.?be.*video_ids=([^$invalid_char]+(?:,[^$invalid_char]+)*)};
    return;
}

sub _vimeo_uploader_id {
    my ( $opt, $id ) = @_;
    return    if ! $opt->{fast_list_vimeo};
    return    if ! $id;
    return $1 if   $id =~ m{https?://vimeo\.com/(?![0-9]+(?:$|[?#/]))([^/]+)(?:/videos|[#?]|$)};
    return;
}


sub _yt_list_info {
    my( $opt, $type, $list_id ) = @_;
    printf "Fetching %s info ... \n", $type eq 'PL' ? 'playlist' : 'channel';
    my $url = URI->new( $type eq 'PL'
        ? 'https://gdata.youtube.com/feeds/api/playlists/' . $list_id
        : 'https://gdata.youtube.com/feeds/api/users/'     . $list_id . '/uploads'
    );
    my $ex = 'youtube';
    my $tmp_info = {};
    my $start_index = 1;
    my $max_results = 50;
    my $count_entries = $max_results;

    while ( $count_entries == $max_results ) {
        $url->query_form( 'start-index' => $start_index, 'max-results' => $max_results, 'v' => $opt->{yt_api_v}, 'alt' => 'json' );
        $start_index += $max_results;
        my $h_ref = get_yt_list_info( $opt, $url->as_string );
        if ( ! defined $h_ref || ! defined $h_ref->{feed}{entry} || ! @{$h_ref->{feed}{entry}} ) {
            push @{$opt->{error_get_download_infos}}, sprintf "%s: %s", $type, $list_id;
            my $prompt = "Error list info: $type - $url";
            choose( [ 'ENTER to continue' ], { prompt => $prompt } );
            return;
        }
        $count_entries = @{$h_ref->{feed}{entry}};
        for my $entry ( @{$h_ref->{feed}{entry}} ) {
            add_entry_to_info( $opt, $tmp_info, $entry, $type, $list_id );
        }
        last if $opt->{max_videos} && $start_index > $opt->{max_videos};
        last if $start_index >= $h_ref->{feed}{'openSearch$totalResults'}{'$t'};
        #last if none { $_->{rel} eq 'next' } @{$h_ref->{feed}{link}};
    }
    if ( ! keys %{$tmp_info->{$ex}} ) {
        my $prompt = "No videos found: $type - $url";
        choose( [ 'Continue with ENTER' ], { prompt => $prompt } );
    }
    my $up = keys %{$tmp_info->{$ex}};
    print up( $up + 2 ), cldown;
    return $tmp_info;
}


sub _vimeo_list_info {
    my( $opt, $list_id ) = @_;
    printf "Fetching info ... \n";
    my $page_nr = 1;
    my $ex = 'vimeo';
    my $tmp_info = {};
    my $videos = 0;
    my $url;
    while ( 1 ) {
        $url = sprintf 'https://vimeo.com/%s/videos/page:%d/sort:date', $list_id, $page_nr;
        my ( $a_ref, $next ) = get_vimeo_list_info( $opt, $url );
        if ( ! defined $a_ref ) {
            push @{$opt->{error_get_download_infos}}, sprintf "%s: error", $list_id;
            return;
        }
        if ( ! @$a_ref ) {
            push @{$opt->{error_get_download_infos}}, sprintf "%s: no videos found.", $url;
        }
        for my $entry ( @{$a_ref} ) {
            add_vimeo_entry_to_info( $opt, $tmp_info, $entry );
        }
        $videos += @{$a_ref};
        last if $opt->{max_videos} && $videos >= $opt->{max_videos} - 2;
        last if ! $next;
        $page_nr++;
    }
    if ( ! keys %{$tmp_info->{$ex}} ) {
        my $prompt = "No videos found: $url";
        choose( [ 'Continue with ENTER' ], { prompt => $prompt } );
    }
    my $up = keys %{$tmp_info->{$ex}};
    print up( $up + 2 ), cldown;
    return $tmp_info;
}


sub _yt_more_url_info {
    my ( $opt, $more_ids ) = @_;
    my $tmp_info = {};
    for my $video_id ( split /,/, $more_ids ) {
        my $url = URI->new( 'https://gdata.youtube.com/feeds/api/videos/' . $video_id );
        $url->query_form( 'v' => $opt->{yt_api_v}, 'alt' => 'json' );
        my $h_ref = get_yt_list_info( $opt, $url );
        if ( ! defined $h_ref ) {
            my $err_msg = 'Video group: ' . $more_ids . ' - ' . $video_id . '   ' . $url;
            push @{$opt->{error_get_download_infos}}, $err_msg;
            next;
        }
        my $entry = $h_ref->{feed}{entry};
        add_entry_to_info( $opt, $tmp_info, $entry );
    }
    return $tmp_info;
}




1;


__END__
