package Math::DWT::Wavelet::Symlet;

=head1 NAME

Math::DWT::Wavelet::Symlet - FIR lo- & hi-pass filter coefficients for the Symlet wavelet.

=head1 VERSION

Version 0.01_1

=cut

our $VERSION = '0.01_1';

=head1 SYNOPSIS

This module provides the lo- and hi-pass decomposition and reconstruction filter coefficients for the Symlet wavelet.  It is meant to be used with other Math::DWT modules:

    use Math::DWT;
    use Math::DWT::UDWT;

    my $dwt = Math::DWT->new('Symlet');
    my $udwt = Math::DWT::UDWT->new('Symlet');


=cut

=head1 SUBROUTINES/METHODS

=head2 new(VAR)

Takes a single argument, var, which determines the particular filter set which the object will provide.  You can get a list of available filter sets with the vars() method.

    my $symlet4 = Math::DWT::Wavelet::Symlet->new(4);

This method returns a Math::DWT::Wavelet::Symlet object;

=head2 vars()

This method returns a list of possible choices for VAR when creating a new object, e.g.:

    my @v = Math::DWT::Wavelet::Symlet->vars();
    print scalar(@v); # 19


=head2 filters()

Depending on the context in which it is called, returns an array or an arrayref containing (lo_d, hi_d, lo_r, hi_r) - the set of filters which are defined with the instantiated object.


=head2 lo_d()

=head2 hi_d()

=head2 lo_r()

=head2 hi_r()

Returns the requested set of filter coefficients as either an array or arrayref, depending on calling context.

=head1 SEE ALSO

Math::DWT(3pm), Math::DWT::UDWT(3pm), Math::DWT::Wavelet::Haar(3pm), Math::DWT::Wavelet::Coiflet(3pm), Math::DWT::Wavelet::Daubechies(3pm), Math::DWT::Wavelet::Biorthogonal(3pm), Math::DWT::Wavelet::ReverseBiorthogonal(3pm), Math::DWT::Wavelet::DiscreteMeyer(3pm), perl(1)


=head1 AUTHOR


Mike Kroh, C<< <kroh at cpan.org> >>

=head1 BUGS


Please report any bugs or feature requests to C<bug-math-dwt-wavelet-symlet at rt.cpan.org>, or through the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Math-DWT-Wavelet-Symlet>.  I will be notified, and then you'll automatically be notified of progress on your bug as I make changes.



=head1 ACKNOWLEDGEMENTS

These wavelet filter coefficients were scraped from this site: L<http://wavelets.pybytes.com/>.

=head1 LICENSE AND COPYRIGHT


Copyright 2016 Mike Kroh.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut





my @vars=qw/2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20/;

my %lo_d; my %hi_d; my %lo_r; my %hi_r;


$lo_d{2}=[qw/-0.12940952255092145 0.22414386804185735 0.836516303737469 0.48296291314469025/];
$hi_d{2}=[qw/-0.48296291314469025 0.836516303737469 -0.22414386804185735 -0.12940952255092145/];
$lo_r{2}=[qw/0.48296291314469025 0.836516303737469 0.22414386804185735 -0.12940952255092145/];
$hi_r{2}=[qw/-0.12940952255092145 -0.22414386804185735 0.836516303737469 -0.48296291314469025/];

$lo_d{3}=[qw/0.035226291882100656 -0.08544127388224149 -0.13501102001039084 0.4598775021193313 0.8068915093133388 0.3326705529509569/];
$hi_d{3}=[qw/-0.3326705529509569 0.8068915093133388 -0.4598775021193313 -0.13501102001039084 0.08544127388224149 0.035226291882100656/];
$lo_r{3}=[qw/0.3326705529509569 0.8068915093133388 0.4598775021193313 -0.13501102001039084 -0.08544127388224149 0.035226291882100656/];
$hi_r{3}=[qw/0.035226291882100656 0.08544127388224149 -0.13501102001039084 -0.4598775021193313 0.8068915093133388 -0.3326705529509569/];

$lo_d{4}=[qw/-0.07576571478927333 -0.02963552764599851 0.49761866763201545 0.8037387518059161 0.29785779560527736 -0.09921954357684722 -0.012603967262037833 0.0322231006040427/];
$hi_d{4}=[qw/-0.0322231006040427 -0.012603967262037833 0.09921954357684722 0.29785779560527736 -0.8037387518059161 0.49761866763201545 0.02963552764599851 -0.07576571478927333/];
$lo_r{4}=[qw/0.0322231006040427 -0.012603967262037833 -0.09921954357684722 0.29785779560527736 0.8037387518059161 0.49761866763201545 -0.02963552764599851 -0.07576571478927333/];
$hi_r{4}=[qw/-0.07576571478927333 0.02963552764599851 0.49761866763201545 -0.8037387518059161 0.29785779560527736 0.09921954357684722 -0.012603967262037833 -0.0322231006040427/];

$lo_d{5}=[qw/0.027333068345077982 0.029519490925774643 -0.039134249302383094 0.1993975339773936 0.7234076904024206 0.6339789634582119 0.01660210576452232 -0.17532808990845047 -0.021101834024758855 0.019538882735286728/];
$hi_d{5}=[qw/-0.019538882735286728 -0.021101834024758855 0.17532808990845047 0.01660210576452232 -0.6339789634582119 0.7234076904024206 -0.1993975339773936 -0.039134249302383094 -0.029519490925774643 0.027333068345077982/];
$lo_r{5}=[qw/0.019538882735286728 -0.021101834024758855 -0.17532808990845047 0.01660210576452232 0.6339789634582119 0.7234076904024206 0.1993975339773936 -0.039134249302383094 0.029519490925774643 0.027333068345077982/];
$hi_r{5}=[qw/0.027333068345077982 -0.029519490925774643 -0.039134249302383094 -0.1993975339773936 0.7234076904024206 -0.6339789634582119 0.01660210576452232 0.17532808990845047 -0.021101834024758855 -0.019538882735286728/];

$lo_d{6}=[qw/0.015404109327027373 0.0034907120842174702 -0.11799011114819057 -0.048311742585633 0.4910559419267466 0.787641141030194 0.3379294217276218 -0.07263752278646252 -0.021060292512300564 0.04472490177066578 0.0017677118642428036 -0.007800708325034148/];
$hi_d{6}=[qw/0.007800708325034148 0.0017677118642428036 -0.04472490177066578 -0.021060292512300564 0.07263752278646252 0.3379294217276218 -0.787641141030194 0.4910559419267466 0.048311742585633 -0.11799011114819057 -0.0034907120842174702 0.015404109327027373/];
$lo_r{6}=[qw/-0.007800708325034148 0.0017677118642428036 0.04472490177066578 -0.021060292512300564 -0.07263752278646252 0.3379294217276218 0.787641141030194 0.4910559419267466 -0.048311742585633 -0.11799011114819057 0.0034907120842174702 0.015404109327027373/];
$hi_r{6}=[qw/0.015404109327027373 -0.0034907120842174702 -0.11799011114819057 0.048311742585633 0.4910559419267466 -0.787641141030194 0.3379294217276218 0.07263752278646252 -0.021060292512300564 -0.04472490177066578 0.0017677118642428036 0.007800708325034148/];

$lo_d{7}=[qw/0.002681814568257878 -0.0010473848886829163 -0.01263630340325193 0.03051551316596357 0.0678926935013727 -0.049552834937127255 0.017441255086855827 0.5361019170917628 0.767764317003164 0.2886296317515146 -0.14004724044296152 -0.10780823770381774 0.004010244871533663 0.010268176708511255/];
$hi_d{7}=[qw/-0.010268176708511255 0.004010244871533663 0.10780823770381774 -0.14004724044296152 -0.2886296317515146 0.767764317003164 -0.5361019170917628 0.017441255086855827 0.049552834937127255 0.0678926935013727 -0.03051551316596357 -0.01263630340325193 0.0010473848886829163 0.002681814568257878/];
$lo_r{7}=[qw/0.010268176708511255 0.004010244871533663 -0.10780823770381774 -0.14004724044296152 0.2886296317515146 0.767764317003164 0.5361019170917628 0.017441255086855827 -0.049552834937127255 0.0678926935013727 0.03051551316596357 -0.01263630340325193 -0.0010473848886829163 0.002681814568257878/];
$hi_r{7}=[qw/0.002681814568257878 0.0010473848886829163 -0.01263630340325193 -0.03051551316596357 0.0678926935013727 0.049552834937127255 0.017441255086855827 -0.5361019170917628 0.767764317003164 -0.2886296317515146 -0.14004724044296152 0.10780823770381774 0.004010244871533663 -0.010268176708511255/];

$lo_d{8}=[qw/-0.0033824159510061256 -0.0005421323317911481 0.03169508781149298 0.007607487324917605 -0.1432942383508097 -0.061273359067658524 0.4813596512583722 0.7771857517005235 0.3644418948353314 -0.05194583810770904 -0.027219029917056003 0.049137179673607506 0.003808752013890615 -0.01495225833704823 -0.0003029205147213668 0.0018899503327594609/];
$hi_d{8}=[qw/-0.0018899503327594609 -0.0003029205147213668 0.01495225833704823 0.003808752013890615 -0.049137179673607506 -0.027219029917056003 0.05194583810770904 0.3644418948353314 -0.7771857517005235 0.4813596512583722 0.061273359067658524 -0.1432942383508097 -0.007607487324917605 0.03169508781149298 0.0005421323317911481 -0.0033824159510061256/];
$lo_r{8}=[qw/0.0018899503327594609 -0.0003029205147213668 -0.01495225833704823 0.003808752013890615 0.049137179673607506 -0.027219029917056003 -0.05194583810770904 0.3644418948353314 0.7771857517005235 0.4813596512583722 -0.061273359067658524 -0.1432942383508097 0.007607487324917605 0.03169508781149298 -0.0005421323317911481 -0.0033824159510061256/];
$hi_r{8}=[qw/-0.0033824159510061256 0.0005421323317911481 0.03169508781149298 -0.007607487324917605 -0.1432942383508097 0.061273359067658524 0.4813596512583722 -0.7771857517005235 0.3644418948353314 0.05194583810770904 -0.027219029917056003 -0.049137179673607506 0.003808752013890615 0.01495225833704823 -0.0003029205147213668 -0.0018899503327594609/];

$lo_d{9}=[qw/0.0014009155259146807 0.0006197808889855868 -0.013271967781817119 -0.01152821020767923 0.03022487885827568 0.0005834627461258068 -0.05456895843083407 0.238760914607303 0.717897082764412 0.6173384491409358 0.035272488035271894 -0.19155083129728512 -0.018233770779395985 0.06207778930288603 0.008859267493400484 -0.010264064027633142 -0.0004731544986800831 0.0010694900329086053/];
$hi_d{9}=[qw/-0.0010694900329086053 -0.0004731544986800831 0.010264064027633142 0.008859267493400484 -0.06207778930288603 -0.018233770779395985 0.19155083129728512 0.035272488035271894 -0.6173384491409358 0.717897082764412 -0.238760914607303 -0.05456895843083407 -0.0005834627461258068 0.03022487885827568 0.01152821020767923 -0.013271967781817119 -0.0006197808889855868 0.0014009155259146807/];
$lo_r{9}=[qw/0.0010694900329086053 -0.0004731544986800831 -0.010264064027633142 0.008859267493400484 0.06207778930288603 -0.018233770779395985 -0.19155083129728512 0.035272488035271894 0.6173384491409358 0.717897082764412 0.238760914607303 -0.05456895843083407 0.0005834627461258068 0.03022487885827568 -0.01152821020767923 -0.013271967781817119 0.0006197808889855868 0.0014009155259146807/];
$hi_r{9}=[qw/0.0014009155259146807 -0.0006197808889855868 -0.013271967781817119 0.01152821020767923 0.03022487885827568 -0.0005834627461258068 -0.05456895843083407 -0.238760914607303 0.717897082764412 -0.6173384491409358 0.035272488035271894 0.19155083129728512 -0.018233770779395985 -0.06207778930288603 0.008859267493400484 0.010264064027633142 -0.0004731544986800831 -0.0010694900329086053/];

$lo_d{10}=[qw/0.0007701598091144901 9.563267072289475e-05 -0.008641299277022422 -0.0014653825813050513 0.0459272392310922 0.011609893903711381 -0.15949427888491757 -0.07088053578324385 0.47169066693843925 0.7695100370211071 0.38382676106708546 -0.03553674047381755 -0.0319900568824278 0.04999497207737669 0.005764912033581909 -0.02035493981231129 -0.0008043589320165449 0.004593173585311828 5.7036083618494284e-05 -0.0004593294210046588/];
$hi_d{10}=[qw/0.0004593294210046588 5.7036083618494284e-05 -0.004593173585311828 -0.0008043589320165449 0.02035493981231129 0.005764912033581909 -0.04999497207737669 -0.0319900568824278 0.03553674047381755 0.38382676106708546 -0.7695100370211071 0.47169066693843925 0.07088053578324385 -0.15949427888491757 -0.011609893903711381 0.0459272392310922 0.0014653825813050513 -0.008641299277022422 -9.563267072289475e-05 0.0007701598091144901/];
$lo_r{10}=[qw/-0.0004593294210046588 5.7036083618494284e-05 0.004593173585311828 -0.0008043589320165449 -0.02035493981231129 0.005764912033581909 0.04999497207737669 -0.0319900568824278 -0.03553674047381755 0.38382676106708546 0.7695100370211071 0.47169066693843925 -0.07088053578324385 -0.15949427888491757 0.011609893903711381 0.0459272392310922 -0.0014653825813050513 -0.008641299277022422 9.563267072289475e-05 0.0007701598091144901/];
$hi_r{10}=[qw/0.0007701598091144901 -9.563267072289475e-05 -0.008641299277022422 0.0014653825813050513 0.0459272392310922 -0.011609893903711381 -0.15949427888491757 0.07088053578324385 0.47169066693843925 -0.7695100370211071 0.38382676106708546 0.03553674047381755 -0.0319900568824278 -0.04999497207737669 0.005764912033581909 0.02035493981231129 -0.0008043589320165449 -0.004593173585311828 5.7036083618494284e-05 0.0004593294210046588/];

$lo_d{11}=[qw/0.00017172195069934854 -3.8795655736158566e-05 -0.0017343662672978692 0.0005883527353969915 0.00651249567477145 -0.009857934828789794 -0.024080841595864003 0.0370374159788594 0.06997679961073414 -0.022832651022562687 0.09719839445890947 0.5720229780100871 0.7303435490883957 0.23768990904924897 -0.2046547944958006 -0.1446023437053156 0.03526675956446655 0.04300019068155228 -0.0020034719001093887 -0.006389603666454892 0.00011053509764272153 0.0004892636102619239/];
$hi_d{11}=[qw/-0.0004892636102619239 0.00011053509764272153 0.006389603666454892 -0.0020034719001093887 -0.04300019068155228 0.03526675956446655 0.1446023437053156 -0.2046547944958006 -0.23768990904924897 0.7303435490883957 -0.5720229780100871 0.09719839445890947 0.022832651022562687 0.06997679961073414 -0.0370374159788594 -0.024080841595864003 0.009857934828789794 0.00651249567477145 -0.0005883527353969915 -0.0017343662672978692 3.8795655736158566e-05 0.00017172195069934854/];
$lo_r{11}=[qw/0.0004892636102619239 0.00011053509764272153 -0.006389603666454892 -0.0020034719001093887 0.04300019068155228 0.03526675956446655 -0.1446023437053156 -0.2046547944958006 0.23768990904924897 0.7303435490883957 0.5720229780100871 0.09719839445890947 -0.022832651022562687 0.06997679961073414 0.0370374159788594 -0.024080841595864003 -0.009857934828789794 0.00651249567477145 0.0005883527353969915 -0.0017343662672978692 -3.8795655736158566e-05 0.00017172195069934854/];
$hi_r{11}=[qw/0.00017172195069934854 3.8795655736158566e-05 -0.0017343662672978692 -0.0005883527353969915 0.00651249567477145 0.009857934828789794 -0.024080841595864003 -0.0370374159788594 0.06997679961073414 0.022832651022562687 0.09719839445890947 -0.5720229780100871 0.7303435490883957 -0.23768990904924897 -0.2046547944958006 0.1446023437053156 0.03526675956446655 -0.04300019068155228 -0.0020034719001093887 0.006389603666454892 0.00011053509764272153 -0.0004892636102619239/];

$lo_d{12}=[qw/0.00011196719424656033 -1.1353928041541452e-05 -0.0013497557555715387 0.00018021409008538188 0.007414965517654251 -0.0014089092443297553 -0.024220722675013445 0.0075537806116804775 0.04917931829966084 -0.03584883073695439 -0.022162306170337816 0.39888597239022 0.7634790977836572 0.46274103121927235 -0.07833262231634322 -0.17037069723886492 0.01530174062247884 0.05780417944550566 -0.0026043910313322326 -0.014589836449234145 0.00030764779631059454 0.002350297614183465 -1.8158078862617515e-05 -0.0001790665869750869/];
$hi_d{12}=[qw/0.0001790665869750869 -1.8158078862617515e-05 -0.002350297614183465 0.00030764779631059454 0.014589836449234145 -0.0026043910313322326 -0.05780417944550566 0.01530174062247884 0.17037069723886492 -0.07833262231634322 -0.46274103121927235 0.7634790977836572 -0.39888597239022 -0.022162306170337816 0.03584883073695439 0.04917931829966084 -0.0075537806116804775 -0.024220722675013445 0.0014089092443297553 0.007414965517654251 -0.00018021409008538188 -0.0013497557555715387 1.1353928041541452e-05 0.00011196719424656033/];
$lo_r{12}=[qw/-0.0001790665869750869 -1.8158078862617515e-05 0.002350297614183465 0.00030764779631059454 -0.014589836449234145 -0.0026043910313322326 0.05780417944550566 0.01530174062247884 -0.17037069723886492 -0.07833262231634322 0.46274103121927235 0.7634790977836572 0.39888597239022 -0.022162306170337816 -0.03584883073695439 0.04917931829966084 0.0075537806116804775 -0.024220722675013445 -0.0014089092443297553 0.007414965517654251 0.00018021409008538188 -0.0013497557555715387 -1.1353928041541452e-05 0.00011196719424656033/];
$hi_r{12}=[qw/0.00011196719424656033 1.1353928041541452e-05 -0.0013497557555715387 -0.00018021409008538188 0.007414965517654251 0.0014089092443297553 -0.024220722675013445 -0.0075537806116804775 0.04917931829966084 0.03584883073695439 -0.022162306170337816 -0.39888597239022 0.7634790977836572 -0.46274103121927235 -0.07833262231634322 0.17037069723886492 0.01530174062247884 -0.05780417944550566 -0.0026043910313322326 0.014589836449234145 0.00030764779631059454 -0.002350297614183465 -1.8158078862617515e-05 0.0001790665869750869/];

$lo_d{13}=[qw/6.820325263075319e-05 -3.573862364868901e-05 -0.0011360634389281183 -0.0001709428585302221 0.0075262253899681 0.005296359738725025 -0.02021676813338983 -0.017211642726299048 0.013862497435849205 -0.0597506277179437 -0.12436246075153011 0.19770481877117801 0.6957391505614964 0.6445643839011856 0.11023022302137217 -0.14049009311363403 0.008819757670420546 0.09292603089913712 0.017618296880653084 -0.020749686325515677 -0.0014924472742598532 0.0056748537601224395 0.00041326119884196064 -0.0007213643851362283 3.690537342319624e-05 7.042986690694402e-05/];
$hi_d{13}=[qw/-7.042986690694402e-05 3.690537342319624e-05 0.0007213643851362283 0.00041326119884196064 -0.0056748537601224395 -0.0014924472742598532 0.020749686325515677 0.017618296880653084 -0.09292603089913712 0.008819757670420546 0.14049009311363403 0.11023022302137217 -0.6445643839011856 0.6957391505614964 -0.19770481877117801 -0.12436246075153011 0.0597506277179437 0.013862497435849205 0.017211642726299048 -0.02021676813338983 -0.005296359738725025 0.0075262253899681 0.0001709428585302221 -0.0011360634389281183 3.573862364868901e-05 6.820325263075319e-05/];
$lo_r{13}=[qw/7.042986690694402e-05 3.690537342319624e-05 -0.0007213643851362283 0.00041326119884196064 0.0056748537601224395 -0.0014924472742598532 -0.020749686325515677 0.017618296880653084 0.09292603089913712 0.008819757670420546 -0.14049009311363403 0.11023022302137217 0.6445643839011856 0.6957391505614964 0.19770481877117801 -0.12436246075153011 -0.0597506277179437 0.013862497435849205 -0.017211642726299048 -0.02021676813338983 0.005296359738725025 0.0075262253899681 -0.0001709428585302221 -0.0011360634389281183 -3.573862364868901e-05 6.820325263075319e-05/];
$hi_r{13}=[qw/6.820325263075319e-05 3.573862364868901e-05 -0.0011360634389281183 0.0001709428585302221 0.0075262253899681 -0.005296359738725025 -0.02021676813338983 0.017211642726299048 0.013862497435849205 0.0597506277179437 -0.12436246075153011 -0.19770481877117801 0.6957391505614964 -0.6445643839011856 0.11023022302137217 0.14049009311363403 0.008819757670420546 -0.09292603089913712 0.017618296880653084 0.020749686325515677 -0.0014924472742598532 -0.0056748537601224395 0.00041326119884196064 0.0007213643851362283 3.690537342319624e-05 -7.042986690694402e-05/];

$lo_d{14}=[qw/-2.5879090265397886e-05 1.1210865808890361e-05 0.00039843567297594335 -6.286542481477636e-05 -0.002579441725933078 0.0003664765736601183 0.01003769371767227 -0.002753774791224071 -0.029196217764038187 0.004280520499019378 0.03743308836285345 -0.057634498351326995 -0.03531811211497973 0.39320152196208885 0.7599762419610909 0.4753357626342066 -0.05811182331771783 -0.15999741114652205 0.02589858753104667 0.06982761636180755 -0.002365048836740385 -0.019439314263626713 0.0010131419871842082 0.004532677471945648 -7.321421356702399e-05 -0.0006057601824664335 1.9329016965523917e-05 4.4618977991475265e-05/];
$hi_d{14}=[qw/-4.4618977991475265e-05 1.9329016965523917e-05 0.0006057601824664335 -7.321421356702399e-05 -0.004532677471945648 0.0010131419871842082 0.019439314263626713 -0.002365048836740385 -0.06982761636180755 0.02589858753104667 0.15999741114652205 -0.05811182331771783 -0.4753357626342066 0.7599762419610909 -0.39320152196208885 -0.03531811211497973 0.057634498351326995 0.03743308836285345 -0.004280520499019378 -0.029196217764038187 0.002753774791224071 0.01003769371767227 -0.0003664765736601183 -0.002579441725933078 6.286542481477636e-05 0.00039843567297594335 -1.1210865808890361e-05 -2.5879090265397886e-05/];
$lo_r{14}=[qw/4.4618977991475265e-05 1.9329016965523917e-05 -0.0006057601824664335 -7.321421356702399e-05 0.004532677471945648 0.0010131419871842082 -0.019439314263626713 -0.002365048836740385 0.06982761636180755 0.02589858753104667 -0.15999741114652205 -0.05811182331771783 0.4753357626342066 0.7599762419610909 0.39320152196208885 -0.03531811211497973 -0.057634498351326995 0.03743308836285345 0.004280520499019378 -0.029196217764038187 -0.002753774791224071 0.01003769371767227 0.0003664765736601183 -0.002579441725933078 -6.286542481477636e-05 0.00039843567297594335 1.1210865808890361e-05 -2.5879090265397886e-05/];
$hi_r{14}=[qw/-2.5879090265397886e-05 -1.1210865808890361e-05 0.00039843567297594335 6.286542481477636e-05 -0.002579441725933078 -0.0003664765736601183 0.01003769371767227 0.002753774791224071 -0.029196217764038187 -0.004280520499019378 0.03743308836285345 0.057634498351326995 -0.03531811211497973 -0.39320152196208885 0.7599762419610909 -0.4753357626342066 -0.05811182331771783 0.15999741114652205 0.02589858753104667 -0.06982761636180755 -0.002365048836740385 0.019439314263626713 0.0010131419871842082 -0.004532677471945648 -7.321421356702399e-05 0.0006057601824664335 1.9329016965523917e-05 -4.4618977991475265e-05/];

$lo_d{15}=[qw/9.712419737963348e-06 -7.35966679891947e-06 -0.00016066186637495343 5.512254785558665e-05 0.0010705672194623959 -0.0002673164464718057 -0.0035901654473726417 0.003423450736351241 0.01007997708790567 -0.01940501143093447 -0.03887671687683349 0.021937642719753955 0.04073547969681068 -0.04108266663538248 0.11153369514261872 0.5786404152150345 0.7218430296361812 0.2439627054321663 -0.1966263587662373 -0.1340562984562539 0.06839331006048024 0.06796982904487918 -0.008744788886477952 -0.01717125278163873 0.0015261382781819983 0.003481028737064895 -0.00010815440168545525 -0.00040216853760293483 2.171789015077892e-05 2.866070852531808e-05/];
$hi_d{15}=[qw/-2.866070852531808e-05 2.171789015077892e-05 0.00040216853760293483 -0.00010815440168545525 -0.003481028737064895 0.0015261382781819983 0.01717125278163873 -0.008744788886477952 -0.06796982904487918 0.06839331006048024 0.1340562984562539 -0.1966263587662373 -0.2439627054321663 0.7218430296361812 -0.5786404152150345 0.11153369514261872 0.04108266663538248 0.04073547969681068 -0.021937642719753955 -0.03887671687683349 0.01940501143093447 0.01007997708790567 -0.003423450736351241 -0.0035901654473726417 0.0002673164464718057 0.0010705672194623959 -5.512254785558665e-05 -0.00016066186637495343 7.35966679891947e-06 9.712419737963348e-06/];
$lo_r{15}=[qw/2.866070852531808e-05 2.171789015077892e-05 -0.00040216853760293483 -0.00010815440168545525 0.003481028737064895 0.0015261382781819983 -0.01717125278163873 -0.008744788886477952 0.06796982904487918 0.06839331006048024 -0.1340562984562539 -0.1966263587662373 0.2439627054321663 0.7218430296361812 0.5786404152150345 0.11153369514261872 -0.04108266663538248 0.04073547969681068 0.021937642719753955 -0.03887671687683349 -0.01940501143093447 0.01007997708790567 0.003423450736351241 -0.0035901654473726417 -0.0002673164464718057 0.0010705672194623959 5.512254785558665e-05 -0.00016066186637495343 -7.35966679891947e-06 9.712419737963348e-06/];
$hi_r{15}=[qw/9.712419737963348e-06 7.35966679891947e-06 -0.00016066186637495343 -5.512254785558665e-05 0.0010705672194623959 0.0002673164464718057 -0.0035901654473726417 -0.003423450736351241 0.01007997708790567 0.01940501143093447 -0.03887671687683349 -0.021937642719753955 0.04073547969681068 0.04108266663538248 0.11153369514261872 -0.5786404152150345 0.7218430296361812 -0.2439627054321663 -0.1966263587662373 0.1340562984562539 0.06839331006048024 -0.06796982904487918 -0.008744788886477952 0.01717125278163873 0.0015261382781819983 -0.003481028737064895 -0.00010815440168545525 0.00040216853760293483 2.171789015077892e-05 -2.866070852531808e-05/];

$lo_d{16}=[qw/6.230006701220761e-06 -3.113556407621969e-06 -0.00010943147929529757 2.8078582128442894e-05 0.0008523547108047095 -0.0001084456223089688 -0.0038809122526038786 0.0007182119788317892 0.012666731659857348 -0.0031265171722710075 -0.031051202843553064 0.004869274404904607 0.032333091610663785 -0.06698304907021778 -0.034574228416972504 0.39712293362064416 0.7565249878756971 0.47534280601152273 -0.054040601387606135 -0.15959219218520598 0.03072113906330156 0.07803785290341991 -0.003510275068374009 -0.024952758046290123 0.001359844742484172 0.0069377611308027096 -0.00022211647621176323 -0.0013387206066921965 3.656592483348223e-05 0.00016545679579108483 -5.396483179315242e-06 -1.0797982104319795e-05/];
$hi_d{16}=[qw/1.0797982104319795e-05 -5.396483179315242e-06 -0.00016545679579108483 3.656592483348223e-05 0.0013387206066921965 -0.00022211647621176323 -0.0069377611308027096 0.001359844742484172 0.024952758046290123 -0.003510275068374009 -0.07803785290341991 0.03072113906330156 0.15959219218520598 -0.054040601387606135 -0.47534280601152273 0.7565249878756971 -0.39712293362064416 -0.034574228416972504 0.06698304907021778 0.032333091610663785 -0.004869274404904607 -0.031051202843553064 0.0031265171722710075 0.012666731659857348 -0.0007182119788317892 -0.0038809122526038786 0.0001084456223089688 0.0008523547108047095 -2.8078582128442894e-05 -0.00010943147929529757 3.113556407621969e-06 6.230006701220761e-06/];
$lo_r{16}=[qw/-1.0797982104319795e-05 -5.396483179315242e-06 0.00016545679579108483 3.656592483348223e-05 -0.0013387206066921965 -0.00022211647621176323 0.0069377611308027096 0.001359844742484172 -0.024952758046290123 -0.003510275068374009 0.07803785290341991 0.03072113906330156 -0.15959219218520598 -0.054040601387606135 0.47534280601152273 0.7565249878756971 0.39712293362064416 -0.034574228416972504 -0.06698304907021778 0.032333091610663785 0.004869274404904607 -0.031051202843553064 -0.0031265171722710075 0.012666731659857348 0.0007182119788317892 -0.0038809122526038786 -0.0001084456223089688 0.0008523547108047095 2.8078582128442894e-05 -0.00010943147929529757 -3.113556407621969e-06 6.230006701220761e-06/];
$hi_r{16}=[qw/6.230006701220761e-06 3.113556407621969e-06 -0.00010943147929529757 -2.8078582128442894e-05 0.0008523547108047095 0.0001084456223089688 -0.0038809122526038786 -0.0007182119788317892 0.012666731659857348 0.0031265171722710075 -0.031051202843553064 -0.004869274404904607 0.032333091610663785 0.06698304907021778 -0.034574228416972504 -0.39712293362064416 0.7565249878756971 -0.47534280601152273 -0.054040601387606135 0.15959219218520598 0.03072113906330156 -0.07803785290341991 -0.003510275068374009 0.024952758046290123 0.001359844742484172 -0.0069377611308027096 -0.00022211647621176323 0.0013387206066921965 3.656592483348223e-05 -0.00016545679579108483 -5.396483179315242e-06 1.0797982104319795e-05/];

$lo_d{17}=[qw/4.297343327345983e-06 2.7801266938414138e-06 -6.293702597554192e-05 -1.3506383399901165e-05 0.0004759963802638669 -0.000138642302680455 -0.0027416759756816018 0.0008567700701915741 0.010482366933031529 -0.004819212803176148 -0.03329138349235933 0.01790395221434112 0.10475461484223211 0.0172711782105185 -0.11856693261143636 0.1423983504146782 0.6507166292045456 0.681488995344925 0.18053958458111286 -0.15507600534974825 -0.08607087472073338 0.016158808725919346 -0.007261634750928767 -0.01803889724191924 0.009952982523509598 0.012396988366648726 -0.001905407689852666 -0.003932325279797902 5.8400428694052584e-05 0.0007198270642148971 2.520793314082878e-05 -7.607124405605129e-05 -2.4527163425833e-06 3.7912531943321266e-06/];
$hi_d{17}=[qw/-3.7912531943321266e-06 -2.4527163425833e-06 7.607124405605129e-05 2.520793314082878e-05 -0.0007198270642148971 5.8400428694052584e-05 0.003932325279797902 -0.001905407689852666 -0.012396988366648726 0.009952982523509598 0.01803889724191924 -0.007261634750928767 -0.016158808725919346 -0.08607087472073338 0.15507600534974825 0.18053958458111286 -0.681488995344925 0.6507166292045456 -0.1423983504146782 -0.11856693261143636 -0.0172711782105185 0.10475461484223211 -0.01790395221434112 -0.03329138349235933 0.004819212803176148 0.010482366933031529 -0.0008567700701915741 -0.0027416759756816018 0.000138642302680455 0.0004759963802638669 1.3506383399901165e-05 -6.293702597554192e-05 -2.7801266938414138e-06 4.297343327345983e-06/];
$lo_r{17}=[qw/3.7912531943321266e-06 -2.4527163425833e-06 -7.607124405605129e-05 2.520793314082878e-05 0.0007198270642148971 5.8400428694052584e-05 -0.003932325279797902 -0.001905407689852666 0.012396988366648726 0.009952982523509598 -0.01803889724191924 -0.007261634750928767 0.016158808725919346 -0.08607087472073338 -0.15507600534974825 0.18053958458111286 0.681488995344925 0.6507166292045456 0.1423983504146782 -0.11856693261143636 0.0172711782105185 0.10475461484223211 0.01790395221434112 -0.03329138349235933 -0.004819212803176148 0.010482366933031529 0.0008567700701915741 -0.0027416759756816018 -0.000138642302680455 0.0004759963802638669 -1.3506383399901165e-05 -6.293702597554192e-05 2.7801266938414138e-06 4.297343327345983e-06/];
$hi_r{17}=[qw/4.297343327345983e-06 -2.7801266938414138e-06 -6.293702597554192e-05 1.3506383399901165e-05 0.0004759963802638669 0.000138642302680455 -0.0027416759756816018 -0.0008567700701915741 0.010482366933031529 0.004819212803176148 -0.03329138349235933 -0.01790395221434112 0.10475461484223211 -0.0172711782105185 -0.11856693261143636 -0.1423983504146782 0.6507166292045456 -0.681488995344925 0.18053958458111286 0.15507600534974825 -0.08607087472073338 -0.016158808725919346 -0.007261634750928767 0.01803889724191924 0.009952982523509598 -0.012396988366648726 -0.001905407689852666 0.003932325279797902 5.8400428694052584e-05 -0.0007198270642148971 2.520793314082878e-05 7.607124405605129e-05 -2.4527163425833e-06 -3.7912531943321266e-06/];

$lo_d{18}=[qw/2.6126125564836423e-06 1.354915761832114e-06 -4.5246757874949856e-05 -1.4020992577726755e-05 0.00039616840638254753 7.021273459036268e-05 -0.002313871814506099 -0.00041152110923597756 0.009502164390962365 0.001642986397278216 -0.030325091089369604 -0.005077085160757053 0.08421992997038655 0.03399566710394736 -0.15993814866932407 -0.052029158983952786 0.47396905989393956 0.7536291401017928 0.40148386057061813 -0.032480573290138676 -0.07379920729060717 0.028529597039037808 0.006277944554311694 -0.03171268473181454 -0.0032607442000749834 0.015012356344250213 0.001087784789595693 -0.005239789683026608 -0.00018877623940755607 0.0014280863270832796 4.741614518373667e-05 -0.0002658301102424104 -9.858816030140058e-06 2.955743762093081e-05 7.847298055831765e-07 -1.5131530692371587e-06/];
$hi_d{18}=[qw/1.5131530692371587e-06 7.847298055831765e-07 -2.955743762093081e-05 -9.858816030140058e-06 0.0002658301102424104 4.741614518373667e-05 -0.0014280863270832796 -0.00018877623940755607 0.005239789683026608 0.001087784789595693 -0.015012356344250213 -0.0032607442000749834 0.03171268473181454 0.006277944554311694 -0.028529597039037808 -0.07379920729060717 0.032480573290138676 0.40148386057061813 -0.7536291401017928 0.47396905989393956 0.052029158983952786 -0.15993814866932407 -0.03399566710394736 0.08421992997038655 0.005077085160757053 -0.030325091089369604 -0.001642986397278216 0.009502164390962365 0.00041152110923597756 -0.002313871814506099 -7.021273459036268e-05 0.00039616840638254753 1.4020992577726755e-05 -4.5246757874949856e-05 -1.354915761832114e-06 2.6126125564836423e-06/];
$lo_r{18}=[qw/-1.5131530692371587e-06 7.847298055831765e-07 2.955743762093081e-05 -9.858816030140058e-06 -0.0002658301102424104 4.741614518373667e-05 0.0014280863270832796 -0.00018877623940755607 -0.005239789683026608 0.001087784789595693 0.015012356344250213 -0.0032607442000749834 -0.03171268473181454 0.006277944554311694 0.028529597039037808 -0.07379920729060717 -0.032480573290138676 0.40148386057061813 0.7536291401017928 0.47396905989393956 -0.052029158983952786 -0.15993814866932407 0.03399566710394736 0.08421992997038655 -0.005077085160757053 -0.030325091089369604 0.001642986397278216 0.009502164390962365 -0.00041152110923597756 -0.002313871814506099 7.021273459036268e-05 0.00039616840638254753 -1.4020992577726755e-05 -4.5246757874949856e-05 1.354915761832114e-06 2.6126125564836423e-06/];
$hi_r{18}=[qw/2.6126125564836423e-06 -1.354915761832114e-06 -4.5246757874949856e-05 1.4020992577726755e-05 0.00039616840638254753 -7.021273459036268e-05 -0.002313871814506099 0.00041152110923597756 0.009502164390962365 -0.001642986397278216 -0.030325091089369604 0.005077085160757053 0.08421992997038655 -0.03399566710394736 -0.15993814866932407 0.052029158983952786 0.47396905989393956 -0.7536291401017928 0.40148386057061813 0.032480573290138676 -0.07379920729060717 -0.028529597039037808 0.006277944554311694 0.03171268473181454 -0.0032607442000749834 -0.015012356344250213 0.001087784789595693 0.005239789683026608 -0.00018877623940755607 -0.0014280863270832796 4.741614518373667e-05 0.0002658301102424104 -9.858816030140058e-06 -2.955743762093081e-05 7.847298055831765e-07 1.5131530692371587e-06/];

$lo_d{19}=[qw/5.487732768215838e-07 -6.463651303345963e-07 -1.1880518269823984e-05 8.873312173729286e-06 0.0001155392333357879 -4.612039600210587e-05 -0.000635764515004334 0.00015915804768084938 0.0021214250281823303 -0.0011607032572062486 -0.005122205002583014 0.007968438320613306 0.01579743929567463 -0.02265199337824595 -0.046635983534938946 0.0070155738571741596 0.008954591173043624 -0.06752505804029409 0.10902582508127781 0.578144945338605 0.7195555257163943 0.2582661692372836 -0.17659686625203097 -0.11624173010739675 0.09363084341589714 0.08407267627924504 -0.016908234861345205 -0.02770989693131125 0.004319351874894969 0.008262236955528255 -0.0006179223277983108 -0.0017049602611649971 0.00012930767650701415 0.0002762187768573407 -1.6821387029373716e-05 -2.8151138661550245e-05 2.0623170632395688e-06 1.7509367995348687e-06/];
$hi_d{19}=[qw/-1.7509367995348687e-06 2.0623170632395688e-06 2.8151138661550245e-05 -1.6821387029373716e-05 -0.0002762187768573407 0.00012930767650701415 0.0017049602611649971 -0.0006179223277983108 -0.008262236955528255 0.004319351874894969 0.02770989693131125 -0.016908234861345205 -0.08407267627924504 0.09363084341589714 0.11624173010739675 -0.17659686625203097 -0.2582661692372836 0.7195555257163943 -0.578144945338605 0.10902582508127781 0.06752505804029409 0.008954591173043624 -0.0070155738571741596 -0.046635983534938946 0.02265199337824595 0.01579743929567463 -0.007968438320613306 -0.005122205002583014 0.0011607032572062486 0.0021214250281823303 -0.00015915804768084938 -0.000635764515004334 4.612039600210587e-05 0.0001155392333357879 -8.873312173729286e-06 -1.1880518269823984e-05 6.463651303345963e-07 5.487732768215838e-07/];
$lo_r{19}=[qw/1.7509367995348687e-06 2.0623170632395688e-06 -2.8151138661550245e-05 -1.6821387029373716e-05 0.0002762187768573407 0.00012930767650701415 -0.0017049602611649971 -0.0006179223277983108 0.008262236955528255 0.004319351874894969 -0.02770989693131125 -0.016908234861345205 0.08407267627924504 0.09363084341589714 -0.11624173010739675 -0.17659686625203097 0.2582661692372836 0.7195555257163943 0.578144945338605 0.10902582508127781 -0.06752505804029409 0.008954591173043624 0.0070155738571741596 -0.046635983534938946 -0.02265199337824595 0.01579743929567463 0.007968438320613306 -0.005122205002583014 -0.0011607032572062486 0.0021214250281823303 0.00015915804768084938 -0.000635764515004334 -4.612039600210587e-05 0.0001155392333357879 8.873312173729286e-06 -1.1880518269823984e-05 -6.463651303345963e-07 5.487732768215838e-07/];
$hi_r{19}=[qw/5.487732768215838e-07 6.463651303345963e-07 -1.1880518269823984e-05 -8.873312173729286e-06 0.0001155392333357879 4.612039600210587e-05 -0.000635764515004334 -0.00015915804768084938 0.0021214250281823303 0.0011607032572062486 -0.005122205002583014 -0.007968438320613306 0.01579743929567463 0.02265199337824595 -0.046635983534938946 -0.0070155738571741596 0.008954591173043624 0.06752505804029409 0.10902582508127781 -0.578144945338605 0.7195555257163943 -0.2582661692372836 -0.17659686625203097 0.11624173010739675 0.09363084341589714 -0.08407267627924504 -0.016908234861345205 0.02770989693131125 0.004319351874894969 -0.008262236955528255 -0.0006179223277983108 0.0017049602611649971 0.00012930767650701415 -0.0002762187768573407 -1.6821387029373716e-05 2.8151138661550245e-05 2.0623170632395688e-06 -1.7509367995348687e-06/];

$lo_d{20}=[qw/3.695537474835221e-07 -1.9015675890554106e-07 -7.919361411976999e-06 3.025666062736966e-06 7.992967835772481e-05 -1.928412300645204e-05 -0.0004947310915672655 7.215991188074035e-05 0.002088994708190198 -0.0003052628317957281 -0.006606585799088861 0.0014230873594621453 0.01700404902339034 -0.003313857383623359 -0.031629437144957966 0.008123228356009682 0.025579349509413946 -0.07899434492839816 -0.02981936888033373 0.4058314443484506 0.75116272842273 0.47199147510148703 -0.0510883429210674 -0.16057829841525254 0.03625095165393308 0.08891966802819956 -0.0068437019650692274 -0.035373336756604236 0.0019385970672402002 0.012157040948785737 -0.0006111263857992088 -0.0034716478028440734 0.0001254409172306726 0.0007476108597820572 -2.6615550335516086e-05 -0.00011739133516291466 4.525422209151636e-06 1.22872527779612e-05 -3.2567026420174407e-07 -6.329129044776395e-07/];
$hi_d{20}=[qw/6.329129044776395e-07 -3.2567026420174407e-07 -1.22872527779612e-05 4.525422209151636e-06 0.00011739133516291466 -2.6615550335516086e-05 -0.0007476108597820572 0.0001254409172306726 0.0034716478028440734 -0.0006111263857992088 -0.012157040948785737 0.0019385970672402002 0.035373336756604236 -0.0068437019650692274 -0.08891966802819956 0.03625095165393308 0.16057829841525254 -0.0510883429210674 -0.47199147510148703 0.75116272842273 -0.4058314443484506 -0.02981936888033373 0.07899434492839816 0.025579349509413946 -0.008123228356009682 -0.031629437144957966 0.003313857383623359 0.01700404902339034 -0.0014230873594621453 -0.006606585799088861 0.0003052628317957281 0.002088994708190198 -7.215991188074035e-05 -0.0004947310915672655 1.928412300645204e-05 7.992967835772481e-05 -3.025666062736966e-06 -7.919361411976999e-06 1.9015675890554106e-07 3.695537474835221e-07/];
$lo_r{20}=[qw/-6.329129044776395e-07 -3.2567026420174407e-07 1.22872527779612e-05 4.525422209151636e-06 -0.00011739133516291466 -2.6615550335516086e-05 0.0007476108597820572 0.0001254409172306726 -0.0034716478028440734 -0.0006111263857992088 0.012157040948785737 0.0019385970672402002 -0.035373336756604236 -0.0068437019650692274 0.08891966802819956 0.03625095165393308 -0.16057829841525254 -0.0510883429210674 0.47199147510148703 0.75116272842273 0.4058314443484506 -0.02981936888033373 -0.07899434492839816 0.025579349509413946 0.008123228356009682 -0.031629437144957966 -0.003313857383623359 0.01700404902339034 0.0014230873594621453 -0.006606585799088861 -0.0003052628317957281 0.002088994708190198 7.215991188074035e-05 -0.0004947310915672655 -1.928412300645204e-05 7.992967835772481e-05 3.025666062736966e-06 -7.919361411976999e-06 -1.9015675890554106e-07 3.695537474835221e-07/];
$hi_r{20}=[qw/3.695537474835221e-07 1.9015675890554106e-07 -7.919361411976999e-06 -3.025666062736966e-06 7.992967835772481e-05 1.928412300645204e-05 -0.0004947310915672655 -7.215991188074035e-05 0.002088994708190198 0.0003052628317957281 -0.006606585799088861 -0.0014230873594621453 0.01700404902339034 0.003313857383623359 -0.031629437144957966 -0.008123228356009682 0.025579349509413946 0.07899434492839816 -0.02981936888033373 -0.4058314443484506 0.75116272842273 -0.47199147510148703 -0.0510883429210674 0.16057829841525254 0.03625095165393308 -0.08891966802819956 -0.0068437019650692274 0.035373336756604236 0.0019385970672402002 -0.012157040948785737 -0.0006111263857992088 0.0034716478028440734 0.0001254409172306726 -0.0007476108597820572 -2.6615550335516086e-05 0.00011739133516291466 4.525422209151636e-06 -1.22872527779612e-05 -3.2567026420174407e-07 6.329129044776395e-07/];
;

sub new {
	my $class=shift;
	my $self={};
	my $var=shift;

	$self={lo_d=>$lo_d{$var},
		hi_d=>$hi_d{$var},
		lo_r=>$lo_r{$var},
		hi_r=>$hi_r{$var}
		};
	
	bless $self,$class;
	return $self;
};

sub vars {
	my $self=shift;
	if (wantarray()) {
		return (@vars);
	};
	return \@vars;
};

sub filters {
	my $self=shift;
	my $lo_d=$self->lo_d;
	my $hi_d=$self->hi_d;
	my $lo_r=$self->lo_r;
	my $hi_r=$self->hi_r;
	my @a=( $lo_d,$hi_d,$lo_r,$hi_r);
	if (wantarray()) {
		return (@a);
	};
	return \@a;
};

sub lo_d {
	my $self=shift;
	my $a=$self->{lo_d};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub hi_d {
	my $self=shift;
	my $a=$self->{hi_d};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub lo_r {
	my $self=shift;
	my $a=$self->{lo_r};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub hi_r {
	my $self=shift;
	my $a=$self->{hi_r};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	

		
1;
