package Gestinanna::Shell::Base;

use Cwd;  # qw(chdir cwd);

our $password;
 
our %EXPORT_COMMANDS = (
    bug => \&do_bug,
    set => \&do_set,
    quit => \&do_quit,
    '?' => \&do_help,
    'cd' => \&do_cd,
    'pwd' => \&do_pwd,
    '.' => \&do_readfile,
    );

sub init_commands {
    my($class, $cmds) = @_;

    @{$cmds}{keys %{"${class}::EXPORT_COMMANDS"}}
        = values %{"${class}::EXPORT_COMMANDS"};
}

sub page {
    my($shell, $string) = @_;

    my $PAGER = (-x $ENV{PAGER} ? $ENV{PAGER} : '') || "/usr/bin/less";
    open my $pager, "|-", $PAGER or do {
        print $string;
        return 1;
    };

    { local($SIG{PIPE}) = 'IGNORE';
        print $pager $string;
    };
    close $pager;
}

sub edit {
    my($shell, $string) = @_;

    open my $fh, ">", "/tmp/gst.edit.$$" or die "Unable to open temporary file for editing.\n";
    print $fh $string;
    close $fh;
    system($ENV{EDITOR}||'vi',"/tmp/gst.edit.$$");
    open $fh, "<", "/tmp/gst.edit.$$" or die "Unable to open temporary file to retrieve edited content.\n";
    local($/);
    my $filled_out_report = <$fh>;
    close $fh;
    return $filled_out_report;
}

sub do_help {
    my($shell, $prefix, $arg) = @_;

    if($arg !~ /^\s*$/) {
        return $shell -> interpret("$arg ?");
    }

    print "The following commands are available: ", join(", ", sort grep { $_ ne '?' } keys %Gestinanna::Shell::COMMANDS), "\n";
}

sub do_readfile {
    my($shell, $prefix, $arg) = @_;

    if($arg =~ /\?$/) {
        print <<1HERE1;
. <file> <args>

Reads and interprets the named file as if its contents were typed on 
the console.  Variables and not localized.  The arguments are available 
as \$(1), \$(2), ....  The name of the file is \$(0).
1HERE1
        return;
    }

    my @bits = split(/\s+/, $arg);

    local(@Gestinanna::Shell::VARIABLES{0..@bits-1}) = (@bits);

    $shell -> read_file($bits[0]);
}

sub do_cd {
    my($shell, $prefix, $arg) = @_;

    chdir($arg) or print "Unable to change to \"$a\".\n";
}

sub do_pwd {
    my($shell, $prefix, $arg) = @_;

    print cwd(), "\n";
}    

sub do_bug {
    my($shell, $prefix, $arg) = @_;
    my $DEV_LIST = "gestinanna-devel\@lists.sourceforge.net";
    my($to, $where) = split(/@/, $DEV_LIST, 2);
    my $date = scalar gmtime() . " GMT";
    my $executable = $0;
    my $config = '';

    if($arg =~ /\?$/) {
        print <<1HERE1;
bug

Sends a bug report to $DEV_LIST.
1HERE1
        return;
    }

    $config = "module\t\tversion\trevision\n\n";
    foreach my $m (sort qw-
        HTML::Mason
        Gestinanna
        Template
        AxKit
        mod_perl
        Uttu
    -) {
        eval "require $m";
        if($@) {
            $config .= "$m\t---\n";
        }
        else {
            $config .= "$m\t" . ${"${m}::VERSION"} . "\t" . ${"${m}::REVISION"} . "\n";
        }
    }

    $bug_report = <<1HERE1;
One line description:
  [ONE LINE DESCRIPTION HERE]

-------------8<---------- Start Bug Report ------------8<----------
1. Problem Description:

  [DESCRIBE THE PROBLEM HERE]

2. Used Components and their Configuration:

$config

  [ADDITIONAL CONFIGURATION INFORMATION HERE]

3. This is the core dump trace: (if you get a core dump):

  [CORE TRACE COMES HERE]

This report was generated by $executable on $date.

-------------8<---------- End Bug Report --------------8<----------

Note: Complete the rest of the details and post this bug report to
$to <at> $where. To subscribe to the list send 
an empty email to $to-subscribe\@$where.

1HERE1

    my $filled_out_report = $self -> edit($bug_report);
    #open my $fh, ">", "/tmp/gst.bug.$$" or return 1;
    #print $fh $bug_report;
    #close $fh;
    #system($ENV{EDITOR}||'vi',"/tmp/gst.bug.$$");
    #open $fh, "<", "/tmp/gst.bug.$$" or return 1;
    #local($/);
    #my $filled_out_report = <$fh>;
    #close $fh;

    if($filled_out_report eq $bug_report) {
        print "Nothing was changed.  Aborting.\n";
    }
    else {
        my $subject;
        if($filled_out_report =~ m{One line description:\n(.*?)\n}) {
            $subject = $1;
            $subject =~ s{^\s*\[}{};
            $subject =~ s{]\s*$}{};
            $subject = "Bug Report" if $subject eq 'ONE LINE DESCRIPTION HERE';
        }
        else {
            $subject = "Bug Report";
        }
        if($filled_out_report =~ m{-------------8<---------- Start Bug Report ------------8<----------(.*)-------------8<---------- End Bug Report --------------8<----------}s) {
            $filled_out_report = $1;
        }
        print "Filled out report:\n\nSubject: $subject\n\n$filled_out_report\n\n";
    }

    1;
}

sub do_set {
    my($self, $prior, $arg) = @_;
    @args = split(/\s/, $arg);
    if(@args) {
        my $v = shift @args;
        my $t = join(' ', @args);
        if($v eq 'password') {
            # need to prompt for it if we have a tty
            $Gestinanna::Shell::password = $t;
            $Gestinanna::Shell::VARIABLES{$v} = $t eq '' ? 'unset' : 'set';
        }
        else {
            $Gestinanna::Shell::VARIABLES{$v} = $t;
        }
    }
    else {
        foreach my $k (sort keys %Gestinanna::Shell::VARIABLES) {
            print "$k [$Gestinanna::Shell::VARIABLES{$k}]\n";
        }
    }
    1;
}

sub do_quit { exit 0; }

sub interpret {
    my($class, $self, $prior, $s) = @_;
    local($_);

    my($cmd, $arg) = split(/\s+/, $s, 2);
    $cmd = lc $cmd;

    my $cmds = \%{"${class}::COMMANDS"};

    unless(defined $cmds->{$cmd}) {
        print STDERR "Unknown command: $prior $cmd\n";
        return 1;
    }

    return $cmds->{$cmd} -> ($self, "$prior $cmd", $arg);
}

1;

__END__

=head1 NAME

Gestinanna::Shell::Base - base commands and support for command extensions

=head1 SYNOPSIS

 package Gestinanna::Shell::MyCommands;

 use base qw(Gestinanna::Shell::Base);

 %COMMANDS = (
    command => \&do_command,
 );

 %EXPORTED_COMMANDS = (
    mycommand => \do_mycommand,
 );

=head1 DESCRIPTION

=head1 AUTHOR
        
James G. Smith, <jsmith@cpan.org>

=head1 COPYRIGHT

Copyright (C) 2002 Texas A&M University.  All Rights Reserved.
        
This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.
