###############################################################################
#                                                                             #
#    Copyright © 2012-2013 -- IRB/INSERM                                      #
#                            (Institut de Recherche en Biothérapie /          #
#                             Institut National de la Santé et de la          #
#                             Recherche Médicale)                             #
#                                                                             #
#  Auteurs/Authors:  Jerôme AUDOUX <jerome.audoux@univ-montp2.fr>             #
#                    Nicolas PHILIPPE <nicolas.philippe@inserm.fr>            #
#                                                                             #
#  -------------------------------------------------------------------------  #
#                                                                             #
#  Ce fichier fait partie de la suite CracTools qui contient plusieurs pipeline# 
#  intégrés permettant de traiter les évênements biologiques présents dans du #
#  RNA-Seq. Les CracTools travaillent à partir d'un fichier SAM de CRAC et d'un# 
#  fichier d'annotation au format GFF3.                                       #
#                                                                             #
#  Ce logiciel est régi  par la licence CeCILL  soumise au droit français et  #
#  respectant les principes  de diffusion des logiciels libres.  Vous pouvez  #
#  utiliser, modifier et/ou redistribuer ce programme sous les conditions de  #
#  la licence CeCILL  telle que diffusée par le CEA,  le CNRS et l'INRIA sur  #
#  le site "http://www.cecill.info".                                          #
#                                                                             #
#  En contrepartie de l'accessibilité au code source et des droits de copie,  #
#  de modification et de redistribution accordés par cette licence, il n'est  #
#  offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,  #
#  seule une responsabilité  restreinte pèse  sur l'auteur du programme,  le  #
#  titulaire des droits patrimoniaux et les concédants successifs.            #
#                                                                             #
#  À  cet égard  l'attention de  l'utilisateur est  attirée sur  les risques  #
#  associés  au chargement,  à  l'utilisation,  à  la modification  et/ou au  #
#  développement  et à la reproduction du  logiciel par  l'utilisateur étant  #
#  donné  sa spécificité  de logiciel libre,  qui peut le rendre  complexe à  #
#  manipuler et qui le réserve donc à des développeurs et des professionnels  #
#  avertis  possédant  des  connaissances  informatiques  approfondies.  Les  #
#  utilisateurs  sont donc  invités  à  charger  et  tester  l'adéquation du  #
#  logiciel  à leurs besoins  dans des conditions  permettant  d'assurer  la  #
#  sécurité de leurs systêmes et ou de leurs données et,  plus généralement,  #
#  à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.         #
#                                                                             #
#  Le fait  que vous puissiez accéder  à cet en-tête signifie  que vous avez  #
#  pris connaissance  de la licence CeCILL,  et que vous en avez accepté les  #
#  termes.                                                                    #
#                                                                             #
#  -------------------------------------------------------------------------  #
#                                                                             #
#  This file is part of the CracTools which provide several integrated        #
#  pipeline to analyze biological events present in RNA-Seq data. CracTools   #
#  work on a SAM file generated by CRAC and an annotation file in GFF3 format.#
#                                                                             #
#  This software is governed by the CeCILL license under French law and       #
#  abiding by the rules of distribution of free software. You can use,        #
#  modify and/ or redistribute the software under the terms of the CeCILL     #
#  license as circulated by CEA, CNRS and INRIA at the following URL          #
#  "http://www.cecill.info".                                                  #
#                                                                             #
#  As a counterpart to the access to the source code and rights to copy,      #
#  modify and redistribute granted by the license, users are provided only    #
#  with a limited warranty and the software's author, the holder of the       #
#  economic rights, and the successive licensors have only limited            #
#  liability.                                                                 #
#                                                                             #
#  In this respect, the user's attention is drawn to the risks associated     #
#  with loading, using, modifying and/or developing or reproducing the        #
#  software by the user in light of its specific status of free software,     #
#  that may mean that it is complicated to manipulate, and that also          #
#  therefore means that it is reserved for developers and experienced         #
#  professionals having in-depth computer knowledge. Users are therefore      #
#  encouraged to load and test the software's suitability as regards their    #
#  requirements in conditions enabling the security of their systems and/or   #
#  data to be ensured and, more generally, to use and operate it in the same  #
#  conditions as regards security.                                            #
#                                                                             #
#  The fact that you are presently reading this means that you have had       #
#  knowledge of the CeCILL license and that you accept its terms.             #
#                                                                             #
###############################################################################


package CracTools::Output;
{
  $CracTools::Output::DIST = 'CracTools';
}
# ABSTRACT: A module to manage CracTools output files.
$CracTools::Output::VERSION = '1.08';
use strict;
use warnings;

use File::Basename;
use CracTools;
use CracTools::Const;

use constant DEFAULT_SEP => "\t";

use strict;
use warnings;


sub new {
  my $class = shift;

  my %args = @_;

  my $sep = DEFAULT_SEP unless $args{sep};
  my $output = $args{file};
  my $out_stream;

  if(defined $output) {
    open($out_stream,">$output") or die ("Enable to open $output file.\n");
  } else {
    $out_stream = \*STDOUT;
  }
  
  my $self = bless {
    sep => $sep, 
    out_stream => $out_stream,
  }, $class;
  
  return $self
}


sub printHeaders {
  my $self = shift;
  my %args = @_;

  my $version = $args{version};
  my $summary = $args{summary};
  my @arguments;
  if(defined $args{args}) {
    @arguments = @{$args{args}};
  }

  $self->printlnOutput("# Date: ".localtime);
  $self->printlnOutput("# Module: $CracTools::DIST (v $CracTools::VERSION)");
  if(defined $version) {
    $self->printlnOutput("# Script: ".basename($0)." (v $version)");
  } else {
    $self->printlnOutput("# Script: ".basename($0));
  }
  if(@arguments > 0) {
    $self->printOutput("# Args: ");
    foreach my $arg (@arguments) {
      $self->printOutput(" $arg");
    }
    $self->printlnOutput();
  }
  if(defined $summary) {
    $self->printlnOutput("# Summary:");
    my @lines = split /\n/, $summary;
    foreach my $line (@lines) {
      $self->printlnOutput("# $line");
    }
  }
  #$self->printlnOutput();
}


sub printHeaderLine {
  my $self = shift;
  my $first_field = shift;
  $first_field = '' unless defined $first_field;
  $self->printLine("# $first_field",@_);
}


sub printLine {
  my $self = shift;
  for(my $cpt = 0; $cpt < scalar @_; $cpt++) {
    if(!defined $_[$cpt]) {
      $_[$cpt] = $CracTools::Const::NOT_AVAILABLE;
    }
  }
  $self->printlnOutput(join($self->{sep},@_));
}


sub printOutput {
  my $self = shift;
  my $stuff = shift;
  my $stream = $self->{out_stream};
  print $stream $stuff;
}


sub printlnOutput {
  my $self = shift;
  my $stuff = shift;
  if(defined $stuff) {
    $self->printOutput("$stuff\n");
  } else {
    $self->printOutput("\n");
  }
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

CracTools::Output - A module to manage CracTools output files.

=head1 VERSION

version 1.08

=head1 SYNOPSIS

  # Creating a default output object.
  # Everything will be print to the standard output
  my $output = CracTools::Output->new();
  
  # Print nice headers
  my $output->printHeaders(version => '1.01',summary => 'blabla', args => @ARGV);
  
  # This will print "foo\tbar\n"
  $output->printLine('foo','bar');
  
  # Using semicolon as separator charcater
  my $output = CracTools::Output->new(sep => ';');
  
  # Print into a file
  my $output = CracTools::Output->new(file => 'foo.bar');

=head1 DESCRIPTION

CracTools::Output is a simple tool to generate Char-Separated files.

=head1 METHODS

=head2 new

  Arg [sep]  : (Optional) Character to use as separator for columns
  Arg [file] : (Optional) String - Ouput file, if not specified
               CracTools::Output prints to STDOUT.

  Example     : $output = CracTools::Output->new(file => 'output.txt', sep => '\t');
  Description : Create a new CracTools::Output object
  ReturnType  : CracTools::Output

=head2 printHeaders

  Arg [version]  : (Optional) Version number of the script that is calling "printHeaders" method.
  Arg [summary]  : (Optional) String - Summary text to print in headers (can have multiple lines

  Example     : $output->printHeaders(version => $version, summary => "Found $n reads");
  Description : Print headers to the output stream with CracTools-core version, date, name of calling script.

=head2 printHeaderLine

  Arg [1] : Array of strings

  Example     : $output->printHeaderLine("Read Id","Read_seq","Nb_occ");
  Description : Print header line to the file (with a "# " append to the start of the line)

=head2 printLine

  Arg [1] : Array of strings

  Example     : $output->printLine("Read Id","Read_seq","Nb_occ");
  Description : Print a line to the file, each string of the array parameter is print
                with the separator defined for the output.

=head2 printOutput

  Arg [1] : String - Value to print

  Example     : $output->printLine("This is a line");
  Description : Print the string in parameter to the output stream.

=head2 printlnOutput

  Arg [1] : String - Value to print

  Example     : $output->printLine("This is a line");
  Description : Print the string in parameter to the output stream with an extra "\n" at the end of the string.

=head1 AUTHORS

=over 4

=item *

Nicolas PHILIPPE <nicolas.philippe@inserm.fr>

=item *

Jérôme AUDOUX <jaudoux@cpan.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2014 by IRB/INSERM (Institut de Recherche en Biothérapie / Institut National de la Santé et de la Recherche Médicale).

This is free software, licensed under:

  CeCILL FREE SOFTWARE LICENSE AGREEMENT, Version 2.1 dated 2013-06-21

=cut
