use 5.010;
use strict;
use warnings;
use open qw< :encoding(utf8) :std >;
use Encode qw<decode_utf8>;
use Data::Section -setup;
use Test::More tests => 2695;
use Hailo::Tokenizer::Words;

my $self = bless {} => __PACKAGE__;

my $text = decode_utf8(${ $self->section_data("UTF-8 encoded sample plain-text file") });
my $perl = eval decode_utf8(${ $self->section_data("Data::Dumper result") });

my $toke = Hailo::Tokenizer::Words->new();
my $parsed = $toke->make_tokens($text);

# How the "Data::Dumper result" was produced:
#use Data::Dumper;
#$Data::Dumper::Terse = 1;
#print STDERR Dumper($parsed);
#exit;

for (my $i = 0; $i < @$perl; $i++) {
    is($parsed->[$i][0], $perl->[$i][0], "Token #$i: type matches");
    is($parsed->[$i][1], $perl->[$i][1], "Token #$i: content matches");
}

is(scalar(@$parsed), scalar(@$perl), "Number of tokens matches");

__DATA__
__[ UTF-8 encoded sample plain-text file ]__
UTF-8 encoded sample plain-text file
‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾

Markus Kuhn [ˈmaʳkʊs kuːn] <http://www.cl.cam.ac.uk/~mgk25/> — 2002-07-25


The ASCII compatible UTF-8 encoding used in this plain-text file
is defined in Unicode, ISO 10646-1, and RFC 2279.


Using Unicode/UTF-8, you can write in emails and source code things such as

Mathematics and sciences:

  ∮ E⋅da = Q,  n → ∞, ∑ f(i) = ∏ g(i),      ⎧⎡⎛┌─────┐⎞⎤⎫
                                            ⎪⎢⎜│a²+b³ ⎟⎥⎪
  ∀x∈ℝ: ⌈x⌉ = −⌊−x⌋, α ∧ ¬β = ¬(¬α ∨ β),    ⎪⎢⎜│───── ⎟⎥⎪
                                            ⎪⎢⎜⎷ c₈   ⎟⎥⎪
  ℕ ⊆ ℕ₀ ⊂ ℤ ⊂ ℚ ⊂ ℝ ⊂ ℂ,                   ⎨⎢⎜       ⎟⎥⎬
                                            ⎪⎢⎜ ∞     ⎟⎥⎪
  ⊥ < a ≠ b ≡ c ≤ d ≪ ⊤ ⇒ (⟦A⟧ ⇔ ⟪B⟫),      ⎪⎢⎜ ⎲     ⎟⎥⎪
                                            ⎪⎢⎜ ⎳aⁱ-bⁱ⎟⎥⎪
  2H₂ + O₂ ⇌ 2H₂O, R = 4.7 kΩ, ⌀ 200 mm     ⎩⎣⎝i=1    ⎠⎦⎭

Linguistics and dictionaries:

  ði ıntəˈnæʃənəl fəˈnɛtık əsoʊsiˈeıʃn
  Y [ˈʏpsilɔn], Yen [jɛn], Yoga [ˈjoːgɑ]

APL:

  ((V⍳V)=⍳⍴V)/V←,V    ⌷←⍳→⍴∆∇⊃‾⍎⍕⌈

Nicer typography in plain text files:

  ╔══════════════════════════════════════════╗
  ║                                          ║
  ║   • ‘single’ and “double” quotes         ║
  ║                                          ║
  ║   • Curly apostrophes: “We’ve been here” ║
  ║                                          ║
  ║   • Latin-1 apostrophe and accents: '´`  ║
  ║                                          ║
  ║   • ‚deutsche‘ „Anführungszeichen“       ║
  ║                                          ║
  ║   • †, ‡, ‰, •, 3–4, —, −5/+5, ™, …      ║
  ║                                          ║
  ║   • ASCII safety test: 1lI|, 0OD, 8B     ║
  ║                      ╭─────────╮         ║
  ║   • the euro symbol: │ 14.95 € │         ║
  ║                      ╰─────────╯         ║
  ╚══════════════════════════════════════════╝

Combining characters:

  STARGΛ̊TE SG-1, a = v̇ = r̈, a⃑ ⊥ b⃑

Greek (in Polytonic):

  The Greek anthem:

  Σὲ γνωρίζω ἀπὸ τὴν κόψη
  τοῦ σπαθιοῦ τὴν τρομερή,
  σὲ γνωρίζω ἀπὸ τὴν ὄψη
  ποὺ μὲ βία μετράει τὴ γῆ.

  ᾿Απ᾿ τὰ κόκκαλα βγαλμένη
  τῶν ῾Ελλήνων τὰ ἱερά
  καὶ σὰν πρῶτα ἀνδρειωμένη
  χαῖρε, ὦ χαῖρε, ᾿Ελευθεριά!

  From a speech of Demosthenes in the 4th century BC:

  Οὐχὶ ταὐτὰ παρίσταταί μοι γιγνώσκειν, ὦ ἄνδρες ᾿Αθηναῖοι,
  ὅταν τ᾿ εἰς τὰ πράγματα ἀποβλέψω καὶ ὅταν πρὸς τοὺς
  λόγους οὓς ἀκούω· τοὺς μὲν γὰρ λόγους περὶ τοῦ
  τιμωρήσασθαι Φίλιππον ὁρῶ γιγνομένους, τὰ δὲ πράγματ᾿
  εἰς τοῦτο προήκοντα,  ὥσθ᾿ ὅπως μὴ πεισόμεθ᾿ αὐτοὶ
  πρότερον κακῶς σκέψασθαι δέον. οὐδέν οὖν ἄλλο μοι δοκοῦσιν
  οἱ τὰ τοιαῦτα λέγοντες ἢ τὴν ὑπόθεσιν, περὶ ἧς βουλεύεσθαι,
  οὐχὶ τὴν οὖσαν παριστάντες ὑμῖν ἁμαρτάνειν. ἐγὼ δέ, ὅτι μέν
  ποτ᾿ ἐξῆν τῇ πόλει καὶ τὰ αὑτῆς ἔχειν ἀσφαλῶς καὶ Φίλιππον
  τιμωρήσασθαι, καὶ μάλ᾿ ἀκριβῶς οἶδα· ἐπ᾿ ἐμοῦ γάρ, οὐ πάλαι
  γέγονεν ταῦτ᾿ ἀμφότερα· νῦν μέντοι πέπεισμαι τοῦθ᾿ ἱκανὸν
  προλαβεῖν ἡμῖν εἶναι τὴν πρώτην, ὅπως τοὺς συμμάχους
  σώσομεν. ἐὰν γὰρ τοῦτο βεβαίως ὑπάρξῃ, τότε καὶ περὶ τοῦ
  τίνα τιμωρήσεταί τις καὶ ὃν τρόπον ἐξέσται σκοπεῖν· πρὶν δὲ
  τὴν ἀρχὴν ὀρθῶς ὑποθέσθαι, μάταιον ἡγοῦμαι περὶ τῆς
  τελευτῆς ὁντινοῦν ποιεῖσθαι λόγον.

  Δημοσθένους, Γ´ ᾿Ολυνθιακὸς

Georgian:

  From a Unicode conference invitation:

  გთხოვთ ახლავე გაიაროთ რეგისტრაცია Unicode-ის მეათე საერთაშორისო
  კონფერენციაზე დასასწრებად, რომელიც გაიმართება 10-12 მარტს,
  ქ. მაინცში, გერმანიაში. კონფერენცია შეჰკრებს ერთად მსოფლიოს
  ექსპერტებს ისეთ დარგებში როგორიცაა ინტერნეტი და Unicode-ი,
  ინტერნაციონალიზაცია და ლოკალიზაცია, Unicode-ის გამოყენება
  ოპერაციულ სისტემებსა, და გამოყენებით პროგრამებში, შრიფტებში,
  ტექსტების დამუშავებასა და მრავალენოვან კომპიუტერულ სისტემებში.

Russian:

  From a Unicode conference invitation:

  Зарегистрируйтесь сейчас на Десятую Международную Конференцию по
  Unicode, которая состоится 10-12 марта 1997 года в Майнце в Германии.
  Конференция соберет широкий круг экспертов по  вопросам глобального
  Интернета и Unicode, локализации и интернационализации, воплощению и
  применению Unicode в различных операционных системах и программных
  приложениях, шрифтах, верстке и многоязычных компьютерных системах.

Thai (UCS Level 2):

  Excerpt from a poetry on The Romance of The Three Kingdoms (a Chinese
  classic 'San Gua'):

  [----------------------------|------------------------]
    ๏ แผ่นดินฮั่นเสื่อมโทรมแสนสังเวช  พระปกเกศกองบู๊กู้ขึ้นใหม่
  สิบสองกษัตริย์ก่อนหน้าแลถัดไป       สององค์ไซร้โง่เขลาเบาปัญญา
    ทรงนับถือขันทีเป็นที่พึ่ง           บ้านเมืองจึงวิปริตเป็นนักหนา
  โฮจิ๋นเรียกทัพทั่วหัวเมืองมา         หมายจะฆ่ามดชั่วตัวสำคัญ
    เหมือนขับไสไล่เสือจากเคหา      รับหมาป่าเข้ามาเลยอาสัญ
  ฝ่ายอ้องอุ้นยุแยกให้แตกกัน          ใช้สาวนั้นเป็นชนวนชื่นชวนใจ
    พลันลิฉุยกุยกีกลับก่อเหตุ          ช่างอาเพศจริงหนาฟ้าร้องไห้
  ต้องรบราฆ่าฟันจนบรรลัย           ฤๅหาใครค้ำชูกู้บรรลังก์ ฯ

  (The above is a two-column text. If combining characters are handled
  correctly, the lines of the second column should be aligned with the
  | character above.)

Ethiopian:

  Proverbs in the Amharic language:

  ሰማይ አይታረስ ንጉሥ አይከሰስ።
  ብላ ካለኝ እንደአባቴ በቆመጠኝ።
  ጌጥ ያለቤቱ ቁምጥና ነው።
  ደሀ በሕልሙ ቅቤ ባይጠጣ ንጣት በገደለው።
  የአፍ ወለምታ በቅቤ አይታሽም።
  አይጥ በበላ ዳዋ ተመታ።
  ሲተረጉሙ ይደረግሙ።
  ቀስ በቀስ፥ ዕንቁላል በእግሩ ይሄዳል።
  ድር ቢያብር አንበሳ ያስር።
  ሰው እንደቤቱ እንጅ እንደ ጉረቤቱ አይተዳደርም።
  እግዜር የከፈተውን ጉሮሮ ሳይዘጋው አይድርም።
  የጎረቤት ሌባ፥ ቢያዩት ይስቅ ባያዩት ያጠልቅ።
  ሥራ ከመፍታት ልጄን ላፋታት።
  ዓባይ ማደሪያ የለው፥ ግንድ ይዞ ይዞራል።
  የእስላም አገሩ መካ የአሞራ አገሩ ዋርካ።
  ተንጋሎ ቢተፉ ተመልሶ ባፉ።
  ወዳጅህ ማር ቢሆን ጨርስህ አትላሰው።
  እግርህን በፍራሽህ ልክ ዘርጋ።

Runes:

  ᚻᛖ ᚳᚹᚫᚦ ᚦᚫᛏ ᚻᛖ ᛒᚢᛞᛖ ᚩᚾ ᚦᚫᛗ ᛚᚪᚾᛞᛖ ᚾᚩᚱᚦᚹᛖᚪᚱᛞᚢᛗ ᚹᛁᚦ ᚦᚪ ᚹᛖᛥᚫ

  (Old English, which transcribed into Latin reads 'He cwaeth that he
  bude thaem lande northweardum with tha Westsae.' and means 'He said
  that he lived in the northern land near the Western Sea.')

Braille:

  ⡌⠁⠧⠑ ⠼⠁⠒  ⡍⠜⠇⠑⠹⠰⠎ ⡣⠕⠌

  ⡍⠜⠇⠑⠹ ⠺⠁⠎ ⠙⠑⠁⠙⠒ ⠞⠕ ⠃⠑⠛⠔ ⠺⠊⠹⠲ ⡹⠻⠑ ⠊⠎ ⠝⠕ ⠙⠳⠃⠞
  ⠱⠁⠞⠑⠧⠻ ⠁⠃⠳⠞ ⠹⠁⠞⠲ ⡹⠑ ⠗⠑⠛⠊⠌⠻ ⠕⠋ ⠙⠊⠎ ⠃⠥⠗⠊⠁⠇ ⠺⠁⠎
  ⠎⠊⠛⠝⠫ ⠃⠹ ⠹⠑ ⠊⠇⠻⠛⠹⠍⠁⠝⠂ ⠹⠑ ⠊⠇⠻⠅⠂ ⠹⠑ ⠥⠝⠙⠻⠞⠁⠅⠻⠂
  ⠁⠝⠙ ⠹⠑ ⠡⠊⠑⠋ ⠍⠳⠗⠝⠻⠲ ⡎⠊⠗⠕⠕⠛⠑ ⠎⠊⠛⠝⠫ ⠊⠞⠲ ⡁⠝⠙
  ⡎⠊⠗⠕⠕⠛⠑⠰⠎ ⠝⠁⠍⠑ ⠺⠁⠎ ⠛⠕⠕⠙ ⠥⠏⠕⠝ ⠰⡡⠁⠝⠛⠑⠂ ⠋⠕⠗ ⠁⠝⠹⠹⠔⠛ ⠙⠑
  ⠡⠕⠎⠑ ⠞⠕ ⠏⠥⠞ ⠙⠊⠎ ⠙⠁⠝⠙ ⠞⠕⠲

  ⡕⠇⠙ ⡍⠜⠇⠑⠹ ⠺⠁⠎ ⠁⠎ ⠙⠑⠁⠙ ⠁⠎ ⠁ ⠙⠕⠕⠗⠤⠝⠁⠊⠇⠲

  ⡍⠔⠙⠖ ⡊ ⠙⠕⠝⠰⠞ ⠍⠑⠁⠝ ⠞⠕ ⠎⠁⠹ ⠹⠁⠞ ⡊ ⠅⠝⠪⠂ ⠕⠋ ⠍⠹
  ⠪⠝ ⠅⠝⠪⠇⠫⠛⠑⠂ ⠱⠁⠞ ⠹⠻⠑ ⠊⠎ ⠏⠜⠞⠊⠊⠥⠇⠜⠇⠹ ⠙⠑⠁⠙ ⠁⠃⠳⠞
  ⠁ ⠙⠕⠕⠗⠤⠝⠁⠊⠇⠲ ⡊ ⠍⠊⠣⠞ ⠙⠁⠧⠑ ⠃⠑⠲ ⠔⠊⠇⠔⠫⠂ ⠍⠹⠎⠑⠇⠋⠂ ⠞⠕
  ⠗⠑⠛⠜⠙ ⠁ ⠊⠕⠋⠋⠔⠤⠝⠁⠊⠇ ⠁⠎ ⠹⠑ ⠙⠑⠁⠙⠑⠌ ⠏⠊⠑⠊⠑ ⠕⠋ ⠊⠗⠕⠝⠍⠕⠝⠛⠻⠹
  ⠔ ⠹⠑ ⠞⠗⠁⠙⠑⠲ ⡃⠥⠞ ⠹⠑ ⠺⠊⠎⠙⠕⠍ ⠕⠋ ⠳⠗ ⠁⠝⠊⠑⠌⠕⠗⠎
  ⠊⠎ ⠔ ⠹⠑ ⠎⠊⠍⠊⠇⠑⠆ ⠁⠝⠙ ⠍⠹ ⠥⠝⠙⠁⠇⠇⠪⠫ ⠙⠁⠝⠙⠎
  ⠩⠁⠇⠇ ⠝⠕⠞ ⠙⠊⠌⠥⠗⠃ ⠊⠞⠂ ⠕⠗ ⠹⠑ ⡊⠳⠝⠞⠗⠹⠰⠎ ⠙⠕⠝⠑ ⠋⠕⠗⠲ ⡹⠳
  ⠺⠊⠇⠇ ⠹⠻⠑⠋⠕⠗⠑ ⠏⠻⠍⠊⠞ ⠍⠑ ⠞⠕ ⠗⠑⠏⠑⠁⠞⠂ ⠑⠍⠏⠙⠁⠞⠊⠊⠁⠇⠇⠹⠂ ⠹⠁⠞
  ⡍⠜⠇⠑⠹ ⠺⠁⠎ ⠁⠎ ⠙⠑⠁⠙ ⠁⠎ ⠁ ⠙⠕⠕⠗⠤⠝⠁⠊⠇⠲

  (The first couple of paragraphs of "A Christmas Carol" by Dickens)

Compact font selection example text:

  ABCDEFGHIJKLMNOPQRSTUVWXYZ /0123456789
  abcdefghijklmnopqrstuvwxyz £©µÀÆÖÞßéöÿ
  –—‘“”„†•…‰™œŠŸž€ ΑΒΓΔΩαβγδω АБВГДабвгд
  ∀∂∈ℝ∧∪≡∞ ↑↗↨↻⇣ ┐┼╔╘░►☺♀ ﬁ�⑀₂ἠḂӥẄɐː⍎אԱა

Greetings in various languages:

  Hello world, Καλημέρα κόσμε, コンニチハ

Box drawing alignment tests:                                          █
                                                                      ▉
  ╔══╦══╗  ┌──┬──┐  ╭──┬──╮  ╭──┬──╮  ┏━━┳━━┓  ┎┒┏┑   ╷  ╻ ┏┯┓ ┌┰┐    ▊ ╱╲╱╲╳╳╳
  ║┌─╨─┐║  │╔═╧═╗│  │╒═╪═╕│  │╓─╁─╖│  ┃┌─╂─┐┃  ┗╃╄┙  ╶┼╴╺╋╸┠┼┨ ┝╋┥    ▋ ╲╱╲╱╳╳╳
  ║│╲ ╱│║  │║   ║│  ││ │ ││  │║ ┃ ║│  ┃│ ╿ │┃  ┍╅╆┓   ╵  ╹ ┗┷┛ └┸┘    ▌ ╱╲╱╲╳╳╳
  ╠╡ ╳ ╞╣  ├╢   ╟┤  ├┼─┼─┼┤  ├╫─╂─╫┤  ┣┿╾┼╼┿┫  ┕┛┖┚     ┌┄┄┐ ╎ ┏┅┅┓ ┋ ▍ ╲╱╲╱╳╳╳
  ║│╱ ╲│║  │║   ║│  ││ │ ││  │║ ┃ ║│  ┃│ ╽ │┃  ░░▒▒▓▓██ ┊  ┆ ╎ ╏  ┇ ┋ ▎
  ║└─╥─┘║  │╚═╤═╝│  │╘═╪═╛│  │╙─╀─╜│  ┃└─╂─┘┃  ░░▒▒▓▓██ ┊  ┆ ╎ ╏  ┇ ┋ ▏
  ╚══╩══╝  └──┴──┘  ╰──┴──╯  ╰──┴──╯  ┗━━┻━━┛  ▗▄▖▛▀▜   └╌╌┘ ╎ ┗╍╍┛ ┋  ▁▂▃▄▅▆▇█
                                               ▝▀▘▙▄▟
__[ Data::Dumper result ]__
[
  [
    0,
    'UTF-8'
  ],
  [
    0,
    'encoded'
  ],
  [
    0,
    'sample'
  ],
  [
    0,
    'plain-text'
  ],
  [
    0,
    'file'
  ],
  [
    0,
    "\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}\x{203e}"
  ],
  [
    0,
    'markus'
  ],
  [
    0,
    'kuhn'
  ],
  [
    1,
    '['
  ],
  [
    0,
    "\x{2c8}ma\x{2b3}k\x{28a}s"
  ],
  [
    0,
    "ku\x{2d0}n"
  ],
  [
    2,
    ']'
  ],
  [
    1,
    '<'
  ],
  [
    0,
    'http://www.cl.cam.ac.uk/~mgk25/'
  ],
  [
    2,
    '>'
  ],
  [
    0,
    "\x{2014}"
  ],
  [
    0,
    '2002-07-25'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'ASCII'
  ],
  [
    0,
    'compatible'
  ],
  [
    0,
    'UTF-8'
  ],
  [
    0,
    'encoding'
  ],
  [
    0,
    'used'
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'this'
  ],
  [
    0,
    'plain-text'
  ],
  [
    0,
    'file'
  ],
  [
    0,
    'is'
  ],
  [
    0,
    'defined'
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'unicode'
  ],
  [
    2,
    ','
  ],
  [
    0,
    'ISO'
  ],
  [
    0,
    '10646-1'
  ],
  [
    2,
    ','
  ],
  [
    0,
    'and'
  ],
  [
    0,
    'RFC'
  ],
  [
    0,
    '2279'
  ],
  [
    2,
    '.'
  ],
  [
    0,
    'using'
  ],
  [
    0,
    'unicode'
  ],
  [
    3,
    '/'
  ],
  [
    0,
    'UTF-8'
  ],
  [
    2,
    ','
  ],
  [
    0,
    'you'
  ],
  [
    0,
    'can'
  ],
  [
    0,
    'write'
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'emails'
  ],
  [
    0,
    'and'
  ],
  [
    0,
    'source'
  ],
  [
    0,
    'code'
  ],
  [
    0,
    'things'
  ],
  [
    0,
    'such'
  ],
  [
    0,
    'as'
  ],
  [
    0,
    'mathematics'
  ],
  [
    0,
    'and'
  ],
  [
    0,
    'sciences'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{222e}"
  ],
  [
    0,
    'E'
  ],
  [
    3,
    "\x{22c5}"
  ],
  [
    0,
    'da'
  ],
  [
    0,
    '='
  ],
  [
    0,
    'Q'
  ],
  [
    2,
    ','
  ],
  [
    0,
    'n'
  ],
  [
    0,
    "\x{2192}"
  ],
  [
    0,
    "\x{221e},"
  ],
  [
    0,
    "\x{2211}"
  ],
  [
    0,
    'f'
  ],
  [
    3,
    '('
  ],
  [
    0,
    'i'
  ],
  [
    2,
    ')'
  ],
  [
    0,
    '='
  ],
  [
    0,
    "\x{220f}"
  ],
  [
    0,
    'g'
  ],
  [
    3,
    '('
  ],
  [
    0,
    'i'
  ],
  [
    2,
    '),'
  ],
  [
    0,
    "\x{23a7}\x{23a1}\x{239b}\x{250c}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2510}\x{239e}\x{23a4}\x{23ab}"
  ],
  [
    1,
    "\x{23aa}\x{23a2}\x{239c}\x{2502}"
  ],
  [
    0,
    'a'
  ],
  [
    3,
    "\x{b2}+"
  ],
  [
    0,
    'b'
  ],
  [
    2,
    "\x{b3}"
  ],
  [
    0,
    "\x{239f}\x{23a5}\x{23aa}"
  ],
  [
    1,
    "\x{2200}"
  ],
  [
    0,
    'x'
  ],
  [
    3,
    "\x{2208}"
  ],
  [
    0,
    "\x{211d}"
  ],
  [
    2,
    ':'
  ],
  [
    1,
    "\x{2308}"
  ],
  [
    0,
    'x'
  ],
  [
    2,
    "\x{2309}"
  ],
  [
    0,
    '='
  ],
  [
    1,
    "\x{2212}\x{230a}\x{2212}"
  ],
  [
    0,
    'x'
  ],
  [
    2,
    "\x{230b},"
  ],
  [
    0,
    "\x{3b1}"
  ],
  [
    0,
    "\x{2227}"
  ],
  [
    1,
    "\x{ac}"
  ],
  [
    0,
    "\x{3b2}"
  ],
  [
    0,
    '='
  ],
  [
    1,
    "\x{ac}(\x{ac}"
  ],
  [
    0,
    "\x{3b1}"
  ],
  [
    0,
    "\x{2228}"
  ],
  [
    0,
    "\x{3b2}"
  ],
  [
    2,
    '),'
  ],
  [
    0,
    "\x{23aa}\x{23a2}\x{239c}\x{2502}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}"
  ],
  [
    0,
    "\x{239f}\x{23a5}\x{23aa}"
  ],
  [
    0,
    "\x{23aa}\x{23a2}\x{239c}\x{23b7}"
  ],
  [
    0,
    'c'
  ],
  [
    2,
    "\x{2088}"
  ],
  [
    0,
    "\x{239f}\x{23a5}\x{23aa}"
  ],
  [
    0,
    "\x{2115}"
  ],
  [
    0,
    "\x{2286}"
  ],
  [
    0,
    "\x{2115}"
  ],
  [
    2,
    "\x{2080}"
  ],
  [
    0,
    "\x{2282}"
  ],
  [
    0,
    "\x{2124}"
  ],
  [
    0,
    "\x{2282}"
  ],
  [
    0,
    "\x{211a}"
  ],
  [
    0,
    "\x{2282}"
  ],
  [
    0,
    "\x{211d}"
  ],
  [
    0,
    "\x{2282}"
  ],
  [
    0,
    "\x{2102}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{23a8}\x{23a2}\x{239c}"
  ],
  [
    0,
    "\x{239f}\x{23a5}\x{23ac}"
  ],
  [
    0,
    "\x{23aa}\x{23a2}\x{239c}"
  ],
  [
    0,
    "\x{221e}"
  ],
  [
    0,
    "\x{239f}\x{23a5}\x{23aa}"
  ],
  [
    0,
    "\x{22a5}"
  ],
  [
    0,
    '<'
  ],
  [
    0,
    'a'
  ],
  [
    0,
    "\x{2260}"
  ],
  [
    0,
    'b'
  ],
  [
    0,
    "\x{2261}"
  ],
  [
    0,
    'c'
  ],
  [
    0,
    "\x{2264}"
  ],
  [
    0,
    'd'
  ],
  [
    0,
    "\x{226a}"
  ],
  [
    0,
    "\x{22a4}"
  ],
  [
    0,
    "\x{21d2}"
  ],
  [
    1,
    "(\x{27e6}"
  ],
  [
    0,
    'A'
  ],
  [
    2,
    "\x{27e7}"
  ],
  [
    0,
    "\x{21d4}"
  ],
  [
    1,
    "\x{27ea}"
  ],
  [
    0,
    'B'
  ],
  [
    2,
    "\x{27eb}),"
  ],
  [
    0,
    "\x{23aa}\x{23a2}\x{239c}"
  ],
  [
    0,
    "\x{23b2}"
  ],
  [
    0,
    "\x{239f}\x{23a5}\x{23aa}"
  ],
  [
    0,
    "\x{23aa}\x{23a2}\x{239c}"
  ],
  [
    1,
    "\x{23b3}"
  ],
  [
    0,
    "a\x{2071}-b\x{2071}"
  ],
  [
    2,
    "\x{239f}\x{23a5}\x{23aa}"
  ],
  [
    0,
    '2'
  ],
  [
    2,
    'H'
  ],
  [
    2,
    "\x{2082}"
  ],
  [
    0,
    '+'
  ],
  [
    0,
    'O'
  ],
  [
    2,
    "\x{2082}"
  ],
  [
    0,
    "\x{21cc}"
  ],
  [
    0,
    '2'
  ],
  [
    2,
    'H'
  ],
  [
    3,
    "\x{2082}"
  ],
  [
    0,
    'O'
  ],
  [
    2,
    ','
  ],
  [
    0,
    'R'
  ],
  [
    0,
    '='
  ],
  [
    0,
    '4.7'
  ],
  [
    0,
    "k\x{3a9}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{2300}"
  ],
  [
    0,
    '200'
  ],
  [
    0,
    'mm'
  ],
  [
    1,
    "\x{23a9}\x{23a3}\x{239d}"
  ],
  [
    0,
    'i'
  ],
  [
    3,
    '='
  ],
  [
    0,
    '1'
  ],
  [
    0,
    "\x{23a0}\x{23a6}\x{23ad}"
  ],
  [
    0,
    'linguistics'
  ],
  [
    0,
    'and'
  ],
  [
    0,
    'dictionaries'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{f0}i"
  ],
  [
    0,
    "\x{131}nt\x{259}\x{2c8}n\x{e6}\x{283}\x{259}n\x{259}l"
  ],
  [
    0,
    "f\x{259}\x{2c8}n\x{25b}t\x{131}k"
  ],
  [
    0,
    "\x{259}so\x{28a}si\x{2c8}e\x{131}\x{283}n"
  ],
  [
    0,
    'Y'
  ],
  [
    1,
    '['
  ],
  [
    0,
    "\x{2c8}\x{28f}psil\x{254}n"
  ],
  [
    2,
    '],'
  ],
  [
    0,
    'yen'
  ],
  [
    1,
    '['
  ],
  [
    0,
    "j\x{25b}n"
  ],
  [
    2,
    '],'
  ],
  [
    0,
    'yoga'
  ],
  [
    1,
    '['
  ],
  [
    0,
    "\x{2c8}jo\x{2d0}g\x{251}"
  ],
  [
    2,
    ']'
  ],
  [
    0,
    'APL'
  ],
  [
    2,
    ':'
  ],
  [
    1,
    '(('
  ],
  [
    0,
    'V'
  ],
  [
    3,
    "\x{2373}"
  ],
  [
    0,
    'V'
  ],
  [
    3,
    ")=\x{2373}\x{2374}"
  ],
  [
    0,
    'V'
  ],
  [
    3,
    ')/'
  ],
  [
    0,
    'V'
  ],
  [
    3,
    "\x{2190},"
  ],
  [
    0,
    'V'
  ],
  [
    0,
    "\x{2337}\x{2190}\x{2373}\x{2192}\x{2374}\x{2206}\x{2207}\x{2283}\x{203e}\x{234e}\x{2355}\x{2308}"
  ],
  [
    0,
    'nicer'
  ],
  [
    0,
    'typography'
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'plain'
  ],
  [
    0,
    'text'
  ],
  [
    0,
    'files'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{2554}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2557}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2022}"
  ],
  [
    1,
    "\x{2018}"
  ],
  [
    0,
    'single'
  ],
  [
    2,
    "\x{2019}"
  ],
  [
    0,
    'and'
  ],
  [
    1,
    "\x{201c}"
  ],
  [
    0,
    'double'
  ],
  [
    2,
    "\x{201d}"
  ],
  [
    0,
    'quotes'
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2022}"
  ],
  [
    0,
    'curly'
  ],
  [
    0,
    'apostrophes'
  ],
  [
    2,
    ':'
  ],
  [
    1,
    "\x{201c}"
  ],
  [
    0,
    "we\x{2019}ve"
  ],
  [
    0,
    'been'
  ],
  [
    0,
    'here'
  ],
  [
    2,
    "\x{201d}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2022}"
  ],
  [
    0,
    'latin-1'
  ],
  [
    0,
    'apostrophe'
  ],
  [
    0,
    'and'
  ],
  [
    0,
    'accents'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "'\x{b4}`"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2022}"
  ],
  [
    1,
    "\x{201a}"
  ],
  [
    0,
    'deutsche'
  ],
  [
    2,
    "\x{2018}"
  ],
  [
    1,
    "\x{201e}"
  ],
  [
    0,
    "anf\x{fc}hrungszeichen"
  ],
  [
    2,
    "\x{201c}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2022}"
  ],
  [
    0,
    "\x{2020},"
  ],
  [
    0,
    "\x{2021},"
  ],
  [
    0,
    "\x{2030},"
  ],
  [
    0,
    "\x{2022},"
  ],
  [
    0,
    '3'
  ],
  [
    3,
    "\x{2013}"
  ],
  [
    0,
    '4'
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{2014},"
  ],
  [
    1,
    "\x{2212}"
  ],
  [
    0,
    '5'
  ],
  [
    3,
    '/+'
  ],
  [
    0,
    '5'
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{2122},"
  ],
  [
    0,
    "\x{2026}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2022}"
  ],
  [
    0,
    'ASCII'
  ],
  [
    0,
    'safety'
  ],
  [
    0,
    'test'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    '1'
  ],
  [
    2,
    'lI'
  ],
  [
    2,
    '|,'
  ],
  [
    0,
    '0'
  ],
  [
    2,
    'OD'
  ],
  [
    2,
    ','
  ],
  [
    0,
    '8'
  ],
  [
    2,
    'B'
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{256d}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{256e}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2022}"
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'euro'
  ],
  [
    0,
    'symbol'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{2502}"
  ],
  [
    0,
    '14.95'
  ],
  [
    0,
    "\x{20ac}"
  ],
  [
    0,
    "\x{2502}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{2570}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{2500}\x{256f}"
  ],
  [
    0,
    "\x{2551}"
  ],
  [
    0,
    "\x{255a}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{2550}\x{255d}"
  ],
  [
    0,
    'combining'
  ],
  [
    0,
    'characters'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "STARG\x{39b}"
  ],
  [
    3,
    "\x{30a}"
  ],
  [
    0,
    'TE'
  ],
  [
    0,
    'SG-1'
  ],
  [
    2,
    ','
  ],
  [
    0,
    'a'
  ],
  [
    0,
    '='
  ],
  [
    0,
    'v'
  ],
  [
    2,
    "\x{307}"
  ],
  [
    0,
    '='
  ],
  [
    0,
    'r'
  ],
  [
    2,
    "\x{308},"
  ],
  [
    0,
    'a'
  ],
  [
    2,
    "\x{20d1}"
  ],
  [
    0,
    "\x{22a5}"
  ],
  [
    0,
    'b'
  ],
  [
    2,
    "\x{20d1}"
  ],
  [
    0,
    'greek'
  ],
  [
    1,
    '('
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'polytonic'
  ],
  [
    2,
    '):'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'greek'
  ],
  [
    0,
    'anthem'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{3c3}\x{1f72}"
  ],
  [
    0,
    "\x{3b3}\x{3bd}\x{3c9}\x{3c1}\x{1f77}\x{3b6}\x{3c9}"
  ],
  [
    0,
    "\x{1f00}\x{3c0}\x{1f78}"
  ],
  [
    0,
    "\x{3c4}\x{1f74}\x{3bd}"
  ],
  [
    0,
    "\x{3ba}\x{1f79}\x{3c8}\x{3b7}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1fe6}"
  ],
  [
    0,
    "\x{3c3}\x{3c0}\x{3b1}\x{3b8}\x{3b9}\x{3bf}\x{1fe6}"
  ],
  [
    0,
    "\x{3c4}\x{1f74}\x{3bd}"
  ],
  [
    0,
    "\x{3c4}\x{3c1}\x{3bf}\x{3bc}\x{3b5}\x{3c1}\x{1f75}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3c3}\x{1f72}"
  ],
  [
    0,
    "\x{3b3}\x{3bd}\x{3c9}\x{3c1}\x{1f77}\x{3b6}\x{3c9}"
  ],
  [
    0,
    "\x{1f00}\x{3c0}\x{1f78}"
  ],
  [
    0,
    "\x{3c4}\x{1f74}\x{3bd}"
  ],
  [
    0,
    "\x{1f44}\x{3c8}\x{3b7}"
  ],
  [
    0,
    "\x{3c0}\x{3bf}\x{1f7a}"
  ],
  [
    0,
    "\x{3bc}\x{1f72}"
  ],
  [
    0,
    "\x{3b2}\x{1f77}\x{3b1}"
  ],
  [
    0,
    "\x{3bc}\x{3b5}\x{3c4}\x{3c1}\x{1f71}\x{3b5}\x{3b9}"
  ],
  [
    0,
    "\x{3c4}\x{1f74}"
  ],
  [
    0,
    "\x{3b3}\x{1fc6}"
  ],
  [
    2,
    '.'
  ],
  [
    1,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{3b1}\x{3c0}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{3c4}\x{1f70}"
  ],
  [
    0,
    "\x{3ba}\x{1f79}\x{3ba}\x{3ba}\x{3b1}\x{3bb}\x{3b1}"
  ],
  [
    0,
    "\x{3b2}\x{3b3}\x{3b1}\x{3bb}\x{3bc}\x{1f73}\x{3bd}\x{3b7}"
  ],
  [
    0,
    "\x{3c4}\x{1ff6}\x{3bd}"
  ],
  [
    1,
    "\x{1ffe}"
  ],
  [
    0,
    "\x{3b5}\x{3bb}\x{3bb}\x{1f75}\x{3bd}\x{3c9}\x{3bd}"
  ],
  [
    0,
    "\x{3c4}\x{1f70}"
  ],
  [
    0,
    "\x{1f31}\x{3b5}\x{3c1}\x{1f71}"
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{1f76}"
  ],
  [
    0,
    "\x{3c3}\x{1f70}\x{3bd}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{1ff6}\x{3c4}\x{3b1}"
  ],
  [
    0,
    "\x{1f00}\x{3bd}\x{3b4}\x{3c1}\x{3b5}\x{3b9}\x{3c9}\x{3bc}\x{1f73}\x{3bd}\x{3b7}"
  ],
  [
    0,
    "\x{3c7}\x{3b1}\x{1fd6}\x{3c1}\x{3b5}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{1f66}"
  ],
  [
    0,
    "\x{3c7}\x{3b1}\x{1fd6}\x{3c1}\x{3b5}"
  ],
  [
    2,
    ','
  ],
  [
    1,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{3b5}\x{3bb}\x{3b5}\x{3c5}\x{3b8}\x{3b5}\x{3c1}\x{3b9}\x{1f71}"
  ],
  [
    2,
    '!'
  ],
  [
    0,
    'from'
  ],
  [
    0,
    'a'
  ],
  [
    0,
    'speech'
  ],
  [
    0,
    'of'
  ],
  [
    0,
    'demosthenes'
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    '4'
  ],
  [
    2,
    'th'
  ],
  [
    0,
    'century'
  ],
  [
    0,
    'BC'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{3bf}\x{1f50}\x{3c7}\x{1f76}"
  ],
  [
    0,
    "\x{3c4}\x{3b1}\x{1f50}\x{3c4}\x{1f70}"
  ],
  [
    0,
    "\x{3c0}\x{3b1}\x{3c1}\x{1f77}\x{3c3}\x{3c4}\x{3b1}\x{3c4}\x{3b1}\x{1f77}"
  ],
  [
    0,
    "\x{3bc}\x{3bf}\x{3b9}"
  ],
  [
    0,
    "\x{3b3}\x{3b9}\x{3b3}\x{3bd}\x{1f7d}\x{3c3}\x{3ba}\x{3b5}\x{3b9}\x{3bd}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{1f66}"
  ],
  [
    0,
    "\x{1f04}\x{3bd}\x{3b4}\x{3c1}\x{3b5}\x{3c2}"
  ],
  [
    1,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{3b1}\x{3b8}\x{3b7}\x{3bd}\x{3b1}\x{1fd6}\x{3bf}\x{3b9}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{1f45}\x{3c4}\x{3b1}\x{3bd}"
  ],
  [
    0,
    "\x{3c4}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{3b5}\x{1f30}\x{3c2}"
  ],
  [
    0,
    "\x{3c4}\x{1f70}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{1f71}\x{3b3}\x{3bc}\x{3b1}\x{3c4}\x{3b1}"
  ],
  [
    0,
    "\x{1f00}\x{3c0}\x{3bf}\x{3b2}\x{3bb}\x{1f73}\x{3c8}\x{3c9}"
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{1f76}"
  ],
  [
    0,
    "\x{1f45}\x{3c4}\x{3b1}\x{3bd}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{1f78}\x{3c2}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1f7a}\x{3c2}"
  ],
  [
    0,
    "\x{3bb}\x{1f79}\x{3b3}\x{3bf}\x{3c5}\x{3c2}"
  ],
  [
    0,
    "\x{3bf}\x{1f53}\x{3c2}"
  ],
  [
    0,
    "\x{1f00}\x{3ba}\x{3bf}\x{1f7b}\x{3c9}"
  ],
  [
    2,
    "\x{387}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1f7a}\x{3c2}"
  ],
  [
    0,
    "\x{3bc}\x{1f72}\x{3bd}"
  ],
  [
    0,
    "\x{3b3}\x{1f70}\x{3c1}"
  ],
  [
    0,
    "\x{3bb}\x{1f79}\x{3b3}\x{3bf}\x{3c5}\x{3c2}"
  ],
  [
    0,
    "\x{3c0}\x{3b5}\x{3c1}\x{1f76}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1fe6}"
  ],
  [
    0,
    "\x{3c4}\x{3b9}\x{3bc}\x{3c9}\x{3c1}\x{1f75}\x{3c3}\x{3b1}\x{3c3}\x{3b8}\x{3b1}\x{3b9}"
  ],
  [
    0,
    "\x{3c6}\x{1f77}\x{3bb}\x{3b9}\x{3c0}\x{3c0}\x{3bf}\x{3bd}"
  ],
  [
    0,
    "\x{1f41}\x{3c1}\x{1ff6}"
  ],
  [
    0,
    "\x{3b3}\x{3b9}\x{3b3}\x{3bd}\x{3bf}\x{3bc}\x{1f73}\x{3bd}\x{3bf}\x{3c5}\x{3c2}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3c4}\x{1f70}"
  ],
  [
    0,
    "\x{3b4}\x{1f72}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{1f71}\x{3b3}\x{3bc}\x{3b1}\x{3c4}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{3b5}\x{1f30}\x{3c2}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1fe6}\x{3c4}\x{3bf}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{3bf}\x{1f75}\x{3ba}\x{3bf}\x{3bd}\x{3c4}\x{3b1}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{1f65}\x{3c3}\x{3b8}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{1f45}\x{3c0}\x{3c9}\x{3c2}"
  ],
  [
    0,
    "\x{3bc}\x{1f74}"
  ],
  [
    0,
    "\x{3c0}\x{3b5}\x{3b9}\x{3c3}\x{1f79}\x{3bc}\x{3b5}\x{3b8}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{3b1}\x{1f50}\x{3c4}\x{3bf}\x{1f76}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{1f79}\x{3c4}\x{3b5}\x{3c1}\x{3bf}\x{3bd}"
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{3ba}\x{1ff6}\x{3c2}"
  ],
  [
    0,
    "\x{3c3}\x{3ba}\x{1f73}\x{3c8}\x{3b1}\x{3c3}\x{3b8}\x{3b1}\x{3b9}"
  ],
  [
    0,
    "\x{3b4}\x{1f73}\x{3bf}\x{3bd}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    "\x{3bf}\x{1f50}\x{3b4}\x{1f73}\x{3bd}"
  ],
  [
    0,
    "\x{3bf}\x{1f56}\x{3bd}"
  ],
  [
    0,
    "\x{1f04}\x{3bb}\x{3bb}\x{3bf}"
  ],
  [
    0,
    "\x{3bc}\x{3bf}\x{3b9}"
  ],
  [
    0,
    "\x{3b4}\x{3bf}\x{3ba}\x{3bf}\x{1fe6}\x{3c3}\x{3b9}\x{3bd}"
  ],
  [
    0,
    "\x{3bf}\x{1f31}"
  ],
  [
    0,
    "\x{3c4}\x{1f70}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{3b9}\x{3b1}\x{1fe6}\x{3c4}\x{3b1}"
  ],
  [
    0,
    "\x{3bb}\x{1f73}\x{3b3}\x{3bf}\x{3bd}\x{3c4}\x{3b5}\x{3c2}"
  ],
  [
    0,
    "\x{1f22}"
  ],
  [
    0,
    "\x{3c4}\x{1f74}\x{3bd}"
  ],
  [
    0,
    "\x{1f51}\x{3c0}\x{1f79}\x{3b8}\x{3b5}\x{3c3}\x{3b9}\x{3bd}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3c0}\x{3b5}\x{3c1}\x{1f76}"
  ],
  [
    0,
    "\x{1f27}\x{3c2}"
  ],
  [
    0,
    "\x{3b2}\x{3bf}\x{3c5}\x{3bb}\x{3b5}\x{1f7b}\x{3b5}\x{3c3}\x{3b8}\x{3b1}\x{3b9}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3bf}\x{1f50}\x{3c7}\x{1f76}"
  ],
  [
    0,
    "\x{3c4}\x{1f74}\x{3bd}"
  ],
  [
    0,
    "\x{3bf}\x{1f56}\x{3c3}\x{3b1}\x{3bd}"
  ],
  [
    0,
    "\x{3c0}\x{3b1}\x{3c1}\x{3b9}\x{3c3}\x{3c4}\x{1f71}\x{3bd}\x{3c4}\x{3b5}\x{3c2}"
  ],
  [
    0,
    "\x{1f51}\x{3bc}\x{1fd6}\x{3bd}"
  ],
  [
    0,
    "\x{1f01}\x{3bc}\x{3b1}\x{3c1}\x{3c4}\x{1f71}\x{3bd}\x{3b5}\x{3b9}\x{3bd}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    "\x{1f10}\x{3b3}\x{1f7c}"
  ],
  [
    0,
    "\x{3b4}\x{1f73}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{1f45}\x{3c4}\x{3b9}"
  ],
  [
    0,
    "\x{3bc}\x{1f73}\x{3bd}"
  ],
  [
    0,
    "\x{3c0}\x{3bf}\x{3c4}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{1f10}\x{3be}\x{1fc6}\x{3bd}"
  ],
  [
    0,
    "\x{3c4}\x{1fc7}"
  ],
  [
    0,
    "\x{3c0}\x{1f79}\x{3bb}\x{3b5}\x{3b9}"
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{1f76}"
  ],
  [
    0,
    "\x{3c4}\x{1f70}"
  ],
  [
    0,
    "\x{3b1}\x{1f51}\x{3c4}\x{1fc6}\x{3c2}"
  ],
  [
    0,
    "\x{1f14}\x{3c7}\x{3b5}\x{3b9}\x{3bd}"
  ],
  [
    0,
    "\x{1f00}\x{3c3}\x{3c6}\x{3b1}\x{3bb}\x{1ff6}\x{3c2}"
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{1f76}"
  ],
  [
    0,
    "\x{3c6}\x{1f77}\x{3bb}\x{3b9}\x{3c0}\x{3c0}\x{3bf}\x{3bd}"
  ],
  [
    0,
    "\x{3c4}\x{3b9}\x{3bc}\x{3c9}\x{3c1}\x{1f75}\x{3c3}\x{3b1}\x{3c3}\x{3b8}\x{3b1}\x{3b9}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{1f76}"
  ],
  [
    0,
    "\x{3bc}\x{1f71}\x{3bb}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{1f00}\x{3ba}\x{3c1}\x{3b9}\x{3b2}\x{1ff6}\x{3c2}"
  ],
  [
    0,
    "\x{3bf}\x{1f36}\x{3b4}\x{3b1}"
  ],
  [
    2,
    "\x{387}"
  ],
  [
    0,
    "\x{1f10}\x{3c0}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{1f10}\x{3bc}\x{3bf}\x{1fe6}"
  ],
  [
    0,
    "\x{3b3}\x{1f71}\x{3c1}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3bf}\x{1f50}"
  ],
  [
    0,
    "\x{3c0}\x{1f71}\x{3bb}\x{3b1}\x{3b9}"
  ],
  [
    0,
    "\x{3b3}\x{1f73}\x{3b3}\x{3bf}\x{3bd}\x{3b5}\x{3bd}"
  ],
  [
    0,
    "\x{3c4}\x{3b1}\x{1fe6}\x{3c4}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{1f00}\x{3bc}\x{3c6}\x{1f79}\x{3c4}\x{3b5}\x{3c1}\x{3b1}"
  ],
  [
    2,
    "\x{387}"
  ],
  [
    0,
    "\x{3bd}\x{1fe6}\x{3bd}"
  ],
  [
    0,
    "\x{3bc}\x{1f73}\x{3bd}\x{3c4}\x{3bf}\x{3b9}"
  ],
  [
    0,
    "\x{3c0}\x{1f73}\x{3c0}\x{3b5}\x{3b9}\x{3c3}\x{3bc}\x{3b1}\x{3b9}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1fe6}\x{3b8}"
  ],
  [
    2,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{1f31}\x{3ba}\x{3b1}\x{3bd}\x{1f78}\x{3bd}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{3bf}\x{3bb}\x{3b1}\x{3b2}\x{3b5}\x{1fd6}\x{3bd}"
  ],
  [
    0,
    "\x{1f21}\x{3bc}\x{1fd6}\x{3bd}"
  ],
  [
    0,
    "\x{3b5}\x{1f36}\x{3bd}\x{3b1}\x{3b9}"
  ],
  [
    0,
    "\x{3c4}\x{1f74}\x{3bd}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{1f7d}\x{3c4}\x{3b7}\x{3bd}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{1f45}\x{3c0}\x{3c9}\x{3c2}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1f7a}\x{3c2}"
  ],
  [
    0,
    "\x{3c3}\x{3c5}\x{3bc}\x{3bc}\x{1f71}\x{3c7}\x{3bf}\x{3c5}\x{3c2}"
  ],
  [
    0,
    "\x{3c3}\x{1f7d}\x{3c3}\x{3bf}\x{3bc}\x{3b5}\x{3bd}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    "\x{1f10}\x{1f70}\x{3bd}"
  ],
  [
    0,
    "\x{3b3}\x{1f70}\x{3c1}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1fe6}\x{3c4}\x{3bf}"
  ],
  [
    0,
    "\x{3b2}\x{3b5}\x{3b2}\x{3b1}\x{1f77}\x{3c9}\x{3c2}"
  ],
  [
    0,
    "\x{1f51}\x{3c0}\x{1f71}\x{3c1}\x{3be}\x{1fc3}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3c4}\x{1f79}\x{3c4}\x{3b5}"
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{1f76}"
  ],
  [
    0,
    "\x{3c0}\x{3b5}\x{3c1}\x{1f76}"
  ],
  [
    0,
    "\x{3c4}\x{3bf}\x{1fe6}"
  ],
  [
    0,
    "\x{3c4}\x{1f77}\x{3bd}\x{3b1}"
  ],
  [
    0,
    "\x{3c4}\x{3b9}\x{3bc}\x{3c9}\x{3c1}\x{1f75}\x{3c3}\x{3b5}\x{3c4}\x{3b1}\x{1f77}"
  ],
  [
    0,
    "\x{3c4}\x{3b9}\x{3c2}"
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{1f76}"
  ],
  [
    0,
    "\x{1f43}\x{3bd}"
  ],
  [
    0,
    "\x{3c4}\x{3c1}\x{1f79}\x{3c0}\x{3bf}\x{3bd}"
  ],
  [
    0,
    "\x{1f10}\x{3be}\x{1f73}\x{3c3}\x{3c4}\x{3b1}\x{3b9}"
  ],
  [
    0,
    "\x{3c3}\x{3ba}\x{3bf}\x{3c0}\x{3b5}\x{1fd6}\x{3bd}"
  ],
  [
    2,
    "\x{387}"
  ],
  [
    0,
    "\x{3c0}\x{3c1}\x{1f76}\x{3bd}"
  ],
  [
    0,
    "\x{3b4}\x{1f72}"
  ],
  [
    0,
    "\x{3c4}\x{1f74}\x{3bd}"
  ],
  [
    0,
    "\x{1f00}\x{3c1}\x{3c7}\x{1f74}\x{3bd}"
  ],
  [
    0,
    "\x{1f40}\x{3c1}\x{3b8}\x{1ff6}\x{3c2}"
  ],
  [
    0,
    "\x{1f51}\x{3c0}\x{3bf}\x{3b8}\x{1f73}\x{3c3}\x{3b8}\x{3b1}\x{3b9}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3bc}\x{1f71}\x{3c4}\x{3b1}\x{3b9}\x{3bf}\x{3bd}"
  ],
  [
    0,
    "\x{1f21}\x{3b3}\x{3bf}\x{1fe6}\x{3bc}\x{3b1}\x{3b9}"
  ],
  [
    0,
    "\x{3c0}\x{3b5}\x{3c1}\x{1f76}"
  ],
  [
    0,
    "\x{3c4}\x{1fc6}\x{3c2}"
  ],
  [
    0,
    "\x{3c4}\x{3b5}\x{3bb}\x{3b5}\x{3c5}\x{3c4}\x{1fc6}\x{3c2}"
  ],
  [
    0,
    "\x{1f41}\x{3bd}\x{3c4}\x{3b9}\x{3bd}\x{3bf}\x{1fe6}\x{3bd}"
  ],
  [
    0,
    "\x{3c0}\x{3bf}\x{3b9}\x{3b5}\x{1fd6}\x{3c3}\x{3b8}\x{3b1}\x{3b9}"
  ],
  [
    0,
    "\x{3bb}\x{1f79}\x{3b3}\x{3bf}\x{3bd}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    "\x{3b4}\x{3b7}\x{3bc}\x{3bf}\x{3c3}\x{3b8}\x{1f73}\x{3bd}\x{3bf}\x{3c5}\x{3c2}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{393}"
  ],
  [
    2,
    "\x{1ffd}"
  ],
  [
    1,
    "\x{1fbf}"
  ],
  [
    0,
    "\x{3bf}\x{3bb}\x{3c5}\x{3bd}\x{3b8}\x{3b9}\x{3b1}\x{3ba}\x{1f78}\x{3c2}"
  ],
  [
    0,
    'georgian'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    'from'
  ],
  [
    0,
    'a'
  ],
  [
    0,
    'unicode'
  ],
  [
    0,
    'conference'
  ],
  [
    0,
    'invitation'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{10d2}\x{10d7}\x{10ee}\x{10dd}\x{10d5}\x{10d7}"
  ],
  [
    0,
    "\x{10d0}\x{10ee}\x{10da}\x{10d0}\x{10d5}\x{10d4}"
  ],
  [
    0,
    "\x{10d2}\x{10d0}\x{10d8}\x{10d0}\x{10e0}\x{10dd}\x{10d7}"
  ],
  [
    0,
    "\x{10e0}\x{10d4}\x{10d2}\x{10d8}\x{10e1}\x{10e2}\x{10e0}\x{10d0}\x{10ea}\x{10d8}\x{10d0}"
  ],
  [
    0,
    "unicode-\x{10d8}\x{10e1}"
  ],
  [
    0,
    "\x{10db}\x{10d4}\x{10d0}\x{10d7}\x{10d4}"
  ],
  [
    0,
    "\x{10e1}\x{10d0}\x{10d4}\x{10e0}\x{10d7}\x{10d0}\x{10e8}\x{10dd}\x{10e0}\x{10d8}\x{10e1}\x{10dd}"
  ],
  [
    0,
    "\x{10d9}\x{10dd}\x{10dc}\x{10e4}\x{10d4}\x{10e0}\x{10d4}\x{10dc}\x{10ea}\x{10d8}\x{10d0}\x{10d6}\x{10d4}"
  ],
  [
    0,
    "\x{10d3}\x{10d0}\x{10e1}\x{10d0}\x{10e1}\x{10ec}\x{10e0}\x{10d4}\x{10d1}\x{10d0}\x{10d3}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{10e0}\x{10dd}\x{10db}\x{10d4}\x{10da}\x{10d8}\x{10ea}"
  ],
  [
    0,
    "\x{10d2}\x{10d0}\x{10d8}\x{10db}\x{10d0}\x{10e0}\x{10d7}\x{10d4}\x{10d1}\x{10d0}"
  ],
  [
    0,
    '10-12'
  ],
  [
    0,
    "\x{10db}\x{10d0}\x{10e0}\x{10e2}\x{10e1}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{10e5}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    "\x{10db}\x{10d0}\x{10d8}\x{10dc}\x{10ea}\x{10e8}\x{10d8}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{10d2}\x{10d4}\x{10e0}\x{10db}\x{10d0}\x{10dc}\x{10d8}\x{10d0}\x{10e8}\x{10d8}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    "\x{10d9}\x{10dd}\x{10dc}\x{10e4}\x{10d4}\x{10e0}\x{10d4}\x{10dc}\x{10ea}\x{10d8}\x{10d0}"
  ],
  [
    0,
    "\x{10e8}\x{10d4}\x{10f0}\x{10d9}\x{10e0}\x{10d4}\x{10d1}\x{10e1}"
  ],
  [
    0,
    "\x{10d4}\x{10e0}\x{10d7}\x{10d0}\x{10d3}"
  ],
  [
    0,
    "\x{10db}\x{10e1}\x{10dd}\x{10e4}\x{10da}\x{10d8}\x{10dd}\x{10e1}"
  ],
  [
    0,
    "\x{10d4}\x{10e5}\x{10e1}\x{10de}\x{10d4}\x{10e0}\x{10e2}\x{10d4}\x{10d1}\x{10e1}"
  ],
  [
    0,
    "\x{10d8}\x{10e1}\x{10d4}\x{10d7}"
  ],
  [
    0,
    "\x{10d3}\x{10d0}\x{10e0}\x{10d2}\x{10d4}\x{10d1}\x{10e8}\x{10d8}"
  ],
  [
    0,
    "\x{10e0}\x{10dd}\x{10d2}\x{10dd}\x{10e0}\x{10d8}\x{10ea}\x{10d0}\x{10d0}"
  ],
  [
    0,
    "\x{10d8}\x{10dc}\x{10e2}\x{10d4}\x{10e0}\x{10dc}\x{10d4}\x{10e2}\x{10d8}"
  ],
  [
    0,
    "\x{10d3}\x{10d0}"
  ],
  [
    0,
    "unicode-\x{10d8}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{10d8}\x{10dc}\x{10e2}\x{10d4}\x{10e0}\x{10dc}\x{10d0}\x{10ea}\x{10d8}\x{10dd}\x{10dc}\x{10d0}\x{10da}\x{10d8}\x{10d6}\x{10d0}\x{10ea}\x{10d8}\x{10d0}"
  ],
  [
    0,
    "\x{10d3}\x{10d0}"
  ],
  [
    0,
    "\x{10da}\x{10dd}\x{10d9}\x{10d0}\x{10da}\x{10d8}\x{10d6}\x{10d0}\x{10ea}\x{10d8}\x{10d0}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "unicode-\x{10d8}\x{10e1}"
  ],
  [
    0,
    "\x{10d2}\x{10d0}\x{10db}\x{10dd}\x{10e7}\x{10d4}\x{10dc}\x{10d4}\x{10d1}\x{10d0}"
  ],
  [
    0,
    "\x{10dd}\x{10de}\x{10d4}\x{10e0}\x{10d0}\x{10ea}\x{10d8}\x{10e3}\x{10da}"
  ],
  [
    0,
    "\x{10e1}\x{10d8}\x{10e1}\x{10e2}\x{10d4}\x{10db}\x{10d4}\x{10d1}\x{10e1}\x{10d0}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{10d3}\x{10d0}"
  ],
  [
    0,
    "\x{10d2}\x{10d0}\x{10db}\x{10dd}\x{10e7}\x{10d4}\x{10dc}\x{10d4}\x{10d1}\x{10d8}\x{10d7}"
  ],
  [
    0,
    "\x{10de}\x{10e0}\x{10dd}\x{10d2}\x{10e0}\x{10d0}\x{10db}\x{10d4}\x{10d1}\x{10e8}\x{10d8}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{10e8}\x{10e0}\x{10d8}\x{10e4}\x{10e2}\x{10d4}\x{10d1}\x{10e8}\x{10d8}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{10e2}\x{10d4}\x{10e5}\x{10e1}\x{10e2}\x{10d4}\x{10d1}\x{10d8}\x{10e1}"
  ],
  [
    0,
    "\x{10d3}\x{10d0}\x{10db}\x{10e3}\x{10e8}\x{10d0}\x{10d5}\x{10d4}\x{10d1}\x{10d0}\x{10e1}\x{10d0}"
  ],
  [
    0,
    "\x{10d3}\x{10d0}"
  ],
  [
    0,
    "\x{10db}\x{10e0}\x{10d0}\x{10d5}\x{10d0}\x{10da}\x{10d4}\x{10dc}\x{10dd}\x{10d5}\x{10d0}\x{10dc}"
  ],
  [
    0,
    "\x{10d9}\x{10dd}\x{10db}\x{10de}\x{10d8}\x{10e3}\x{10e2}\x{10d4}\x{10e0}\x{10e3}\x{10da}"
  ],
  [
    0,
    "\x{10e1}\x{10d8}\x{10e1}\x{10e2}\x{10d4}\x{10db}\x{10d4}\x{10d1}\x{10e8}\x{10d8}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    'russian'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    'from'
  ],
  [
    0,
    'a'
  ],
  [
    0,
    'unicode'
  ],
  [
    0,
    'conference'
  ],
  [
    0,
    'invitation'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{437}\x{430}\x{440}\x{435}\x{433}\x{438}\x{441}\x{442}\x{440}\x{438}\x{440}\x{443}\x{439}\x{442}\x{435}\x{441}\x{44c}"
  ],
  [
    0,
    "\x{441}\x{435}\x{439}\x{447}\x{430}\x{441}"
  ],
  [
    0,
    "\x{43d}\x{430}"
  ],
  [
    0,
    "\x{434}\x{435}\x{441}\x{44f}\x{442}\x{443}\x{44e}"
  ],
  [
    0,
    "\x{43c}\x{435}\x{436}\x{434}\x{443}\x{43d}\x{430}\x{440}\x{43e}\x{434}\x{43d}\x{443}\x{44e}"
  ],
  [
    0,
    "\x{43a}\x{43e}\x{43d}\x{444}\x{435}\x{440}\x{435}\x{43d}\x{446}\x{438}\x{44e}"
  ],
  [
    0,
    "\x{43f}\x{43e}"
  ],
  [
    0,
    'unicode'
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{43a}\x{43e}\x{442}\x{43e}\x{440}\x{430}\x{44f}"
  ],
  [
    0,
    "\x{441}\x{43e}\x{441}\x{442}\x{43e}\x{438}\x{442}\x{441}\x{44f}"
  ],
  [
    0,
    '10-12'
  ],
  [
    0,
    "\x{43c}\x{430}\x{440}\x{442}\x{430}"
  ],
  [
    0,
    '1997'
  ],
  [
    0,
    "\x{433}\x{43e}\x{434}\x{430}"
  ],
  [
    0,
    "\x{432}"
  ],
  [
    0,
    "\x{43c}\x{430}\x{439}\x{43d}\x{446}\x{435}"
  ],
  [
    0,
    "\x{432}"
  ],
  [
    0,
    "\x{433}\x{435}\x{440}\x{43c}\x{430}\x{43d}\x{438}\x{438}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    "\x{43a}\x{43e}\x{43d}\x{444}\x{435}\x{440}\x{435}\x{43d}\x{446}\x{438}\x{44f}"
  ],
  [
    0,
    "\x{441}\x{43e}\x{431}\x{435}\x{440}\x{435}\x{442}"
  ],
  [
    0,
    "\x{448}\x{438}\x{440}\x{43e}\x{43a}\x{438}\x{439}"
  ],
  [
    0,
    "\x{43a}\x{440}\x{443}\x{433}"
  ],
  [
    0,
    "\x{44d}\x{43a}\x{441}\x{43f}\x{435}\x{440}\x{442}\x{43e}\x{432}"
  ],
  [
    0,
    "\x{43f}\x{43e}"
  ],
  [
    0,
    "\x{432}\x{43e}\x{43f}\x{440}\x{43e}\x{441}\x{430}\x{43c}"
  ],
  [
    0,
    "\x{433}\x{43b}\x{43e}\x{431}\x{430}\x{43b}\x{44c}\x{43d}\x{43e}\x{433}\x{43e}"
  ],
  [
    0,
    "\x{438}\x{43d}\x{442}\x{435}\x{440}\x{43d}\x{435}\x{442}\x{430}"
  ],
  [
    0,
    "\x{438}"
  ],
  [
    0,
    'unicode'
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{43b}\x{43e}\x{43a}\x{430}\x{43b}\x{438}\x{437}\x{430}\x{446}\x{438}\x{438}"
  ],
  [
    0,
    "\x{438}"
  ],
  [
    0,
    "\x{438}\x{43d}\x{442}\x{435}\x{440}\x{43d}\x{430}\x{446}\x{438}\x{43e}\x{43d}\x{430}\x{43b}\x{438}\x{437}\x{430}\x{446}\x{438}\x{438}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{432}\x{43e}\x{43f}\x{43b}\x{43e}\x{449}\x{435}\x{43d}\x{438}\x{44e}"
  ],
  [
    0,
    "\x{438}"
  ],
  [
    0,
    "\x{43f}\x{440}\x{438}\x{43c}\x{435}\x{43d}\x{435}\x{43d}\x{438}\x{44e}"
  ],
  [
    0,
    'unicode'
  ],
  [
    0,
    "\x{432}"
  ],
  [
    0,
    "\x{440}\x{430}\x{437}\x{43b}\x{438}\x{447}\x{43d}\x{44b}\x{445}"
  ],
  [
    0,
    "\x{43e}\x{43f}\x{435}\x{440}\x{430}\x{446}\x{438}\x{43e}\x{43d}\x{43d}\x{44b}\x{445}"
  ],
  [
    0,
    "\x{441}\x{438}\x{441}\x{442}\x{435}\x{43c}\x{430}\x{445}"
  ],
  [
    0,
    "\x{438}"
  ],
  [
    0,
    "\x{43f}\x{440}\x{43e}\x{433}\x{440}\x{430}\x{43c}\x{43c}\x{43d}\x{44b}\x{445}"
  ],
  [
    0,
    "\x{43f}\x{440}\x{438}\x{43b}\x{43e}\x{436}\x{435}\x{43d}\x{438}\x{44f}\x{445}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{448}\x{440}\x{438}\x{444}\x{442}\x{430}\x{445}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{432}\x{435}\x{440}\x{441}\x{442}\x{43a}\x{435}"
  ],
  [
    0,
    "\x{438}"
  ],
  [
    0,
    "\x{43c}\x{43d}\x{43e}\x{433}\x{43e}\x{44f}\x{437}\x{44b}\x{447}\x{43d}\x{44b}\x{445}"
  ],
  [
    0,
    "\x{43a}\x{43e}\x{43c}\x{43f}\x{44c}\x{44e}\x{442}\x{435}\x{440}\x{43d}\x{44b}\x{445}"
  ],
  [
    0,
    "\x{441}\x{438}\x{441}\x{442}\x{435}\x{43c}\x{430}\x{445}"
  ],
  [
    2,
    '.'
  ],
  [
    0,
    'thai'
  ],
  [
    1,
    '('
  ],
  [
    0,
    'UCS'
  ],
  [
    0,
    'level'
  ],
  [
    0,
    '2'
  ],
  [
    2,
    '):'
  ],
  [
    0,
    'excerpt'
  ],
  [
    0,
    'from'
  ],
  [
    0,
    'a'
  ],
  [
    0,
    'poetry'
  ],
  [
    0,
    'on'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'romance'
  ],
  [
    0,
    'of'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'three'
  ],
  [
    0,
    'kingdoms'
  ],
  [
    1,
    '('
  ],
  [
    0,
    'a'
  ],
  [
    0,
    'chinese'
  ],
  [
    0,
    'classic'
  ],
  [
    1,
    '\''
  ],
  [
    0,
    'san'
  ],
  [
    0,
    'gua'
  ],
  [
    2,
    '\'):'
  ],
  [
    0,
    '[----------------------------|------------------------]'
  ],
  [
    0,
    "\x{e4f}"
  ],
  [
    0,
    "\x{e41}\x{e1c}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e19}\x{e14}\x{e34}\x{e19}\x{e2e}\x{e31}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e19}\x{e40}\x{e2a}\x{e37}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e2d}\x{e21}\x{e42}\x{e17}\x{e23}\x{e21}\x{e41}\x{e2a}\x{e19}\x{e2a}\x{e31}\x{e07}\x{e40}\x{e27}\x{e0a}"
  ],
  [
    0,
    "\x{e1e}\x{e23}\x{e30}\x{e1b}\x{e01}\x{e40}\x{e01}\x{e28}\x{e01}\x{e2d}\x{e07}\x{e1a}\x{e39}"
  ],
  [
    3,
    "\x{e4a}"
  ],
  [
    0,
    "\x{e01}\x{e39}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e02}\x{e36}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e19}\x{e43}\x{e2b}\x{e21}"
  ],
  [
    2,
    "\x{e48}"
  ],
  [
    0,
    "\x{e2a}\x{e34}\x{e1a}\x{e2a}\x{e2d}\x{e07}\x{e01}\x{e29}\x{e31}\x{e15}\x{e23}\x{e34}\x{e22}"
  ],
  [
    3,
    "\x{e4c}"
  ],
  [
    0,
    "\x{e01}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e2d}\x{e19}\x{e2b}\x{e19}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e32}\x{e41}\x{e25}\x{e16}\x{e31}\x{e14}\x{e44}\x{e1b}"
  ],
  [
    0,
    "\x{e2a}\x{e2d}\x{e07}\x{e2d}\x{e07}\x{e04}"
  ],
  [
    3,
    "\x{e4c}"
  ],
  [
    0,
    "\x{e44}\x{e0b}\x{e23}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e42}\x{e07}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e40}\x{e02}\x{e25}\x{e32}\x{e40}\x{e1a}\x{e32}\x{e1b}\x{e31}\x{e0d}\x{e0d}\x{e32}"
  ],
  [
    0,
    "\x{e17}\x{e23}\x{e07}\x{e19}\x{e31}\x{e1a}\x{e16}\x{e37}\x{e2d}\x{e02}\x{e31}\x{e19}\x{e17}\x{e35}\x{e40}\x{e1b}"
  ],
  [
    3,
    "\x{e47}"
  ],
  [
    0,
    "\x{e19}\x{e17}\x{e35}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e1e}\x{e36}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e07}"
  ],
  [
    0,
    "\x{e1a}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e32}\x{e19}\x{e40}\x{e21}\x{e37}\x{e2d}\x{e07}\x{e08}\x{e36}\x{e07}\x{e27}\x{e34}\x{e1b}\x{e23}\x{e34}\x{e15}\x{e40}\x{e1b}"
  ],
  [
    3,
    "\x{e47}"
  ],
  [
    0,
    "\x{e19}\x{e19}\x{e31}\x{e01}\x{e2b}\x{e19}\x{e32}"
  ],
  [
    0,
    "\x{e42}\x{e2e}\x{e08}\x{e34}"
  ],
  [
    3,
    "\x{e4b}"
  ],
  [
    0,
    "\x{e19}\x{e40}\x{e23}\x{e35}\x{e22}\x{e01}\x{e17}\x{e31}\x{e1e}\x{e17}\x{e31}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e27}\x{e2b}\x{e31}\x{e27}\x{e40}\x{e21}\x{e37}\x{e2d}\x{e07}\x{e21}\x{e32}"
  ],
  [
    0,
    "\x{e2b}\x{e21}\x{e32}\x{e22}\x{e08}\x{e30}\x{e06}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e32}\x{e21}\x{e14}\x{e0a}\x{e31}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e27}\x{e15}\x{e31}\x{e27}\x{e2a}\x{e33}\x{e04}\x{e31}\x{e0d}"
  ],
  [
    0,
    "\x{e40}\x{e2b}\x{e21}\x{e37}\x{e2d}\x{e19}\x{e02}\x{e31}\x{e1a}\x{e44}\x{e2a}\x{e44}\x{e25}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e40}\x{e2a}\x{e37}\x{e2d}\x{e08}\x{e32}\x{e01}\x{e40}\x{e04}\x{e2b}\x{e32}"
  ],
  [
    0,
    "\x{e23}\x{e31}\x{e1a}\x{e2b}\x{e21}\x{e32}\x{e1b}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e32}\x{e40}\x{e02}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e32}\x{e21}\x{e32}\x{e40}\x{e25}\x{e22}\x{e2d}\x{e32}\x{e2a}\x{e31}\x{e0d}"
  ],
  [
    0,
    "\x{e1d}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e32}\x{e22}\x{e2d}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e2d}\x{e07}\x{e2d}\x{e38}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e19}\x{e22}\x{e38}\x{e41}\x{e22}\x{e01}\x{e43}\x{e2b}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e41}\x{e15}\x{e01}\x{e01}\x{e31}\x{e19}"
  ],
  [
    0,
    "\x{e43}\x{e0a}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e2a}\x{e32}\x{e27}\x{e19}\x{e31}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e19}\x{e40}\x{e1b}"
  ],
  [
    3,
    "\x{e47}"
  ],
  [
    0,
    "\x{e19}\x{e0a}\x{e19}\x{e27}\x{e19}\x{e0a}\x{e37}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e19}\x{e0a}\x{e27}\x{e19}\x{e43}\x{e08}"
  ],
  [
    0,
    "\x{e1e}\x{e25}\x{e31}\x{e19}\x{e25}\x{e34}\x{e09}\x{e38}\x{e22}\x{e01}\x{e38}\x{e22}\x{e01}\x{e35}\x{e01}\x{e25}\x{e31}\x{e1a}\x{e01}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e2d}\x{e40}\x{e2b}\x{e15}\x{e38}"
  ],
  [
    0,
    "\x{e0a}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e32}\x{e07}\x{e2d}\x{e32}\x{e40}\x{e1e}\x{e28}\x{e08}\x{e23}\x{e34}\x{e07}\x{e2b}\x{e19}\x{e32}\x{e1f}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e32}\x{e23}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e2d}\x{e07}\x{e44}\x{e2b}"
  ],
  [
    2,
    "\x{e49}"
  ],
  [
    0,
    "\x{e15}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e2d}\x{e07}\x{e23}\x{e1a}\x{e23}\x{e32}\x{e06}"
  ],
  [
    3,
    "\x{e48}"
  ],
  [
    0,
    "\x{e32}\x{e1f}\x{e31}\x{e19}\x{e08}\x{e19}\x{e1a}\x{e23}\x{e23}\x{e25}\x{e31}\x{e22}"
  ],
  [
    0,
    "\x{e24}\x{e45}\x{e2b}\x{e32}\x{e43}\x{e04}\x{e23}\x{e04}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e33}\x{e0a}\x{e39}\x{e01}\x{e39}"
  ],
  [
    3,
    "\x{e49}"
  ],
  [
    0,
    "\x{e1a}\x{e23}\x{e23}\x{e25}\x{e31}\x{e07}\x{e01}"
  ],
  [
    2,
    "\x{e4c}"
  ],
  [
    0,
    "\x{e2f}"
  ],
  [
    1,
    '('
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'above'
  ],
  [
    0,
    'is'
  ],
  [
    0,
    'a'
  ],
  [
    0,
    'two-column'
  ],
  [
    0,
    'text'
  ],
  [
    2,
    '.'
  ],
  [
    0,
    'if'
  ],
  [
    0,
    'combining'
  ],
  [
    0,
    'characters'
  ],
  [
    0,
    'are'
  ],
  [
    0,
    'handled'
  ],
  [
    0,
    'correctly'
  ],
  [
    2,
    ','
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'lines'
  ],
  [
    0,
    'of'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'second'
  ],
  [
    0,
    'column'
  ],
  [
    0,
    'should'
  ],
  [
    0,
    'be'
  ],
  [
    0,
    'aligned'
  ],
  [
    0,
    'with'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    '|'
  ],
  [
    0,
    'character'
  ],
  [
    0,
    'above'
  ],
  [
    2,
    '.)'
  ],
  [
    0,
    'ethiopian'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    'proverbs'
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'amharic'
  ],
  [
    0,
    'language'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{1230}\x{121b}\x{12ed}"
  ],
  [
    0,
    "\x{12a0}\x{12ed}\x{1273}\x{1228}\x{1235}"
  ],
  [
    0,
    "\x{1295}\x{1309}\x{1225}"
  ],
  [
    0,
    "\x{12a0}\x{12ed}\x{12a8}\x{1230}\x{1235}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{1265}\x{120b}"
  ],
  [
    0,
    "\x{12ab}\x{1208}\x{129d}"
  ],
  [
    0,
    "\x{12a5}\x{1295}\x{12f0}\x{12a0}\x{1263}\x{1274}"
  ],
  [
    0,
    "\x{1260}\x{1246}\x{1218}\x{1320}\x{129d}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{130c}\x{1325}"
  ],
  [
    0,
    "\x{12eb}\x{1208}\x{1264}\x{1271}"
  ],
  [
    0,
    "\x{1241}\x{121d}\x{1325}\x{1293}"
  ],
  [
    0,
    "\x{1290}\x{12cd}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12f0}\x{1200}"
  ],
  [
    0,
    "\x{1260}\x{1215}\x{120d}\x{1219}"
  ],
  [
    0,
    "\x{1245}\x{1264}"
  ],
  [
    0,
    "\x{1263}\x{12ed}\x{1320}\x{1323}"
  ],
  [
    0,
    "\x{1295}\x{1323}\x{1275}"
  ],
  [
    0,
    "\x{1260}\x{1308}\x{12f0}\x{1208}\x{12cd}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12e8}\x{12a0}\x{134d}"
  ],
  [
    0,
    "\x{12c8}\x{1208}\x{121d}\x{1273}"
  ],
  [
    0,
    "\x{1260}\x{1245}\x{1264}"
  ],
  [
    0,
    "\x{12a0}\x{12ed}\x{1273}\x{123d}\x{121d}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12a0}\x{12ed}\x{1325}"
  ],
  [
    0,
    "\x{1260}\x{1260}\x{120b}"
  ],
  [
    0,
    "\x{12f3}\x{12cb}"
  ],
  [
    0,
    "\x{1270}\x{1218}\x{1273}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{1232}\x{1270}\x{1228}\x{1309}\x{1219}"
  ],
  [
    0,
    "\x{12ed}\x{12f0}\x{1228}\x{130d}\x{1219}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{1240}\x{1235}"
  ],
  [
    0,
    "\x{1260}\x{1240}\x{1235}"
  ],
  [
    2,
    "\x{1365}"
  ],
  [
    0,
    "\x{12d5}\x{1295}\x{1241}\x{120b}\x{120d}"
  ],
  [
    0,
    "\x{1260}\x{12a5}\x{130d}\x{1229}"
  ],
  [
    0,
    "\x{12ed}\x{1204}\x{12f3}\x{120d}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12f5}\x{122d}"
  ],
  [
    0,
    "\x{1262}\x{12eb}\x{1265}\x{122d}"
  ],
  [
    0,
    "\x{12a0}\x{1295}\x{1260}\x{1233}"
  ],
  [
    0,
    "\x{12eb}\x{1235}\x{122d}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{1230}\x{12cd}"
  ],
  [
    0,
    "\x{12a5}\x{1295}\x{12f0}\x{1264}\x{1271}"
  ],
  [
    0,
    "\x{12a5}\x{1295}\x{1305}"
  ],
  [
    0,
    "\x{12a5}\x{1295}\x{12f0}"
  ],
  [
    0,
    "\x{1309}\x{1228}\x{1264}\x{1271}"
  ],
  [
    0,
    "\x{12a0}\x{12ed}\x{1270}\x{12f3}\x{12f0}\x{122d}\x{121d}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12a5}\x{130d}\x{12dc}\x{122d}"
  ],
  [
    0,
    "\x{12e8}\x{12a8}\x{1348}\x{1270}\x{12cd}\x{1295}"
  ],
  [
    0,
    "\x{1309}\x{122e}\x{122e}"
  ],
  [
    0,
    "\x{1233}\x{12ed}\x{12d8}\x{130b}\x{12cd}"
  ],
  [
    0,
    "\x{12a0}\x{12ed}\x{12f5}\x{122d}\x{121d}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12e8}\x{130e}\x{1228}\x{1264}\x{1275}"
  ],
  [
    0,
    "\x{120c}\x{1263}"
  ],
  [
    2,
    "\x{1365}"
  ],
  [
    0,
    "\x{1262}\x{12eb}\x{12e9}\x{1275}"
  ],
  [
    0,
    "\x{12ed}\x{1235}\x{1245}"
  ],
  [
    0,
    "\x{1263}\x{12eb}\x{12e9}\x{1275}"
  ],
  [
    0,
    "\x{12eb}\x{1320}\x{120d}\x{1245}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{1225}\x{122b}"
  ],
  [
    0,
    "\x{12a8}\x{1218}\x{134d}\x{1273}\x{1275}"
  ],
  [
    0,
    "\x{120d}\x{1304}\x{1295}"
  ],
  [
    0,
    "\x{120b}\x{134b}\x{1273}\x{1275}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12d3}\x{1263}\x{12ed}"
  ],
  [
    0,
    "\x{121b}\x{12f0}\x{122a}\x{12eb}"
  ],
  [
    0,
    "\x{12e8}\x{1208}\x{12cd}"
  ],
  [
    2,
    "\x{1365}"
  ],
  [
    0,
    "\x{130d}\x{1295}\x{12f5}"
  ],
  [
    0,
    "\x{12ed}\x{12de}"
  ],
  [
    0,
    "\x{12ed}\x{12de}\x{122b}\x{120d}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12e8}\x{12a5}\x{1235}\x{120b}\x{121d}"
  ],
  [
    0,
    "\x{12a0}\x{1308}\x{1229}"
  ],
  [
    0,
    "\x{1218}\x{12ab}"
  ],
  [
    0,
    "\x{12e8}\x{12a0}\x{121e}\x{122b}"
  ],
  [
    0,
    "\x{12a0}\x{1308}\x{1229}"
  ],
  [
    0,
    "\x{12cb}\x{122d}\x{12ab}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{1270}\x{1295}\x{130b}\x{120e}"
  ],
  [
    0,
    "\x{1262}\x{1270}\x{1349}"
  ],
  [
    0,
    "\x{1270}\x{1218}\x{120d}\x{1236}"
  ],
  [
    0,
    "\x{1263}\x{1349}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12c8}\x{12f3}\x{1305}\x{1205}"
  ],
  [
    0,
    "\x{121b}\x{122d}"
  ],
  [
    0,
    "\x{1262}\x{1206}\x{1295}"
  ],
  [
    0,
    "\x{1328}\x{122d}\x{1235}\x{1205}"
  ],
  [
    0,
    "\x{12a0}\x{1275}\x{120b}\x{1230}\x{12cd}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    "\x{12a5}\x{130d}\x{122d}\x{1205}\x{1295}"
  ],
  [
    0,
    "\x{1260}\x{134d}\x{122b}\x{123d}\x{1205}"
  ],
  [
    0,
    "\x{120d}\x{12ad}"
  ],
  [
    0,
    "\x{12d8}\x{122d}\x{130b}"
  ],
  [
    2,
    "\x{1362}"
  ],
  [
    0,
    'runes'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{16bb}\x{16d6}"
  ],
  [
    0,
    "\x{16b3}\x{16b9}\x{16ab}\x{16a6}"
  ],
  [
    0,
    "\x{16a6}\x{16ab}\x{16cf}"
  ],
  [
    0,
    "\x{16bb}\x{16d6}"
  ],
  [
    0,
    "\x{16d2}\x{16a2}\x{16de}\x{16d6}"
  ],
  [
    0,
    "\x{16a9}\x{16be}"
  ],
  [
    0,
    "\x{16a6}\x{16ab}\x{16d7}"
  ],
  [
    0,
    "\x{16da}\x{16aa}\x{16be}\x{16de}\x{16d6}"
  ],
  [
    0,
    "\x{16be}\x{16a9}\x{16b1}\x{16a6}\x{16b9}\x{16d6}\x{16aa}\x{16b1}\x{16de}\x{16a2}\x{16d7}"
  ],
  [
    0,
    "\x{16b9}\x{16c1}\x{16a6}"
  ],
  [
    0,
    "\x{16a6}\x{16aa}"
  ],
  [
    0,
    "\x{16b9}\x{16d6}\x{16e5}\x{16ab}"
  ],
  [
    1,
    '('
  ],
  [
    0,
    'old'
  ],
  [
    0,
    'english'
  ],
  [
    2,
    ','
  ],
  [
    0,
    'which'
  ],
  [
    0,
    'transcribed'
  ],
  [
    0,
    'into'
  ],
  [
    0,
    'latin'
  ],
  [
    0,
    'reads'
  ],
  [
    1,
    '\''
  ],
  [
    0,
    'he'
  ],
  [
    0,
    'cwaeth'
  ],
  [
    0,
    'that'
  ],
  [
    0,
    'he'
  ],
  [
    0,
    'bude'
  ],
  [
    0,
    'thaem'
  ],
  [
    0,
    'lande'
  ],
  [
    0,
    'northweardum'
  ],
  [
    0,
    'with'
  ],
  [
    0,
    'tha'
  ],
  [
    0,
    'westsae'
  ],
  [
    2,
    '.\''
  ],
  [
    0,
    'and'
  ],
  [
    0,
    'means'
  ],
  [
    1,
    '\''
  ],
  [
    0,
    'he'
  ],
  [
    0,
    'said'
  ],
  [
    0,
    'that'
  ],
  [
    0,
    'he'
  ],
  [
    0,
    'lived'
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'northern'
  ],
  [
    0,
    'land'
  ],
  [
    0,
    'near'
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'western'
  ],
  [
    0,
    'sea'
  ],
  [
    2,
    '.\')'
  ],
  [
    0,
    'braille'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{284c}\x{2801}\x{2827}\x{2811}"
  ],
  [
    0,
    "\x{283c}\x{2801}\x{2812}"
  ],
  [
    0,
    "\x{284d}\x{281c}\x{2807}\x{2811}\x{2839}\x{2830}\x{280e}"
  ],
  [
    0,
    "\x{2863}\x{2815}\x{280c}"
  ],
  [
    0,
    "\x{284d}\x{281c}\x{2807}\x{2811}\x{2839}"
  ],
  [
    0,
    "\x{283a}\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{2819}\x{2811}\x{2801}\x{2819}\x{2812}"
  ],
  [
    0,
    "\x{281e}\x{2815}"
  ],
  [
    0,
    "\x{2803}\x{2811}\x{281b}\x{2814}"
  ],
  [
    0,
    "\x{283a}\x{280a}\x{2839}\x{2832}"
  ],
  [
    0,
    "\x{2879}\x{283b}\x{2811}"
  ],
  [
    0,
    "\x{280a}\x{280e}"
  ],
  [
    0,
    "\x{281d}\x{2815}"
  ],
  [
    0,
    "\x{2819}\x{2833}\x{2803}\x{281e}"
  ],
  [
    0,
    "\x{2831}\x{2801}\x{281e}\x{2811}\x{2827}\x{283b}"
  ],
  [
    0,
    "\x{2801}\x{2803}\x{2833}\x{281e}"
  ],
  [
    0,
    "\x{2839}\x{2801}\x{281e}\x{2832}"
  ],
  [
    0,
    "\x{2879}\x{2811}"
  ],
  [
    0,
    "\x{2817}\x{2811}\x{281b}\x{280a}\x{280c}\x{283b}"
  ],
  [
    0,
    "\x{2815}\x{280b}"
  ],
  [
    0,
    "\x{2819}\x{280a}\x{280e}"
  ],
  [
    0,
    "\x{2803}\x{2825}\x{2817}\x{280a}\x{2801}\x{2807}"
  ],
  [
    0,
    "\x{283a}\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{280e}\x{280a}\x{281b}\x{281d}\x{282b}"
  ],
  [
    0,
    "\x{2803}\x{2839}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{280a}\x{2807}\x{283b}\x{281b}\x{2839}\x{280d}\x{2801}\x{281d}\x{2802}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{280a}\x{2807}\x{283b}\x{2805}\x{2802}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{2825}\x{281d}\x{2819}\x{283b}\x{281e}\x{2801}\x{2805}\x{283b}\x{2802}"
  ],
  [
    0,
    "\x{2801}\x{281d}\x{2819}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{2821}\x{280a}\x{2811}\x{280b}"
  ],
  [
    0,
    "\x{280d}\x{2833}\x{2817}\x{281d}\x{283b}\x{2832}"
  ],
  [
    0,
    "\x{284e}\x{280a}\x{2817}\x{2815}\x{2815}\x{281b}\x{2811}"
  ],
  [
    0,
    "\x{280e}\x{280a}\x{281b}\x{281d}\x{282b}"
  ],
  [
    0,
    "\x{280a}\x{281e}\x{2832}"
  ],
  [
    0,
    "\x{2841}\x{281d}\x{2819}"
  ],
  [
    0,
    "\x{284e}\x{280a}\x{2817}\x{2815}\x{2815}\x{281b}\x{2811}\x{2830}\x{280e}"
  ],
  [
    0,
    "\x{281d}\x{2801}\x{280d}\x{2811}"
  ],
  [
    0,
    "\x{283a}\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{281b}\x{2815}\x{2815}\x{2819}"
  ],
  [
    0,
    "\x{2825}\x{280f}\x{2815}\x{281d}"
  ],
  [
    0,
    "\x{2830}\x{2861}\x{2801}\x{281d}\x{281b}\x{2811}\x{2802}"
  ],
  [
    0,
    "\x{280b}\x{2815}\x{2817}"
  ],
  [
    0,
    "\x{2801}\x{281d}\x{2839}\x{2839}\x{2814}\x{281b}"
  ],
  [
    0,
    "\x{2819}\x{2811}"
  ],
  [
    0,
    "\x{2821}\x{2815}\x{280e}\x{2811}"
  ],
  [
    0,
    "\x{281e}\x{2815}"
  ],
  [
    0,
    "\x{280f}\x{2825}\x{281e}"
  ],
  [
    0,
    "\x{2819}\x{280a}\x{280e}"
  ],
  [
    0,
    "\x{2819}\x{2801}\x{281d}\x{2819}"
  ],
  [
    0,
    "\x{281e}\x{2815}\x{2832}"
  ],
  [
    0,
    "\x{2855}\x{2807}\x{2819}"
  ],
  [
    0,
    "\x{284d}\x{281c}\x{2807}\x{2811}\x{2839}"
  ],
  [
    0,
    "\x{283a}\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{2819}\x{2811}\x{2801}\x{2819}"
  ],
  [
    0,
    "\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{2801}"
  ],
  [
    0,
    "\x{2819}\x{2815}\x{2815}\x{2817}\x{2824}\x{281d}\x{2801}\x{280a}\x{2807}\x{2832}"
  ],
  [
    0,
    "\x{284d}\x{2814}\x{2819}\x{2816}"
  ],
  [
    0,
    "\x{284a}"
  ],
  [
    0,
    "\x{2819}\x{2815}\x{281d}\x{2830}\x{281e}"
  ],
  [
    0,
    "\x{280d}\x{2811}\x{2801}\x{281d}"
  ],
  [
    0,
    "\x{281e}\x{2815}"
  ],
  [
    0,
    "\x{280e}\x{2801}\x{2839}"
  ],
  [
    0,
    "\x{2839}\x{2801}\x{281e}"
  ],
  [
    0,
    "\x{284a}"
  ],
  [
    0,
    "\x{2805}\x{281d}\x{282a}\x{2802}"
  ],
  [
    0,
    "\x{2815}\x{280b}"
  ],
  [
    0,
    "\x{280d}\x{2839}"
  ],
  [
    0,
    "\x{282a}\x{281d}"
  ],
  [
    0,
    "\x{2805}\x{281d}\x{282a}\x{2807}\x{282b}\x{281b}\x{2811}\x{2802}"
  ],
  [
    0,
    "\x{2831}\x{2801}\x{281e}"
  ],
  [
    0,
    "\x{2839}\x{283b}\x{2811}"
  ],
  [
    0,
    "\x{280a}\x{280e}"
  ],
  [
    0,
    "\x{280f}\x{281c}\x{281e}\x{280a}\x{280a}\x{2825}\x{2807}\x{281c}\x{2807}\x{2839}"
  ],
  [
    0,
    "\x{2819}\x{2811}\x{2801}\x{2819}"
  ],
  [
    0,
    "\x{2801}\x{2803}\x{2833}\x{281e}"
  ],
  [
    0,
    "\x{2801}"
  ],
  [
    0,
    "\x{2819}\x{2815}\x{2815}\x{2817}\x{2824}\x{281d}\x{2801}\x{280a}\x{2807}\x{2832}"
  ],
  [
    0,
    "\x{284a}"
  ],
  [
    0,
    "\x{280d}\x{280a}\x{2823}\x{281e}"
  ],
  [
    0,
    "\x{2819}\x{2801}\x{2827}\x{2811}"
  ],
  [
    0,
    "\x{2803}\x{2811}\x{2832}"
  ],
  [
    0,
    "\x{2814}\x{280a}\x{2807}\x{2814}\x{282b}\x{2802}"
  ],
  [
    0,
    "\x{280d}\x{2839}\x{280e}\x{2811}\x{2807}\x{280b}\x{2802}"
  ],
  [
    0,
    "\x{281e}\x{2815}"
  ],
  [
    0,
    "\x{2817}\x{2811}\x{281b}\x{281c}\x{2819}"
  ],
  [
    0,
    "\x{2801}"
  ],
  [
    0,
    "\x{280a}\x{2815}\x{280b}\x{280b}\x{2814}\x{2824}\x{281d}\x{2801}\x{280a}\x{2807}"
  ],
  [
    0,
    "\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{2819}\x{2811}\x{2801}\x{2819}\x{2811}\x{280c}"
  ],
  [
    0,
    "\x{280f}\x{280a}\x{2811}\x{280a}\x{2811}"
  ],
  [
    0,
    "\x{2815}\x{280b}"
  ],
  [
    0,
    "\x{280a}\x{2817}\x{2815}\x{281d}\x{280d}\x{2815}\x{281d}\x{281b}\x{283b}\x{2839}"
  ],
  [
    0,
    "\x{2814}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{281e}\x{2817}\x{2801}\x{2819}\x{2811}\x{2832}"
  ],
  [
    0,
    "\x{2843}\x{2825}\x{281e}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{283a}\x{280a}\x{280e}\x{2819}\x{2815}\x{280d}"
  ],
  [
    0,
    "\x{2815}\x{280b}"
  ],
  [
    0,
    "\x{2833}\x{2817}"
  ],
  [
    0,
    "\x{2801}\x{281d}\x{280a}\x{2811}\x{280c}\x{2815}\x{2817}\x{280e}"
  ],
  [
    0,
    "\x{280a}\x{280e}"
  ],
  [
    0,
    "\x{2814}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{280e}\x{280a}\x{280d}\x{280a}\x{2807}\x{2811}\x{2806}"
  ],
  [
    0,
    "\x{2801}\x{281d}\x{2819}"
  ],
  [
    0,
    "\x{280d}\x{2839}"
  ],
  [
    0,
    "\x{2825}\x{281d}\x{2819}\x{2801}\x{2807}\x{2807}\x{282a}\x{282b}"
  ],
  [
    0,
    "\x{2819}\x{2801}\x{281d}\x{2819}\x{280e}"
  ],
  [
    0,
    "\x{2829}\x{2801}\x{2807}\x{2807}"
  ],
  [
    0,
    "\x{281d}\x{2815}\x{281e}"
  ],
  [
    0,
    "\x{2819}\x{280a}\x{280c}\x{2825}\x{2817}\x{2803}"
  ],
  [
    0,
    "\x{280a}\x{281e}\x{2802}"
  ],
  [
    0,
    "\x{2815}\x{2817}"
  ],
  [
    0,
    "\x{2839}\x{2811}"
  ],
  [
    0,
    "\x{284a}\x{2833}\x{281d}\x{281e}\x{2817}\x{2839}\x{2830}\x{280e}"
  ],
  [
    0,
    "\x{2819}\x{2815}\x{281d}\x{2811}"
  ],
  [
    0,
    "\x{280b}\x{2815}\x{2817}\x{2832}"
  ],
  [
    0,
    "\x{2879}\x{2833}"
  ],
  [
    0,
    "\x{283a}\x{280a}\x{2807}\x{2807}"
  ],
  [
    0,
    "\x{2839}\x{283b}\x{2811}\x{280b}\x{2815}\x{2817}\x{2811}"
  ],
  [
    0,
    "\x{280f}\x{283b}\x{280d}\x{280a}\x{281e}"
  ],
  [
    0,
    "\x{280d}\x{2811}"
  ],
  [
    0,
    "\x{281e}\x{2815}"
  ],
  [
    0,
    "\x{2817}\x{2811}\x{280f}\x{2811}\x{2801}\x{281e}\x{2802}"
  ],
  [
    0,
    "\x{2811}\x{280d}\x{280f}\x{2819}\x{2801}\x{281e}\x{280a}\x{280a}\x{2801}\x{2807}\x{2807}\x{2839}\x{2802}"
  ],
  [
    0,
    "\x{2839}\x{2801}\x{281e}"
  ],
  [
    0,
    "\x{284d}\x{281c}\x{2807}\x{2811}\x{2839}"
  ],
  [
    0,
    "\x{283a}\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{2819}\x{2811}\x{2801}\x{2819}"
  ],
  [
    0,
    "\x{2801}\x{280e}"
  ],
  [
    0,
    "\x{2801}"
  ],
  [
    0,
    "\x{2819}\x{2815}\x{2815}\x{2817}\x{2824}\x{281d}\x{2801}\x{280a}\x{2807}\x{2832}"
  ],
  [
    1,
    '('
  ],
  [
    0,
    'the'
  ],
  [
    0,
    'first'
  ],
  [
    0,
    'couple'
  ],
  [
    0,
    'of'
  ],
  [
    0,
    'paragraphs'
  ],
  [
    0,
    'of'
  ],
  [
    1,
    '"'
  ],
  [
    0,
    'A'
  ],
  [
    0,
    'christmas'
  ],
  [
    0,
    'carol'
  ],
  [
    2,
    '"'
  ],
  [
    0,
    'by'
  ],
  [
    0,
    'dickens'
  ],
  [
    2,
    ')'
  ],
  [
    0,
    'compact'
  ],
  [
    0,
    'font'
  ],
  [
    0,
    'selection'
  ],
  [
    0,
    'example'
  ],
  [
    0,
    'text'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    'ABCDEFGHIJKLMNOPQRSTUVWXYZ'
  ],
  [
    1,
    '/'
  ],
  [
    0,
    '0123456789'
  ],
  [
    0,
    'abcdefghijklmnopqrstuvwxyz'
  ],
  [
    1,
    "\x{a3}\x{a9}"
  ],
  [
    0,
    "\x{b5}\x{c0}\x{c6}\x{d6}\x{de}\x{df}\x{e9}\x{f6}\x{ff}"
  ],
  [
    1,
    "\x{2013}\x{2014}\x{2018}\x{201c}\x{201d}\x{201e}\x{2020}\x{2022}\x{2026}\x{2030}\x{2122}"
  ],
  [
    0,
    "\x{153}\x{160}\x{178}\x{17e}"
  ],
  [
    2,
    "\x{20ac}"
  ],
  [
    0,
    "\x{3b1}\x{3b2}\x{3b3}\x{3b4}\x{3c9}\x{3b1}\x{3b2}\x{3b3}\x{3b4}\x{3c9}"
  ],
  [
    0,
    "\x{430}\x{431}\x{432}\x{433}\x{434}\x{430}\x{431}\x{432}\x{433}\x{434}"
  ],
  [
    1,
    "\x{2200}\x{2202}\x{2208}"
  ],
  [
    0,
    "\x{211d}"
  ],
  [
    2,
    "\x{2227}\x{222a}\x{2261}\x{221e}"
  ],
  [
    0,
    "\x{2191}\x{2197}\x{21a8}\x{21bb}\x{21e3}"
  ],
  [
    0,
    "\x{2510}\x{253c}\x{2554}\x{2558}\x{2591}\x{25ba}\x{263a}\x{2640}"
  ],
  [
    0,
    "\x{fb01}"
  ],
  [
    3,
    "\x{fffd}\x{2440}\x{2082}"
  ],
  [
    0,
    "\x{1f20}\x{1e02}\x{4e5}\x{1e84}\x{250}\x{2d0}"
  ],
  [
    3,
    "\x{234e}"
  ],
  [
    0,
    "\x{5d0}\x{531}\x{10d0}"
  ],
  [
    0,
    'greetings'
  ],
  [
    0,
    'in'
  ],
  [
    0,
    'various'
  ],
  [
    0,
    'languages'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    'hello'
  ],
  [
    0,
    'world'
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{3ba}\x{3b1}\x{3bb}\x{3b7}\x{3bc}\x{1f73}\x{3c1}\x{3b1}"
  ],
  [
    0,
    "\x{3ba}\x{1f79}\x{3c3}\x{3bc}\x{3b5}"
  ],
  [
    2,
    ','
  ],
  [
    0,
    "\x{30b3}\x{30f3}\x{30cb}\x{30c1}\x{30cf}"
  ],
  [
    0,
    'box'
  ],
  [
    0,
    'drawing'
  ],
  [
    0,
    'alignment'
  ],
  [
    0,
    'tests'
  ],
  [
    2,
    ':'
  ],
  [
    0,
    "\x{2588}"
  ],
  [
    0,
    "\x{2589}"
  ],
  [
    0,
    "\x{2554}\x{2550}\x{2550}\x{2566}\x{2550}\x{2550}\x{2557}"
  ],
  [
    0,
    "\x{250c}\x{2500}\x{2500}\x{252c}\x{2500}\x{2500}\x{2510}"
  ],
  [
    0,
    "\x{256d}\x{2500}\x{2500}\x{252c}\x{2500}\x{2500}\x{256e}"
  ],
  [
    0,
    "\x{256d}\x{2500}\x{2500}\x{252c}\x{2500}\x{2500}\x{256e}"
  ],
  [
    0,
    "\x{250f}\x{2501}\x{2501}\x{2533}\x{2501}\x{2501}\x{2513}"
  ],
  [
    0,
    "\x{250e}\x{2512}\x{250f}\x{2511}"
  ],
  [
    0,
    "\x{2577}"
  ],
  [
    0,
    "\x{257b}"
  ],
  [
    0,
    "\x{250f}\x{252f}\x{2513}"
  ],
  [
    0,
    "\x{250c}\x{2530}\x{2510}"
  ],
  [
    0,
    "\x{258a}"
  ],
  [
    0,
    "\x{2571}\x{2572}\x{2571}\x{2572}\x{2573}\x{2573}\x{2573}"
  ],
  [
    0,
    "\x{2551}\x{250c}\x{2500}\x{2568}\x{2500}\x{2510}\x{2551}"
  ],
  [
    0,
    "\x{2502}\x{2554}\x{2550}\x{2567}\x{2550}\x{2557}\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2552}\x{2550}\x{256a}\x{2550}\x{2555}\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2553}\x{2500}\x{2541}\x{2500}\x{2556}\x{2502}"
  ],
  [
    0,
    "\x{2503}\x{250c}\x{2500}\x{2542}\x{2500}\x{2510}\x{2503}"
  ],
  [
    0,
    "\x{2517}\x{2543}\x{2544}\x{2519}"
  ],
  [
    0,
    "\x{2576}\x{253c}\x{2574}\x{257a}\x{254b}\x{2578}\x{2520}\x{253c}\x{2528}"
  ],
  [
    0,
    "\x{251d}\x{254b}\x{2525}"
  ],
  [
    0,
    "\x{258b}"
  ],
  [
    0,
    "\x{2572}\x{2571}\x{2572}\x{2571}\x{2573}\x{2573}\x{2573}"
  ],
  [
    0,
    "\x{2551}\x{2502}\x{2572}"
  ],
  [
    0,
    "\x{2571}\x{2502}\x{2551}"
  ],
  [
    0,
    "\x{2502}\x{2551}"
  ],
  [
    0,
    "\x{2551}\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2502}"
  ],
  [
    0,
    "\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2551}"
  ],
  [
    0,
    "\x{2503}"
  ],
  [
    0,
    "\x{2551}\x{2502}"
  ],
  [
    0,
    "\x{2503}\x{2502}"
  ],
  [
    0,
    "\x{257f}"
  ],
  [
    0,
    "\x{2502}\x{2503}"
  ],
  [
    0,
    "\x{250d}\x{2545}\x{2546}\x{2513}"
  ],
  [
    0,
    "\x{2575}"
  ],
  [
    0,
    "\x{2579}"
  ],
  [
    0,
    "\x{2517}\x{2537}\x{251b}"
  ],
  [
    0,
    "\x{2514}\x{2538}\x{2518}"
  ],
  [
    0,
    "\x{258c}"
  ],
  [
    0,
    "\x{2571}\x{2572}\x{2571}\x{2572}\x{2573}\x{2573}\x{2573}"
  ],
  [
    0,
    "\x{2560}\x{2561}"
  ],
  [
    0,
    "\x{2573}"
  ],
  [
    0,
    "\x{255e}\x{2563}"
  ],
  [
    0,
    "\x{251c}\x{2562}"
  ],
  [
    0,
    "\x{255f}\x{2524}"
  ],
  [
    0,
    "\x{251c}\x{253c}\x{2500}\x{253c}\x{2500}\x{253c}\x{2524}"
  ],
  [
    0,
    "\x{251c}\x{256b}\x{2500}\x{2542}\x{2500}\x{256b}\x{2524}"
  ],
  [
    0,
    "\x{2523}\x{253f}\x{257e}\x{253c}\x{257c}\x{253f}\x{252b}"
  ],
  [
    0,
    "\x{2515}\x{251b}\x{2516}\x{251a}"
  ],
  [
    0,
    "\x{250c}\x{2504}\x{2504}\x{2510}"
  ],
  [
    0,
    "\x{254e}"
  ],
  [
    0,
    "\x{250f}\x{2505}\x{2505}\x{2513}"
  ],
  [
    0,
    "\x{250b}"
  ],
  [
    0,
    "\x{258d}"
  ],
  [
    0,
    "\x{2572}\x{2571}\x{2572}\x{2571}\x{2573}\x{2573}\x{2573}"
  ],
  [
    0,
    "\x{2551}\x{2502}\x{2571}"
  ],
  [
    0,
    "\x{2572}\x{2502}\x{2551}"
  ],
  [
    0,
    "\x{2502}\x{2551}"
  ],
  [
    0,
    "\x{2551}\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2502}"
  ],
  [
    0,
    "\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2551}"
  ],
  [
    0,
    "\x{2503}"
  ],
  [
    0,
    "\x{2551}\x{2502}"
  ],
  [
    0,
    "\x{2503}\x{2502}"
  ],
  [
    0,
    "\x{257d}"
  ],
  [
    0,
    "\x{2502}\x{2503}"
  ],
  [
    0,
    "\x{2591}\x{2591}\x{2592}\x{2592}\x{2593}\x{2593}\x{2588}\x{2588}"
  ],
  [
    0,
    "\x{250a}"
  ],
  [
    0,
    "\x{2506}"
  ],
  [
    0,
    "\x{254e}"
  ],
  [
    0,
    "\x{254f}"
  ],
  [
    0,
    "\x{2507}"
  ],
  [
    0,
    "\x{250b}"
  ],
  [
    0,
    "\x{258e}"
  ],
  [
    0,
    "\x{2551}\x{2514}\x{2500}\x{2565}\x{2500}\x{2518}\x{2551}"
  ],
  [
    0,
    "\x{2502}\x{255a}\x{2550}\x{2564}\x{2550}\x{255d}\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2558}\x{2550}\x{256a}\x{2550}\x{255b}\x{2502}"
  ],
  [
    0,
    "\x{2502}\x{2559}\x{2500}\x{2540}\x{2500}\x{255c}\x{2502}"
  ],
  [
    0,
    "\x{2503}\x{2514}\x{2500}\x{2542}\x{2500}\x{2518}\x{2503}"
  ],
  [
    0,
    "\x{2591}\x{2591}\x{2592}\x{2592}\x{2593}\x{2593}\x{2588}\x{2588}"
  ],
  [
    0,
    "\x{250a}"
  ],
  [
    0,
    "\x{2506}"
  ],
  [
    0,
    "\x{254e}"
  ],
  [
    0,
    "\x{254f}"
  ],
  [
    0,
    "\x{2507}"
  ],
  [
    0,
    "\x{250b}"
  ],
  [
    0,
    "\x{258f}"
  ],
  [
    0,
    "\x{255a}\x{2550}\x{2550}\x{2569}\x{2550}\x{2550}\x{255d}"
  ],
  [
    0,
    "\x{2514}\x{2500}\x{2500}\x{2534}\x{2500}\x{2500}\x{2518}"
  ],
  [
    0,
    "\x{2570}\x{2500}\x{2500}\x{2534}\x{2500}\x{2500}\x{256f}"
  ],
  [
    0,
    "\x{2570}\x{2500}\x{2500}\x{2534}\x{2500}\x{2500}\x{256f}"
  ],
  [
    0,
    "\x{2517}\x{2501}\x{2501}\x{253b}\x{2501}\x{2501}\x{251b}"
  ],
  [
    0,
    "\x{2597}\x{2584}\x{2596}\x{259b}\x{2580}\x{259c}"
  ],
  [
    0,
    "\x{2514}\x{254c}\x{254c}\x{2518}"
  ],
  [
    0,
    "\x{254e}"
  ],
  [
    0,
    "\x{2517}\x{254d}\x{254d}\x{251b}"
  ],
  [
    0,
    "\x{250b}"
  ],
  [
    0,
    "\x{2581}\x{2582}\x{2583}\x{2584}\x{2585}\x{2586}\x{2587}\x{2588}"
  ],
  [
    0,
    "\x{259d}\x{2580}\x{2598}\x{2599}\x{2584}\x{259f}"
  ]
]
