package Prty::JQuery::Form::ViewEdit;
use base qw/Prty::Hash/;

use strict;
use warnings;
use utf8;

our $VERSION = 1.087;

use Prty::Hash;
use Prty::Unindent;
use Prty::Html::Widget::Button;
use Prty::Html::Widget::CheckBox;
use Prty::Html::Form::Layout;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::JQuery::Form::ViewEdit - Formular zum Ansehen und Bearbeiten von persistenten Daten

=head1 BASE CLASS

L<Prty::Hash>

=head1 DESCRIPTION

Die Klasse implementiert ein jQuery UI Plugin zum Ansehen und
Bearbeiten von persistenten Daten, die typischerweise aus
einer Datenbank stammen.

Das Layout muss die Platzhalter __SAVE__, __DELETE__ und __EDIT__
enthalten. Für diese werden intern drei Widgets generiert: für
__SAVE__ und __DELETE__ ein Button zum Speichern bzw. Löschen, für
__EDIT__ eine Checkbox zum Umschalten zwischen Ansehen und Bearbeiten.

Bei Betätigung einer der beiden Button werden die Formulardaten
an den action-URL gepostet. Im Erfolgsfall wird anschließend
die onSuccess-Methode aufgerufen.

=head2 CSS-Klassen

=over 2

=item *

editCheckbox

=item *

saveButton

=item *

deleteButton

=item *

enabled

=item *

disabled

=back

=head2 Beschriftungen

=over 2

=item *

Speichern

=item *

Löschen

=item *

Bearbeiten

=back

=head1 ATTRIBUTES

=over 4

=item action => $url (Default: undef)

URL, an den die Daten bei bei Betätigung des Save- oder
des Delete-Buttons geschickt werden.

=item id (Default: undef)

Die DOM-Id des Formulars.

=item instantiate => $bool (Default: 0)

Füge die Plugin-Instantiierung beim Aufruf von html()
zum HTML-Code hinzu.

=item layout => $html (Default: '')

Der HTML-Code des Layouts. In das Layout wird der HTML-Code der
Widgets eingesetzt.

=item onSucces => $javaScript (Default: undef)

JavaScript-Methode, die nach einem erfolgreichen Ajax-Aufruf
ausgeführt wird. Parameter: onSuccess(data,textStatus,jqXHR,op),
wobei op 'save' oder 'delete' ist.

=item state => 'update' | 'insert' (Default: 'update')

Anfänglicher Zusatand des Formulars:

=over 4

=item 'update'

Der Save- und der Delete-Button werden im Edit-Modus freigeschaltet.

=item 'insert'

Nur der Save-Button wird im Edit-Modus freigeschaltet.

=back

=item text => \@keyVal (Default: s. Text)

Die Beschriftungen der intern generierten Widgets:

    saveButton=>'Speichern',
    deleteButton=>'Löschen',
    editCheckBox=>'Bearbeiten',

=item widgets => \@widgets (Default: [])

Liste der Widgets, die in das Layout eingesetzt werden.

=back

=head1 EXAMPLE

    $html = Prty::JQuery::Form::ViewEdit->html($h,
        instantiate=>1,
        id=>'basisdatenForm',
        state=>'insert',
        action=>$c->url_for('/mandant/speichern'),
        onSuccess=>q|
            function () {
                var d = new Date;
                var date = d.getFullYear()+'-'
                    +('0'+(d.getMonth()+1)).slice(-2)+'-'
                    +('0'+d.getDate()).slice(-2)+' '
                    +('0'+d.getHours()).slice(-2)+':'
                    +('0'+d.getMinutes()).slice(-2)+':'
                    +('0'+d.getSeconds()).slice(-2);
                $('input[name=zeit]').val(date);
            }
        |,
        text=>{
            saveButton=>'Speichern',
            deleteButton=>'Löschen',
            editCheckbox=>'Bearbeiten',
        },
        layout=>$h->cat(
            Prty::Html::Table::Simple->html($h,
                class=>'form',
                rows=>[
                    ['form-section',[colspan=>2,'Basisdaten']],
                    ['form-widget',['Mandanten-Id:'],['__MANID__']],
                    ['form-widget',['Mandanten-Kürzel:'],['__KUERZEL__']],
                    ['form-widget',['Mandanten-Parent:'],['__PARENT__']],
                    ['form-widget',['Mandant/Firma:'],['__NAMEINTERN__']],
                    ['form-widget',['Zusatz:'],['__ZUSATZ__']],
                    ['form-widget',['Portal:'],['__PORTALNAME__']],
                    ['form-widget',['Faktura-Mandant:'],
                        ['__FAKTURAMANDANT__']],
                    ['form-widget',['Deaktiviert:'],['__INAKTIV__']],
                    ['form-widget',['Austausch-Kennwort:'],
                        ['__AUSTAUSCHKENNWORT__']],
                ],
            ),
            Prty::Html::Table::Simple->html($h,
                class=>'form',
                rows=>[
                    [['__SAVE__ __DELETE__ __EDIT__']],
                ],
            ),
        ),
        widgets=>[
            Prty::Html::Widget::Hidden->new(
                name=>'formular',
                value=>'basisdaten',
            ),
            Prty::Html::Widget::Hidden->new(
                name=>'zeit',
                value=>$zeit,
            ),
            Prty::Html::Widget::ReadOnly->new(
                name=>'manId',
                value=>$man->id,
            ),
            Prty::Html::Widget::TextField->new(
                name=>'kuerzel',
                size=>10,
                value=>$man->kuerzel,
            ),
            Prty::Html::Widget::SelectMenu->new(
                name=>'parent',
                options=>[$tab->values('id')],
                texts=>[$tab->values('kuerzelPfad')],
                value=>$man->parent,
            ),
            Prty::Html::Widget::TextField->new(
                name=>'nameintern',
                size=>45,
                maxLength=>100,
                value=>$man->nameintern,
            ),
            Prty::Html::Widget::TextField->new(
                name=>'zusatz',
                size=>45,
                maxLength=>4000,
                value=>$man->zusatz,
            ),
            Prty::Html::Widget::TextField->new(
                name=>'portalname',
                size=>45,
                maxLength=>50,
                value=>$man->portalname,
            ),
            Prty::Html::Widget::CheckBox->new(
                name=>'fakturamandant',
                option=>1,
                value=>$man->fakturamandant,
            ),
            Prty::Html::Widget::CheckBox->new(
                name=>'inaktiv',
                option=>1,
                value=>$man->inaktiv,
            ),
            Prty::Html::Widget::TextField->new(
                name=>'austauschkennwort',
                size=>15,
                maxLength=>30,
                value=>$man->austauschkennwort,
            ),
        ],
    );

=head1 METHODS

=head2 Konstruktor

=head3 new() - Instantiiere Objekt

=head4 Synopsis

    $e = $class->new(@keyVal);

=head4 Description

Instantiiere ein Formular-Objekt und liefere eine Referenz auf
dieses Objekt zurück.

=cut

# -----------------------------------------------------------------------------

sub new {
    my $class = shift;
    # @_: @keyVal

    my $self = $class->SUPER::new(
        action=>undef,
        id=>undef,
        instantiate=>0,
        layout=>'',
        onSuccess=>undef,
        state=>'update',
        text=>Prty::Hash->new(
            saveButton=>'Speichern',
            deleteButton=>'Löschen',
            editCheckbox=>'Bearbeiten',
        ),
        widgets=>[],
    );

    while (@_) {
        my $key = shift;
        if ($key eq 'text') {
            $self->get($key)->join(shift);
            next;
        }
        $self->set($key=>shift);
    }

    return $self;
}

# -----------------------------------------------------------------------------

=head2 Klassenmethoden

=head3 pluginCode() - JavaScript-Code des Plugin

=head4 Synopsis

    $javascript = $e->pluginCode;

=head4 Description

Liefere den JavaScript-Code des jQuery UI Plugin. Dieser Code
kann auf einer HTML-Seite inline verwendet oder - besser -
vom Webserver ausgeliefert werden.

=cut

# -----------------------------------------------------------------------------

sub pluginCode {
    my $this = shift;

    return Prty::Unindent->hereDoc(q~
    $.widget('prty.viewEditForm',{
        options: {
            action: null,
            onSuccess: null,
            state: null,
        },
        editCheckbox: null,
        widgets: null,
        _create: function () {
            this.element.addClass('viewEditForm');

            this._on(this.element,{
                'click .editCheckbox': function (event) {
                    this.render();
                },
                'click .saveButton': function (event) {
                    this.execute('save');
                },
                'click .deleteButton': function (event) {
                    this.execute('delete');
                },
            });

            this.editCheckbox = $('.editCheckbox',this.element);

            var editCheckbox = this.editCheckbox.get(0);
            this.widgets = $(':input',this.element).filter(function (i) {
                return this != editCheckbox;
            });

            this.render();
        },
        render: function () {
            var edit = this.editCheckbox.is(':checked');

            if (edit) {
                // Widgets enablen

                this.widgets.prop('disabled',false);
                this.widgets.removeClass('disabled');
                this.widgets.addClass('enabled');

                if (this.options.state == 'insert') {
                    // Im Insert-State wird die Delete-Operation
                    // nicht angeboten

                    var $deleteButton = $('.deleteButton');
                    $deleteButton.prop('disabled',true);
                    $deleteButton.removeClass('enabled');
                    $deleteButton.addClass('disabled');
                }
            }
            else {
                // Widgets disablen

                this.widgets.prop('disabled',true);
                this.widgets.removeClass('enabled');
                this.widgets.addClass('disabled');
            }
        },
        execute: function (op) {
            var instance = this;
            var url = this.options.action;
            var data = 'op='+op+'&'+$(':input',this.element).serialize();

            $.ajax({
                url: url,
                type: 'POST',
                data: data,
                async: false,
                success: function (data,textStatus,jqXHR) {
                    if (data) {
                        // Fehlermeldung
                        alert(data);
                        return;
                    }
                    instance.editCheckbox.prop('checked',false);
                    if (instance.options.onSuccess)
                        instance.options.onSuccess(data,textStatus,jqXHR,op);
                    instance.render();
                },
                error: function () {
                    alert('FEHLER: Speichern fehlgeschlagen');
                },
            });
        },
    });
    ~);
}

# -----------------------------------------------------------------------------

=head2 Objektmethoden

=head3 html() - Generiere HTML für die Widget-Instanz

=head4 Synopsis

    $html = $e->html($h);
    $html = $class->html($h,@keyVal);

=head4 Description

Generiere den HTML-Code des Formular-Objekts und liefere diesen
zurück. Als Klassenmethode gerufen, wird das Objekt intern erzeugt
und mit den Attributen @keyVal instantiiert.

=cut

# -----------------------------------------------------------------------------

sub html {
    my $this = shift;
    my $h = shift;

    my $self = ref $this? $this: $this->new(@_);

    my ($id,$instantiate,$layout,$state,$textH,$widgetA) = $self->get(qw/id
        instantiate layout state text widgets/);

    # Liste der Widgets kopieren und Save-, Delete-, Edit-Widgets hinzufügen

    my @widgets = @$widgetA;
    push @widgets,Prty::Html::Widget::Button->new(
        class=>'saveButton',
        name=>'save',
        content=>$textH->get('saveButton'),
    );
    push @widgets,Prty::Html::Widget::Button->new(
        class=>'deleteButton',
        name=>'delete',
        content=>$textH->get('deleteButton'),
    );
    push @widgets,Prty::Html::Widget::CheckBox->new(
        class=>'editCheckbox',
        name=>'edit',
        label=>$textH->get('editCheckbox'),
    );

    my $html = $h->tag('div',
        id=>$id,
        Prty::Html::Form::Layout->html($h,
            layout=>$layout,
            widgets=>\@widgets,
        ),
    );

    if ($instantiate) {
        $html .= $h->tag('script',
            $self->instantiate,
        );
    }

    return $html;
}

# -----------------------------------------------------------------------------

=head3 instantiate() - JavaScript-Code, der das Widget instantiiert

=head4 Synopsis

    $javaScript = $e->instantiate;

=cut

# -----------------------------------------------------------------------------

sub instantiate {
    my $self = shift;

    my ($action,$id,$onSuccess,$state) = $self->get(qw/action id onSuccess
        state/);

    my @att;
    if ($state) {
        push @att,"state: '$state'";
    }
    if ($action) {
        push @att,"action: '$action'";
    }
    if ($onSuccess) {
        $onSuccess = Prty::Unindent->trim($onSuccess);
        $onSuccess =~ s/\n/\n    /g;
        push @att,"onSuccess: $onSuccess";
    }

    return sprintf q|$('#%s').viewEditForm({%s});|,$id,
        "\n    ".join(",\n    ",@att)."\n";
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.087

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
