=head1 NAME

CGI::WPM::Base - Perl module that defines the API for subclasses, which are
miniature applications called "web page makers", and provides them with a
hierarchical environment that handles details for obtaining program settings,
resolving file system or web site contexts, obtaining user input, and sending new
web pages to the user.

=cut

######################################################################

package CGI::WPM::Base;
require 5.004;

# Copyright (c) 1999-2000, Darren R. Duncan. All rights reserved. This module is
# free software; you can redistribute it and/or modify it under the same terms as
# Perl itself.  However, I do request that this copyright information remain
# attached to the file.  If you modify this module and redistribute a changed
# version then please attach a note listing the modifications.

use strict;
use vars qw($VERSION);
$VERSION = '0.2';

######################################################################

=head1 DEPENDENCIES

=head2 Perl Version

	5.004

=head2 Standard Modules

	I<none>

=head2 Nonstandard Modules

	CGI::WPM::Content
	CGI::WPM::Globals

=cut

######################################################################

use CGI::WPM::Content;
use CGI::WPM::Globals;

######################################################################
# Names of properties for objects of this class are declared here:
my $KEY_SITE_GLOBALS = 'site_globals';  # hold global site values
my $KEY_PAGE_CONTENT = 'page_content';  # hold return values

# Keys for items in site global preferences:
my $PKEY_AMEND_MSG = 'amend_msg';  # personalized html appears on error page

# Keys for items in site page preferences:
my $PKEY_PAGE_HEADER = 'page_header'; # content goes above our subclass's
my $PKEY_PAGE_FOOTER = 'page_footer'; # content goes below our subclass's
my $PKEY_PAGE_TITLE = 'page_title';  # title for this document
my $PKEY_PAGE_AUTHOR = 'page_author';  # author for this document
my $PKEY_PAGE_META = 'page_meta';  # meta tags for this document
my $PKEY_PAGE_CSS_SRC = 'page_css_src';  # stylesheet urls to link in
my $PKEY_PAGE_CSS_CODE = 'page_css_code';  # css code to embed in head
my $PKEY_PAGE_BODY_ATTR = 'page_body_attr';  # params to put in <BODY>
my $PKEY_PAGE_REPLACE = 'page_replace';  # replacements to perform

######################################################################

sub new {
	my $class = shift( @_ );
	my $self = {};
	bless( $self, ref($class) || $class );
	$self->initialize( @_ );
	return( $self );
}

######################################################################

sub initialize {
	my ($self, $globals) = @_;

	ref($globals) eq 'CGI::WPM::Globals' or 
		die "initializer is not a valid CGI::WPM::Globals object";

	%{$self} = (
		$KEY_SITE_GLOBALS => $globals,
	);

	$self->_initialize( @_ );
}

# subclass should have their own of these, if needed
sub _initialize {
}

######################################################################

sub dispatch_by_user {
	my $self = shift( @_ );
	if( $self->{$KEY_SITE_GLOBALS}->get_error() ) {  # prefs not open
		$self->_set_to_init_error_page();
		return( 0 );
	}
	return( $self->_dispatch_by_user( @_ ) );
}

# subclass should have their own of these
sub _dispatch_by_user {
	my $self = shift( @_ );
	my $webpage = CGI::WPM::Content->new();

	$webpage->title( 'Web Page For Users' );

	$webpage->body_content( <<__endquote );
<H2 ALIGN="center">@{[$webpage->title()]}</H2>

<P>This web page has been generated by CGI::WPM::Base, which is 
copyright (c) 1999-2000, Darren R. Duncan.  This Perl Class 
is intended to be subclassed before it is used.</P>

<P>You are reading this message because either no subclass is in use 
or that subclass hasn't declared the _dispatch_by_user() method, 
which is required to generate the web pages that normal visitors 
would see.</P>
__endquote

	$self->{$KEY_PAGE_CONTENT} = $webpage;
}

######################################################################

sub dispatch_by_admin {
	my $self = shift( @_ );
	if( $self->{$KEY_SITE_GLOBALS}->get_error() ) {  # prefs not open
		$self->_set_to_init_error_page();
		return( 0 );
	}
	return( $self->_dispatch_by_admin( @_ ) );
}

# subclass should have their own of these, if needed
sub _dispatch_by_admin {
	my $self = shift( @_ );
	my $webpage = CGI::WPM::Content->new();

	$webpage->title( 'Web Page For Administrators' );

	$webpage->body_content( <<__endquote );
<H2 ALIGN="center">@{[$webpage->title()]}</H2>

<P>This web page has been generated by CGI::WPM::Base, which is 
copyright (c) 1999-2000, Darren R. Duncan.  This Perl Class 
is intended to be subclassed before it is used.</P>

<P>You are reading this message because either no subclass is in use 
or that subclass hasn't declared the _dispatch_by_admin() method, 
which is required to generate the web pages that site administrators 
would use to administrate site content using their web browsers.</P>
__endquote

	$self->{$KEY_PAGE_CONTENT} = $webpage;
}

######################################################################

sub get_page_content {   # should be called after "dispatch" methods
	my $self = shift( @_ );
	my $webpage = $self->{$KEY_PAGE_CONTENT} ||= CGI::WPM::Content->new();
	my $rh_prefs = $self->{$KEY_SITE_GLOBALS}->site_prefs();
		# note that we don't see parent prefs here, only current level

	$webpage->body_prepend( $rh_prefs->{$PKEY_PAGE_HEADER} );
	$webpage->body_append( $rh_prefs->{$PKEY_PAGE_FOOTER} );

	$webpage->title() or $webpage->title( $rh_prefs->{$PKEY_PAGE_TITLE} );
	$webpage->author() or $webpage->author( $rh_prefs->{$PKEY_PAGE_AUTHOR} );
	
	if( ref( my $rh_meta = $rh_prefs->{$PKEY_PAGE_META} ) eq 'HASH' ) {
		@{$webpage->meta()}{keys %{$rh_meta}} = values %{$rh_meta};
	}	

	if( defined( my $css_urls_pref = $rh_prefs->{$PKEY_PAGE_CSS_SRC} ) ) {
		push( @{$webpage->style_sources()}, 
			ref($css_urls_pref) eq 'ARRAY' ? @{$css_urls_pref} : () );
	}
	if( defined( my $css_code_pref = $rh_prefs->{$PKEY_PAGE_CSS_CODE} ) ) {
		push( @{$webpage->style_code()}, 
			ref($css_code_pref) eq 'ARRAY' ? @{$css_code_pref} : () );
	}

	if( ref(my $rh_body = $rh_prefs->{$PKEY_PAGE_BODY_ATTR}) eq 'HASH' ) {
		@{$webpage->body_attributes()}{keys %{$rh_body}} = 
			values %{$rh_body};
	}	

	$webpage->add_later_replace( $rh_prefs->{$PKEY_PAGE_REPLACE} );

	$self->_get_page_content();
	return( $webpage );  # returns ref to original
}

# subclass should have their own of these, if needed
sub _get_page_content {
}

######################################################################

sub get_page_string {   # should be called after "dispatch" methods
	my $self = shift( @_ );
	return( $self->get_page_content()->to_string() );
}

######################################################################
# This is meant to be called after the global "is error" is set

sub _set_to_init_error_page {
	my $self = @_;
	my $globals = $self->{$KEY_SITE_GLOBALS};
	my $webpage = CGI::WPM::Content->new();

	$webpage->title( 'Error Initializing Page Maker' );

	$webpage->body_content( <<__endquote );
<H2 ALIGN="center">@{[$webpage->title()]}</H2>

<P>I'm sorry, but an error has occurred while trying to initialize 
a required program module, "@{[ref($self)]}".  The file that 
contains its preferences couldn't be opened.</P>  

@{[$self->_get_amendment_message()]}

<P>Details: @{[$globals->get_error()]}</P>
__endquote

	$self->{$KEY_PAGE_CONTENT} = $webpage;
}

######################################################################

sub _get_amendment_message {
	my $self = shift( @_ );
	my $globals = $self->{$KEY_SITE_GLOBALS};
	return( $globals->site_pref( $PKEY_AMEND_MSG ) || <<__endquote );
<P>This should be temporary, the result of a transient server problem
or a site update being performed at the moment.  Click 
@{[$globals->self_html('here')]} to automatically try again.  
If the problem persists, please try again later, or send an
@{[$globals->site_owner_email_html('e-mail')]}
message about the problem, so it can be fixed.</P>
__endquote
}

######################################################################

1;
__END__

=head1 AUTHOR

Copyright (c) 1999-2000, Darren R. Duncan. All rights reserved. This module is
free software; you can redistribute it and/or modify it under the same terms as
Perl itself.  However, I do request that this copyright information remain
attached to the file.  If you modify this module and redistribute a changed
version then please attach a note listing the modifications.

I am always interested in knowing how my work helps others, so if you put this
module to use in any of your own code then please send me the URL.  Also, if you
make modifications to the module because it doesn't work the way you need, please
send me a copy so that I can roll desirable changes into the main release.

Address comments, suggestions, and bug reports to B<perl@DarrenDuncan.net>.

=head1 SEE ALSO

perl(1).

=cut
