# ABSTRACT: A Perl Catalyst Utility class (to be a model) for Tumblr.
package Thorny::Tumblr;

use Moose;

use WWW::Tumblr;
use Date::Parse;
use Array::Shuffle qw(shuffle_array);

use namespace::autoclean;

=head1 NAME

Thorny::Tumblr - Catalyst Utility for Tumblr

=head1 DESCRIPTION

This class is a Tumblr utility class for Perl Catalyst.

Try the following in a Perl Catalyst class based on Tumblr:

use base 'Catalyst::Model::Adaptor';
__PACKAGE__->config(class => 'SpeedDateCanada::Util::Tumblr', args => { tag => 'content_hashtag', type => 'photo' }); 
 
Here's an example of the root page :

sub index :Path :Args(0) {
    my ( $self, $c) = @_;
    
    # create an instance of the Tumblr util
    my $jumbotrons = $c->model('Site::Jumbotrons')->get_tagged_posts;
    
    my $events = $c->model('Site::Events')->get_tagged_posts;
    
    my $testimonials = $c->model('Site::Testimonials')->get_tagged_posts;
    
    my $questions = $c->model('Site::Questions')->get_tagged_posts;
    
    my $abouts = $c->model('Site::Abouts')->get_tagged_posts;
    
    my @blockquotes = shuffle(@{$testimonials});
    my @stripped_blockquotes;    
    my $hs = HTML::Strip->new();
    for my $bq (@blockquotes) {
        push @stripped_blockquotes, {  voice => $hs->parse($bq->{text}),handle => $hs->parse($bq->{source}) };
    }
    my $stripped_blockquotes_ref = \@stripped_blockquotes;
    
    # root page from template toolkit
    $c->stash(template => 'root.tt', jumbotrons => $jumbotrons, blockquotes => $stripped_blockquotes_ref, events => $events, questions => $questions, abouts => $abouts);
}

=encoding utf8

=head1 AUTHOR

Jason McVeigh

=head1 LICENSE

This library is free software. You can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

has 'blog' => (
    is => 'rw',
    isa => 'Object',
);

has 'tumblr' => (
    is => 'rw',
    isa => 'Object',
);

has 'posts' => (
    is => 'rw',
    isa => 'ArrayRef',
);

has 'tagged' => (
    is => 'rw',
    isa => 'ArrayRef',
);

sub BUILD {
    my ($self,$args) = @_;
    
    $self->tumblr(WWW::Tumblr->new(
        consumer_key => '',
        secret_key => '',
        token => '',
        token_secret => '',
    ));
    
    $self->blog($self->tumblr->blog('speed-date-ca.tumblr.com'));    
    $self->posts($self->blog->posts(type => $args->{'type'})->{posts});
        
    my @t = @{$self->get_posts_by_tag($self->posts,$args->{'tag'})};
    
    $self->tagged(\@t);
}

sub get_posts {
    my ($self,$c) = @_;
    return($self->posts);
}

sub get_tagged_posts {
    my ($self,$c) = @_;
    return($self->tagged);
}

sub get_posts_by_tag {
    my $self = shift;
    my $c = shift;
    my $q = shift;
    
    my @posts;
    
    for my $post (@{$self->posts}) {        
        if (scalar(grep(/^$q$/, @{$post->{tags}})) > 0) {
            push @posts,$post;                    
        }
    }
    
    return(\@posts);
}

__PACKAGE__->meta->make_immutable;

1;
