# Spelling.pm - support for the `spell', `spellin' and `spellout' utilities.
# Copyright 1999 Nathan Scott Thompson ( quimby at city-net dot com )
# You may use this according to the GNU Public License: see http://www.gnu.org

use strict;
package Stems;

# My interpretation of Doug McIlroy's selected word stemming rules.
# Described in M. D. McIlroy, "Development of a spelling list",
# IEEE Transactions on Communications 30 (1982) 91-99.

my $IZE;
my $IZATION;

sub Americanize
{
    $IZE = 'ize';
    $IZATION = 'ization';
}

sub Britishise
{
    $IZE = 'ise';
    $IZATION = 'isation';
}

# Given a ["word","affix rule"] pair, 
# return a list of ["word", "affix rule"] pairs after applying prefix rules.
# These prefix rules are applied as many times as possible to each word.

my $prefix = join '|', qw/ anti auto intra inter mono multi re semi bio
                           counter dis electro en fore geo hyper iso kilo
                           magneto meta micro mid milli mis non out over
                           photo poly pre pseudo psycho stereo sub super
                           tele thermo ultra under un /;

sub prefix_stems
{
    my $w = shift;
    my $r = $w->[1];
    local $_ = $w->[0];
    my @stems;
    $r = "+$1$r", push( @stems, [ $_, $r ] ) while s/^($prefix)//;
    @stems;
}

# Given a ["word","affix rule"] pair, 
# return a list of ["word", "affix rule"] pairs after applying suffix rules.
# These suffix rules are applied as many times as possible to each word.

sub suffixes
{
    my $w = shift;
    my $r = $w->[1];
    local $_ = $w->[0];
    my $x;

    s/ie(r|d|st)$/y/    and return ($w=[ $_, "-y+ie$1$r"    ], suffixes($w));
    ! /[ac]ter$/ and s/e(r|d|st)$//
                        and return ($w=[ "${_}e", "+$1$r"   ],
                                    $x=[ $_, "+e$1$r"       ], suffixes($w),
                                                               suffixes($x));
    s/bility$/ble/      and return ($w=[ $_, "-e+ity$r"     ], suffixes($w));
    s/fiable$/fy/       and return ($w=[ $_, "-y+iable$r"   ], suffixes($w));
    s/logist$/logy/     and return ($w=[ $_, "-y+ist$r"     ], suffixes($w));
    ! /blity$/ and s/(ing|ist|ism|ity)$//
                        and return ($w=[ "${_}e", "-e+$1$r" ],
                                    $x=[ $_, "+$1$r"        ], suffixes($w),
                                                               suffixes($x));
    s/([^cg])able$/$1/  and return ($w=[ "${_}e","-e+able$r"],
                                    $x=[ $_, "+able$r"      ], suffixes($w),
                                                               suffixes($x));
    s/graphic$/graphy/  and return ($w=[ $_, "-y+ic$r"      ], suffixes($w));
    s/logic$/logy/      and return ($w=[ $_, "-y+ic$r"      ], suffixes($w));
    s/istic$/ist/       and return ($w=[ $_, "+ic$r"        ], suffixes($w));
    s/itic$/ite/        and return ($w=[ $_, "-e+ic$r"      ], suffixes($w));
    s/(like|ment|ship)$//
	                and return ($w=[ $_, "+$1$r"        ], suffixes($w));
    s/metry$/meter/     and return ($w=[ $_, "-er+ry$r"     ], suffixes($w));
    s/nc(e|y)$/nt/      and return ($w=[ $_, "-t+c$1$r"     ], suffixes($w));
    s/mental$/ment/     and return ($w=[ $_, "+al$r"        ], suffixes($w));
    s/ical$/ic/         and return ($w=[ $_, "+al$r"        ], suffixes($w));
    s/ional$/ion/       and return ($w=[ $_, "+al$r"        ], suffixes($w));
    s/bly$/ble/         and return ($w=[ $_, "-e+y$r"       ], suffixes($w));
    s/i(ly|ful|hood|less|ness)$/y/
                        and return ($w=[ $_, "-y+i$1$r"     ], suffixes($w));
    s/(ly|ful|hood|less|ness)$//
                        and return ($w=[ $_, "+$1$r"        ], suffixes($w));
    s/ification$/ify/   and return ($w=[ $_, "-y+ication$r" ], suffixes($w));
    s/([cr])tion$/$1t/  and return ($w=[ $_, "+ion$r"       ], suffixes($w));
    $IZATION and s/$IZATION$/$IZE/
                        and return ($w=[ $_, "-e+ation$r"   ], suffixes($w));
    s/ation$/ate/       and return ($w=[ $_, "-e+ion$r"     ], suffixes($w));
    s/ator$/ate/        and return ($w=[ $_, "-e+or$r"      ], suffixes($w));
    s/ctor$/ct/         and return ($w=[ $_, "+or$r"        ], suffixes($w));
    s/ive$/ion/         and return ($w=[ $_, "-ion+ive$r"   ], suffixes($w));
    s/ive$/ion/         and return ($w=[ $_, "-ion+ive$r"   ], suffixes($w));
    $IZE and s/$IZE$//  and return ($w=[ "${_}e", "-e+$IZE$r" ],
                                    $x=[ $_, "+$IZE$r"      ], suffixes($w),
                                                               suffixes($x));
    /^[A-Z]/ and s/ian$// and return ( [ $_, "+ian$r"       ] );
    /^[A-Z]/ and s/an$/a/ and return ( [ $_, "+n$r"         ] );
    ();
}

# Given a ["word","affix rule"] pair, 
# return a list of ["word", "affix rule"] pairs after applying suffix rules.
# Plural suffixes are applied just once.

sub suffix_stems
{
    my $w = shift;
    my $r = $w->[1];
    local $_ = $w->[0];
    my $x;

    s/([hsxz])es$/$1/   and return ($w=[ "${_}e", "$r+s"     ],
                                    $x=[ $_,      "$r+es"    ], suffixes($w),
                                                                suffixes($x));
    s/ies$/y/           and return ($w=[ $_,      "$r-y+ies" ], suffixes($w));
    s/'s$//             and return ($w=[ $_,      "$r+'s"    ], suffixes($w));
    s/([^s])s$/$1/      and return ($w=[ $_,      "$r+s"     ], suffixes($w));
    suffixes($w);
}

# Given a word, return a list of ["word", "affix rule"] pairs.
# If no affix rules apply, an empty list is returned.

sub stems
{
    my @s = ( [ shift, '' ] );
    push( @s, suffix_stems( $s[0] ) );
    @s = map { ( $_, prefix_stems( $_ ) ) } @s;
    shift @s;   # ditch the original word, return just derived stems
    @s;
}

# ----------------------------------------------------------------------------

package Hlist;

# My interpretation of superimposed hash coding, described in 
# "Development of a spelling list" (cited above) and also in D. E. Knuth,
# The Art of Computer Programming, Vol. 3: Sorting and Searching.
#
# The idea is to use N independent hash functions to set N bits in a vector
# to represent each entry.  We take a regular 32-bit CRC as a preliminary
# hash value and use the remainder modulo a prime number as a hash key.
# 16 more-or-less independent hash keys are produced using 16 prime numbers
# close to the size of our bit-vector.
#
# McIlroy chose Huffman compressed hashing over this method in his
# implementation to save a little space.
# The original `spell' had a designed hashing error rate of 1 in 2^12
# with a word list of 30,000 entries.  The compressed hashed list fit into
# a cozy 64k bytes.
#
# Our designed error rate is 1 in 2^16 with a word list of 60,000 entries.
# Probably overkill, considering the error rate from stemming.
# The hashed list is around 200k bytes.

sub NumberOfKeys { 16 }

my @table = (
    0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f, 
    0xe963a535, 0x9e6495a3, 0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988, 
    0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91, 0x1db71064, 0x6ab020f2, 
    0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7, 
    0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9, 
    0xfa0f3d63, 0x8d080df5, 0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172, 
    0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b, 0x35b5a8fa, 0x42b2986c, 
    0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59, 
    0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423, 
    0xcfba9599, 0xb8bda50f, 0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924, 
    0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d, 0x76dc4190, 0x01db7106, 
    0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433, 
    0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d, 
    0x91646c97, 0xe6635c01, 0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e, 
    0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457, 0x65b0d9c6, 0x12b7e950, 
    0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65, 
    0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7, 
    0xa4d1c46d, 0xd3d6f4fb, 0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0, 
    0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9, 0x5005713c, 0x270241aa, 
    0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f, 
    0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81, 
    0xb7bd5c3b, 0xc0ba6cad, 0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a, 
    0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683, 0xe3630b12, 0x94643b84, 
    0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1, 
    0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb, 
    0x196c3671, 0x6e6b06e7, 0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc, 
    0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5, 0xd6d6a3e8, 0xa1d1937e, 
    0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b, 
    0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55, 
    0x316e8eef, 0x4669be79, 0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236, 
    0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f, 0xc5ba3bbe, 0xb2bd0b28, 
    0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d, 
    0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f, 
    0x72076785, 0x05005713, 0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38, 
    0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21, 0x86d3d2d4, 0xf1d4e242, 
    0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777, 
    0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69, 
    0x616bffd3, 0x166ccf45, 0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2, 
    0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db, 0xaed16a4a, 0xd9d65adc, 
    0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9, 
    0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693, 
    0x54de5729, 0x23d967bf, 0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94, 
    0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d, 
);

sub hashes
{
    my $self = shift;
    my $crc;
    foreach ( split //, shift )
    {
        $crc = ( $crc >> 8 ) ^ $table[ ord($_) ^ $crc & 0xff ];
    }
    $crc = ~$crc if ( $crc < $self->{VectorBits} );
    map { $crc % $_ } @{$self->{Primes}};
}

sub contains
{
    my $self = shift;    
    map { return 0 unless vec($self->{Vector}, $_, 1); } $self->hashes(shift);
    1;
}

sub insert
{
    my $self = shift;    
    map { vec( $self->{Vector}, $_, 1 ) = 1; } $self->hashes( shift );
    ++$self->{Entries} < $self->{Capacity};
}

# Use a tag (exactly 20 characters) to identify the hash file.
# We can change this tag to reject old, incompatible files.

sub FileTag { pack( 'A20', 'spell hash file' ); }

sub load
{
    my $self = shift;
    my $filehandle = shift;
    binmode $filehandle;
    my ($chunk,$tag,$vector_length);
    read( $filehandle, $chunk, (20 + (10 * (4+NumberOfKeys))) );
    ($tag,
     $self->{Capacity},
     $self->{Entries},
     $self->{VectorBits},
     $vector_length,
     @{$self->{Primes}} ) = unpack( 'a20'.('a10' x (4+NumberOfKeys)), $chunk );
    return 0 if ( $tag ne FileTag );
    read( $filehandle, $self->{Vector}, $vector_length );
}

sub dump
{
    my $self = shift;
    my $filehandle = shift;
    binmode $filehandle;
    print $filehandle FileTag, sprintf( '%10d' x (4+NumberOfKeys),
                                        $self->{Capacity},
                                        $self->{Entries},
                                        $self->{VectorBits},
				        length( $self->{Vector} ),
                                        @{$self->{Primes}} );
    print $filehandle $self->{Vector};
}

sub new
{
    my $proto = shift;
    my %args = @_;
    my $self = bless( {}, (ref($proto) || $proto) );
    if ( $args{FileHandle} )
    {
        return 0 unless $self->load( $args{FileHandle} );
    }
    elsif ( $args{Capacity} )
    {
        $self->{Capacity} = $args{Capacity};
        $self->{Entries} = 0;
        $self->{VectorBits} = int($self->{Capacity} * NumberOfKeys()/log(2));
        $self->{Primes} = [ primes( $self->{VectorBits} ) ];
        vec( $self->{Vector}, $self->{VectorBits}-1, 1 ) = 0;
    }
    else
    {
        return 0;
    }
    return $self;
}

# Return a list of the N=NumberOfKeys largest primes less than
# the given number.
# Slow for large numbers but ok in the range we want (less than 2,000,000.)

sub primes
{
    my $end = shift;
    my $limit = int( sqrt( $end ) );
    my ($n, @primes, @result);

TRY:
    for ( $n=3; $n<=$limit; $n+=2 )
    {
        foreach ( @primes )
        {
            next TRY unless ($n % $_);
        }
        push @primes, $n;
    }

TRY:
    for ( $n=$end-500; $n<=$end; $n+=2 )
    {
        foreach ( @primes )
        {
            next TRY unless ($n % $_) and ($_ < $n);
        }
        push @result, $n;
    }
    splice( @result, - NumberOfKeys );
}

package Spelling;

sub import
{
    no strict 'refs';
    *{ caller() . '::stems' } = \&Stems::stems;
}

1;
