#!perl
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   file: t/die-hard.pm
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Tester-DieHard.
#
#   perl-Dist-Zilla-Tester-DieHard is free software: you can redistribute it and/or modify it under
#   the terms of the GNU General Public License as published by the Free Software Foundation,
#   either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Tester-DieHard is distributed in the hope that it will be useful, but WITHOUT
#   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#   PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Tester-DieHard. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

use lib 't';
use strict;
use warnings;

use DistZillaPlugin;
use File::Temp qw{ tempdir };
use Test::DZil qw{ dist_ini };
use Test::More;

#   `AutoPrereqs` hints:
use Software::License::GPL_3::or_later ();
use Path::Tiny ();
    # ^ The test fails on some Windows machines because of exception
    #       Can't locate Unicode/UTF8.pm in @INC
    #   trown from `Path::Tiny`. Let us temporary add `Path::Tiny` to the requirements, so test
    #   infrastructure will report its version.

my $ini_head = {
    name                => 'Dummy',
    version             => '0.007',
    author              => 'John Doe',
    license             => 'GPL_3::or_later',
    copyright_holder    => 'John Doe',
    copyright_year      => '2007',
};

my @ini_body = (
    [ '=DistZillaPlugin' ],
);

my $args = {
    dist_root => tempdir( CLEANUP => 1 ),
};

my $tester_args = {
    add_files => {
        'source/lib/Dummy.pm' =>
            "package Dummy;\n" .
            "\n" .                  # Blank line for `PkgVersion`.
            "# ABSTRACT: Dummy\n" .
            "# VERSION\n" .
            "1;\n",
        'source/dist.ini' => dist_ini( $ini_head, @ini_body ),
    },
};

my $error = qr{Died at t/DistZillaPlugin\.pm line };

plan tests => 2;

# --------------------------------------------------------------------------------------------------

subtest 'regular' => sub {

    package Regular;            # Localize `use Dist::Zilla::Tester` effect.

    use Dist::Zilla::Tester;
    use Test::Fatal;
    use Test::More;

    plan tests => 2;

    my ( $tzil, $exception );

    $exception = exception { $tzil = Builder->from_config( $args, $tester_args ); };
    like( $exception, qr{^$error}, 'exception' );
    is( $tzil, undef, '$tzil undefined' );

    #   Ok, we have exception and can check it. That's all. `DistZillaPlugin` logged a message
    #   before dying, but we cannot check it: all the messages are lost.

    done_testing;

};

# --------------------------------------------------------------------------------------------------

subtest 'die hard' => sub {

    package DieHard;            # Localize `use Dist::Zilla::Tester::DieHard` effect.

    use Dist::Zilla::Tester::DieHard;
    use Scalar::Util qw{ blessed };
    use Test::Deep qw{ cmp_deeply };
    use Test::Fatal;
    use Test::More;

    plan tests => 6;

    my ( $tzil, $exception, $messages );

    $exception = exception { $tzil = Builder->from_config( $args, $tester_args ); };
    is( $exception, undef, 'no exception' );
    isnt( $tzil, undef, '$tzil defined' );
    ok( blessed( $tzil ), '$tzil blessed' );

    #   No exception. `$tzil` is defined, we can retrieve messages.

    my @messages = @{ $tzil->log_messages };    # Create a copy.
    cmp_deeply( \@messages, [ '[=DistZillaPlugin] before die' ], 'messages' )
        or diag( "Log:\n" . join( '', map( "    $_\n", @messages ) ) );

    #   Saved exception will be rethrown by `build`. Messages are still available.

    $exception = exception { $tzil->build(); };
    like( $exception, qr{^$error}, 'exception' );
    cmp_deeply( $tzil->log_messages, \@messages, 'messages are not changed' )
        or diag( "Log:\n" . join( '', map( "    $_\n", @messages ) ) );

    done_testing;

};

# --------------------------------------------------------------------------------------------------

done_testing;

exit( 0 );

# end of file #
