/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) 1998 - 2018, Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.haxx.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 ***************************************************************************/
#include "curlcheck.h"

#include "urldata.h"
#include "curl/urlapi.h"
#include "urlapi-int.h"


static CURLU *u;

static CURLcode
unit_setup(void)
{
  return CURLE_OK;
}

static void
unit_stop(void)
{
  curl_global_cleanup();
}

UNITTEST_START

  CURLUcode ret;
  char *ipv6port;
  char *portnum;

  /* Valid IPv6 */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15]");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret == CURLUE_OK, "Curl_parse_port returned error");
  ret = curl_url_get(u, CURLUPART_PORT, &portnum, CURLU_NO_DEFAULT_PORT);
  fail_unless(ret != CURLUE_OK, "curl_url_get portnum returned something");
  free(ipv6port);
  curl_url_cleanup(u);

  /* Invalid IPv6 */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15|");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret != CURLUE_OK, "Curl_parse_port true on error");
  free(ipv6port);
  curl_url_cleanup(u);

  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff;fea7:da15]:80");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret != CURLUE_OK, "Curl_parse_port true on error");
  free(ipv6port);
  curl_url_cleanup(u);

  /* Valid IPv6 with zone index and port number */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15%25eth3]:80");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret == CURLUE_OK, "Curl_parse_port returned error");
  ret = curl_url_get(u, CURLUPART_PORT, &portnum, 0);
  fail_unless(ret == CURLUE_OK, "curl_url_get portnum returned error");
  fail_unless(strcmp(portnum, "80") == 0, "Check portnumber");
  curl_free(portnum);
  free(ipv6port);
  curl_url_cleanup(u);

  /* Valid IPv6 with zone index without port number */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15%25eth3]");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret == CURLUE_OK, "Curl_parse_port returned error");
  free(ipv6port);
  curl_url_cleanup(u);

  /* Valid IPv6 with port number */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15]:81");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret == CURLUE_OK, "Curl_parse_port returned error");
  ret = curl_url_get(u, CURLUPART_PORT, &portnum, 0);
  fail_unless(ret == CURLUE_OK, "curl_url_get portnum returned error");
  fail_unless(strcmp(portnum, "81") == 0, "Check portnumber");
  curl_free(portnum);
  free(ipv6port);
  curl_url_cleanup(u);

  /* Valid IPv6 with syntax error in the port number */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15];81");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret != CURLUE_OK, "Curl_parse_port true on error");
  free(ipv6port);
  curl_url_cleanup(u);

  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15]80");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret != CURLUE_OK, "Curl_parse_port true on error");
  free(ipv6port);
  curl_url_cleanup(u);

  /* Valid IPv6 with no port after the colon, should use default */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15]:");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret == CURLUE_OK, "Curl_parse_port returned error");
  free(ipv6port);
  curl_url_cleanup(u);

  /* Incorrect zone index syntax */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15%!25eth3]:80");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret != CURLUE_OK, "Curl_parse_port returned non-error");
  free(ipv6port);
  curl_url_cleanup(u);

  /* Non percent-encoded zone index */
  u = curl_url();
  ipv6port = strdup("[fe80::250:56ff:fea7:da15%eth3]:80");
  ret = Curl_parse_port(u, ipv6port);
  fail_unless(ret != CURLUE_OK, "Curl_parse_port returned non-error");
  free(ipv6port);
  curl_url_cleanup(u);

UNITTEST_STOP
