package Acme::Albedo;
use 5.20.2;
use strict;
use warnings;
use Encode;
use parent 'Exporter';

our @EXPORT = qw( to_albedo to_spira );

our $VERSION = "0.01";

my %dict_spira = (
  "ワ" => "あ", "ミ" => "い", "フ" => "う", "ネ" => "え", "ト" => "お",
  "ア" => "か", "チ" => "き", "ル" => "く", "テ" => "け", "ヨ" => "こ",
  "ラ" => "さ", "キ" => "し", "ヌ" => "す", "ヘ" => "せ", "ホ" => "そ",
  "サ" => "た", "ヒ" => "ち", "ユ" => "つ", "セ" => "て", "ソ" => "と",
  "ハ" => "な", "シ" => "に", "ス" => "ぬ", "メ" => "ね", "オ" => "の",
  "マ" => "は", "リ" => "ひ", "ク" => "ふ", "ケ" => "へ", "ロ" => "ほ",
  "ヤ" => "ま", "イ" => "み", "ツ" => "む", "レ" => "め", "コ" => "も",
  "タ" => "や", "ヲ" => "ゆ", "モ" => "よ",
  "ナ" => "ら", "二" => "り", "ウ" => "る", "エ" => "れ", "ノ" => "ろ",
  "カ" => "わ", "ム" => "を", "ン" => "ん",
  "ダ" => "が", "ジ" => "ぎ", "ヅ" => "ぐ", "デ" => "げ", "ゾ" => "ご",
  "バ" => "ざ", "ギ" => "じ", "ブ" => "ず", "ゲ" => "ぜ", "ボ" => "ぞ",
  "ガ" => "だ", "ビ" => "ぢ", "グ" => "づ", "ベ" => "で", "ゴ" => "ど",
  "ザ" => "ば", "ヂ" => "び", "ズ" => "ぶ", "ゼ" => "べ", "ド" => "ぼ",
  "プ" => "ぱ", "ペ" => "ぴ", "パ" => "ぷ", "ポ" => "ぺ", "ピ" => "ぽ",
  "ァ" => "ぁ", "ィ" => "ぃ", "ゥ" => "ぅ", "ェ" => "ぇ", "ォ" => "ぉ",
  "ッ" => "っ", "ャ" => "ゃ", "ュ" => "ゅ", "ョ" => "ょ",
);

my %dict_albedo = (
  "あ" => "ワ", "い" => "ミ", "う" => "フ", "え" => "ネ", "お" => "ト",
  "か" => "ア", "き" => "チ", "く" => "ル", "け" => "テ", "こ" => "ヨ",
  "さ" => "ラ", "し" => "キ", "す" => "ヌ", "せ" => "ヘ", "そ" => "ホ",
  "た" => "サ", "ち" => "ヒ", "つ" => "ユ", "て" => "セ", "と" => "ソ",
  "な" => "ハ", "に" => "シ", "ぬ" => "ス", "ね" => "メ", "の" => "オ",
  "は" => "マ", "ひ" => "リ", "ふ" => "ク", "へ" => "ケ", "ほ" => "ロ",
  "ま" => "ヤ", "み" => "イ", "む" => "ツ", "め" => "レ", "も" => "コ",
  "や" => "タ", "ゆ" => "ヲ", "よ" => "モ",
  "ら" => "ナ", "り" => "二", "る" => "ウ", "れ" => "エ", "ろ" => "ノ",
  "わ" => "カ", "を" => "ム", "ん" => "ン",
  "が" => "ダ", "ぎ" => "ジ", "ぐ" => "ヅ", "げ" => "デ", "ご" => "ゾ",
  "ざ" => "バ", "じ" => "ギ", "ず" => "ブ", "ぜ" => "ゲ", "ぞ" => "ボ",
  "だ" => "ガ", "ぢ" => "ビ", "づ" => "グ", "で" => "ベ", "ど" => "ゴ",
  "ば" => "ザ", "び" => "ヂ", "ぶ" => "ズ", "べ" => "ゼ", "ぼ" => "ド",
  "ぱ" => "プ", "ぴ" => "ペ", "ぷ" => "パ", "ぺ" => "ポ", "ぽ" => "ピ",
  "ぁ" => "ァ", "ぃ" => "ィ", "ぅ" => "ゥ", "ぇ" => "ェ", "ぉ" => "ォ",
  "っ" => "ッ", "ゃ" => "ャ", "ゅ" => "ュ", "ょ" => "ョ",
);

sub translate {
  my ($in, %dict) = @_;
  my $out;

  utf8::decode($in);

  foreach (split //, $in) {
    utf8::encode($_);
    $out .= exists $dict{$_} ? $dict{$_} : $_;
  }

  $out;
}

sub to_albedo {
  my ($in) = @_;
  &translate($in, %dict_albedo);
}

sub to_spira {
  my ($in) = @_;
  &translate($in, %dict_spira);
}

1;
__END__

=encoding utf-8

=head1 NAME

Acme::Albedo - Translation module between Albedo and Spira.

=head1 SYNOPSIS

    use Acme::Albedo
    print to_albedo("ごっどはんど") # => ゾッゴマンゴ

=head1 DESCRIPTION

This module can translate Albedo to Spira, and vice versa.

=head1 METHODS

=head2 to_albedo

Translate Albedo to Spira.
Albedo must be written in Hiragana.

    print to_albedo("いせき") # => ミヘチ

=head2 to_spira

Translate Spira to Albedo.
Spira must be written in Katakana.

    print to_albedo("ミヘチ") # => いせき

=head1 LICENSE

Copyright (c), tenmihi.

This module is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=head1 AUTHOR

tenmihi E<lt>tenmihi @ cpan.orgE<gt>

=cut
