use BSON::Document;
use Data::Dump;
use MongoDB::Client;
use MongoDB::Database;
use MongoDB::Collection;
use MongoDB::Cursor;
use Data::Dump::Tree;
use Art::Behavior::CRUD;

# Provide basic CRUD operation for entities
role Art::Behavior::Crudable does Art::Behavior::CRUD {

    has MongoDB::Database $.database;
    # $.type-for-document must be taken from the class we try to work
    # on. It correspond to a collection in the database
    has Str $.type-for-document = self!get-type.lc;

    multi method save (Str $name) {
        my BSON::Document $req .= new: (
        insert => $.type-for-document ~ 's', # We add a 's'
        documents => [
                      BSON::Document.new((
                                            name => $name,
                                        )),
                  ]);
        my BSON::Document $doc = $.database.run-command($req);
    }

    # Return the last element in a module package like "Artist" in
    # "Art::Entities::artist"
    method !get-type {
        my @mods = split("::", self.WHAT.^name);
        return @mods.pop;
    }

    method introspect-attributes {
        my @attributes = self.^attributes;
        for @attributes {
            $_.=Str; # Transtype to Str and remove type
            $_ ~~ s/ [ '$!' || '@!' ] //; # Remove sigils, keep attribut name only
            say "introspecting " ~ $_;
        }
        return @attributes;
    }

    # This method retrieves all attributes of an entity and return
    # only the one that have the 'is crud' trait (Art::Behavior::CRUD)
    method introspect-crud-attributes {
        my @attrs = self.^attributes;
        my @crud-attrs;

        for @attrs.grep: Art::Behavior::CRUD -> $attr {
             @crud-attrs.push: $attr;
        }

        return @crud-attrs;
    }

    # We need a flag that set class attributes as "crudable". What is
    # mean by "crudable" is that some objects attributes must be taken
    # in consideration while storing objects to a database while other
    # won't. It must be independant of the attribute value and is
    # stored this way:
    #
    #     .Attribute+{Art::Behavior::CRUD}
    #
    # This process invole a custom trait (is crud), a role
    # (Art::Behavior::CRUD) and the mixin of the role inside a dynamic
    # attribute.
    multi trait_mod:<is>(Attribute:D $a, :$crud!) is export {
        $a.^mixin: Art::Behavior::CRUD if $crud;
    }

}

=begin pod

=head1 NAME

Crudable - Role providing common functionnalities for all kind of
objects that can interact with the DB.

=head1 SYNOPSIS

# For the Agent class
use Art::Behavior::Crudable;
class Art::Entities::Agent does Art::Behavior::Crudable {
      has $attr is crud;
}

# For Agent sublasses
use Art::Behavior::Crud;
class Art::Entities::Artist is Art::Entities::Agent {
      has $attr is crud;
}

=head1 DESCRIPTION

The crudable role provide all kind of tools to facilitate
communication of the Art::World with it's database.

Note that subclasses of generic classes that implement the
Art::Behavior::Crudable role must use Art::Behavior::Crud to access
the "is crud" trait, even if Art::Behavior::Crudable already "does"
it.

=head1 AUTHOR

Seb. Hu-Rillettes <shr@balik.network>

=head1 COPYRIGHT AND LICENSE

Copyright 2017 Seb. Hu-Rillettes

This library is free software; you can redistribute it and/or modify it under the Artistic License 2.0.

=end pod
