package Firewall::PaloAlto::Routes;

use Moose;

use 5.006;
use strict;
use warnings;
use Modern::Perl;

use Data::Dumper;

our $VERSION = '0.01';

=head1 NAME

Firewall::PaloAlto::Routes- Class returned when requesting the routing table from a firewall.

=cut

=head1 CLASS METHODS

=head2 new($http_response)

Taken an HTTP::Response object, which is returned from the API call.

=cut

has status => ( is => 'ro' );
has entry => (  is => 'ro',
                reader => 'raw');
                        

around BUILDARGS => sub {
    my $orig = shift;
    my $class = shift;
    my $http_response = shift;

    if ($http_response->{status} eq 'success') {
        $http_response->{result}->{system}->{status} = 1;
    }
    else {
        $http_response->{result}->{system}->{status} = 0;
    }

    return $class->$orig($http_response->{result});
};

=head1 OBJECT METHODS

=head2 print_routes

Prints the routes from the routing table in a format similar to Cisco's 'show ip route' out.

    $route_object->print_routes;

=cut

sub print_routes {
    my $self = shift;

    for my $route (@{ $self->raw }) {
        say "$route->{flags}\t$route->{destination} via $route->{nexthop}". (!ref $route->{interface} ? " $route->{interface}" : "") ." [$route->{metric}]";
    }
}


=head1 AUTHOR

Greg Foletta, C<< <greg at foletta.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-firewall-paloalto-common at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Firewall-PaloAlto-Common>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Firewall::PaloAlto::Common


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Firewall-PaloAlto-Common>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Firewall-PaloAlto-Common>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Firewall-PaloAlto-Common>

=item * Search CPAN

L<http://search.cpan.org/dist/Firewall-PaloAlto-Common/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2015 Greg Foletta.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Firewall::PaloAlto::Common
