package Bencher::Scenario::URIEscaping;

our $DATE = '2016-06-17'; # DATE
our $VERSION = '0.003'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

# we do this so URI::XSEscape does not override URI::Escape's functions, because
# obviously we want to test both
$ENV{PERL_URI_XSESCAPE} = 0;

our $scenario = {
    summary => 'Benchmark URI escaping using various modules',

    precision => 0.001,

    participants => [
        {fcall_template => 'URI::Escape::uri_escape(<str>)', tags=>['escape']},
        {fcall_template => 'URI::Escape::uri_escape_utf8(<str>)', tags=>['escape', 'utf8']},
        {fcall_template => 'URI::Escape::uri_unescape(<str>)', tags=>['unescape']},
        {fcall_template => 'URI::XSEscape::uri_escape(<str>)', tags=>['escape']},
        {fcall_template => 'URI::XSEscape::uri_escape_utf8(<str>)', tags=>['escape', 'utf8']},
        {fcall_template => 'URI::XSEscape::uri_unescape(<str>)', tags=>['unescape']},
        {fcall_template => 'URI::Escape::XS::uri_escape(<str>)', tags=>['escape']},
        #{fcall_template => 'URI::Escape::XS::uri_escape_utf8(<str>)', tags=>['escape', 'utf8']},
        {fcall_template => 'URI::Escape::XS::uri_unescape(<str>)', tags=>['unescape']},

        {fcall_template => 'URI::Encode::uri_encode(<str>)', tags=>['escape']},
        {fcall_template => 'URI::Encode::uri_decode(<str>)', tags=>['unescape']},
        {fcall_template => 'URI::Encode::XS::uri_encode(<str>)', tags=>['escape']},
        {fcall_template => 'URI::Encode::XS::uri_decode(<str>)', tags=>['unescape']},
    ],

    datasets => [
        {
            name => 'empty',
            tags => ['escape'],
            include_participant_tags => ['escape'],
            args => {str=>''},
        },
        # sample data from URI-XSEscape distribution
        {
            name => 'ascii53',
            tags => ['escape'],
            include_participant_tags => ['escape'],
            args => {str=>'I said this: you / them ~ us & me _will_ "do-it" NOW!'},
        },
        # sample data from URI-XSEscape distribution
        {
            name => 'utf36',
            tags => ['escape', 'utf8'],
            include_participant_tags => ['escape & utf8'],
            args => {str=>'http://www.google.co.jp/search?q=小飼弾'},
        },
        # sample data from URI-XSEscape distribution
        {
            name => 'u_ascii53',
            tags => ['unescape'],
            include_participant_tags => ['unescape'],
            args => {str=>'I%20said%20this%3a%20you%20%2f%20them%20~%20us%20%26%20me%20_will_%20%22do-it%22%20NOW%21'},
        },

        # sample data from URI-Escape-XS distribution
        {
            name => 'ascii66',
            tags => ['escape'],
            include_participant_tags => ['escape'],
            args => {str=>'https://stackoverflow.com/questions/3629212/how can perls xsub die'},
        },
        # sample data from URI-Escape-XS distribution
        {
            name => 'u_ascii66',
            tags => ['unescape'],
            include_participant_tags => ['unescape'],
            args => {str=>'https%3A%2F%2Fstackoverflow.com%2Fquestions%2F3629212%2Fhow%20can%20perls%20xsub%20die'},
        },
    ],
};

1;
# ABSTRACT: Benchmark URI escaping using various modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::URIEscaping - Benchmark URI escaping using various modules

=head1 VERSION

This document describes version 0.003 of Bencher::Scenario::URIEscaping (from Perl distribution Bencher-Scenario-URIEscaping), released on 2016-06-17.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m URIEscaping

To run module startup overhead benchmark:

 % bencher --module-startup -m URIEscaping

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<URI::Escape> 3.31

L<URI::XSEscape> 0.000007

L<URI::Escape::XS> 0.14

L<URI::Encode> 1.1.1

L<URI::Encode::XS> 0.08

=head1 BENCHMARK PARTICIPANTS

=over

=item * URI::Escape::uri_escape (perl_code) [escape]

Function call template:

 URI::Escape::uri_escape(<str>)



=item * URI::Escape::uri_escape_utf8 (perl_code) [escape, utf8]

Function call template:

 URI::Escape::uri_escape_utf8(<str>)



=item * URI::Escape::uri_unescape (perl_code) [unescape]

Function call template:

 URI::Escape::uri_unescape(<str>)



=item * URI::XSEscape::uri_escape (perl_code) [escape]

Function call template:

 URI::XSEscape::uri_escape(<str>)



=item * URI::XSEscape::uri_escape_utf8 (perl_code) [escape, utf8]

Function call template:

 URI::XSEscape::uri_escape_utf8(<str>)



=item * URI::XSEscape::uri_unescape (perl_code) [unescape]

Function call template:

 URI::XSEscape::uri_unescape(<str>)



=item * URI::Escape::XS::uri_escape (perl_code) [escape]

Function call template:

 URI::Escape::XS::uri_escape(<str>)



=item * URI::Escape::XS::uri_unescape (perl_code) [unescape]

Function call template:

 URI::Escape::XS::uri_unescape(<str>)



=item * URI::Encode::uri_encode (perl_code) [escape]

Function call template:

 URI::Encode::uri_encode(<str>)



=item * URI::Encode::uri_decode (perl_code) [unescape]

Function call template:

 URI::Encode::uri_decode(<str>)



=item * URI::Encode::XS::uri_encode (perl_code) [escape]

Function call template:

 URI::Encode::XS::uri_encode(<str>)



=item * URI::Encode::XS::uri_decode (perl_code) [unescape]

Function call template:

 URI::Encode::XS::uri_decode(<str>)



=back

=head1 BENCHMARK DATASETS

=over

=item * empty [escape]

=item * ascii53 [escape]

=item * utf36 [escape, utf8]

=item * u_ascii53 [unescape]

=item * ascii66 [escape]

=item * u_ascii66 [unescape]

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.1 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m URIEscaping >>):

 +--------------------------------+-----------+-----------+-----------+------------+---------+---------+
 | participant                    | dataset   | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +--------------------------------+-----------+-----------+-----------+------------+---------+---------+
 | URI::Encode::uri_decode        | u_ascii53 |      2330 |  429      |       1    | 4.1e-07 |      28 |
 | URI::Encode::uri_encode        | ascii53   |      2340 |  427      |       1    | 4.3e-07 |      20 |
 | URI::Encode::uri_encode        | ascii66   |      2410 |  416      |       1.03 | 2.1e-07 |      20 |
 | URI::Encode::uri_decode        | u_ascii66 |      2410 |  415      |       1.03 | 4.1e-07 |     208 |
 | URI::Encode::uri_encode        | empty     |      2470 |  404      |       1.06 | 2.1e-07 |      20 |
 | URI::Escape::uri_escape_utf8   | ascii53   |     82000 |   12.2    |      35.1  | 3.3e-09 |      20 |
 | URI::Escape::uri_escape        | ascii53   |     85700 |   11.7    |      36.8  | 1.1e-08 |     136 |
 | URI::Escape::uri_escape_utf8   | utf36     |     96000 |   10      |      41    |   1e-08 |     166 |
 | URI::Escape::uri_escape_utf8   | ascii66   |    117000 |    8.54   |      50.2  | 3.1e-09 |      23 |
 | URI::Escape::uri_unescape      | u_ascii53 |    120000 |    8.5    |      50    | 8.6e-09 |     192 |
 | URI::Escape::uri_escape        | ascii66   |    120000 |    8.1    |      53    | 8.1e-09 |     215 |
 | URI::Escape::uri_unescape      | u_ascii66 |    180000 |    5.5    |      77    | 5.6e-09 |      28 |
 | URI::Escape::uri_escape_utf8   | empty     |    900000 |    1.1    |     390    | 1.2e-09 |     155 |
 | URI::Escape::uri_escape        | empty     |   1310000 |    0.765  |     561    | 7.6e-10 |     150 |
 | URI::Escape::XS::uri_escape    | ascii66   |   1340000 |    0.748  |     573    | 4.2e-10 |      20 |
 | URI::Escape::XS::uri_escape    | ascii53   |   1350000 |    0.738  |     581    |   4e-10 |      22 |
 | URI::XSEscape::uri_escape_utf8 | ascii66   |   1380000 |    0.726  |     590    | 4.1e-10 |      21 |
 | URI::XSEscape::uri_escape_utf8 | ascii53   |   1400000 |    0.714  |     601    | 3.8e-10 |      25 |
 | URI::XSEscape::uri_escape_utf8 | utf36     |   1600000 |    0.64   |     670    | 6.9e-10 |     307 |
 | URI::Escape::XS::uri_unescape  | u_ascii53 |   1600000 |    0.626  |     684    | 6.3e-10 |      20 |
 | URI::Escape::XS::uri_escape    | empty     |   1670000 |    0.6    |     715    | 2.1e-10 |      20 |
 | URI::Escape::XS::uri_unescape  | u_ascii66 |   1840000 |    0.542  |     791    | 2.1e-10 |      20 |
 | URI::XSEscape::uri_escape_utf8 | empty     |   1900000 |    0.52   |     830    |   6e-10 |      22 |
 | URI::XSEscape::uri_unescape    | u_ascii53 |   2700000 |    0.36   |    1200    | 3.9e-10 |      93 |
 | URI::XSEscape::uri_escape      | ascii66   |   2820000 |    0.355  |    1210    | 4.5e-11 |      20 |
 | URI::XSEscape::uri_escape      | ascii53   |   2920000 |    0.343  |    1250    | 1.9e-10 |      26 |
 | URI::XSEscape::uri_unescape    | u_ascii66 |   2900000 |    0.34   |    1300    | 4.2e-10 |      78 |
 | URI::Encode::XS::uri_decode    | u_ascii66 |   4700000 |    0.213  |    2020    | 1.1e-10 |      20 |
 | URI::XSEscape::uri_escape      | empty     |   5400000 |    0.19   |    2300    | 2.6e-10 |      81 |
 | URI::Encode::XS::uri_decode    | u_ascii53 |   5720000 |    0.175  |    2450    | 4.5e-11 |      20 |
 | URI::Encode::XS::uri_encode    | ascii66   |   7370000 |    0.136  |    3160    |   1e-10 |      26 |
 | URI::Encode::XS::uri_encode    | ascii53   |   7850000 |    0.127  |    3370    | 3.9e-11 |      20 |
 | URI::Encode::XS::uri_encode    | empty     |  13400000 |    0.0745 |    5760    | 6.5e-11 |      20 |
 +--------------------------------+-----------+-----------+-----------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m URIEscaping --module-startup >>):

 +---------------------+-----------+------------------------+------------+---------+---------+
 | participant         | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors | samples |
 +---------------------+-----------+------------------------+------------+---------+---------+
 | URI::Encode         | 19.2      |                  14.5  | 1          | 1.9e-05 |     723 |
 | URI::Escape         | 11.6      |                   6.9  | 1.65       | 1.1e-05 |    1557 |
 | URI::Escape::XS     | 9.26      |                   4.56 | 2.07       | 8.8e-06 |     805 |
 | URI::XSEscape       | 9.11      |                   4.41 | 2.1        | 8.8e-06 |     596 |
 | URI::Encode::XS     | 8.2       |                   3.5  | 2.34       | 8.1e-06 |     824 |
 | perl -e1 (baseline) | 4.7       |                   0    | 4.08       | 4.7e-06 |    5156 |
 +---------------------+-----------+------------------------+------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-URIEscaping>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-URIEscaping>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-URIEscaping>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
